#include "Mix/Private/Scene/Common/BillboardParticleSystem.h"

#ifdef _DEBUG
	#include "Mix/Graphics/Utility/IPerspectiveRenderer.h"
#endif //_DEBUG

namespace Mix{ namespace Scene{ namespace Common{

const wchar_t* BillboardParticleSystem::FAILED_CREATE = L"r{[hp[eBNVXe̍쐬Ɏs";

////////////////////////////////////////////////////////////////////////////////////////////////////
// BillboardParticleSystem::InternalGenerator
////////////////////////////////////////////////////////////////////////////////////////////////////

BillboardParticleSystem::InternalGenerator* BillboardParticleSystem::InternalGenerator::CreateInstance( void )
{
	return MIX_LIB_NEW_T( Mix::Memory::SECTION_SCENE, BillboardParticleSystem::InternalGenerator );
}

BillboardParticleSystem::InternalGenerator::InternalGenerator( void ) :
m_MinAngularImpulse( -4.0f ),
m_MaxAngularImpulse( +4.0f ),
m_DiffAngularImpulse( m_MaxAngularImpulse - m_MinAngularImpulse ),
m_MinAngularAcc( 0.0f ),
m_MaxAngularAcc( 0.0f ),
m_DiffAngularAcc( m_MaxAngularAcc - m_MinAngularAcc ),
m_MinAngularVelDamp( 0.0f ),
m_MaxAngularVelDamp( 0.0f ),
m_DiffAngularVelDamp( m_MaxAngularVelDamp - m_MinAngularVelDamp )
{
}

BillboardParticleSystem::InternalGenerator::~InternalGenerator( void )
{
}

Float32 BillboardParticleSystem::InternalGenerator::GetMinAngularImpulse( void ) const
{
	return m_MinAngularImpulse;
}

Float32 BillboardParticleSystem::InternalGenerator::GetMaxAngularImpulse( void ) const
{
	return m_MaxAngularImpulse;
}

void BillboardParticleSystem::InternalGenerator::SetAngularImpulse( Float32 minImpulse, Float32 maxImpulse )
{
	m_MinAngularImpulse = minImpulse;
	m_MaxAngularImpulse = max( m_MinAngularImpulse, maxImpulse );
	m_DiffAngularImpulse = m_MaxAngularImpulse - m_MinAngularImpulse;
}

Float32 BillboardParticleSystem::InternalGenerator::GetMinAngularAcceleration( void ) const
{
	return m_MinAngularAcc;
}

Float32 BillboardParticleSystem::InternalGenerator::GetMaxAngularAcceleration( void ) const
{
	return m_MaxAngularAcc;
}

void BillboardParticleSystem::InternalGenerator::SetAngularAcceleration( Float32 minAcc, Float32 maxAcc )
{
	m_MinAngularAcc = minAcc;
	m_MaxAngularAcc = max( m_MinAngularAcc, maxAcc );
	m_DiffAngularAcc = m_MaxAngularAcc - m_MinAngularAcc;
}

Float32 BillboardParticleSystem::InternalGenerator::GetMinAngularVelocityDamping( void ) const
{
	return m_MinAngularVelDamp;
}

Float32 BillboardParticleSystem::InternalGenerator::GetMaxAngularVelocityDamping( void ) const
{
	return m_MaxAngularVelDamp;
}

void BillboardParticleSystem::InternalGenerator::SetAngularVelocityDamping( Float32 minDamping, Float32 maxDamping )
{
	m_MinAngularVelDamp = max( 0.0f, minDamping );
	m_MaxAngularVelDamp = max( m_MinAngularVelDamp, maxDamping );
	m_DiffAngularVelDamp = m_MaxAngularVelDamp - m_MinAngularVelDamp;
}

void BillboardParticleSystem::InternalGenerator::Internal_GetData( UInt32 dataSize, void* pData )
{
	MIX_ASSERT( dataSize == sizeof( BillboardParticleSystem::DATA ) );
	MIX_ASSERT( pData != NULL );

	BillboardParticleSystem::DATA* pDst = reinterpret_cast<BillboardParticleSystem::DATA*>( pData );

	Float32 mass;
	Mix::Vector2 middleSize;
	Mix::Matrix4x4 invInertia;
	Mix::Vector3 angularImpulse;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// f[^̎擾P : 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	SimpleParticleGenerator::GetData(	mass,
										pDst->pos,
										pDst->life,
										pDst->invLifeMax,
										pDst->constLinearAcceleration,
										pDst->constLinearVelocity,
										pDst->linearAcceleration,
										pDst->linearVelocity,
										pDst->linearVelocityDamping,
										pDst->angularVelocityDamping,
										pDst->initalSize,
										middleSize,
										pDst->lastSize,
										pDst->initalColor,
										pDst->middleColor,
										pDst->lastColor,
										pDst->colorCtrlPoints,
										pDst->tex[0],
										pDst->tex[1] );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// f[^̎擾Q : e\̎擾
	////////////////////////////////////////////////////////////////////////////////////////////////////

	SimpleParticleGenerator::GetData(	mass,
										middleSize.x,
										1.0f,
										middleSize.y,
										invInertia );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// f[^̎擾R : ]֌W
	////////////////////////////////////////////////////////////////////////////////////////////////////

	angularImpulse.x = 0.0f;
	angularImpulse.y = 0.0f;
	angularImpulse.z = m_MinAngularImpulse + m_DiffAngularImpulse * Mix::RandF();

	pDst->rot = 0.0f;
	pDst->angularAcceleration = m_MinAngularAcc + m_DiffAngularAcc * Mix::RandF();
	pDst->angularVelocity = invInertia.TransformSR( angularImpulse ).z;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// BillboardParticleSystem::InternalRenderer
////////////////////////////////////////////////////////////////////////////////////////////////////

BillboardParticleSystem::InternalProcessor* BillboardParticleSystem::InternalProcessor::CreateInstance( const wchar_t* pDebugName )
{
	return MIX_LIB_NEW_T( Mix::Memory::SECTION_SCENE, BillboardParticleSystem::InternalProcessor, pDebugName );
}

BillboardParticleSystem::InternalProcessor::InternalProcessor( const wchar_t* pDebugName ) :
m_FaceNum( 0 )
{
#ifdef _DEBUG

	Mix::StringW tempStr;

	tempStr.Sprintf( L"%s/DataList", MIX_SAFE_NAME( pDebugName ) );
	m_DataList.Initialize( BillboardParticleSystem::LIST_DEF_SIZE, BillboardParticleSystem::LIST_RESIZE_STEP, tempStr.GetConstPtr() );

	tempStr.Sprintf( L"%s/FacePool", MIX_SAFE_NAME( pDebugName ) );
	m_FacePool.Initialize( BillboardParticleSystem::LIST_DEF_SIZE, BillboardParticleSystem::LIST_RESIZE_STEP, tempStr.GetConstPtr() );

#else //_DEBUG

	m_DataList.Initialize( BillboardParticleSystem::LIST_DEF_SIZE, BillboardParticleSystem::LIST_RESIZE_STEP );
	m_FacePool.Initialize( BillboardParticleSystem::LIST_DEF_SIZE, BillboardParticleSystem::LIST_RESIZE_STEP );

#endif //_DEBUG
}

BillboardParticleSystem::InternalProcessor::~InternalProcessor( void )
{
}

UInt32 BillboardParticleSystem::InternalProcessor::Internal_GetCount( void ) const
{
	return ( m_DataList.GetCount() > 0 );
}

void BillboardParticleSystem::InternalProcessor::Internal_Clear( void )
{
	m_DataList.Clear();
	m_FacePool.Clear();
	m_FaceNum = 0;
}

UInt32 BillboardParticleSystem::InternalProcessor::Internal_Add( Mix::Scene::IParticleGenerator* pGenerator, UInt32 genCount )
{
	MIX_ASSERT( genCount > 0 );

	UInt32 dataSize = sizeof( BillboardParticleSystem::DATA );
	BillboardParticleSystem::DATA* pData = m_DataList.Add( genCount );
	
	UInt32 i;
	UInt32 dataNum;
	UInt32 faceCapacity;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// f[^擾
	////////////////////////////////////////////////////////////////////////////////////////////////////

	for( i = 0; i < genCount; i++ )
	{
		pGenerator->Internal_GetData( dataSize, pData );
		pData++;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// tFCXv[KvɉĊg
	////////////////////////////////////////////////////////////////////////////////////////////////////

	dataNum = m_DataList.GetCount();
	faceCapacity = m_FacePool.GetCount();

	if( faceCapacity < dataNum )
	{
		m_FacePool.Add( dataNum - faceCapacity );
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return genCount;
}

UInt32 BillboardParticleSystem::InternalProcessor::Internal_Add( const Mix::Matrix4x4& worldMat, Mix::Scene::IParticleGenerator* pGenerator, UInt32 genCount )
{
	MIX_ASSERT( genCount > 0 );

	UInt32 dataSize = sizeof( BillboardParticleSystem::DATA );
	BillboardParticleSystem::DATA* pData = m_DataList.Add( genCount );
	
	UInt32 i;
	UInt32 dataNum;
	UInt32 faceCapacity;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// f[^擾
	////////////////////////////////////////////////////////////////////////////////////////////////////

	for( i = 0; i < genCount; i++ )
	{
		pGenerator->Internal_GetData( dataSize, pData );

		pData->pos = worldMat * pData->pos;
		pData->linearAcceleration = worldMat.TransformSR( pData->linearAcceleration );
		pData->linearVelocity = worldMat.TransformSR( pData->linearVelocity );

		pData++;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// tFCXv[KvɉĊg
	////////////////////////////////////////////////////////////////////////////////////////////////////

	dataNum = m_DataList.GetCount();
	faceCapacity = m_FacePool.GetCount();

	if( faceCapacity < dataNum )
	{
		m_FacePool.Add( dataNum - faceCapacity );
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return genCount;
}

Boolean BillboardParticleSystem::InternalProcessor::Internal_Update( Float32 dt )
{
	UInt32 dataNum;

	UInt32 dataIndex;
	BillboardParticleSystem::DATA* dats;
	BillboardParticleSystem::DATA* pData;

	Mix::Scene::IParticleProcessor::FACE* faces;
	Mix::Scene::IParticleProcessor::FACE* pFace;

	const Float32* colCtrlPoints;
	Mix::Vector3* localPoints;
	const Mix::Vector2* texTL;
	const Mix::Vector2* texBR;
	Mix::Vector2* texCoords;

	Float32 lifeRatio;
	Mix::Vector2 size;
	Mix::Matrix3x3 rotMat;

	Float32 a;
	Float32 b;
	Float32 a2;
	Float32 b2;
	Float32 t;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	m_FaceNum = 0;

	dataNum = m_DataList.GetCount();
	if( dataNum == 0 )
	{
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////

	dats = m_DataList.GetBeginPtr();
	dataIndex = 0;

	faces = m_FacePool.GetBeginPtr();

	while( dataIndex < dataNum )
	{
		pData = &( dats[dataIndex] );

		pData->life -= dt;

		if( pData->life >= 0.0f )
		{
			pFace = &( faces[m_FaceNum] );

			lifeRatio = MIX_FLOAT_SATURATE( 1.0f - pData->life * pData->invLifeMax );

			// ]

			pData->angularVelocity += pData->angularAcceleration * dt;
			pData->angularVelocity -= pData->angularVelocity * pData->angularVelocityDamping;

			pData->rot += pData->angularVelocity * dt;

			// ړ

			pData->linearVelocity += pData->constLinearVelocity * dt;
			pData->linearVelocity += pData->constLinearAcceleration * dt;
			pData->linearVelocity += pData->linearAcceleration * dt;
			pData->linearVelocity -= pData->linearVelocity * pData->linearVelocityDamping;

			pData->pos += pData->linearVelocity * dt;

			// tFCX : ʒu

			pFace->pos = pData->pos;

			// tFCX : J[

			colCtrlPoints = pData->colorCtrlPoints;

			a = 1.0f - lifeRatio;
			b = lifeRatio;
			a2 = a * a;
			b2 = b * b;

			t = a2 * a * colCtrlPoints[0] +
				3.0f * a2 * b * colCtrlPoints[1] +
				3.0f * a * b2 * colCtrlPoints[2] +
				b2 * b * colCtrlPoints[3];

			if( t < 0.5f )
			{
				t = MIX_FLOAT_SATURATE( t * 2.0f );
				pFace->color = Mix::Vector4::Lerp( pData->initalColor, pData->middleColor, t );
			}
			else
			{
				t = MIX_FLOAT_SATURATE( ( t - 0.5f ) * 2.0f );
				pFace->color = Mix::Vector4::Lerp( pData->middleColor, pData->lastColor, t );
			}

			// tFCX : [J|Cg

			localPoints = pFace->localPoints;
			size = Mix::Vector2::Lerp( pData->initalSize, pData->lastSize, lifeRatio );

			localPoints[0].Set( -size.x, -size.y, 0.0f );
			localPoints[1].Set( +size.x, -size.y, 0.0f );
			localPoints[2].Set( +size.x, +size.y, 0.0f );
			localPoints[3].Set( -size.x, +size.y, 0.0f );

			rotMat.SetRotation( pData->rot );
			rotMat.Transform( localPoints, 4 );

			// tFCX : eNX`W

			texCoords = pFace->texCoords;

			texTL = &( pData->tex[0] );
			texBR = &( pData->tex[1] );

			texCoords[0].Set( texTL->x, texTL->y );
			texCoords[1].Set( texBR->x, texTL->y );
			texCoords[2].Set( texBR->x, texBR->y );
			texCoords[3].Set( texTL->x, texBR->y );

			m_FaceNum++;
			dataIndex++;
		}
		else
		{
			m_DataList.FastRemoveByIndex( dataIndex ); // Ō̗vfƓւč폜̂ dataIndex ̓CNgȂ
			dataNum = m_DataList.GetCount(); // f[^擾Ȃ
		}
	}

	return True;
}

Boolean BillboardParticleSystem::InternalProcessor::Internal_Duplicate( Mix::Scene::IParticleProcessor** ppProcessor, const wchar_t* pDebugName )
{
	BillboardParticleSystem::InternalProcessor* pProcessor = BillboardParticleSystem::InternalProcessor::CreateInstance( MIX_SAFE_NAME( pDebugName ) );
	if( pProcessor == NULL )
	{
		return False;
	}

	UInt32 dataNum = m_DataList.GetCount();
	UInt32 faceCapacity = m_FacePool.GetCount();

	if( dataNum > 0 )
	{
		pProcessor->m_DataList.Add( m_DataList.GetConstBeginPtr(), dataNum );
	}

	if( faceCapacity > 0 )
	{
		pProcessor->m_FacePool.Add( faceCapacity );
	}

	( *ppProcessor ) = pProcessor;

	return True;
}

UInt32 BillboardParticleSystem::InternalProcessor::Internal_GetFaceNum( void ) const
{
	return m_FaceNum;
}

const Mix::Scene::IParticleProcessor::FACE* BillboardParticleSystem::InternalProcessor::Internal_GetFaces( void ) const
{
	return m_FacePool.GetBeginPtr();
}

#ifdef _DEBUG

void BillboardParticleSystem::InternalProcessor::Debug_Draw( const Mix::Matrix4x4& worldMat, Mix::Graphics::Utility::IPerspectiveRenderer* pPerspectiveRenderer )
{
}

#endif //_DEBUG

////////////////////////////////////////////////////////////////////////////////////////////////////
// BillboardParticleSystem
////////////////////////////////////////////////////////////////////////////////////////////////////

BillboardParticleSystem* BillboardParticleSystem::CreateInstance( Boolean bSimWorldSpace, const wchar_t* pDebugName )
{
	return MIX_LIB_NEW_T( Mix::Memory::SECTION_SCENE, BillboardParticleSystem, bSimWorldSpace, pDebugName );
}

BillboardParticleSystem::BillboardParticleSystem( Boolean bSimWorldSpace, const wchar_t* pDebugName ) :
m_pGenerator( NULL ),
m_MinEmitInterval( 0.1f ),
m_MaxEmitInterval( 0.2f ),
m_DiffEmitInterval( m_MaxEmitInterval - m_MinEmitInterval ),
m_MinEmitCount( 2 ),
m_MaxEmitCount( 4 ),
m_DiffEmitCount( m_MaxEmitCount - m_MinEmitCount )
{
	m_Config.baseFlags = ( bSimWorldSpace == True )? ( IParticleSystem::BILLBOARD | IParticleSystem::SIMULATION_WORLD_SPACE ) : IParticleSystem::BILLBOARD;
	m_Config.behaviorFlags = IParticleSystem::INVISIBLE_SLEEP | IParticleSystem::FAR_SKIP_FRAMES;
	m_Config.farMinDist = 30.0f;
	m_Config.farMaxDist = 50.0f;
	m_Config.maxSkipFrames = 4;

#ifdef _DEBUG
	m_DebugName = MIX_SAFE_NAME( pDebugName );
#endif //_DEBUG
}

BillboardParticleSystem::~BillboardParticleSystem( void )
{
	MIX_RELEASE( m_pGenerator );
}

Boolean BillboardParticleSystem::Initialize( void )
{
	MIX_ASSERT( m_pGenerator == NULL );

#ifdef _DEBUG
	const wchar_t* pDebugName = m_DebugName.GetConstPtr();
#else //_DEBUG
	const wchar_t* pDebugName = Mix::STR_UNKNOWN;
#endif //_DEBUG

	m_pGenerator = BillboardParticleSystem::InternalGenerator::CreateInstance();
	if( m_pGenerator == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s]", BillboardParticleSystem::FAILED_CREATE, Mix::STR_OUTOFMEMORY, pDebugName );
		return False;
	}

	return True;
}

Float32 BillboardParticleSystem::GetMinAngularImpulse( void ) const
{
	return m_pGenerator->GetMinAngularImpulse();
}

Float32 BillboardParticleSystem::GetMaxAngularImpulse( void ) const
{
	return m_pGenerator->GetMaxAngularImpulse();
}

void BillboardParticleSystem::SetAngularImpulse( Float32 minImpulse, Float32 maxImpulse )
{
	m_pGenerator->SetAngularImpulse( minImpulse, maxImpulse );
}

Float32 BillboardParticleSystem::GetMinAngularAcceleration( void ) const
{
	return m_pGenerator->GetMinAngularAcceleration();
}

Float32 BillboardParticleSystem::GetMaxAngularAcceleration( void ) const
{
	return m_pGenerator->GetMaxAngularAcceleration();
}

void BillboardParticleSystem::SetAngularAcceleration( Float32 minAcc, Float32 maxAcc )
{
	m_pGenerator->SetAngularAcceleration( minAcc, maxAcc );
}

Boolean BillboardParticleSystem::IsBillboard( void ) const
{
	return ( MIX_TESTBIT( m_Config.baseFlags, IParticleSystem::BILLBOARD ) == IParticleSystem::BILLBOARD );
}

Boolean BillboardParticleSystem::IsSimulationWorldSpace( void ) const
{
	return ( MIX_TESTBIT( m_Config.baseFlags, IParticleSystem::SIMULATION_WORLD_SPACE ) == IParticleSystem::SIMULATION_WORLD_SPACE );
}

Boolean BillboardParticleSystem::IsInvisibleSleep( void ) const
{
	return ( MIX_TESTBIT( m_Config.behaviorFlags, IParticleSystem::INVISIBLE_SLEEP ) == IParticleSystem::INVISIBLE_SLEEP );
}

void BillboardParticleSystem::SetInvisibleSleep( Boolean state )
{
	if( state == True )
	{
		MIX_SETBIT( m_Config.behaviorFlags, IParticleSystem::INVISIBLE_SLEEP );
	}
	else
	{
		MIX_RESETBIT( m_Config.behaviorFlags, IParticleSystem::INVISIBLE_SLEEP );
	}
}

Float32 BillboardParticleSystem::GetMinFarDist( void ) const
{
	return m_Config.farMinDist;
}

Float32 BillboardParticleSystem::GetMaxFarDist( void ) const
{
	return m_Config.farMaxDist;
}

void BillboardParticleSystem::SetFarDist( Float32 minDist, Float32 maxDist )
{
	m_Config.farMinDist = max( 0.0f, minDist );
	m_Config.farMaxDist = max( m_Config.farMinDist, maxDist );
}

Boolean BillboardParticleSystem::IsFarSkipFrames( void ) const
{
	return ( MIX_TESTBIT( m_Config.behaviorFlags, IParticleSystem::FAR_SKIP_FRAMES ) == IParticleSystem::FAR_SKIP_FRAMES );
}

void BillboardParticleSystem::SetFarSkipFrames( Boolean state )
{
	if( state == True )
	{
		MIX_SETBIT( m_Config.behaviorFlags, IParticleSystem::FAR_SKIP_FRAMES );
	}
	else
	{
		MIX_RESETBIT( m_Config.behaviorFlags, IParticleSystem::FAR_SKIP_FRAMES );
	}
}

UInt32 BillboardParticleSystem::GetMaxFarSkipFrames( void ) const
{
	return m_Config.maxSkipFrames;
}

void BillboardParticleSystem::SetMaxFarSkipFrames( UInt32 value )
{
	m_Config.maxSkipFrames = value;
}

Float32 BillboardParticleSystem::GetMinEmitInterval( void ) const
{
	return m_MinEmitInterval;
}

Float32 BillboardParticleSystem::GetMaxEmitInterval( void ) const
{
	return m_MaxEmitInterval;
}

void BillboardParticleSystem::SetEmitInterval( Float32 minEI, Float32 maxEI )
{
	m_MinEmitInterval = max( 0.0f, minEI );
	m_MaxEmitInterval = max( m_MinEmitInterval, maxEI );
	m_DiffEmitInterval = m_MaxEmitInterval - m_MinEmitInterval;
}

UInt32 BillboardParticleSystem::GetMinEmitCount( void ) const
{
	return m_MinEmitCount;
}

UInt32 BillboardParticleSystem::GetMaxEmitCount( void ) const
{
	return m_MaxEmitCount;
}

void BillboardParticleSystem::SetEmitCount( UInt32 minEC, UInt32 maxEC )
{
	m_MinEmitCount = max( 0, minEC );
	m_MaxEmitCount = max( m_MinEmitCount, maxEC );
	m_DiffEmitCount = m_MaxEmitCount - m_MinEmitCount;
}

const Mix::Vector3& BillboardParticleSystem::GetEmitRadius( void ) const
{
	return m_pGenerator->GetEmitRadius();
}

void BillboardParticleSystem::SetEmitRadius( const Mix::Vector3& radius )
{
	m_pGenerator->SetEmitRadius( radius );
}

Float32 BillboardParticleSystem::GetEmitOffset( void ) const
{
	return m_pGenerator->GetEmitOffset();
}

void BillboardParticleSystem::SetEmitOffset( Float32 offset )
{
	m_pGenerator->SetEmitOffset( offset );
}

Float32 BillboardParticleSystem::GetMinLife( void ) const
{
	return m_pGenerator->GetMinLife();
}

Float32 BillboardParticleSystem::GetMaxLife( void ) const
{
	return m_pGenerator->GetMaxLife();
}

void BillboardParticleSystem::SetLife( Float32 minLife, Float32 maxLife )
{
	m_pGenerator->SetLife( minLife, maxLife );
}

Float32 BillboardParticleSystem::GetMinMass( void ) const
{
	return m_pGenerator->GetMinMass();
}

Float32 BillboardParticleSystem::GetMaxMass( void ) const
{
	return m_pGenerator->GetMaxMass();
}

void BillboardParticleSystem::SetMass( Float32 minMass, Float32 maxMass )
{
	m_pGenerator->SetMass( minMass, maxMass );
}

const Mix::Vector3& BillboardParticleSystem::GetGravity( void ) const
{
	return m_pGenerator->GetGravity();
}

void BillboardParticleSystem::SetGravity( const Mix::Vector3& gravity )
{
	m_pGenerator->SetGravity( gravity );
}

const Mix::Vector3& BillboardParticleSystem::GetConstantLinearVelocity( void ) const
{
	return m_pGenerator->GetConstantLinearVelocity();
}

void BillboardParticleSystem::SetConstantLinearVelocity( const Mix::Vector3& vel )
{
	m_pGenerator->SetConstantLinearVelocity( vel );
}

const Mix::Vector3& BillboardParticleSystem::GetMinLinearImpulse( void ) const
{
	return m_pGenerator->GetMinLinearImpulse();
}

const Mix::Vector3& BillboardParticleSystem::GetMaxLinearImpulse( void ) const
{
	return m_pGenerator->GetMaxLinearImpulse();
}

void BillboardParticleSystem::SetLinearImpulse( const Mix::Vector3& minImpulse, const Mix::Vector3& maxImpulse )
{
	m_pGenerator->SetLinearImpulse( minImpulse, maxImpulse );
}

const Mix::Vector3& BillboardParticleSystem::GetMinLinearAcceleration( void ) const
{
	return m_pGenerator->GetMinLinearAcceleration();
}

const Mix::Vector3& BillboardParticleSystem::GetMaxLinearAcceleration( void ) const
{
	return m_pGenerator->GetMaxLinearAcceleration();
}

void BillboardParticleSystem::SetLinearAcceleration( const Mix::Vector3& minAcc, const Mix::Vector3& maxAcc )
{
	m_pGenerator->SetLinearAcceleration( minAcc, maxAcc );
}

Float32 BillboardParticleSystem::GetMinLinearVelocityDamping( void ) const
{
	return m_pGenerator->GetMinLinearVelocityDamping();
}

Float32 BillboardParticleSystem::GetMaxLinearVelocityDamping( void ) const
{
	return m_pGenerator->GetMaxLinearVelocityDamping();
}

void BillboardParticleSystem::SetLinearVelocityDamping( Float32 minDamping, Float32 maxDamping )
{
	m_pGenerator->SetLinearVelocityDamping( minDamping, maxDamping );
}

Float32 BillboardParticleSystem::GetMinAngularVelocityDamping( void ) const
{
	return m_pGenerator->GetMinAngularVelocityDamping();
}

Float32 BillboardParticleSystem::GetMaxAngularVelocityDamping( void ) const
{
	return m_pGenerator->GetMaxAngularVelocityDamping();
}

void BillboardParticleSystem::SetAngularVelocityDamping( Float32 minDamping, Float32 maxDamping )
{
	m_pGenerator->SetAngularVelocityDamping( minDamping, maxDamping );
}

const Mix::Vector2& BillboardParticleSystem::GetInitalSize( void ) const
{
	return m_pGenerator->GetInitalSize();
}

const Mix::Vector2& BillboardParticleSystem::GetLastSize( void ) const
{
	return m_pGenerator->GetLastSize();
}

Float32 BillboardParticleSystem::GetMinSizeRand( void ) const
{
	return m_pGenerator->GetMinSizeRand();
}

Float32 BillboardParticleSystem::GetMaxSizeRand( void ) const
{
	return m_pGenerator->GetMaxSizeRand();
}

void BillboardParticleSystem::SetSize( const Mix::Vector2& initalSize, const Mix::Vector2& lastSize, Float32 minSizeScale, Float32 maxSizeScale )
{
	m_pGenerator->SetSize( initalSize, lastSize, minSizeScale, maxSizeScale );
}

const Mix::Vector4& BillboardParticleSystem::GetInitalColor( void ) const
{
	return m_pGenerator->GetInitalColor();
}

const Mix::Vector4& BillboardParticleSystem::GetMiddleColor( void ) const
{
	return m_pGenerator->GetMiddleColor();
}

const Mix::Vector4& BillboardParticleSystem::GetLastColor( void ) const
{
	return m_pGenerator->GetLastColor();
}

void BillboardParticleSystem::SetColor( const Mix::Vector4& initalColor, const Mix::Vector4& middleColor, const Mix::Vector4& lastColor )
{
	m_pGenerator->SetColor( initalColor, middleColor, lastColor );
}

Float32 BillboardParticleSystem::GetColorControlPoint1( void ) const
{
	return m_pGenerator->GetColorControlPoint1();
}

Float32 BillboardParticleSystem::GetColorControlPoint2( void ) const
{
	return m_pGenerator->GetColorControlPoint2();
}

void BillboardParticleSystem::SetColorControlPoints( Float32 p1, Float32 p2 )
{
	m_pGenerator->SetColorControlPoints( p1, p2 );
}

const Mix::Vector2& BillboardParticleSystem::GetTexTL( void ) const
{
	return m_pGenerator->GetTexTL();
}

const Mix::Vector2& BillboardParticleSystem::GetTexBR( void ) const
{
	return m_pGenerator->GetTexBR();
}

void BillboardParticleSystem::SetTexCoords( const Mix::Vector2& tl, const Mix::Vector2& br )
{
	m_pGenerator->SetTexCoords( tl, br );
}

const Mix::Scene::IParticleSystem::CONFIG& BillboardParticleSystem::GetConfig( void ) const
{
	return m_Config;
}

void BillboardParticleSystem::Internal_GetGenerator( Mix::Scene::IParticleGenerator** ppGenerator )
{
	MIX_ASSERT( ppGenerator != NULL );

	MIX_ADD_REF( m_pGenerator );
	( *ppGenerator ) = m_pGenerator;
}

Boolean BillboardParticleSystem::Internal_CreateProcessor( Mix::Scene::IParticleProcessor** ppProcessor )
{
	MIX_ASSERT( ppProcessor != NULL );

#ifdef _DEBUG
	const wchar_t* pDebugName = m_DebugName.GetConstPtr();
#else //_DEBUG
	const wchar_t* pDebugName = NULL;
#endif //_DEBUG

	BillboardParticleSystem::InternalProcessor* pInternalProcessor = BillboardParticleSystem::InternalProcessor::CreateInstance( pDebugName );
	if( pInternalProcessor == NULL )
	{
		return False;
	}

	( *ppProcessor ) = pInternalProcessor;

	return True;
}

Float32 BillboardParticleSystem::Internal_GetGenerateInterval( void ) const
{
	return m_MinEmitInterval + m_DiffEmitInterval * Mix::RandF();
}

UInt32 BillboardParticleSystem::Internal_GetGenerateCount( void ) const
{
	if( m_DiffEmitCount == 0 )
	{
		return m_MinEmitCount;
	}

	return m_MinEmitCount + ( Mix::Rand() % m_DiffEmitCount );
}

}}}
