#include "Mix/Private/Scene/Common/Manager.h"

#include "Mix/IO/IManager.h"
#include "Mix/IO/IReader.h"
#include "Mix/Graphics/IManager.h"
#include "Mix/Graphics/IDevice.h"
#include "Mix/Scene/IMaterial.h"

#include "Mix/Private/IO/MemoryReader.h"

#include "Mix/Private/Scene/Common/Renderer.h"

#include "Mix/Private/Scene/Standard/DX9/EffectPackage.h"
#include "Mix/Private/Scene/Standard/DX10/EffectPackage.h"
#include "Mix/Private/Scene/Standard/DX11/EffectPackage.h"

#include "Mix/Private/Scene/Standard/DX9/Renderer.h"
#include "Mix/Private/Scene/Standard/DX11/Renderer.h"

#include "Mix/Private/Scene/Common/UniversalCamera.h"
#include "Mix/Private/Scene/Common/ViewCamera.h"

#include "Mix/Private/Scene/Common/HemisphereLight.h"
#include "Mix/Private/Scene/Common/DirectionalLight.h"
#include "Mix/Private/Scene/Common/PointLight.h"
#include "Mix/Private/Scene/Common/SpotLight.h"

#include "Mix/Private/Scene/Common/DefaultParticleSystem.h"
#include "Mix/Private/Scene/Common/BillboardParticleSystem.h"
#include "Mix/Private/Scene/Common/LeavingParticle.h"
#include "Mix/Private/Scene/Common/ScatterParticle.h"

#include "Mix/Private/Scene/Common/Planter.h"

#include "Mix/Private/Scene/Common/LightShaftsCaster.h"
#include "Mix/Private/Scene/Common/WaterPool.h"

#include "Mix/Private/Scene/Common/Floor.h"

#include "Mix/Private/Scene/Common/BasisGhost.h"
#include "Mix/Private/Scene/Common/KinematicGhost.h"

#include "Mix/Private/Scene/Common/ActorLoader.h"
#include "Mix/Private/Scene/Common/TerrainLoader.h"

#include "Mix/Private/Scene/Standard/DX9/DefaultMaterial.h"
#include "Mix/Private/Scene/Standard/DX11/DefaultMaterial.h"

#include "Mix/Private/Scene/Common/Motion.h"

#ifdef _DEBUG
	#include "Mix/Private/Scene/Common/Debug.h"
#endif //_DEBUG

namespace Mix{ namespace Scene{ namespace Common{

////////////////////////////////////////////////////////////////////////////////////////////////////
// Manager : 萔
////////////////////////////////////////////////////////////////////////////////////////////////////

const wchar_t* Manager::MSG_NOT_SUPPORT_SHADERMODEL = L"vꂽVF[_[f̓T|[gĂ܂";
const wchar_t* Manager::MSP_EP_NOT_SUPPORT_RENDERER = L"GtFNgpbP[Wv郌_[̓T|[gĂ܂";
const wchar_t* Manager::MSP_EP_NOT_SUPPORT_MATERIAL = L"GtFNgpbP[Wv}eA̓T|[gĂ܂";

////////////////////////////////////////////////////////////////////////////////////////////////////
// Manager : NX
////////////////////////////////////////////////////////////////////////////////////////////////////

Manager* Manager::CreateInstance( void )
{
	return MIX_LIB_NEW_T( Mix::Memory::SECTION_SCENE, Manager );
}

Manager::Manager( void ) :
m_pFileManager( NULL ),
m_pGraphicsDevice( NULL ),
m_pActorLoader( NULL ),
m_pTerrainLoader( NULL )
{
	////////////////////////////////////////////////////////////////////////////////////////////////////

	Mix::IO::IManager* pFileMgr = Mix::IO::GetManagerPtr();

	MIX_ASSERT( pFileMgr != NULL );

	MIX_ADD_REF( pFileMgr );
	m_pFileManager = pFileMgr;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	Mix::Graphics::IManager* pGraphcisMgr = Mix::Graphics::GetManagerPtr();

	MIX_ASSERT( pGraphcisMgr != NULL );

	pGraphcisMgr->GetDevice( &m_pGraphicsDevice );
	MIX_ASSERT( m_pGraphicsDevice != NULL );
}

Manager::~Manager( void )
{
	MIX_ASSERT( m_RendererList.size() == 0 );
	MIX_ASSERT( m_UpdateMaterialMap.size() == 0 );
	MIX_ASSERT( m_pTerrainLoader == NULL );
	MIX_ASSERT( m_pActorLoader == NULL );
	MIX_ASSERT( m_pGraphicsDevice == NULL );
	MIX_ASSERT( m_pFileManager == NULL );
}

Boolean Manager::Initialize( void )
{
	MIX_ASSERT( m_pActorLoader == NULL );
	MIX_ASSERT( m_pTerrainLoader == NULL );

	// AN^[[_[̍쐬

	m_pActorLoader = MIX_LIB_NEW_T( Mix::Memory::SECTION_SCENE, Mix::Scene::Common::ActorLoader );
	if( m_pActorLoader == NULL )
	{
		return False;
	}

	// eC[_[̍쐬

	m_pTerrainLoader = MIX_LIB_NEW_T( Mix::Memory::SECTION_SCENE, Mix::Scene::Common::TerrainLoader );
	if( m_pTerrainLoader == NULL )
	{
		return False;
	}

	return True;
}

void Manager::Update( Float32 dt )
{
	Manager::RendererList::iterator it_rdr_begin = m_RendererList.begin();
	Manager::RendererList::iterator it_rdr_end = m_RendererList.end();
	Manager::RendererList::iterator it_rdr;

	Manager::UpdateMaterialMap::iterator it_um_begin = m_UpdateMaterialMap.begin();
	Manager::UpdateMaterialMap::iterator it_um_end = m_UpdateMaterialMap.end();
	Manager::UpdateMaterialMap::iterator it_um;

	Mix::Scene::Common::Renderer* pRenderer;
	Mix::Scene::IMaterial* pMaterial;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// _[̃tBjbV
	////////////////////////////////////////////////////////////////////////////////////////////////////

	for( it_rdr = it_rdr_begin; it_rdr != it_rdr_end; ++it_rdr )
	{
		pRenderer = ( *it_rdr );
		pRenderer->Finish();
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// }eA(gAj[V)̃Abvf[g
	////////////////////////////////////////////////////////////////////////////////////////////////////

	for( it_um = it_um_begin; it_um != it_um_end; ++it_um )
	{
		pMaterial = it_um->first;
		pMaterial->Update( dt );
	}
}

Mix::Scene::IMaterial* Manager::CreateMaterial(	Mix::Scene::Common::EffectPackage* pEffectPackage,
												Mix::IO::IReader* pReader,
												Boolean bWithMagicNumber,
												const wchar_t* pNameLabel,
												const wchar_t* pName )
{
	MIX_ASSERT( pEffectPackage != NULL );
	MIX_ASSERT( pReader != NULL );

	Mix::Scene::IRenderer::TYPE rendererType = pEffectPackage->GetRendererType();
	Mix::Scene::IMaterial::TYPE materialType = pEffectPackage->GetMaterialType();
	Mix::Scene::IMaterial* pMaterial = NULL;

	if( rendererType == Mix::Scene::IRenderer::FORWARD )
	{
		if( materialType == Mix::Scene::IMaterial::DEFAULT )
		{
			switch( m_pGraphicsDevice->GetShaderModel() )
			{
			case Mix::Graphics::SHADER_MODEL_3:
				pMaterial = Mix::Scene::Standard::DX9::DefaultMaterial::CreateInstance( m_pGraphicsDevice, pEffectPackage, pReader, bWithMagicNumber, pNameLabel, pName );
				break;
			case Mix::Graphics::SHADER_MODEL_4:
			case Mix::Graphics::SHADER_MODEL_5:
				pMaterial = Mix::Scene::Standard::DX11::DefaultMaterial::CreateInstance( m_pGraphicsDevice, pEffectPackage, pReader, bWithMagicNumber, pNameLabel, pName );
				break;

			default:
				MIX_LOG_ERROR( L"%s : %s[%s]", Manager::MSG_NOT_SUPPORT_SHADERMODEL, pNameLabel, pName );
				break;
			}
		}
		else
		{
			MIX_LOG_ERROR( L"%s : %s[%s]", Manager::MSP_EP_NOT_SUPPORT_MATERIAL, pNameLabel, pName );
		}
	}
	else
	{
		MIX_LOG_ERROR( L"%s : %s[%s]", Manager::MSP_EP_NOT_SUPPORT_RENDERER, pNameLabel, pName );
	}

	return pMaterial;
}

Mix::Scene::IMaterial* Manager::CreateMaterialWithoutMagicNumber(	Mix::Scene::Common::EffectPackage* pEffectPackage,
																	Mix::IO::IReader* pReader,
																	const wchar_t* pName )
{
	MIX_ASSERT( pEffectPackage != NULL );
	MIX_ASSERT( pReader != NULL );

	Mix::Scene::IMaterial* pMaterial;

	MIX_LOG_INFO_SECT_START( L"}eA̍쐬 : Name[%s] Type[Standard( Default )]", MIX_SAFE_NAME( pName ) );

	pMaterial = CreateMaterial( pEffectPackage, pReader, False, L"Name", pName );

	MIX_LOG_INFO_SECT_END();

	return pMaterial;
}

void Manager::AddUpdateMaterial( Mix::Scene::IMaterial* pMaterial )
{
	MIX_ASSERT( pMaterial != NULL );

	Manager::UpdateMaterialMap::iterator it = m_UpdateMaterialMap.find( pMaterial );

	MIX_ADD_REF( pMaterial );

	if( it == m_UpdateMaterialMap.end() )
	{
		m_UpdateMaterialMap.insert( Manager::UpdateMaterialMap::value_type( pMaterial, Mix::AtomicValue( 1 ) ) );

		//QƂ
		it = m_UpdateMaterialMap.find( pMaterial );
		MIX_ASSERT( it != m_UpdateMaterialMap.end() );
	}
	else
	{
		//QƂCNg
		it->second.Increment();
	}
}

void Manager::RemoveUpdateMaterial( Mix::Scene::IMaterial* pMaterial )
{
	MIX_ASSERT( pMaterial != NULL );

	Manager::UpdateMaterialMap::iterator it = m_UpdateMaterialMap.find( pMaterial );

	if( it != m_UpdateMaterialMap.end() )
	{
		if( it->second.Get() == 1 )
		{
			m_UpdateMaterialMap.erase( it );
		}
		else
		{
			it->second.Decrement();
		}

		MIX_RELEASE( pMaterial );
	}
}

void Manager::AddRenderer( Mix::Scene::Common::Renderer* pRenderer )
{
	MIX_ASSERT( pRenderer != NULL );
	MIX_ASSERT( std::find( m_RendererList.begin(), m_RendererList.end(), pRenderer ) == m_RendererList.end() );

	m_RendererList.push_back( pRenderer );
}

void Manager::RemoveRenderer( Mix::Scene::Common::Renderer* pRenderer )
{
	MIX_ASSERT( pRenderer != NULL );
	MIX_ASSERT( std::find( m_RendererList.begin(), m_RendererList.end(), pRenderer ) != m_RendererList.end() );

	Mix::STL::Vector_FirstErase( m_RendererList, pRenderer );
}

void Manager::Dispose( void )
{
	MIX_LIB_DELETE_T( TerrainLoader, m_pTerrainLoader );
	MIX_LIB_DELETE_T( ActorLoader, m_pActorLoader );

	MIX_RELEASE( m_pGraphicsDevice );
	MIX_RELEASE( m_pFileManager );
}

Boolean Manager::CreateEffectPackage( const wchar_t* pDirPath, const wchar_t* pName, Mix::Scene::IEffectPackage** ppEffectPackage )
{
	if( ( pDirPath == NULL ) ||
		( pName == NULL ) ||
		( wcslen( pName ) == 0 ) ||
		( ppEffectPackage == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pDirPath[%s] pName[%s] ppEffectPackage[%s]",
			EffectPackage::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_STR( pDirPath ),
			MIX_LOG_STR( pName ),
			MIX_LOG_PTR( ppEffectPackage ) );

		return False;
	}

	Mix::Graphics::SHADER_MODEL shaderModel = m_pGraphicsDevice->GetShaderModel();
	Mix::StringW fileName;
	Mix::StringW filePath;
	Mix::IO::IReader* pReader = NULL;
	EffectPackage::MEP_FILE_HEADER fileHeader;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"GtFNgpbP[W̍쐬" );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	/*
		t@CpX̍쐬
	*/

	fileName.Sprintf( L"%s%s", pName, EffectPackage::MEP_FILEEXT_TABLE[shaderModel] );

	if( Mix::IO::CombinePath( pDirPath, False, fileName.GetConstPtr(), filePath ) == False )
	{
		MIX_LOG_ERROR( L"%s : GtFNgpbP[W̃t@CpX쐬ł܂ł : %s[%s]",
			EffectPackage::FAILED_CREATE,
			Mix::STR_FILEPATH,
			filePath.GetConstPtr() );

		MIX_LOG_INFO_SECT_END();

		return False;
	}

	/*
		ނ𔻕
	*/

	// t@C[_[̍쐬
	if( m_pFileManager->CreateFileReader( filePath.GetConstPtr(), &pReader ) == False )
	{
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	// t@Cwb_ǂݍ݁A`FbN
	if( EffectPackage::LoadFileHeader( filePath.GetConstPtr(), pReader, fileHeader ) == False )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pReader );
		return False;
	}

	// C^[tF[X̍쐬A
	if( fileHeader.version == EffectPackage::FILE_VERSION )
	{
		// ݂̃o[W

		Mix::Scene::Standard::Common::EffectPackage* pEffectPackage = NULL;

		switch( shaderModel )
		{
		case Mix::Graphics::SHADER_MODEL_3:
			pEffectPackage = Mix::Scene::Standard::DX9::EffectPackage::CreateInstance( pDirPath, pName, filePath.GetConstPtr() );
			break;
		case Mix::Graphics::SHADER_MODEL_4:
			pEffectPackage = Mix::Scene::Standard::DX10::EffectPackage::CreateInstance( pDirPath, pName, filePath.GetConstPtr() );
			break;
		case Mix::Graphics::SHADER_MODEL_5:
			pEffectPackage = Mix::Scene::Standard::DX11::EffectPackage::CreateInstance( pDirPath, pName, filePath.GetConstPtr() );
			break;

		default:
			MIX_LOG_ERROR( L"%s : T|[gȂVF[_[fw肳Ă܂", EffectPackage::FAILED_CREATE );
			MIX_LOG_INFO_SECT_END();
			MIX_RELEASE( pReader );
			return False;
		}

		if( pEffectPackage == NULL )
		{
			MIX_LOG_ERROR( L"%s : %s", EffectPackage::FAILED_CREATE, Mix::STR_OUTOFMEMORY );
			MIX_LOG_INFO_SECT_END();
			MIX_RELEASE( pReader );
			return False;
		}

		if( pEffectPackage->Initialize( pReader ) == False )
		{
			MIX_LOG_INFO_SECT_END();
			MIX_RELEASE( pReader );
			return False;
		}

		( *ppEffectPackage ) = pEffectPackage;
	}
	else
	{
		MIX_LOG_ERROR( L"%s : ̃o[W͎gpł܂ : FileVersion[0x%08x] CurrentVersion[0x%08x]",
			EffectPackage::FAILED_CREATE,
			fileHeader.version,
			EffectPackage::FILE_VERSION );

		MIX_LOG_INFO_SECT_END();

		MIX_RELEASE( pReader );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ㏈
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateRenderer( Mix::Scene::IEffectPackage* pEffectPackage, const Mix::Scene::RENDERER_CONFIG& config, Mix::Scene::IRenderer** ppRenderer, const wchar_t* pDebugName )
{
	if( pEffectPackage == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : pEffectPackage[%s]",
			Renderer::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( pEffectPackage ) );

		return False;
	}

	Mix::Scene::IRenderer::TYPE rendererType = pEffectPackage->GetRendererType();
	Mix::Scene::IMaterial::TYPE materialType = pEffectPackage->GetMaterialType();

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"_[̍쐬 : Name[%s]", MIX_SAFE_NAME( pDebugName ) );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( rendererType == Mix::Scene::IRenderer::FORWARD )
	{
		if( materialType == Mix::Scene::IMaterial::DEFAULT )
		{
			Mix::Scene::Standard::Common::EffectPackage* pEffectPackageFD = static_cast<Mix::Scene::Standard::Common::EffectPackage*>( pEffectPackage );
			Mix::Scene::Standard::Common::Renderer* pRendererFD = NULL;

			switch( m_pGraphicsDevice->GetShaderModel() )
			{
			case  Mix::Graphics::SHADER_MODEL_3:
				pRendererFD = Mix::Scene::Standard::DX9::Renderer::CreateInstance( pEffectPackageFD, MIX_SAFE_NAME( pDebugName ) );
				break;
			case  Mix::Graphics::SHADER_MODEL_4:
			case  Mix::Graphics::SHADER_MODEL_5:
				pRendererFD = Mix::Scene::Standard::DX11::Renderer::CreateInstance( pEffectPackageFD, MIX_SAFE_NAME( pDebugName ) );
				break;

			default:
				MIX_LOG_ERROR( L"%s : Name[%s]", Manager::MSG_NOT_SUPPORT_SHADERMODEL, MIX_SAFE_NAME( pDebugName ) );
				MIX_LOG_INFO_SECT_END();
				return False;
			}

			if( pRendererFD == NULL )
			{
				MIX_LOG_ERROR( Mix::STR_OUTOFMEMORY );
				MIX_LOG_INFO_SECT_END();
				return False;
			}

			if( pRendererFD->Initialize( config, NULL ) == False )
			{
				MIX_LOG_INFO_SECT_END();
				return False;
			}

			( *ppRenderer ) = pRendererFD;
		}
		else
		{
			MIX_LOG_ERROR( L"%s : Name[%s]", Manager::MSP_EP_NOT_SUPPORT_MATERIAL, MIX_SAFE_NAME( pDebugName ) );
			MIX_LOG_INFO_SECT_END();
			return False;
		}
	}
	else
	{
		MIX_LOG_ERROR( L"%s : Name[%s]", Manager::MSP_EP_NOT_SUPPORT_RENDERER, MIX_SAFE_NAME( pDebugName ) );
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateUniversalCamera( const Mix::Scene::CAMERA_CONFIG& config, Mix::Scene::IUniversalCamera** ppCamera, const wchar_t* pDebugName )
{
	if( ppCamera == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppCamera[%s]",
			UniversalCamera::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( ppCamera ) );

		return False;
	}

	Mix::Scene::Common::UniversalCamera* pCamera = NULL;

	pCamera = Mix::Scene::Common::UniversalCamera::CreateInstance( config );
	if( pCamera == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", UniversalCamera::FAILED_CREATE, Mix::STR_OUTOFMEMORY, MIX_SAFE_NAME( pDebugName ) );
		return False;
	}

#ifdef _DEBUG
	pCamera->Debug_SetName( MIX_SAFE_NAME( pDebugName ) );
#endif //_DEBUG

	( *ppCamera ) = pCamera;

	return True;
}

Boolean Manager::CreateViewCamera( const Mix::Scene::CAMERA_CONFIG& config, Mix::Scene::IViewCamera** ppCamera, const wchar_t* pDebugName )
{
	if( ppCamera == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppCamera[%s]",
			ViewCamera::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( ppCamera ) );

		return False;
	}

	Mix::Scene::Common::ViewCamera* pCamera = NULL;

	pCamera = Mix::Scene::Common::ViewCamera::CreateInstance( config );
	if( pCamera == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", ViewCamera::FAILED_CREATE, Mix::STR_OUTOFMEMORY, MIX_SAFE_NAME( pDebugName ) );
		return False;
	}

#ifdef _DEBUG
	pCamera->Debug_SetName( MIX_SAFE_NAME( pDebugName ) );
#endif //_DEBUG

	( *ppCamera ) = pCamera;

	return True;
}

Boolean Manager::CreateHemisphereLight( Mix::Scene::IHemisphereLight** ppLight, const wchar_t* pDebugName )
{
	if( ppLight == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppLight[%s]",
			HemisphereLight::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( ppLight ) );

		return False;
	}

	Mix::Scene::Common::HemisphereLight* pLight = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pLight = Mix::Scene::Common::HemisphereLight::CreateInstance();
	if( pLight == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]",
			HemisphereLight::FAILED_CREATE,
			Mix::STR_OUTOFMEMORY,
			MIX_SAFE_NAME( pDebugName ) );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppLight ) = pLight;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateDirectionalLight( Mix::Scene::IDirectionalLight** ppLight, const wchar_t* pDebugName )
{
	if( ppLight == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppLight[%s]",
			DirectionalLight::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( ppLight ) );

		return False;
	}

	Mix::Scene::Common::DirectionalLight* pLight = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pLight = Mix::Scene::Common::DirectionalLight::CreateInstance();
	if( pLight == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]",
			DirectionalLight::FAILED_CREATE,
			Mix::STR_OUTOFMEMORY,
			MIX_SAFE_NAME( pDebugName ) );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppLight ) = pLight;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreatePointLight( Mix::Scene::IPointLight** ppLight, const wchar_t* pDebugName )
{
	if( ppLight == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppLight[%s]",
			PointLight::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( ppLight ) );

		return False;
	}

	Mix::Scene::Common::PointLight* pLight = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pLight = Mix::Scene::Common::PointLight::CreateInstance();
	if( pLight == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]",
			PointLight::FAILED_CREATE,
			Mix::STR_OUTOFMEMORY,
			MIX_SAFE_NAME( pDebugName ) );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppLight ) = pLight;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateSpotLight( Mix::Scene::ISpotLight** ppLight, const wchar_t* pDebugName )
{
	if( ppLight == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppLight[%s]",
			SpotLight::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( ppLight ) );

		return False;
	}

	Mix::Scene::Common::SpotLight* pLight = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pLight = Mix::Scene::Common::SpotLight::CreateInstance();
	if( pLight == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]",
			SpotLight::FAILED_CREATE,
			Mix::STR_OUTOFMEMORY,
			MIX_SAFE_NAME( pDebugName ) );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppLight ) = pLight;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateSkyDome(	Float32 radius, Float32 height,
								Mix::Scene::ISkyDome::QUALITY heQuality,
								Mix::Scene::ISkyDome** ppSkyDome,
								const wchar_t* pDebugName )
{
	if( ppSkyDome == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppSkyDome[%s]",
			SkyDome::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( ppSkyDome ) );

		return False;
	}

	Mix::Scene::Common::SkyDome* pSkyDome = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"XJCh[̍쐬 : DebugName[%s]", MIX_SAFE_NAME( pDebugName ) );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pSkyDome = Mix::Scene::Common::SkyDome::CreateInstance( radius, height );
	if( pSkyDome == NULL )
	{
		MIX_LOG_ERROR( L"%s", Mix::STR_OUTOFMEMORY );
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	if( pSkyDome->Initialize( m_pGraphicsDevice, heQuality, Mix::Scene::Common::SkyDome::QL_MAX, pDebugName ) == False )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pSkyDome );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppSkyDome ) = pSkyDome;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateSkyDome(	Float32 radius, Float32 height,
								Mix::Scene::ISkyDome::QUALITY heQuality,
								Mix::Scene::ISkyDome::QUALITY paQuality,
								Mix::Scene::ISkyDome** ppSkyDome,
								const wchar_t* pDebugName )
{
	if( ppSkyDome == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppSkyDome[%s]",
			SkyDome::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( ppSkyDome ) );

		return False;
	}

	Mix::Scene::Common::SkyDome* pSkyDome = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"XJCh[̍쐬 : DebugName[%s]", MIX_SAFE_NAME( pDebugName ) );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pSkyDome = Mix::Scene::Common::SkyDome::CreateInstance( radius, height );
	if( pSkyDome == NULL )
	{
		MIX_LOG_ERROR( L"%s", Mix::STR_OUTOFMEMORY );
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	if( pSkyDome->Initialize( m_pGraphicsDevice, heQuality, paQuality, pDebugName ) == False )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pSkyDome );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppSkyDome ) = pSkyDome;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateDefaultParticleSystem( Boolean bSimWorldSpace, Mix::Scene::IDefaultParticleSystem::PLANE_TYPE planeType, Mix::Scene::IDefaultParticleSystem** ppSystem, const wchar_t* pDebugName )
{
	if( ppSystem == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppProducer[%s]",
			MIX_SAFE_NAME( pDebugName ),
			DefaultParticleSystem::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_PTR( ppSystem ) );

		return False;
	}

	Mix::Scene::Common::DefaultParticleSystem* pSystem = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pSystem = Mix::Scene::Common::DefaultParticleSystem::CreateInstance( bSimWorldSpace, MIX_SAFE_NAME( pDebugName ) );
	if( pSystem == NULL )
	{
		MIX_LOG_ERROR( L"%s : DebugName[%s]", Mix::STR_OUTOFMEMORY, MIX_SAFE_NAME( pDebugName ) );

		return False;
	}

	if( pSystem->Initialize( planeType ) == False )
	{
		MIX_RELEASE( pSystem );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppSystem ) = pSystem;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateBillboardParticleSystem( Boolean bSimWorldSpace, Mix::Scene::IBillboardParticleSystem** ppSystem, const wchar_t* pDebugName )
{
	if( ppSystem == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppProducer[%s]",
			MIX_SAFE_NAME( pDebugName ),
			BillboardParticleSystem::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_PTR( ppSystem ) );

		return False;
	}

	Mix::Scene::Common::BillboardParticleSystem* pSystem = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pSystem = Mix::Scene::Common::BillboardParticleSystem::CreateInstance( bSimWorldSpace, MIX_SAFE_NAME( pDebugName ) );
	if( pSystem == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]",
			BillboardParticleSystem::FAILED_CREATE,
			Mix::STR_OUTOFMEMORY,
			MIX_SAFE_NAME( pDebugName ) );

		return False;
	}

	if( pSystem->Initialize() == False )
	{
		//G[bZ[W Initialize \bhŏoĂ܂
		MIX_RELEASE( pSystem );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppSystem ) = pSystem;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateLeavingParticle( Mix::Scene::IMaterial* pMaterial, Mix::Scene::IParticleSystem* pSystem, Mix::Scene::ILeavingParticle** ppParticle, const wchar_t* pDebugName )
{
	if( ( pMaterial == NULL ) ||
		( pSystem == NULL ) ||
		( ppParticle == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : pMaterial[%s] pSystem[%s] ppParticle[%s]",
			Mix::Scene::Common::LeavingParticle::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( pMaterial ),
			MIX_LOG_PTR( pSystem ),
			MIX_LOG_PTR( ppParticle ) );

		return False;
	}

	Mix::Scene::Common::LeavingParticle* pParticle = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"[rOp[eBN̍쐬 : DebugName[%s]", MIX_SAFE_NAME( pDebugName ) );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// }eÃ`FbN
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( InternalParticle::CheckMaterial( pMaterial, NULL, MIX_SAFE_NAME( pDebugName ) ) == False )
	{
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pParticle = Mix::Scene::Common::LeavingParticle::CreateInstance( pMaterial, pSystem, MIX_SAFE_NAME( pDebugName ) );
	if( pParticle == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_LOG_ERROR( L"%s", Mix::STR_OUTOFMEMORY );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pParticle->Initialize() == False )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pParticle );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppParticle ) = pParticle;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateScatterParticle( Mix::Scene::IMaterial* pMaterial, Mix::Scene::IParticleSystem* pSystem, Mix::Scene::IScatterParticle** ppParticle, const wchar_t* pDebugName )
{
	if( ( pMaterial == NULL ) ||
		( pSystem == NULL ) ||
		( ppParticle == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : pMaterial[%s] pSystem[%s] ppParticle[%s]",
			Mix::Scene::Common::ScatterParticle::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( pMaterial ),
			MIX_LOG_PTR( pSystem ),
			MIX_LOG_PTR( ppParticle ) );

		return False;
	}

	Mix::Scene::Common::ScatterParticle* pParticle = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"XLb^[p[eBN̍쐬 : DebugName[%s]", MIX_SAFE_NAME( pDebugName ) );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// }eÃ`FbN
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( InternalParticle::CheckMaterial( pMaterial, NULL, MIX_SAFE_NAME( pDebugName ) ) == False )
	{
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pParticle = Mix::Scene::Common::ScatterParticle::CreateInstance( pMaterial, pSystem, 32, 256, 16, MIX_SAFE_NAME( pDebugName ) );
	if( pParticle == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s", Mix::Scene::Common::ScatterParticle::FAILED_CREATE, Mix::STR_OUTOFMEMORY );
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppParticle ) = pParticle;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateFlowerPackage(	Mix::Scene::IMaterial* pMaterial,
										UInt32 numModel, const Mix::Scene::IFlowerPackage::MODEL* models,
										Mix::Scene::IFlowerPackage** ppFlowerPackage,
										const wchar_t* pDebugName )
{
	if( ( pMaterial == NULL ) ||
		( numModel == 0 ) ||
		( models == NULL ) ||
		( ppFlowerPackage == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pMaterial[%s] numModel[%d] models[%s] ppFlowerPackage[%s]",
			FlowerPackage::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( pMaterial ),
			numModel,
			MIX_LOG_PTR( models ),
			MIX_LOG_PTR( ppFlowerPackage ) );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// }eÃ`FbN
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( FlowerPackage::CheckMaterial( pMaterial, FlowerPackage::FAILED_CREATE, Mix::STR_DEBUGNAME, MIX_SAFE_NAME( pDebugName ) ) == False )
	{
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	Mix::Scene::Common::FlowerPackage* pFlowerPackage = NULL;

	pFlowerPackage = FlowerPackage::CreateInstance( pMaterial, numModel, models, MIX_SAFE_NAME( pDebugName ) );
	if( pFlowerPackage == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FlowerPackage::FAILED_CREATE, Mix::STR_OUTOFMEMORY, MIX_SAFE_NAME( pDebugName ) );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppFlowerPackage ) = pFlowerPackage;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreatePlanter(	Mix::Scene::IFlowerPackage* pFlowerPackage,
								Mix::Scene::IPlanter** ppPlanter,
								const wchar_t* pDebugName )
{
	if( ( pFlowerPackage == NULL ) ||
		( ppPlanter == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pFlowerPackage[%s] ppPlanter[%s]",
			Planter::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( pFlowerPackage ),
			MIX_LOG_PTR( ppPlanter ) );

		return False;
	}

	Mix::Scene::Common::Planter* pPlanter = NULL;

	pPlanter = Planter::CreateInstance( pFlowerPackage, 0, NULL, Mix::Matrix4x4::Identity(), pDebugName );
	if( pPlanter == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", Planter::FAILED_CREATE, Mix::STR_OUTOFMEMORY, MIX_SAFE_NAME( pDebugName ) );
		return False;
	}

	( *ppPlanter ) = pPlanter;

	return True;
}

Boolean Manager::CreatePlanter(	Mix::Scene::IFlowerPackage* pFlowerPackage,
								UInt32 numUnit, const Mix::Scene::IPlanter::UNIT* units,
								const Mix::Matrix4x4& worldMat,
								Mix::Scene::IPlanter** ppPlanter,
								const wchar_t* pDebugName )
{
	if( ( pFlowerPackage == NULL ) ||
		( numUnit == 0 ) ||
		( units == NULL ) ||
		( ppPlanter == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pFlowerPackage[%s] numUnit[%d] units[%s] ppPlanter[%s]",
			Planter::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( pFlowerPackage ),
			numUnit,
			MIX_LOG_PTR( units ),
			MIX_LOG_PTR( ppPlanter ) );

		return False;
	}

	Mix::Scene::Common::Planter* pPlanter = NULL;

	pPlanter = Planter::CreateInstance( pFlowerPackage, numUnit, units, Mix::Matrix4x4::Identity(), pDebugName );
	if( pPlanter == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", Planter::FAILED_CREATE, Mix::STR_OUTOFMEMORY, MIX_SAFE_NAME( pDebugName ) );
		return False;
	}

	( *ppPlanter ) = pPlanter;

	return True;
}

Boolean Manager::CreateActorModelFromFile( Mix::Scene::IEffectPackage* pEffectPackage, const wchar_t* pFilePath, Mix::Scene::IActorModel** ppModel )
{
	if( ( pEffectPackage == NULL ) ||
		( pFilePath == NULL ) ||
		( ::wcslen( pFilePath ) == 0 ) ||
		( ppModel == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pEffectPackage[%s] pFilePath[%s] ppModel[%s]",
			Mix::Scene::Common::ActorLoader::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_PTR( pEffectPackage ),
			MIX_LOG_STR( pFilePath ),
			MIX_LOG_PTR( ppModel ) );

		return False;
	}

	Mix::IO::IReader* pReader = NULL;
	Mix::Scene::IActorModel* pModel = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"AN^[f̍쐬 : FilePath[%s]", pFilePath );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// f̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pFileManager->CreateFileReader( pFilePath, &pReader ) == False )
	{
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	pModel = m_pActorLoader->Create( static_cast<Mix::Scene::Common::EffectPackage*>( pEffectPackage ), pReader );
	if( pModel == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pReader );
		return False;
	}

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppModel ) = pModel;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateActorModelFromReader( Mix::Scene::IEffectPackage* pEffectPackage, Mix::IO::IReader* pReader, Mix::Scene::IActorModel** ppModel )
{
	if( ( pEffectPackage == NULL ) ||
		( pReader == NULL ) ||
		( ppModel == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pEffectPackage[%s] pReader[%s] ppModel[%s]",
			Mix::Scene::Common::ActorLoader::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_PTR( pEffectPackage ),
			MIX_LOG_PTR( pReader ),
			MIX_LOG_PTR( ppModel ) );

		return False;
	}

	Mix::Scene::IActorModel* pModel = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"AN^[f̍쐬 : FilePath[%s]", pReader->GetFilePath() );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// f̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pModel = m_pActorLoader->Create( static_cast<Mix::Scene::Common::EffectPackage*>( pEffectPackage ), pReader );
	if( pModel == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppModel ) = pModel;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateActorModelFromMemory( Mix::Scene::IEffectPackage* pEffectPackage, const wchar_t* pFilePath, const void* pSrc, UInt32 srcSize, Mix::Scene::IActorModel** ppModel )
{
	if( ( pEffectPackage == NULL ) ||
		( pSrc == NULL ) ||
		( srcSize == 0 ) ||
		( ppModel == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pEffectPackage[%s] pFilePath[%s] pSrc[%s] srcSize[%d] ppModel[%s]",
			Mix::Scene::Common::ActorLoader::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_PTR( pEffectPackage ),
			MIX_SAFE_NAME( pFilePath ),
			MIX_LOG_PTR( pSrc ),
			srcSize,
			MIX_LOG_PTR( ppModel ) );

		return False;
	}

	Mix::IO::MemoryReader* pReader = NULL;
	Mix::Scene::IActorModel* pModel = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( Jn )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"AN^[f̍쐬 : FilePath[%s]", MIX_SAFE_NAME( pFilePath ) );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// f̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pReader = Mix::IO::MemoryReader::CreateInstance( MIX_SAFE_NAME( pFilePath ), pSrc, srcSize );
	if( pReader == NULL )
	{
		MIX_LOG_ERROR( Mix::STR_OUTOFMEMORY );
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	pModel = m_pActorLoader->Create( static_cast<Mix::Scene::Common::EffectPackage*>( pEffectPackage ), pReader );
	if( pModel == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pReader );
		return False;
	}

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( I )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppModel ) = pModel;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateTerrainModelFromFile( Mix::Scene::IEffectPackage* pEffectPackage, const wchar_t* pFilePath, Mix::Scene::ITerrainModel** ppModel )
{
	if( ( pEffectPackage == NULL ) ||
		( pFilePath == NULL ) ||
		( ::wcslen( pFilePath ) == 0 ) ||
		( ppModel == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pEffectPackage[%s] pFilePath[%s] ppModel[%s]",
			Mix::Scene::Common::TerrainLoader::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_PTR( pEffectPackage ),
			MIX_LOG_STR( pFilePath ),
			MIX_LOG_PTR( ppModel ) );

		return False;
	}

	Mix::IO::IReader* pReader = NULL;
	Mix::Scene::ITerrainModel* pModel = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( Jn )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"eCf̍쐬 : FilePath[%s]", pFilePath );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// f̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pFileManager->CreateFileReader( pFilePath, &pReader ) == False )
	{
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	pModel = m_pTerrainLoader->Create( static_cast<Mix::Scene::Common::EffectPackage*>( pEffectPackage ), pReader );
	if( pModel == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pReader );
		return False;
	}

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( I )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppModel ) = pModel;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateTerrainModelFromReader( Mix::Scene::IEffectPackage* pEffectPackage, Mix::IO::IReader* pReader, Mix::Scene::ITerrainModel** ppModel )
{
	if( ( pEffectPackage == NULL ) ||
		( pReader == NULL ) ||
		( ppModel == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pEffectPackage[%s] pReader[%s] ppModel[%s]",
			Mix::Scene::Common::TerrainLoader::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_PTR( pEffectPackage ),
			MIX_LOG_PTR( pReader ),
			MIX_LOG_PTR( ppModel ) );

		return False;
	}

	Mix::Scene::ITerrainModel* pModel = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( Jn )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"eCf̍쐬 : FilePath[%s]", pReader->GetFilePath() );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// f̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pModel = m_pTerrainLoader->Create( static_cast<Mix::Scene::Common::EffectPackage*>( pEffectPackage ), pReader );
	if( pModel == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( I )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppModel ) = pModel;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateTerrainModelFromMemory( Mix::Scene::IEffectPackage* pEffectPackage, const wchar_t* pFilePath, const void* pSrc, UInt32 srcSize, Mix::Scene::ITerrainModel** ppModel )
{
	if( ( pEffectPackage == NULL ) ||
		( pFilePath == NULL ) ||
		( pSrc == NULL ) ||
		( srcSize == 0 ) ||
		( ppModel == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pEffectPackage[%s] pFilePath[%s] pSrc[%s] srcSize[%d] ppModel[%s]",
			Mix::Scene::Common::TerrainLoader::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_PTR( pEffectPackage ),
			MIX_SAFE_NAME( pFilePath ),
			MIX_LOG_PTR( pSrc ),
			srcSize,
			MIX_LOG_PTR( ppModel ) );

		return False;
	}

	Mix::IO::MemoryReader* pReader = NULL;
	Mix::Scene::ITerrainModel* pModel = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( Jn )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"eCf̍쐬 : FilePath[%s]", pFilePath );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// f̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pReader = Mix::IO::MemoryReader::CreateInstance( MIX_SAFE_NAME( pFilePath ), pSrc, srcSize );
	if( pReader == NULL )
	{
		MIX_LOG_ERROR( L"%s", Mix::STR_OUTOFMEMORY );
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	pModel = m_pTerrainLoader->Create( static_cast<Mix::Scene::Common::EffectPackage*>( pEffectPackage ), pReader );
	if( pModel == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pReader );
		return False;
	}

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( I )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppModel ) = pModel;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateLightShaftsCaster( Mix::Scene::ILightShaftsCaster** ppLightShaftsCaster, const wchar_t* pDebugName )
{
	if( ppLightShaftsCaster == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppLightShaftsCaster[%s]",
			LightShaftsCaster::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( ppLightShaftsCaster ) );

		return False;
	}

	Mix::Scene::Common::LightShaftsCaster* pLightShaftsCaster;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pLightShaftsCaster = LightShaftsCaster::CreateInstance();
	if( pLightShaftsCaster == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", LightShaftsCaster::FAILED_CREATE, Mix::STR_OUTOFMEMORY, MIX_SAFE_NAME( pDebugName ) );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppLightShaftsCaster ) = pLightShaftsCaster;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateWaterPool( const Mix::Vector3& halfExtents, Mix::Vector3& worldPos, Mix::Scene::IWaterPool** ppWaterPool, const wchar_t* pDebugName )
{
	if( ppWaterPool == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppWaterPool[%s]",
			WaterPool::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( ppWaterPool ) );

		return False;
	}

	Mix::Scene::Common::DefaultWaterPoolContext* pContext;
	Mix::Scene::Common::WaterPool* pWaterPool;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ftHgReLXg̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pContext = Mix::Scene::Common::DefaultWaterPoolContext::CreateInstance( halfExtents );
	if( pContext == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", WaterPool::FAILED_CREATE, Mix::STR_OUTOFMEMORY, MIX_SAFE_NAME( pDebugName ) );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pWaterPool = Mix::Scene::Common::WaterPool::CreateInstance( pContext );
	if( pWaterPool != NULL )
	{
		Mix::Matrix4x4 worldMat;

		worldMat.SetTranslation( worldPos );

		pWaterPool->SetWorldMatrix( worldMat );
		pWaterPool->Update();
	}
	else
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", WaterPool::FAILED_CREATE, Mix::STR_OUTOFMEMORY, MIX_SAFE_NAME( pDebugName ) );
		MIX_RELEASE( pContext );
		return False;
	}

	MIX_RELEASE( pContext );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppWaterPool ) = pWaterPool;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateWaterPool( Mix::Scene::IWaterPoolContext* pContext, const Mix::Matrix4x4& worldMat, Mix::Scene::IWaterPool** ppWaterPool, const wchar_t* pDebugName )
{
	if( ppWaterPool == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pContext[%s] ppWaterPool[%s]",
			WaterPool::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( pContext ),
			MIX_LOG_PTR( ppWaterPool ) );

		return False;
	}

	Mix::Scene::Common::WaterPool* pWaterPool;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pWaterPool = Mix::Scene::Common::WaterPool::CreateInstance( pContext );
	if( pWaterPool != NULL )
	{
		pWaterPool->SetWorldMatrix( worldMat );
		pWaterPool->Update();
	}
	else
	{
		MIX_LOG_ERROR( L"%s : %s : %s[%s]",
			WaterPool::FAILED_CREATE,
			Mix::STR_OUTOFMEMORY,
			Mix::STR_DEBUGNAME,
			MIX_SAFE_NAME( pDebugName ) );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppWaterPool ) = pWaterPool;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateFloor( Mix::Scene::IFloor** ppFloor, const wchar_t* pDebugName )
{
	if( ppFloor == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppFloor[%s]",
			Floor::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( ppFloor ) );

		return False;
	}

	Mix::Scene::Common::Floor* pFloor = NULL;

	pFloor = Mix::Scene::Common::Floor::CreateInstance();
	if( pFloor == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : %s[%s]",
			Floor::FAILED_CREATE,
			Mix::STR_OUTOFMEMORY,
			Mix::STR_DEBUGNAME,
			MIX_SAFE_NAME( pDebugName ) );

		return False;
	}

	if( pFloor->Initialize( Mix::Vector3::YAxis(), 0.0f, MIX_SAFE_NAME( pDebugName ) ) == False )
	{
		MIX_RELEASE( pFloor );
		return False;
	}

	( *ppFloor ) = pFloor;

	return True;
}

Boolean Manager::CreateFloor( const Mix::Vector3& normal, Float32 offset, Mix::Scene::IFloor** ppFloor, const wchar_t* pDebugName )
{
	if( ppFloor == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppFloor[%s]",
			Floor::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( ppFloor ) );

		return False;
	}

	Mix::Scene::Common::Floor* pFloor = NULL;

	pFloor = Mix::Scene::Common::Floor::CreateInstance();
	if( pFloor == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : %s[%s]",
			Floor::FAILED_CREATE,
			Mix::STR_OUTOFMEMORY,
			Mix::STR_DEBUGNAME,
			MIX_SAFE_NAME( pDebugName ) );

		return False;
	}

	if( pFloor->Initialize( normal, offset, MIX_SAFE_NAME( pDebugName ) ) == False )
	{
		MIX_RELEASE( pFloor );
		return False;
	}

	( *ppFloor ) = pFloor;

	return True;
}

Boolean Manager::CreateBasisGhost( Mix::Dynamics::IShape* pShape, Boolean bCollide, Mix::Scene::IGhost** ppGhost, const wchar_t* pDebugName )
{
	if( ( pShape == NULL ) ||
		( ppGhost == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : %s[%s] : pShape[%s] ppGhost[%s]",
			BasisGhost::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			Mix::STR_DEBUGNAME,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( pShape ),
			MIX_LOG_PTR( ppGhost ) );

		return False;
	}

	Mix::Scene::Common::Ghost* pGhost = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( bCollide == True )
	{
		Mix::Scene::Common::CollideGhost* pBasisGhost = Mix::Scene::Common::CollideGhost::CreateInstance();

		if( pBasisGhost != NULL )
		{
			if( pBasisGhost->Initialize( pShape, MIX_SAFE_NAME( pDebugName ) ) == True )
			{
				pGhost = pBasisGhost;
			}
			else
			{
				MIX_RELEASE( pBasisGhost );
				return False;
			}
		}
		else
		{
			MIX_LOG_ERROR( L"%s : %s : %s[Basis] %s[%s] Collide[]",
				BasisGhost::FAILED_CREATE,
				Mix::STR_OUTOFMEMORY,
				Mix::STR_TYPE,
				Mix::STR_DEBUGNAME,
				MIX_SAFE_NAME( pDebugName ) );

			return False;
		}
	}
	else
	{
		Mix::Scene::Common::NoCollideGhost* pBasisGhost = Mix::Scene::Common::NoCollideGhost::CreateInstance();

		if( pBasisGhost != NULL )
		{
			if( pBasisGhost->Initialize( pShape, MIX_SAFE_NAME( pDebugName ) ) == True )
			{
				pGhost = pBasisGhost;
			}
			else
			{
				MIX_RELEASE( pBasisGhost );
				return False;
			}
		}
		else
		{
			MIX_LOG_ERROR( L"%s : %s : %s[Basis] %s[%s] Collide[~]",
				BasisGhost::FAILED_CREATE,
				Mix::STR_OUTOFMEMORY,
				Mix::STR_TYPE,
				Mix::STR_DEBUGNAME,
				MIX_SAFE_NAME( pDebugName ) );

			return False;
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppGhost ) = pGhost;

	return True;
}

Boolean Manager::CreateKinematicGhost( Float32 height, Float32 radius, Float32 stepHeight, Mix::Scene::IGhost** ppGhost, const wchar_t* pDebugName )
{
	if( ( height <= 0.0f ) ||
		( radius <= 0.0f ) ||
		( stepHeight < 0.0f ) ||
		( ppGhost == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : %s[%s] : height[%f > 0.0f] radius[%f > 0.0f] stepHeight[%f >= 0.0f] ppGhost[%s]",
			KinematicGhost::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			Mix::STR_DEBUGNAME,
			MIX_SAFE_NAME( pDebugName ),
			height,
			radius,
			stepHeight,
			MIX_LOG_PTR( ppGhost ) );

		return False;
	}

	Mix::Scene::Common::KinematicGhost* pGhost = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pGhost = Mix::Scene::Common::KinematicGhost::CreateInstance();
	if( pGhost != NULL )
	{
		if( pGhost->Initialize( height, radius, stepHeight, MIX_SAFE_NAME( pDebugName ) ) == False )
		{
			MIX_RELEASE( pGhost );
			return False;
		}
	}
	else
	{
		MIX_LOG_ERROR( L"%s : %s : %s[%s]",
			KinematicGhost::FAILED_CREATE,
			Mix::STR_OUTOFMEMORY,
			Mix::STR_DEBUGNAME,
			MIX_SAFE_NAME( pDebugName ) );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppGhost ) = pGhost;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateMaterialFromFile( Mix::Scene::IEffectPackage* pEffectPackage, const wchar_t* pFilePath, Mix::Scene::IMaterial** ppMaterial )
{
	if( ( pEffectPackage == NULL ) ||
		( MIX_STR_LENGTH( pFilePath ) == 0 ) ||
		( ppMaterial == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pEffectPackage[%s] pFilePath[%s] ppMaterial[%s]",
			Mix::Scene::Standard::Common::DefaultMaterial::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_PTR( pEffectPackage ),
			MIX_SAFE_STR( pFilePath ),
			MIX_LOG_PTR( ppMaterial ) );

		return False;
	}

	Mix::IO::IReader* pReader = NULL;
	Mix::Scene::IMaterial* pMaterial = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"}eA̍쐬 : FilePath[%s] Type[Standard( Default )]", pFilePath );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// t@C[_[쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pFileManager->CreateFileReader( pFilePath, &pReader ) == False )
	{
		MIX_LOG_ERROR( L"%s : %s : FilePath[%s]",
			Mix::Scene::Standard::Common::DefaultMaterial::FAILED_CREATE,
			Mix::STR_FILENOTFOUND,
			pFilePath );

		MIX_LOG_INFO_SECT_END();

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pMaterial = CreateMaterial( static_cast<Mix::Scene::Common::EffectPackage*>( pEffectPackage ),
								pReader,
								True,
								L"FilePath",
								pFilePath );

	if( pMaterial == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pReader );
		return False;
	}

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppMaterial ) = pMaterial;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateMaterialFromReader( Mix::Scene::IEffectPackage* pEffectPackage, Mix::IO::IReader* pReader, Mix::Scene::IMaterial** ppMaterial )
{
	if( ( pEffectPackage == NULL ) ||
		( pReader == NULL ) ||
		( ppMaterial == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pEffectPackage[%s] pReader[%s] ppMaterial[%s]",
			Mix::Scene::Standard::Common::DefaultMaterial::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_PTR( pEffectPackage ),
			MIX_LOG_PTR( pReader ),
			MIX_LOG_PTR( ppMaterial ) );

		return False;
	}

	Mix::Scene::IMaterial* pMaterial = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"}eA̍쐬 : FilePath[%s] Type[Standard( Default )]", pReader->GetFilePath() );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// }eA쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pMaterial = CreateMaterial( static_cast<Mix::Scene::Common::EffectPackage*>( pEffectPackage ),
								pReader,
								True,
								L"FilePath",
								pReader->GetFilePath() );

	if( pMaterial == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppMaterial ) = pMaterial;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateMaterialFromMemory( Mix::Scene::IEffectPackage* pEffectPackage, const wchar_t* pFilePath, const void* pSrc, UInt32 srcSize, Mix::Scene::IMaterial** ppMaterial )
{
	if(	( pEffectPackage == NULL ) ||
		( MIX_STR_LENGTH( pFilePath ) == 0 ) ||
		( pSrc == NULL ) ||
		( srcSize == 0 ) ||
		( ppMaterial == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pEffectPackage[%s] pFilePath[%s] : pSrc[%s] srcSize[%d] ppMaterial[%s]",
			Mix::Scene::Standard::Common::DefaultMaterial::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_PTR( pEffectPackage ),
			MIX_SAFE_STR( pFilePath ),
			MIX_LOG_PTR( pSrc ),
			srcSize,
			MIX_LOG_PTR( ppMaterial ) );

		return False;
	}

	Mix::IO::MemoryReader* pReader = NULL;
	Mix::Scene::IMaterial* pMaterial = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"}eA̍쐬 : FilePath[%s] Type[Standard( Default )]", MIX_SAFE_NAME( pFilePath ) );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// }eA쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pReader = Mix::IO::MemoryReader::CreateInstance( pFilePath, pSrc, srcSize );
	if( pReader == NULL )
	{
		MIX_LOG_ERROR( Mix::STR_OUTOFMEMORY );
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	pMaterial = CreateMaterial( static_cast<Mix::Scene::Common::EffectPackage*>( pEffectPackage ),
								pReader,
								True,
								L"FilePath",
								pFilePath );

	if( pMaterial == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pReader );
		return False;
	}

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppMaterial ) = pMaterial;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateMotionFromFile( const wchar_t* pFilePath, Mix::Scene::IMotion** ppMotion )
{
	if( ( pFilePath == NULL ) ||
		( ::wcslen( pFilePath ) == 0 ) ||
		( ppMotion == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pFilePath[%s], ppMotion[%s]",
			Mix::Scene::Common::Motion::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_STR( pFilePath ),
			MIX_LOG_PTR( ppMotion ) );

		return False;
	}

	Mix::IO::IReader* pReader = NULL;
	Mix::Scene::Common::Motion* pMotion = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( Jn )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"[V̍쐬 : FilePath[%s]", pFilePath );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// [_[̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pFileManager->CreateFileReader( pFilePath, &pReader ) == False )
	{
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pMotion = Mix::Scene::Common::Motion::CreateInstance( pReader, True, L"FilePath", pFilePath );
	if( pMotion == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pReader );
		return False;
	}

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( I )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppMotion ) = pMotion;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return False;
}

Boolean Manager::CreateMotionFromReader( Mix::IO::IReader* pReader, Mix::Scene::IMotion** ppMotion )
{
	if( ( pReader == NULL ) ||
		( ppMotion == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pReader[%s], ppMotion[%s]",
			Mix::Scene::Common::Motion::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_PTR( pReader ),
			MIX_LOG_PTR( ppMotion ) );

		return False;
	}

	Mix::Scene::Common::Motion* pMotion = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( Jn )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"[V̍쐬 : FilePath[%s]", pReader->GetFilePath() );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// [V̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pMotion = Mix::Scene::Common::Motion::CreateInstance( pReader, True, L"FilePath", pReader->GetFilePath() );
	if( pMotion == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( I )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppMotion ) = pMotion;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return False;
}

Boolean Manager::CreateMotionFromMemory( const wchar_t* pFilePath, const void* pSrc, UInt32 srcSize, Mix::Scene::IMotion** ppMotion )
{
	if( ( pFilePath == NULL ) ||
		( ::wcslen( pFilePath ) == 0 ) ||
		( pSrc == NULL ) ||
		( srcSize == 0 ) ||
		( ppMotion == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pFilePath[%s] pSrc[%s] srcSize[%d] ppMotion[%s]",
			Mix::Scene::Common::Motion::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_STR( pFilePath ),
			MIX_LOG_PTR( pSrc ),
			srcSize,
			MIX_LOG_PTR( ppMotion ) );

		return False;
	}

	Mix::IO::MemoryReader* pReader = NULL;
	Mix::Scene::Common::Motion* pMotion = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( Jn )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"[V̍쐬 : FilePath[%s]", pFilePath );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// [V̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pReader = Mix::IO::MemoryReader::CreateInstance( pFilePath, pSrc, srcSize );
	if( pReader == NULL )
	{
		MIX_LOG_ERROR( Mix::STR_OUTOFMEMORY );
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	pMotion = Mix::Scene::Common::Motion::CreateInstance( pReader, True, L"FilePath", pFilePath );
	if( pMotion == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pReader );
		return False;
	}

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( I )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppMotion ) = pMotion;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return False;
}

const Mix::Vector4& Manager::Debug_GetDrawColor( Mix::Scene::DEBUG_DRAW_COLOR_TYPE type ) const
{
#ifdef _DEBUG
	return Mix::Scene::Common::Debug::GetDrawColor( type );
#else //_DEBUG
	return Mix::Vector4::Zero();
#endif //_DEBUG
}

void Manager::Debug_SetDrawColor( Mix::Scene::DEBUG_DRAW_COLOR_TYPE type, const Mix::Vector4& color )
{
#ifdef _DEBUG
	return Mix::Scene::Common::Debug::SetDrawColor( type, color );
#endif //_DEBUG
}

}}}
