#include "Mix/Tool/Win32/Core/Dynamics/Impl/HingeJoint.h"

#include "Mix/Tool/Win32/Core/Graphics/LineHelper.h"

#include "Mix/Tool/Win32/Core/Dynamics/Impl/Utility.h"
#include "Mix/Tool/Win32/Core/Dynamics/Impl/Types.h"
#include "Mix/Tool/Win32/Core/Dynamics/Impl/World.h"
#include "Mix/Tool/Win32/Core/Dynamics/Impl/RigidBody.h"

#include "btBulletDynamicsCommon.h"

namespace Mix{ namespace Tool{ namespace Win32{ namespace Dynamics{ namespace Impl{
		
HingeJoint::HingeJoint( void ) :
m_pConstraint( NULL ),
m_pWorld( NULL ),
m_pRigidBodyA( NULL ),
m_pRigidBodyB( NULL ),
m_bInitalRigidBodyTransform( false ),
m_InitalRigidBodyRotA( 0.0f, 0.0f, 0.0f, 1.0f ),
m_InitalRigidBodyPosA( 0.0f, 0.0f, 0.0f ),
m_InitalRigidBodyRotB( 0.0f, 0.0f, 0.0f, 1.0f ),
m_InitalRigidBodyPosB( 0.0f, 0.0f, 0.0f ),
m_PivotA( 0.0f, 0.0f, 0.0f ),
m_PivotB( 0.0f, 0.0f, 0.0f ),
m_bCollisionDisabled( false ),
m_Axis( 0.0f, 0.0f, 1.0f ),
m_NormAxis( 0.0f, 0.0f, 1.0f ),
m_LowerLimit( -1.0f ),
m_UpperLimit( +1.0f ),
m_ParamFlags( 0 )
{
	for( unsigned int i = 0; i < HingeJoint::PT_MAX; i++ )
	{
		m_ParamValues[i] = 0.0f;
	}
}

HingeJoint::HingeJoint(	const D3DXVECTOR3& axis,
						float lowerLimit,
						float upperLimit,
						bool collisionDisabled,
						float limitSpring,
						float limitDamper ) :
m_pConstraint( NULL ),
m_pWorld( NULL ),
m_pRigidBodyA( NULL ),
m_pRigidBodyB( NULL ),
m_bInitalRigidBodyTransform( false ),
m_InitalRigidBodyRotA( 0.0f, 0.0f, 0.0f, 1.0f ),
m_InitalRigidBodyPosA( 0.0f, 0.0f, 0.0f ),
m_InitalRigidBodyRotB( 0.0f, 0.0f, 0.0f, 1.0f ),
m_InitalRigidBodyPosB( 0.0f, 0.0f, 0.0f ),
m_PivotA( 0.0f, 0.0f, 0.0f ),
m_PivotB( 0.0f, 0.0f, 0.0f ),
m_bCollisionDisabled( collisionDisabled ),
m_Axis( axis ),
m_LowerLimit( lowerLimit ),
m_UpperLimit( upperLimit ),
m_ParamFlags( 0 )
{
	D3DXVec3Normalize( &m_NormAxis, &m_Axis );

	for( int i = 0; i < HingeJoint::PT_MAX; i++ )
	{
		m_ParamValues[i] = 0.0f;
	}

	SetLimitSpring( limitSpring );
	SetLimitDamper( limitDamper );
}

HingeJoint::~HingeJoint( void )
{
	MIX_DELETE( m_pConstraint );
}

void HingeJoint::SetWorldPtr( Mix::Tool::Win32::Dynamics::Impl::World* pWorld )
{
	m_pWorld = pWorld;
}

const D3DXVECTOR3& HingeJoint::GetNormalizeAxis( void ) const
{
	return m_NormAxis;
}

const D3DXVECTOR3& HingeJoint::GetAxis( void ) const
{
	return m_Axis;
}

void HingeJoint::SetAxis( const D3DXVECTOR3& axis )
{
	m_Axis = axis;
	D3DXVec3Normalize( &m_NormAxis, &m_Axis );

	UpdateFrames();
}

float HingeJoint::GetLowerLimit( void ) const
{
	return m_LowerLimit;
}

void HingeJoint::SetLowerLimit( float rad )
{
	m_LowerLimit = rad;

	UpdateLimit();
}

float HingeJoint::GetUpperLimit( void ) const
{
	return m_UpperLimit;
}

void HingeJoint::SetUpperLimit( float rad )
{
	m_UpperLimit = rad;

	UpdateLimit();
}

float HingeJoint::GetLimitSpring( void ) const
{
	return GetParam( HingeJoint::PT_STOP_CFM, HingeJoint::PA_NONE );
}

void HingeJoint::SetLimitSpring( float value )
{
	if( value > SIMD_EPSILON )
	{
		SetParam( HingeJoint::PT_STOP_CFM, HingeJoint::PA_NONE, ( value > ( 1.0f - SIMD_EPSILON ) )? 1.0f : value );
	}
	else
	{
		ResetParam( HingeJoint::PT_STOP_CFM, HingeJoint::PA_NONE );
	}
}

float HingeJoint::GetLimitDamper( void ) const
{
	return GetParam( HingeJoint::PT_STOP_ERP, HingeJoint::PA_NONE );
}

void HingeJoint::SetLimitDamper( float value )
{
	if( value > SIMD_EPSILON )
	{
		SetParam( HingeJoint::PT_STOP_ERP, HingeJoint::PA_NONE, ( value > ( 1.0f - SIMD_EPSILON ) )? 1.0f : value );
	}
	else
	{
		ResetParam( HingeJoint::PT_STOP_ERP, HingeJoint::PA_NONE );
	}
}

float HingeJoint::GetNormalSpring( void ) const
{
	return GetParam( HingeJoint::PT_NORMAL_CFM, HingeJoint::PA_NONE );
}

void HingeJoint::SetNormalSpring( float value )
{
	if( value > SIMD_EPSILON )
	{
		SetParam( HingeJoint::PT_NORMAL_CFM, HingeJoint::PA_NONE, ( value > ( 1.0f - SIMD_EPSILON ) )? 1.0f : value );
	}
	else
	{
		ResetParam( HingeJoint::PT_NORMAL_CFM, HingeJoint::PA_NONE );
	}
}

void HingeJoint::ResetInitalRigidBodyTransform( void )
{
	m_bInitalRigidBodyTransform = false;

	D3DXQuaternionIdentity( &m_InitalRigidBodyRotA );
	m_InitalRigidBodyPosA = D3DXVECTOR3( 0.0f, 0.0f, 0.0f );

	D3DXQuaternionIdentity( &m_InitalRigidBodyRotB );
	m_InitalRigidBodyPosB = D3DXVECTOR3( 0.0f, 0.0f, 0.0f );
}

void HingeJoint::SetInitalRigidBodyTransform(	const D3DXQUATERNION& rotA,
												const D3DXVECTOR3& posA,
												const D3DXQUATERNION& rotB,
												const D3DXVECTOR3& posB )
{
	m_bInitalRigidBodyTransform = true;

	m_InitalRigidBodyRotA = rotA;
	m_InitalRigidBodyPosA = posA;

	m_InitalRigidBodyRotB = rotB;
	m_InitalRigidBodyPosB = posB;
}

bool HingeJoint::SetFrame(	Mix::Tool::Win32::Dynamics::RigidBody* pRigidBodyA,
								const D3DXVECTOR3& pivotA )
{
	bool ret = false;

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// 쐬
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////

	BeginRefresh();

	MIX_DELETE( m_pConstraint );
	m_pRigidBodyA = NULL;
	m_pRigidBodyB = NULL;

	if( ( pRigidBodyA != NULL ) &&
		( pRigidBodyA->GetShapeHandleCount() > 0 ) )
	{
		RigidBody::STATUS statusA = pRigidBodyA->GetStatus();

		if( ( statusA == RigidBody::DEFAULT ) ||
			( statusA == RigidBody::KINEMATIC ) )
		{
			Mix::Tool::Win32::Dynamics::Impl::RigidBody* pImplRigidBodyA = static_cast<Mix::Tool::Win32::Dynamics::Impl::RigidBody*>( pRigidBodyA );
			btRigidBody* pBulletRigidBodyA = pImplRigidBodyA->Bullet_GetRigidBodyPtr();

			if( pBulletRigidBodyA != NULL )
			{
				btTransform frameA;
				btTransform frameB;

				Bullet_ComputeFrames(	pBulletRigidBodyA,
										&btVector3( pivotA.x, pivotA.y, pivotA.z ),
										&btVector3( m_NormAxis.x, m_NormAxis.y, m_NormAxis.z ),
										&frameA,
										&frameB );

				m_pConstraint = new btHingeConstraint( *pBulletRigidBodyA, btTypedConstraint::getFixedBody(), frameA, frameB );

				if( m_pConstraint != NULL )
				{
					const btTransform& frameB = m_pConstraint->getBFrame();
					const btVector3& pivotB = frameB.getOrigin();

					m_pConstraint->buildJacobian();

					m_pRigidBodyA = pRigidBodyA;
					m_pRigidBodyB = NULL;

					m_PivotA = pivotA;
					m_PivotB = D3DXVECTOR3( pivotB.x(), pivotB.y(), pivotB.z() );

					ret = true;
				}
			}
		}
	}

	UpdateLimit();
	UpdateParams();

	EndRefresh();

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////

	return ret;
}

bool HingeJoint::SetFrame(	Mix::Tool::Win32::Dynamics::RigidBody* pRigidBodyA,
								Mix::Tool::Win32::Dynamics::RigidBody* pRigidBodyB,
								const D3DXVECTOR3& pivotA,
								const D3DXVECTOR3& pivotB )
{
	bool ret = false;

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// 쐬
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////

	BeginRefresh();

	MIX_DELETE( m_pConstraint );
	m_pRigidBodyA = NULL;
	m_pRigidBodyB = NULL;

	if( ( pRigidBodyA != NULL ) &&
		( pRigidBodyB != NULL ) &&
		( pRigidBodyA->GetShapeHandleCount() > 0 ) &&
		( pRigidBodyB->GetShapeHandleCount() > 0 ) )
	{
		Mix::Tool::Win32::Dynamics::RigidBody::STATUS statusA = pRigidBodyA->GetStatus();
		Mix::Tool::Win32::Dynamics::RigidBody::STATUS statusB = pRigidBodyB->GetStatus();

		if( ( statusA == Mix::Tool::Win32::Dynamics::RigidBody::DEFAULT ) ||
			( statusB == Mix::Tool::Win32::Dynamics::RigidBody::DEFAULT ) ||
			( statusA == Mix::Tool::Win32::Dynamics::RigidBody::KINEMATIC ) ||
			( statusB == Mix::Tool::Win32::Dynamics::RigidBody::KINEMATIC ) )
		{
			Mix::Tool::Win32::Dynamics::Impl::RigidBody* pImplRigidBodyA = static_cast<Mix::Tool::Win32::Dynamics::Impl::RigidBody*>( pRigidBodyA );
			Mix::Tool::Win32::Dynamics::Impl::RigidBody* pImplRigidBodyB = static_cast<Mix::Tool::Win32::Dynamics::Impl::RigidBody*>( pRigidBodyB );
			btRigidBody* pBulletRigidBodyA = pImplRigidBodyA->Bullet_GetRigidBodyPtr();
			btRigidBody* pBulletRigidBodyB = pImplRigidBodyB->Bullet_GetRigidBodyPtr();

			if( ( pBulletRigidBodyA != NULL ) &&
				( pBulletRigidBodyB != NULL ) )
			{
				btTransform frameA;
				btTransform frameB;

				Bullet_ComputeFrames(	pBulletRigidBodyA,
										pBulletRigidBodyB,
										&btVector3( pivotA.x, pivotA.y, pivotA.z ),
										&btVector3( pivotB.x, pivotB.y, pivotB.z ),
										&btVector3( m_NormAxis.x, m_NormAxis.y, m_NormAxis.z ),
										&frameA,
										&frameB );

				m_pConstraint = new btHingeConstraint( *pBulletRigidBodyA, *pBulletRigidBodyB, frameA, frameB );

				if( m_pConstraint != NULL )
				{
					m_pConstraint->buildJacobian();

					m_pRigidBodyA = pRigidBodyA;
					m_pRigidBodyB = pRigidBodyB;

					m_PivotA = pivotA;
					m_PivotB = pivotB;

					ret = true;
				}
			}
		}
	}

	UpdateLimit();
	UpdateParams();

	EndRefresh();

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////

	return ret;
}

const Mix::Tool::Win32::Dynamics::RigidBody* HingeJoint::GetRigidBodyA( void ) const
{
	return m_pRigidBodyA;
}

D3DXVECTOR3 HingeJoint::GetPivotA( void )
{
	return m_PivotA;
}

void HingeJoint::SetPivotA( const D3DXVECTOR3& pivot )
{
	m_PivotA = pivot;

	UpdatePivot();
}

bool HingeJoint::GetCollisionDisabled( void ) const
{
	return m_bCollisionDisabled;
}

void HingeJoint::SetCollisionDisabled( bool state )
{
	m_bCollisionDisabled = state;

	Refresh();

	if( m_pRigidBodyA != NULL ) { m_pRigidBodyA->Activate(); }
	if( m_pRigidBodyB != NULL ) { m_pRigidBodyB->Activate(); }
}

const Mix::Tool::Win32::Dynamics::RigidBody* HingeJoint::GetRigidBodyB( void ) const
{
	return m_pRigidBodyB;
}

D3DXVECTOR3 HingeJoint::GetPivotB( void )
{
	return m_PivotB;
}

void HingeJoint::SetPivotB( const D3DXVECTOR3& pivot )
{
	m_PivotB = pivot;

	UpdatePivot();
}

bool HingeJoint::IsParamSupported( int type, int axis ) const
{
	if( ( HingeJoint::PT_FIRST <= type ) &&
		( HingeJoint::PT_LAST >= type ) &&
		( axis == HingeJoint::PA_NONE ) )
	{
		return true;
	}

	return false;
}

bool HingeJoint::IsParamEnabled( int type, int axis ) const
{
	if( IsParamSupported( type, axis ) == false )
	{
		return false;
	}

	unsigned int bit = 1 << type;

	return ( MIX_TEST_BIT( m_ParamFlags, bit ) == bit );
}

float HingeJoint::GetParam( int type, int axis ) const
{
	if( IsParamSupported( type, axis ) == false )
	{
		return 0.0f;
	}

	return m_ParamValues[type];
}

void HingeJoint::SetParam( int type, int axis, float value )
{
	if( IsParamSupported( type, axis ) == true )
	{
		MIX_SET_BIT( m_ParamFlags, ( 1 << type ) );
		m_ParamValues[type] = value;

		if( m_pConstraint != NULL )
		{
			m_pConstraint->setParam( type, value, axis );
		}
	}
}

void HingeJoint::ResetParam( int type, int axis )
{
	if( IsParamSupported( type, axis ) == true )
	{
		unsigned int bit = 1 << type;

		if( MIX_TEST_BIT( m_ParamFlags, bit ) == bit )
		{
			MIX_RESET_BIT( m_ParamFlags, bit );
			m_ParamValues[type] = 0.0f;

			if( m_pConstraint != NULL )
			{
				if( ( m_pRigidBodyA != NULL ) &&
					( m_pRigidBodyB == NULL ) )
				{
					SetFrame( m_pRigidBodyA, m_PivotA );
				}
				else if(	( m_pRigidBodyA != NULL ) &&
							( m_pRigidBodyB != NULL ) )
				{
					SetFrame( m_pRigidBodyA, m_pRigidBodyB, m_PivotA, m_PivotB );
				}
			}
		}
	}
}

void HingeJoint::Draw(	Mix::Tool::Win32::Graphics::LineHelper* pLineHelper,
						const D3DXVECTOR4& axisColor,
						const D3DXVECTOR4& ballColor,
						const D3DXVECTOR4& limitColor,
						float scale )
{
	if( ( pLineHelper == NULL ) ||
		( m_pConstraint == NULL ) )
	{
		return;
	}

	const btTransform& frameA = m_pConstraint->getAFrame();
	const btTransform& frameB = m_pConstraint->getBFrame();
	const btVector3& pivotA = frameA.getOrigin();
	const btVector3& pivotB = frameB.getOrigin();
	const btRigidBody* pRigidBodyA = &( m_pConstraint->getRigidBodyA() );
	const btRigidBody* pRigidBodyB = &( m_pConstraint->getRigidBodyB() );

	D3DXVECTOR3 pointA;
	D3DXVECTOR3 pointB;

	Mix::Tool::Win32::Geometry::SPHERE sphere;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pLineHelper->IdentityMatrix();

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pLineHelper->SetColor( axisColor );

	if( pRigidBodyA != &( btTypedConstraint::getFixedBody() ) )
	{
		btTransform tr;
		btQuaternion rot;
		btVector3 pos;
		D3DXMATRIX mat;

		if( pRigidBodyA->isKinematicObject() == true )
		{
			pRigidBodyA->getMotionState()->getWorldTransform( tr );
		}
		else
		{
			tr = pRigidBodyA->getWorldTransform();
		}

		rot = tr.getRotation();
		pos = tr.getOrigin();

		D3DXMatrixRotationQuaternion( &mat, &D3DXQUATERNION( rot.x(), rot.y(), rot.z(), rot.w() ) );
		mat._41 = pos.x(); mat._42 = pos.y(); mat._43 = pos.z(); mat._44 = 1.0f;

		D3DXVec3TransformCoord( &pointA, &D3DXVECTOR3( pivotA.x(), pivotA.y(), pivotA.z() ), &mat );

		pLineHelper->AddPoints( pointA, D3DXVECTOR3( pos.x(), pos.y(), pos.z() ) );
	}
	else
	{
		pointA = D3DXVECTOR3( pivotA.x(), pivotA.y(), pivotA.z() );
	}

	if( pRigidBodyB != &( btTypedConstraint::getFixedBody() ) )
	{
		btTransform tr;
		btQuaternion rot;
		btVector3 pos;
		D3DXMATRIX mat;
		D3DXVECTOR3 point;

		if( pRigidBodyB->isKinematicObject() == true )
		{
			pRigidBodyB->getMotionState()->getWorldTransform( tr );
		}
		else
		{
			tr = pRigidBodyB->getWorldTransform();
		}

		rot = tr.getRotation();
		pos = tr.getOrigin();

		D3DXMatrixRotationQuaternion( &mat, &D3DXQUATERNION( rot.x(), rot.y(), rot.z(), rot.w() ) );
		mat._41 = pos.x(); mat._42 = pos.y(); mat._43 = pos.z(); mat._44 = 1.0f;

		D3DXVec3TransformCoord( &pointB, &D3DXVECTOR3( pivotB.x(), pivotB.y(), pivotB.z() ), &mat );

		pLineHelper->AddPoints( pointB, D3DXVECTOR3( pos.x(), pos.y(), pos.z() ) );
	}
	else
	{
		pointB = D3DXVECTOR3( pivotB.x(), pivotB.y(), pivotB.z() );
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// s{bg
	////////////////////////////////////////////////////////////////////////////////////////////////////

	sphere.pos = ( pointA + pointB ) * 0.5f;
	sphere.radius = max( Mix::Tool::Win32::Dynamics::Impl::CONSTRAINT_MIN_JOINT_RADIUS, D3DXVec3Length( &( pointA - pointB ) ) * 0.5f );

	pLineHelper->SetColor( ballColor );
	pLineHelper->AddSphere( sphere );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ~bg
	////////////////////////////////////////////////////////////////////////////////////////////////////

	float minAngle = m_pConstraint->getLowerLimit();
	float maxAngle = m_pConstraint->getUpperLimit();

	if( minAngle != maxAngle )
	{
		const float stepDegrees = 10.0f;
		const float step = stepDegrees * SIMD_RADS_PER_DEG;

		float radiusA = scale;
		float radiusB = scale;

		btTransform tr = m_pConstraint->getRigidBodyB().getCenterOfMassTransform() * m_pConstraint->getBFrame();

		if( MIX_TEST_BIT( pRigidBodyB->getCollisionFlags(), btCollisionObject::CF_KINEMATIC_OBJECT ) == btCollisionObject::CF_KINEMATIC_OBJECT )
		{
			btTransform mstr;
			pRigidBodyB->getMotionState()->getWorldTransform( mstr );
			tr = mstr * m_pConstraint->getBFrame();
		}
		else
		{
			tr = pRigidBodyB->getCenterOfMassTransform() * m_pConstraint->getBFrame();
		}

		D3DXVECTOR3 center = ToD3DXVector( tr.getOrigin() );
		D3DXVECTOR3 normal = ToD3DXVector( tr.getBasis().getColumn( 2 ) );
		D3DXVECTOR3 axis = ToD3DXVector( tr.getBasis().getColumn( 0 ) );
		bool bDrawSect = ( minAngle <= maxAngle );

		if( bDrawSect == false )
		{
			minAngle = 0.0f;
			maxAngle = SIMD_2_PI;
		}

		////////////////////////////////////////////////////////////////////////////////////////////////////

		int stepCount = max( 1, static_cast<int>( ( maxAngle - minAngle ) / step ) );
		const D3DXVECTOR3& vx = axis;
		D3DXVECTOR3 vy;
		D3DXVECTOR3 prev;

		D3DXVec3Cross( &vy, &normal, &axis );

		pLineHelper->SetColor( limitColor );
		pLineHelper->IdentityMatrix();
	
		prev = center + radiusA * vx * ::cosf( minAngle ) + radiusB * vy * ::sinf( minAngle );
		if( bDrawSect == true )
		{
			pLineHelper->AddPoints( center, prev );
		}

		for( int i = 1; i <= stepCount; i++ )
		{
			float angle = minAngle + ( maxAngle - minAngle ) * static_cast<float>( i ) / static_cast<float>( stepCount );
			D3DXVECTOR3 next = center + radiusA * vx * ::cosf( angle ) + radiusB * vy * ::sinf( angle );

			pLineHelper->AddPoints( prev, next );

			prev = next;
		}

		if( bDrawSect == true )
		{
			pLineHelper->AddPoints( center, prev );
		}
	}
}

void* HingeJoint::Bullet_GetTypedConstraintPtr( void )
{
	return m_pConstraint;
}

void HingeJoint::BeginRefresh( void )
{
	if( m_pWorld != NULL )
	{
		m_pWorld->DetachJoint( this, false );
	}
}

void HingeJoint::EndRefresh( void )
{
	if( m_pWorld != NULL )
	{
		m_pWorld->AttachJoint( this, false );
	}
}

void HingeJoint::Refresh( void )
{
	if( m_pWorld != NULL )
	{
		m_pWorld->DetachJoint( this, false );
		m_pWorld->AttachJoint( this, false );
	}
}

void HingeJoint::UpdateLimit( void )
{
	if( m_pConstraint != NULL )
	{
		m_pConstraint->setLimit( m_LowerLimit, m_UpperLimit );
	}
}

void HingeJoint::UpdatePivot( void )
{
	if( m_pConstraint != NULL )
	{
		btTransform frameA = m_pConstraint->getAFrame();
		btTransform frameB = m_pConstraint->getBFrame();

		frameA.setOrigin( btVector3( m_PivotA.x, m_PivotA.y, m_PivotA.z ) );
		frameB.setOrigin( btVector3( m_PivotB.x, m_PivotB.y, m_PivotB.z ) );

		m_pConstraint->setFrames( frameA, frameB );
	}
}

void HingeJoint::UpdateFrames( void )
{
	if( ( m_pRigidBodyA != NULL ) &&
		( m_pRigidBodyB == NULL ) )
	{
		D3DXVECTOR3 pivotA = m_PivotA;

		SetFrame( m_pRigidBodyA, pivotA );
	}
	else if(	( m_pRigidBodyA != NULL ) &&
				( m_pRigidBodyB != NULL ) )
	{
		D3DXVECTOR3 pivotA = m_PivotA;
		D3DXVECTOR3 pivotB = m_PivotB;

		SetFrame( m_pRigidBodyA, m_pRigidBodyB, pivotA, pivotB );
	}
}

void HingeJoint::UpdateParams( void )
{
	if( m_pConstraint != NULL )
	{
		for( unsigned int i = 0; i < HingeJoint::PT_MAX; i++ )
		{
			int num = static_cast<int>( i );
			unsigned int bit = ( 1 << i );

			if( MIX_TEST_BIT( m_ParamFlags, bit ) == bit )
			{
				m_pConstraint->setParam( num, m_ParamValues[i] );
			}
		}
	}
}

void HingeJoint::Bullet_ComputeFrames(	const btRigidBody* rbA,
										const btVector3* pivotA,
										const btVector3* axis,
										btTransform* frameA,
										btTransform* frameB )
{
	////////////////////////////////////////////////////////////////////////////////////////////////////
	// At[( Wbh{fB`̋ )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	const btVector3& rbAxisA0 = *axis;
	btVector3 rbAxisA1;
	btVector3 rbAxisA2;

	btPlaneSpace1( rbAxisA0, rbAxisA1, rbAxisA2 );

	frameA->getOrigin() = *pivotA;
	frameA->getBasis().setValue(	rbAxisA1.getX(), rbAxisA2.getX(), rbAxisA0.getX(),
									rbAxisA1.getY(), rbAxisA2.getY(), rbAxisA0.getY(),
									rbAxisA1.getZ(), rbAxisA2.getZ(), rbAxisA0.getZ() );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// Bt[( [h )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_bInitalRigidBodyTransform == true )
	{
		btTransform tr;

		tr.setRotation( btQuaternion( m_InitalRigidBodyRotA.x, m_InitalRigidBodyRotA.y, m_InitalRigidBodyRotA.z, m_InitalRigidBodyRotA.w ) );
		tr.setOrigin( btVector3( m_InitalRigidBodyPosA.x, m_InitalRigidBodyPosA.y, m_InitalRigidBodyPosA.z ) );

		*frameB = tr * *frameB;
	}
	else
	{
		*frameB = rbA->getCenterOfMassTransform() * *frameA;
	}
}

void HingeJoint::Bullet_ComputeFrames(	const btRigidBody* rbA,
										const btRigidBody* rbB,
										const btVector3* pivotA,
										const btVector3* pivotB,
										const btVector3* axis,
										btTransform* frameA,
										btTransform* frameB )
{
	////////////////////////////////////////////////////////////////////////////////////////////////////
	// At[
	////////////////////////////////////////////////////////////////////////////////////////////////////

	const btVector3& rbAxisA0 = *axis;
	btVector3 rbAxisA1;
	btVector3 rbAxisA2;

	btPlaneSpace1( rbAxisA0, rbAxisA1, rbAxisA2 );

	frameA->getOrigin() = *pivotA;
	frameA->getBasis().setValue(	rbAxisA1.getX(), rbAxisA2.getX(), rbAxisA0.getX(),
									rbAxisA1.getY(), rbAxisA2.getY(), rbAxisA0.getY(),
									rbAxisA1.getZ(), rbAxisA2.getZ(), rbAxisA0.getZ() );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// Bt[
	////////////////////////////////////////////////////////////////////////////////////////////////////

	btMatrix3x3 basisA;
	btMatrix3x3 basisB;

	if( m_bInitalRigidBodyTransform == true )
	{
		//pݒ肳Ăꍇ
		basisA.setRotation( btQuaternion( m_InitalRigidBodyRotA.x, m_InitalRigidBodyRotA.y, m_InitalRigidBodyRotA.z, m_InitalRigidBodyRotA.w ) );
		basisB.setRotation( btQuaternion( m_InitalRigidBodyRotB.x, m_InitalRigidBodyRotB.y, m_InitalRigidBodyRotB.z, m_InitalRigidBodyRotB.w ) );
	}
	else
	{
		//pݒ肳ĂȂꍇ́AWbh{fB璼ڎ擾
		basisA = rbA->getCenterOfMassTransform().getBasis();
		basisB = rbB->getCenterOfMassTransform().getBasis();
	}

	frameB->getOrigin() = *pivotB;
	frameB->getBasis() = basisA * frameA->getBasis();
	frameB->getBasis() = basisB.inverse() * frameB->getBasis();
}

}}}}}
