﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices;

namespace Mix.Tool.ModelEditor
{
    /// <summary>
    /// メインフォームクラス
    /// </summary>
    public partial class MainForm : Form
    {
        #region Private Constant

        private const UInt32 SC_CLOSE = 0x0000F060;
        private const UInt32 MF_BYCOMMAND = 0x00000000;
        private const UInt32 MF_ENABLED   = 0x00000000;
        private const UInt32 MF_GRAYED    = 0x00000001;

        [DllImport("USER32.DLL")]
        private static extern IntPtr GetSystemMenu(IntPtr hWnd, UInt32 bRevert);
        [DllImport("USER32.DLL")]
        private static extern UInt32 EnableMenuItem(IntPtr hMenu, UInt32 uIDEnableItem, UInt32 uEnable);

        #endregion

        #region Private Class

        /// <summary>
        /// エディットスタイルアイテムクラス
        /// </summary>
        private class EditStyleItem
        {
            private string display;
            private Project.EditStyle value;

            public EditStyleItem(string _display, Project.EditStyle _value)
            {
                this.display = _display;
                this.value = _value;
            }

            public string Display
            {
                get { return this.display; }
            }

            public Project.EditStyle Value
            {
                get { return this.value; }
            }
        }

        #endregion

        #region Private Member

        private Mix.Tool.ModelEditor.Project.BackgroundWorker backgroundWorker = null;

        private Mix.Tool.Graphics.Manager graphicsMgr = null;
        private Mix.Tool.Graphics.Scene scene = null;

        private Mix.Tool.Dynamics.World dynamicsWorld = null;
        private Mix.Tool.Dynamics.StaticPlane staticPlane = null;

        private Project.Manager projectMgr = null;

        private PreviewForm previewForm = null;
        private OutlinerForm outlinerForm = null;
        private AssetForm assetForm = null;
        private PreferenceForm preferenceForm = null;
        private LogForm logForm = null;
        private MaterialEditorForm materialEditorForm = null;
        private MotionEditorForm motionEditorForm = null;
        private MotionManagerForm motionMgrForm = null;
        private MotionOutlinerForm motOutlinerForm = null;
        private PhysicsManagerForm physicsMgrForm = null;
        private PhysicsMaterialEditorForm physicsMaterialEditorForm = null;
        private KinematicCharacterEditorForm objCharacterEditorForm = null;
        private ActorRigidBodyEditorForm actorRigidBodyEditorForm = null;
        private TerrainRigidBodyEditorForm terrainRigidBodyEditorForm = null;
        private ShapeEditorForm shapeEditorForm = null;
        private JointEditorForm objJointEditorForm = null;

        private bool closeEnabled = true;

        #endregion

        #region MainForm Method

        public MainForm()
        {
            InitializeComponent();

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // コントロールの初期化
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            this.mmiSaveProject.Enabled = false;
            this.mmiPropProject.Enabled = false;
            this.mmiView.Enabled = false;
            this.mmiTool.Enabled = false;
            this.mmiEditStyle.Enabled = false;

            this.mmiEditStyle.ComboBox.SelectedValueChanged += new EventHandler(this.mmiEditStyle_SelectedValueChanged);
        }

        public void Initialize(Mix.Tool.Graphics.Manager _graphicsMgr, Mix.Tool.Dynamics.World _dynamicsWorld)
        {
            this.backgroundWorker = new Mix.Tool.ModelEditor.Project.BackgroundWorker(32);

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // シーンの初期化
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            this.graphicsMgr = _graphicsMgr;

            this.scene = new Mix.Tool.Graphics.Scene();

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ダイナミクスワールドの初期化
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            this.dynamicsWorld = _dynamicsWorld;

            this.staticPlane = new Mix.Tool.Dynamics.StaticPlane();
            this.dynamicsWorld.CollisionObjects.Add(this.staticPlane);
        }

        public void Poll()
        {
            //バックグラウンドワーカー
            this.backgroundWorker.Poll();

            //モーションアウトライナ
            if ((this.motOutlinerForm != null) &&
                (this.motOutlinerForm.IsDisposed == false))
            {
                this.motOutlinerForm.Poll();
            }

            //モーションエディタ
            if ((this.motionEditorForm != null) &&
                (this.motionEditorForm.IsDisposed == false))
            {
                this.motionEditorForm.Poll();
            }
        }

        private void MainForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (this.closeEnabled == true)
            {
                if (Mix.Tool.MessageDialog.ShowYN(
                    Properties.Resources.PRE_EXIT_MES,
                    Properties.Resources.TITLE,
                    MessageDialogIcon.Question) == DialogResult.No)
                {
                    e.Cancel = true;
                }
            }
            else
            {
                e.Cancel = true;
            }
        }

        private void MainForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            this.backgroundWorker.Dispose();

            if (this.projectMgr != null)
            {
                if (this.scene != null)
                {
                    this.scene.DrawObject = null;
                }

                this.projectMgr.Dispose();
                this.projectMgr = null;
            }

            if (this.scene != null)
            {
                this.scene.Dispose();
                this.scene = null;
            }

            if (this.staticPlane != null)
            {
                if (this.dynamicsWorld != null)
                {
                    this.dynamicsWorld.CollisionObjects.Remove(this.staticPlane);
                }

                this.staticPlane.Dispose();
                this.staticPlane = null;
            }

            this.graphicsMgr = null;
            this.dynamicsWorld = null;
        }

        protected override void WndProc(ref Message m)
        {
            Mix.Tool.Logger.WndProc(m);

            base.WndProc(ref m);
        }

        #endregion

        #region MainMenu Method

        private void mmiNewProject_Click(object sender, EventArgs e)
        {
            Project.Manager nextProject = CreateProject();

            using (NewProjectForm form = new NewProjectForm(nextProject))
            {
                if (form.ShowDialog() == DialogResult.OK)
                {
                    this.SetCloseEnabled( false );

                    if (this.projectMgr != null)
                    {
                        this.projectMgr.Dispose();
                        this.projectMgr = null;
                    }

                    this.projectMgr = nextProject;

                    try
                    {
                        this.projectMgr.Load();
                    }
                    catch (System.Exception excep)
                    {
                        Mix.Tool.MessageDialog.ShowOK(
                            Properties.Resources.PROJECT_NEW_ERROR + excep.Message,
                            Properties.Resources.TITLE,
                            MessageDialogIcon.Error);
                    }
                }
            }
        }

        private void mmiOpenPeoject_Click(object sender, EventArgs e)
        {
            using (OpenFileDialog dlg = new OpenFileDialog())
            {
                dlg.Filter = Properties.Resources.FILE_FILTER_PRJ;
                dlg.FilterIndex = 1;

                if (dlg.ShowDialog() == DialogResult.OK)
                {
                    this.SetCloseEnabled(false);

                    if (this.projectMgr != null)
                    {
                        this.projectMgr.Dispose();
                        this.projectMgr = null;
                    }

                    this.projectMgr = this.CreateProject();

                    try
                    {
                        this.projectMgr.Load(dlg.FileName);
                    }
                    catch (System.Exception excep)
                    {
                        Mix.Tool.MessageDialog.ShowOK(
                            Properties.Resources.PROJECT_OPEN_ERROR + excep.Message,
                            Properties.Resources.TITLE,
                            MessageDialogIcon.Error);
                    }
                }
            }
        }

        private void mmiSaveProject_Click(object sender, EventArgs e)
        {
//            this.dockPanel.SaveAsXml("F:\\defPhysics_ObjectModel.es.txt");
//            return;

            try
            {
                this.projectMgr.Save();
            }
            catch (System.Exception excep)
            {
                Mix.Tool.MessageDialog.ShowOK(
                    Properties.Resources.PROJECT_SAVE_ERROR + excep.Message,
                    Properties.Resources.TITLE,
                    MessageDialogIcon.Error);
            }
        }

        private void mmiPropProject_Click(object sender, EventArgs e)
        {
            using (ProjectPropertyForm form = new ProjectPropertyForm(this.projectMgr))
            {
                if (form.ShowDialog() == DialogResult.OK)
                {
                    this.projectMgr.Refresh();
                }
            }
        }

        private void mmiExist_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        private void mmiPreview_Click(object sender, EventArgs e)
        {
            if ((this.previewForm == null) ||
                (this.previewForm.IsDisposed == true))
            {
                this.CreatePreview(true);
            }
            else
            {
                this.ClosePreview();
            }
        }

        private void mmiOutliner_Click(object sender, EventArgs e)
        {
            if ((this.outlinerForm == null) ||
                (this.outlinerForm.IsDisposed == true))
            {
                this.CreateOutliner(true);
            }
            else
            {
                this.CloseOutliner();
            }
        }

        private void mmiAsset_Click(object sender, EventArgs e)
        {
            if ((this.assetForm == null) ||
                (this.assetForm.IsDisposed == true))
            {
                this.CreateAsset(true);
            }
            else
            {
                this.CloseAsset();
            }
        }

        private void mmiPreference_Click(object sender, EventArgs e)
        {
            if ((this.preferenceForm == null) ||
                (this.preferenceForm.IsDisposed == true))
            {
                this.CreatePreference(true);
            }
            else
            {
                this.ClosePreference();
            }
        }

        private void mmiLog_Click(object sender, EventArgs e)
        {
            if ((this.logForm == null) ||
                (this.logForm.IsDisposed == true))
            {
                this.CreateLog(true);
            }
            else
            {
                this.CloseLog();
            }
        }

        private void mmiMaterialEditor_Click(object sender, EventArgs e)
        {
            if ((this.materialEditorForm == null) ||
                (this.materialEditorForm.IsDisposed == true))
            {
                this.CreateMaterialEditor(true);
            }
            else
            {
                this.CloseMaterialEditor();
            }
        }

        private void mmiMotionManager_Click(object sender, EventArgs e)
        {
            if ((this.motionMgrForm == null) ||
                (this.motionMgrForm.IsDisposed == true))
            {
                this.CreateMotionManager(true);
            }
            else
            {
                this.CloseMotionManager();
            }
        }

        private void mmiMotionOutliner_Click(object sender, EventArgs e)
        {
            if ((this.motOutlinerForm == null) ||
                (this.motOutlinerForm.IsDisposed == true))
            {
                this.CreateMotionOutliner(true);
            }
            else
            {
                this.CloseMotionOutliner();
            }
        }

        private void mmiMotionEditor_Click(object sender, EventArgs e)
        {
            if ((this.motionEditorForm == null) ||
                (this.motionEditorForm.IsDisposed == true))
            {
                this.CreateMotionEditor(true);
            }
            else
            {
                this.CloseMotionEditor();
            }
        }

        private void mmiPhysicsMgr_Click(object sender, EventArgs e)
        {
            if ((this.physicsMgrForm == null) ||
                (this.physicsMgrForm.IsDisposed == true))
            {
                this.CreatePhysicsMgr(true);
            }
            else
            {
                this.ClosePhysicsMgr();
            }
        }

        private void mmiPhysicsMaterialEditor_Click(object sender, EventArgs e)
        {
            if ((this.physicsMaterialEditorForm == null) ||
                (this.physicsMaterialEditorForm.IsDisposed == true))
            {
                this.CreatePhysicsMaterialEditor(true);
            }
            else
            {
                this.ClosePhysicsMaterialEditor();
            }
        }

        private void mmiCharacterEditor_Click(object sender, EventArgs e)
        {
            if ((this.objCharacterEditorForm == null) ||
                (this.objCharacterEditorForm.IsDisposed == true))
            {
                this.CreateCharacterEditor(true);
            }
            else
            {
                this.CloseCharacterEditor();
            }
        }

        private void mmiActorRBEditor_Click(object sender, EventArgs e)
        {
            if ((this.actorRigidBodyEditorForm == null) ||
                (this.actorRigidBodyEditorForm.IsDisposed == true))
            {
                this.CreateActorRigidBodyEditor(true);
            }
            else
            {
                this.CloseActorRigidBodyEditor();
            }
        }

        private void mmiShapeEditor_Click(object sender, EventArgs e)
        {
            if ((this.shapeEditorForm == null) ||
                (this.shapeEditorForm.IsDisposed == true))
            {
                this.CreateShapeEditor(true);
            }
            else
            {
                this.CloseShapeEditor();
            }
        }

        private void mmiJoint_Click(object sender, EventArgs e)
        {
            if ((this.objJointEditorForm == null) ||
                (this.objJointEditorForm.IsDisposed == true))
            {
                this.CreateJointEditor(true);
            }
            else
            {
                this.CloseJointEditor();
            }
        }

        private void mmiTerrainRBEditor_Click(object sender, EventArgs e)
        {
            if ((this.terrainRigidBodyEditorForm == null) ||
                (this.terrainRigidBodyEditorForm.IsDisposed == true))
            {
                this.CreateTerrainRigidBodyEditor(true);
            }
            else
            {
                this.CloseTerrainRigidBodyEditor();
            }
        }

        private void mmiMEP_Recompile_Click(object sender, EventArgs e)
        {
            this.graphicsMgr.EffectPackage.Compile();
        }

        private void mmiMEPCleanup_Click(object sender, EventArgs e)
        {
            this.graphicsMgr.EffectPackage.Compile(this.projectMgr.Template.EffectSearchDirCollection);
        }

        private void mmiAbout_Click(object sender, EventArgs e)
        {
            using (AboutForm form = new AboutForm())
            {
                form.ShowDialog();
            }
        }

        private void mmiEditStyle_SelectedValueChanged(object sender, EventArgs e)
        {
            if ((this.projectMgr != null) &&
                (this.mmiEditStyle.ComboBox.SelectedValue != null))
            {
                this.projectMgr.EditStyle = (Project.EditStyle)(this.mmiEditStyle.ComboBox.SelectedValue);
            }
        }

        #endregion

        #region DockPanel Method

        private Mix.Tool.Docking.IDockContent dockPanel_DeserializeDockContent(string persistString)
        {
            if (persistString.Equals(typeof(PreviewForm).ToString()) == true)
            {
                return this.CreatePreview(false);
            }
            else if (persistString.Equals(typeof(OutlinerForm).ToString()) == true)
            {
                return this.CreateOutliner(false);
            }
            else if (persistString.Equals(typeof(AssetForm).ToString()) == true)
            {
                return this.CreateAsset(false);
            }
            else if (persistString.Equals(typeof(PreferenceForm).ToString()) == true)
            {
                return this.CreatePreference(false);
            }
            else if (persistString.Equals(typeof(LogForm).ToString()) == true)
            {
                return this.CreateLog(false);
            }
            else if (persistString.Equals(typeof(MaterialEditorForm).ToString()) == true)
            {
                return this.CreateMaterialEditor(false);
            }
            else if (persistString.Equals(typeof(MotionManagerForm).ToString()) == true)
            {
                return this.CreateMotionManager(false);
            }
            else if (persistString.Equals(typeof(MotionOutlinerForm).ToString()) == true)
            {
                return this.CreateMotionOutliner(false);
            }
            else if (persistString.Equals(typeof(MotionEditorForm).ToString()) == true)
            {
                return this.CreateMotionEditor(false);
            }
            else if (persistString.Equals(typeof(PhysicsManagerForm).ToString()) == true)
            {
                return this.CreatePhysicsMgr(false);
            }
            else if (persistString.Equals(typeof(PhysicsMaterialEditorForm).ToString()) == true)
            {
                return this.CreatePhysicsMaterialEditor(false);
            }
            else if (persistString.Equals(typeof(KinematicCharacterEditorForm).ToString()) == true)
            {
                return this.CreateCharacterEditor(false);
            }
            else if (persistString.Equals(typeof(ShapeEditorForm).ToString()) == true)
            {
                return this.CreateShapeEditor(false);
            }
            else if (persistString.Equals(typeof(TerrainRigidBodyEditorForm).ToString()) == true)
            {
                return this.CreateTerrainRigidBodyEditor(false);
            }
            else if (persistString.Equals(typeof(ActorRigidBodyEditorForm).ToString()) == true)
            {
                return this.CreateActorRigidBodyEditor(false);
            }
            else if (persistString.Equals(typeof(JointEditorForm).ToString()) == true)
            {
                return this.CreateJointEditor(false);
            }

            return null;
        }

        #endregion

        #region PreviewForm Method

        private Mix.Tool.Docking.IDockContent CreatePreview(bool show)
        {
            if ((this.previewForm == null) ||
                (this.previewForm.IsDisposed == true))
            {
                this.previewForm = new PreviewForm(this.scene, this.projectMgr.DrawObject, this.dynamicsWorld, this.projectMgr.Preview);
                this.previewForm.HandleCreated += new EventHandler(this.previewForm_HandleCreated);
                this.previewForm.HandleDestroyed += new EventHandler(this.previewForm_HandleDestroyed);
                this.previewForm.MeshSelected += new PreviewForm.MeshSelectEventHandler(this.previewForm_MeshSelected);
                this.previewForm.MeshStateChanged += new PreviewForm.MeshStateEventHandler(this.previewForm_MeshStateChanged);

                if (show == true)
                {
                    this.previewForm.Show(this.dockPanel);
                }

                return this.previewForm;
            }

            return null;
        }

        private void ClosePreview()
        {
            if (this.previewForm != null)
            {
                this.previewForm.Close();
                this.previewForm.Dispose();
                this.previewForm = null;
            }
        }

        private void previewForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiPreview.Checked = true;
        }

        private void previewForm_HandleDestroyed(object sender, EventArgs e)
        {
            this.mmiPreview.Checked = false;
        }

        private void previewForm_MeshSelected(object sender, PreviewForm.MeshSelectEventArgs e)
        {
            if (this.projectMgr.DrawObject is Mix.Tool.Graphics.Model)
            {
                Mix.Tool.Graphics.Model model = this.projectMgr.DrawObject as Mix.Tool.Graphics.Model;

                //キーで選択
                model.SelectedMeshKey = e.Key;

                //アウトライナーに通知
                if ((this.outlinerForm != null) &&
                    (this.outlinerForm.IsDisposed == false))
                {
                    this.outlinerForm.UpdateSelectedNode();
                }
            }
        }

        private void previewForm_MeshStateChanged(object sender, PreviewForm.MeshStateEventArgs e)
        {
            if (this.projectMgr.DrawObject is Mix.Tool.Graphics.Model)
            {
                Mix.Tool.Graphics.Model model = this.projectMgr.DrawObject as Mix.Tool.Graphics.Model;
                Mix.Tool.Graphics.MeshNode meshNode = model.SelectedNode as Mix.Tool.Graphics.MeshNode;

                if (meshNode != null)
                {
                    //表示状態の変更
                    meshNode.Visible = e.Visible;

                    //ノードアウトライナーに通知
                    if ((this.outlinerForm != null) &&
                        (this.outlinerForm.IsDisposed == false))
                    {
                        this.outlinerForm.UpdateSelectedNode();
                    }

                    //非表示だったら、メッシュの選択を解除
                    if (e.Visible == false)
                    {
                        model.SelectedNode = null;
                    }
                }
            }
        }

        #endregion

        #region OutlinerForm Method

        private Mix.Tool.Docking.IDockContent CreateOutliner(bool show)
        {
            if (this.projectMgr.DrawObject == null)
            {
                return null;
            }

            if (this.projectMgr.DrawObject is Mix.Tool.Graphics.Model)
            {
                if ((this.outlinerForm == null) ||
                    (this.outlinerForm.IsDisposed == true))
                {
                    this.outlinerForm = new OutlinerForm();
                    this.outlinerForm.HandleCreated += new EventHandler(this.outlinerForm_HandleCreated);
                    this.outlinerForm.HandleDestroyed += new EventHandler(this.outlinerForm_HandleDestroyed);
                    this.outlinerForm.SelectedObjectChanged += new OutlinerForm.ObjectEventHandler(this.outlinerForm_SelectedObjectChanged);
                    this.outlinerForm.Source = this.projectMgr.DrawObject as Mix.Tool.Graphics.Model;

                    if (show == true)
                    {
                        this.outlinerForm.Show(this.dockPanel);
                        this.outlinerForm.DockState = Mix.Tool.Docking.DockState.Float;
                    }

                    return this.outlinerForm;
                }
            }

            return null;
        }

        private void CloseOutliner()
        {
            if (this.outlinerForm != null)
            {
                this.outlinerForm.Close();
                this.outlinerForm.Dispose();
                this.outlinerForm = null;
            }
        }

        private void outlinerForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiNodeOutliner.Checked = true;
        }

        private void outlinerForm_HandleDestroyed(object sender, EventArgs e)
        {
            this.mmiNodeOutliner.Checked = false;
        }

        private void outlinerForm_SelectedObjectChanged(object sender, OutlinerForm.ObjectEventArg e)
        {
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ノードの選択
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            if (this.projectMgr.DrawObject is Mix.Tool.Graphics.Model)
            {
                Mix.Tool.Graphics.Model model = this.projectMgr.DrawObject as Mix.Tool.Graphics.Model;

                if (e.Object is Mix.Tool.Graphics.Node)
                {
                    model.SelectedNode = e.Object as Mix.Tool.Graphics.Node;
                }
                else
                {
                    model.SelectedNode = null;
                }
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // アセットフォームの設定
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            if ((this.assetForm != null) &&
                (this.assetForm.IsDisposed == false))
            {
                this.assetForm.Source = e.Object;
            }
        }

        #endregion

        #region AssetForm Method

        private Mix.Tool.Docking.IDockContent CreateAsset(bool show)
        {
            if ((this.assetForm == null) ||
                (this.assetForm.IsDisposed == true))
            {
                this.assetForm = new AssetForm(this.backgroundWorker);
                this.assetForm.HandleCreated += new EventHandler(this.assetForm_HandleCreated);
                this.assetForm.HandleDestroyed += new EventHandler(this.assetForm_HandleDestroyed);
                this.assetForm.SelectedMaterialChanged += new AssetForm.MaterialEventHandler(this.assetForm_SelectedMaterialChanged);
                this.assetForm.SelectedKinematicCharacterChanged += new AssetForm.KinematicCharacterEventHandler(this.assetForm_SelectedKinematicCharacterChanged);
                this.assetForm.SelectedTerrainChanged += new AssetForm.TerrainEventHandler(this.assetForm_SelectedTerrainChanged);
                this.assetForm.SelectedRigidBodyChanged += new AssetForm.RigidBodyEventHandler(this.assetForm_SelectedRigidBodyChanged);
                this.assetForm.SelectedShapeChanged += new AssetForm.ShapeEventHandler(this.assetForm_SelectedShapeChanged);
                this.assetForm.SelectedJointChanged += new AssetForm.JointEventHandler(this.assetForm_SelectedJointChanged);

                if ((this.outlinerForm != null) &&
                    (this.outlinerForm.IsDisposed == false))
                {
                    this.outlinerForm.RequestSelectedObjectChangedEvent();
                }

                if (show == true)
                {
                    this.assetForm.Show(this.dockPanel);
                }

                return this.assetForm;
            }

            return null;
        }

        private void CloseAsset()
        {
            if (this.assetForm != null)
            {
                this.assetForm.Close();
                this.assetForm.Dispose();
                this.assetForm = null;
            }
        }

        private void assetForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiAsset.Checked = true;
        }

        private void assetForm_HandleDestroyed(object sender, EventArgs e)
        {
            this.mmiAsset.Checked = false;
        }

        private void assetForm_SelectedMaterialChanged(object sender, AssetForm.MaterialEventArgs e)
        {
            if ((this.materialEditorForm != null) &&
                (this.materialEditorForm.IsDisposed == false))
            {
                if (e.IsAvailable == true)
                {
                    this.materialEditorForm.Source = new MaterialEditorSource(e.MaterialSlot, e.Material);

                    if ((this.projectMgr.DrawObject is Mix.Tool.Graphics.Model) &&
                        (this.scene.General.SelectedAlwaysDisplay == false))
                    {
                        Mix.Tool.Graphics.Model model = this.projectMgr.DrawObject as Mix.Tool.Graphics.Model;
                        model.SelectedNode = null;
                    }
                }
                else
                {
                    this.materialEditorForm.Source = null;
                }
            }
        }

        private void assetForm_SelectedKinematicCharacterChanged(object sender, AssetForm.KinematicCharacterEventArgs e)
        {
            if ((this.objCharacterEditorForm != null) &&
                (this.objCharacterEditorForm.IsDisposed == false))
            {
                this.objCharacterEditorForm.Source = e.DynamicsDesigner;
            }
        }

        private void assetForm_SelectedRigidBodyChanged(object sender, AssetForm.RigidBodyEventArgs e)
        {
            if ((this.actorRigidBodyEditorForm != null) &&
                (this.actorRigidBodyEditorForm.IsDisposed == false))
            {
                this.actorRigidBodyEditorForm.Source = e.BasicChild;
            }
        }

        private void assetForm_SelectedShapeChanged(object sender, AssetForm.ShapeEventArgs e)
        {
            if ((this.shapeEditorForm != null) &&
                (this.shapeEditorForm.IsDisposed == false))
            {
                if (e.IsAvailable == true)
                {
                    this.shapeEditorForm.Source = new ShapeEditorSource(e.Document, e.Shape);
                }
                else
                {
                    this.shapeEditorForm.Source = null;
                }
            }
        }

        private void assetForm_SelectedJointChanged(object sender, AssetForm.JointEventArgs e)
        {
            if ((this.objJointEditorForm != null) &&
                (this.objJointEditorForm.IsDisposed == false))
            {
                this.objJointEditorForm.Source = e.DynamicsDesigner;
            }
        }

        private void assetForm_SelectedTerrainChanged(object sender, AssetForm.TerrainEventArgs e)
        {
            if ((this.terrainRigidBodyEditorForm != null) &&
                (this.terrainRigidBodyEditorForm.IsDisposed == false))
            {
                if (e.IsAvailable == true)
                {
                    Mix.Tool.Dynamics.Design.ITerrainDocument dynDoc = e.Document as Mix.Tool.Dynamics.Design.ITerrainDocument;
                    Mix.Tool.Dynamics.Design.Terrain dynDesigner = dynDoc.DynamicsDesigner;

                    this.terrainRigidBodyEditorForm.Source = (dynDesigner.Status == Mix.Tool.Dynamics.Design.TerrainStatus.Available) ? dynDesigner : null;
                }
                else
                {
                    this.terrainRigidBodyEditorForm.Source = null;
                }
            }
        }

        #endregion

        #region PreferenceForm Method

        private Mix.Tool.Docking.IDockContent CreatePreference(bool show)
        {
            if ((this.preferenceForm == null) ||
                (this.preferenceForm.IsDisposed == true))
            {
                this.preferenceForm = new PreferenceForm(this.graphicsMgr, this.scene);
                this.preferenceForm.HandleCreated += new EventHandler(this.preferenceForm_HandleCreated);
                this.preferenceForm.HandleDestroyed += new EventHandler(this.preferenceForm_HandleDestroyed);

                if (show == true)
                {
                    this.preferenceForm.Show(this.dockPanel);
                }

                return this.preferenceForm;
            }

            return null;
        }

        private void ClosePreference()
        {
            if (this.preferenceForm != null)
            {
                this.preferenceForm.Close();
                this.preferenceForm.Dispose();
                this.preferenceForm = null;
            }
        }

        private void preferenceForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiPreference.Checked = true;
        }

        private void preferenceForm_HandleDestroyed(object sender, EventArgs e)
        {
            this.mmiPreference.Checked = false;
        }

        #endregion

        #region LogForm Method

        private Mix.Tool.Docking.IDockContent CreateLog(bool show)
        {
            if ((this.logForm == null) ||
                (this.logForm.IsDisposed == true))
            {
                this.logForm = new LogForm(this.projectMgr.Log);
                this.logForm.HandleCreated += new EventHandler(this.logForm_HandleCreated);
                this.logForm.HandleDestroyed += new EventHandler(this.logForm_HandleDestroyed);

                if (show == true)
                {
                    this.logForm.Show(this.dockPanel);
                }

                return this.logForm;
            }

            return null;
        }

        private void CloseLog()
        {
            if (this.logForm != null)
            {
                this.logForm.Close();
                this.logForm.Dispose();
                this.logForm = null;
            }
        }

        private void logForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiLog.Checked = true;
        }

        private void logForm_HandleDestroyed(object sender, EventArgs e)
        {
            this.mmiLog.Checked = false;
        }

        #endregion

        #region MaterialEditorForm Method

        private Mix.Tool.Docking.IDockContent CreateMaterialEditor(bool show)
        {
            if (this.projectMgr.DrawObject == null)
            {
                return null;
            }

            if ((this.materialEditorForm == null) ||
                (this.materialEditorForm.IsDisposed == true))
            {
                this.materialEditorForm = new MaterialEditorForm(this.graphicsMgr);
                this.materialEditorForm.HandleCreated += new EventHandler(this.materialEditorForm_HandleCreated);
                this.materialEditorForm.HandleDestroyed += new EventHandler(this.materialEditorForm_HandleDestroyed);

                if ((this.assetForm != null) &&
                    (this.assetForm.IsDisposed == false))
                {
                    this.assetForm.RequestMaterialEvent();
                }

                if (show == true)
                {
                    this.materialEditorForm.Show(this.dockPanel);
                }

                return this.materialEditorForm;
            }

            return null;
        }

        private void CloseMaterialEditor()
        {
            if (this.materialEditorForm != null)
            {
                this.materialEditorForm.Close();
                this.materialEditorForm.Dispose();
                this.materialEditorForm = null;
            }
        }

        private void materialEditorForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiMaterialEditor.Checked = true;
        }

        private void materialEditorForm_HandleDestroyed(object sender, EventArgs e)
        {
            this.mmiMaterialEditor.Checked = false;
        }

        #endregion

        #region MotionManagerForm Method

        private Mix.Tool.Docking.IDockContent CreateMotionManager(bool show)
        {
            if (this.projectMgr.DrawObject == null)
            {
                return null;
            }

            if ((this.motionMgrForm == null) ||
                (this.motionMgrForm.IsDisposed == true))
            {
                this.motionMgrForm = new MotionManagerForm(this.projectMgr.DrawObject as Mix.Tool.Graphics.ObjectModel);
                this.motionMgrForm.HandleCreated += new EventHandler(this.motionMgrForm_HandleCreated);
                this.motionMgrForm.HandleDestroyed += new EventHandler(this.motionMgrForm_HandleDestroyed);

                if (show == true)
                {
                    this.motionMgrForm.Show(this.dockPanel);
                }

                return this.motionMgrForm;
            }

            return null;
        }

        private void CloseMotionManager()
        {
            if (this.motionMgrForm != null)
            {
                this.motionMgrForm.Close();
                this.motionMgrForm.Dispose();
                this.motionMgrForm = null;
            }
        }

        private void motionMgrForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiMotionManager.Checked = true;
        }

        private void motionMgrForm_HandleDestroyed(object sender, EventArgs e)
        {
            this.mmiMotionManager.Checked = false;
        }

        #endregion

        #region MotionOutlinerForm Method

        private Mix.Tool.Docking.IDockContent CreateMotionOutliner(bool show)
        {
            if (this.projectMgr.DrawObject == null)
            {
                return null;
            }

            if ((this.motOutlinerForm == null) ||
                (this.motOutlinerForm.IsDisposed == true))
            {
                this.motOutlinerForm = new MotionOutlinerForm();
                this.motOutlinerForm.HandleCreated += new EventHandler(this.motOutlinerForm_HandleCreated);
                this.motOutlinerForm.HandleDestroyed += new EventHandler(this.motOutlinerForm_HandleDestroyed);
                this.motOutlinerForm.SelectedMotionChanged += new MotionOutlinerForm.SelectedMotionChangedEventHandler(this.motOutlinerForm_SelectedMotionChanged);
                this.motOutlinerForm.Model = this.projectMgr.DrawObject as Mix.Tool.Graphics.ObjectModel;

                if (show == true)
                {
                    this.motOutlinerForm.Show(this.dockPanel);
                }

                return this.motOutlinerForm;
            }

            return null;
        }

        private void CloseMotionOutliner()
        {
            if (this.motOutlinerForm != null)
            {
                this.motOutlinerForm.Close();
                this.motOutlinerForm.Dispose();
                this.motOutlinerForm = null;
            }
        }

        private void motOutlinerForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiMotionOutliner.Checked = true;
        }

        private void motOutlinerForm_HandleDestroyed(object sender, EventArgs e)
        {
            if ((this.motionEditorForm != null) &&
                (this.motionEditorForm.IsDisposed == false))
            {
                this.motionEditorForm.Motion = null;
            }

            this.mmiMotionOutliner.Checked = false;
        }

        private void motOutlinerForm_SelectedMotionChanged(Mix.Tool.Graphics.Motion motion)
        {
            if ((this.motionEditorForm != null) &&
                (this.motionEditorForm.IsDisposed == false))
            {
                this.motionEditorForm.Motion = motion;
            }
        }

        #endregion

        #region MotionEditoForm Method

        private Mix.Tool.Docking.IDockContent CreateMotionEditor(bool show)
        {
            if (this.projectMgr.DrawObject == null)
            {
                return null;
            }

            if ((this.motionEditorForm == null) ||
                (this.motionEditorForm.IsDisposed == true))
            {
                this.motionEditorForm = new MotionEditorForm();
                this.motionEditorForm.HandleCreated += new EventHandler(this.motionEditorForm_HandleCreated);
                this.motionEditorForm.HandleDestroyed += new EventHandler(this.motionEditorForm_HandleDestroyed);

                if( ( this.motOutlinerForm != null ) &&
                    ( this.motOutlinerForm.IsDisposed == false ) )
                {
                    this.motionEditorForm.Motion = this.motOutlinerForm.SelectedMotion;
                }

                if (show == true)
                {
                    this.motionEditorForm.Show(this.dockPanel);
                }

                return this.motionEditorForm;
            }

            return null;
        }

        private void CloseMotionEditor()
        {
            if (this.motionEditorForm != null)
            {
                this.motionEditorForm.Close();
                this.motionEditorForm.Dispose();
                this.motionEditorForm = null;
            }
        }

        private void motionEditorForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiMotionEditor.Checked = true;
        }

        private void motionEditorForm_HandleDestroyed(object sender, EventArgs e)
        {
            this.mmiMotionEditor.Checked = false;
        }

        #endregion

        #region PhysicsManagerForm Method

        private Mix.Tool.Docking.IDockContent CreatePhysicsMgr(bool show)
        {
            if ((this.physicsMgrForm == null) ||
                (this.physicsMgrForm.IsDisposed == true))
            {
                this.physicsMgrForm = new PhysicsManagerForm(this.projectMgr.Dynamics);
                this.physicsMgrForm.HandleCreated += new EventHandler(this.physicsMgrForm_HandleCreated);
                this.physicsMgrForm.HandleDestroyed += new EventHandler(this.physicsMgrForm_HandleDestroyed);

                if (show == true)
                {
                    this.physicsMgrForm.Show(this.dockPanel);
                }

                return this.physicsMgrForm;
            }

            return null;
        }

        private void ClosePhysicsMgr()
        {
            if (this.physicsMgrForm != null)
            {
                this.physicsMgrForm.Close();
                this.physicsMgrForm.Dispose();
                this.physicsMgrForm = null;
            }
        }

        private void physicsMgrForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiDynamicsMgr.Checked = true;
        }

        private void physicsMgrForm_HandleDestroyed(object sender, EventArgs e)
        {
            this.mmiDynamicsMgr.Checked = false;
        }

        #endregion

        #region PhysicsMaterialEditorForm Method

        private Mix.Tool.Docking.IDockContent CreatePhysicsMaterialEditor(bool show)
        {
            if (this.projectMgr.DrawObject == null)
            {
                return null;
            }

            if ((this.physicsMaterialEditorForm == null) ||
                (this.physicsMaterialEditorForm.IsDisposed == true))
            {
                this.physicsMaterialEditorForm = new PhysicsMaterialEditorForm(this.projectMgr.Dynamics);
                this.physicsMaterialEditorForm.HandleCreated += new EventHandler(this.physicsMaterialEditorForm_HandleCreated);
                this.physicsMaterialEditorForm.HandleDestroyed += new EventHandler(this.physicsMaterialEditorForm_HandleDestroyed);

                if (show == true)
                {
                    this.physicsMaterialEditorForm.Show(this.dockPanel);
                }

                return this.physicsMaterialEditorForm;
            }

            return null;
        }

        private void ClosePhysicsMaterialEditor()
        {
            if (this.physicsMaterialEditorForm != null)
            {
                this.physicsMaterialEditorForm.Close();
                this.physicsMaterialEditorForm.Dispose();
                this.physicsMaterialEditorForm = null;
            }
        }

        private void physicsMaterialEditorForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiPhysicsMaterialEditor.Checked = true;
        }

        private void physicsMaterialEditorForm_HandleDestroyed(object sender, EventArgs e)
        {
            this.mmiPhysicsMaterialEditor.Checked = false;
        }

        #endregion

        #region KinematicCharacterEditorForm Method

        private Mix.Tool.Docking.IDockContent CreateCharacterEditor(bool show)
        {
            if (this.projectMgr.DrawObject == null)
            {
                return null;
            }

            if ((this.objCharacterEditorForm == null) ||
                (this.objCharacterEditorForm.IsDisposed == true))
            {
                this.objCharacterEditorForm = new KinematicCharacterEditorForm(this.projectMgr.Dynamics);
                this.objCharacterEditorForm.HandleCreated += new EventHandler(this.objCharacterEditorForm_HandleCreated);
                this.objCharacterEditorForm.HandleDestroyed += new EventHandler(this.objCharacterEditorForm_HandleDestroyed);

                if ((this.assetForm != null) &&
                    (this.assetForm.IsDisposed == false))
                {
                    this.assetForm.RequestKinematicCharacterEvent();
                }

                if (show == true)
                {
                    this.objCharacterEditorForm.Show(this.dockPanel);
                }

                return this.objCharacterEditorForm;
            }

            return null;
        }

        private void CloseCharacterEditor()
        {
            if (this.objCharacterEditorForm != null)
            {
                this.objCharacterEditorForm.Close();
                this.objCharacterEditorForm.Dispose();
                this.objCharacterEditorForm = null;
            }
        }

        private void objCharacterEditorForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiCharacterEditor.Checked = true;
        }

        private void objCharacterEditorForm_HandleDestroyed(object sender, EventArgs e)
        {
            this.mmiCharacterEditor.Checked = false;
        }

        #endregion

        #region Actor RigidBodyEditorForm Method

        private Mix.Tool.Docking.IDockContent CreateActorRigidBodyEditor(bool show)
        {
            if ((this.actorRigidBodyEditorForm == null) ||
                (this.actorRigidBodyEditorForm.IsDisposed == true))
            {
                this.actorRigidBodyEditorForm = new ActorRigidBodyEditorForm(this.projectMgr.Dynamics);
                this.actorRigidBodyEditorForm.HandleCreated += new EventHandler(this.actorRigidBodyEditorForm_HandleCreated);
                this.actorRigidBodyEditorForm.HandleDestroyed += new EventHandler(this.actorRigidBodyEditorForm_HandleDestroyed);

                if ((this.assetForm != null) &&
                    (this.assetForm.IsDisposed == false))
                {
                    this.assetForm.RequestRigidBodyEvent();
                }

                if (show == true)
                {
                    this.actorRigidBodyEditorForm.Show(this.dockPanel);
                }

                return this.actorRigidBodyEditorForm;
            }

            return null;
        }

        private void CloseActorRigidBodyEditor()
        {
            if (this.actorRigidBodyEditorForm != null)
            {
                this.actorRigidBodyEditorForm.Close();
                this.actorRigidBodyEditorForm.Dispose();
                this.actorRigidBodyEditorForm = null;
            }
        }

        private void actorRigidBodyEditorForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiActorRBEditor.Checked = true;
        }

        private void actorRigidBodyEditorForm_HandleDestroyed(object sender, EventArgs e)
        {
            this.mmiActorRBEditor.Checked = false;
        }

        #endregion

        #region ShapeEditorForm Method

        private Mix.Tool.Docking.IDockContent CreateShapeEditor(bool show)
        {
            if (this.projectMgr.DrawObject == null)
            {
                return null;
            }

            if (this.projectMgr.DrawObject is Mix.Tool.Graphics.ObjectModel)
            {
                if ((this.shapeEditorForm == null) ||
                    (this.shapeEditorForm.IsDisposed == true))
                {
                    this.shapeEditorForm = new ShapeEditorForm(this.projectMgr.Dynamics);
                    this.shapeEditorForm.HandleCreated += new EventHandler(this.shapeEditorForm_HandleCreated);
                    this.shapeEditorForm.HandleDestroyed += new EventHandler(this.shapeEditorForm_HandleDestroyed);

                    if ((this.assetForm != null) &&
                        (this.assetForm.IsDisposed == false))
                    {
                        this.assetForm.RequestShapeEvent();
                    }

                    if (show == true)
                    {
                        this.shapeEditorForm.Show(this.dockPanel);
                    }

                    return this.shapeEditorForm;
                }
            }

            return null;
        }

        private void CloseShapeEditor()
        {
            if (this.shapeEditorForm != null)
            {
                this.shapeEditorForm.Close();
                this.shapeEditorForm.Dispose();
                this.shapeEditorForm = null;
            }
        }

        private void shapeEditorForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiShapeEditor.Checked = true;
        }

        private void shapeEditorForm_HandleDestroyed(object sender, EventArgs e)
        {
            this.mmiShapeEditor.Checked = false;
        }

        #endregion

        #region JointEditorForm Method

        private Mix.Tool.Docking.IDockContent CreateJointEditor(bool show)
        {
            if ((this.objJointEditorForm == null) ||
                (this.objJointEditorForm.IsDisposed == true))
            {
                this.objJointEditorForm = new JointEditorForm();
                this.objJointEditorForm.HandleCreated += new EventHandler(this.objJointEditorForm_HandleCreated);
                this.objJointEditorForm.HandleDestroyed += new EventHandler(this.objJointEditorForm_HandleDestroyed);

                if ((this.assetForm != null) &&
                    (this.assetForm.IsDisposed == false))
                {
                    this.assetForm.RequestJointEvent();
                }

                if (show == true)
                {
                    this.objJointEditorForm.Show(this.dockPanel);
                }

                return this.objJointEditorForm;
            }

            return null;
        }

        private void CloseJointEditor()
        {
            if (this.objJointEditorForm != null)
            {
                this.objJointEditorForm.Close();
                this.objJointEditorForm.Dispose();
                this.objJointEditorForm = null;
            }
        }

        private void objJointEditorForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiJointEditor.Checked = true;
        }

        private void objJointEditorForm_HandleDestroyed(object sender, EventArgs e)
        {
            this.mmiJointEditor.Checked = false;
        }

        #endregion

        #region Terrain RigidBodyEditorForm Method

        private Mix.Tool.Docking.IDockContent CreateTerrainRigidBodyEditor(bool show)
        {
            if (this.projectMgr.DrawObject == null)
            {
                return null;
            }

            if (this.projectMgr.DrawObject is Mix.Tool.Graphics.MapModel)
            {
                if ((this.terrainRigidBodyEditorForm == null) ||
                    (this.terrainRigidBodyEditorForm.IsDisposed == true))
                {
                    this.terrainRigidBodyEditorForm = new TerrainRigidBodyEditorForm(this.projectMgr.Dynamics.PhysicsMaterials);
                    this.terrainRigidBodyEditorForm.HandleCreated += new EventHandler(terrainRigidBodyEditorForm_HandleCreated);
                    this.terrainRigidBodyEditorForm.HandleDestroyed += new EventHandler(terrainRigidBodyEditorForm_HandleDestroyed);

                    if ((this.assetForm != null) &&
                        (this.assetForm.IsDisposed == false))
                    {
                        this.assetForm.RequestCollisionEvent();
                    }

                    if (show == true)
                    {
                        this.terrainRigidBodyEditorForm.Show(this.dockPanel);
                    }

                    return this.terrainRigidBodyEditorForm;
                }
            }

            return null;
        }

        private void CloseTerrainRigidBodyEditor()
        {
            if (this.terrainRigidBodyEditorForm != null)
            {
                this.terrainRigidBodyEditorForm.Close();
                this.terrainRigidBodyEditorForm.Dispose();
                this.terrainRigidBodyEditorForm = null;
            }
        }

        private void terrainRigidBodyEditorForm_HandleCreated(object sender, EventArgs e)
        {
            this.mmiTerrainRBEditor.Checked = true;
        }

        private void terrainRigidBodyEditorForm_HandleDestroyed(object sender, EventArgs e)
        {
            this.mmiTerrainRBEditor.Checked = false;
        }

        #endregion

        #region Project Method

        private Project.Manager CreateProject()
        {
            Project.Manager proj = new Project.Manager(
                this.backgroundWorker,
                this.graphicsMgr,
                this.scene,
                this,
                this.dockPanel,
                this.dockPanel_DeserializeDockContent);

            proj.BeginLoad += new System.EventHandler(this.project_BeginLoad);
            proj.EndLoad += new System.EventHandler(this.project_EndLoad);

            return proj;
        }

        private void project_BeginLoad(object sender, System.EventArgs e)
        {
            this.mmiFile.Enabled = false;
            this.mmiView.Enabled = false;
            this.mmiTool.Enabled = false;
            this.mmiHelp.Enabled = false;
            this.mmiEditStyle.Enabled = false;
        }

        private void project_EndLoad(object sender, System.EventArgs e)
        {
            if (this.projectMgr.DrawObject != null)
            {
                Mix.Tool.Graphics.DrawObject drawObj = this.projectMgr.DrawObject;

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // シーンの設定
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                this.scene.DrawObject = drawObj;

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // ダイナミクスワールドの設定
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                if (drawObj is Mix.Tool.Dynamics.Design.IBasicDocument)
                {
                    Mix.Tool.Dynamics.Design.IBasicDocument dynDoc = drawObj as Mix.Tool.Dynamics.Design.IBasicDocument;

                    if (dynDoc.DynamicsDesigner is Mix.Tool.Dynamics.Design.Actor)
                    {
                        Mix.Tool.Dynamics.Design.Actor dynActor = dynDoc.DynamicsDesigner as Mix.Tool.Dynamics.Design.Actor;

                        dynActor.World = this.dynamicsWorld;
                    }
                }

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // メニューの設定
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                if (this.projectMgr.DrawObject is Mix.Tool.Graphics.ObjectModel)
                {
                    this.mmiMotionSep.Available = true;
                    this.mmiMotionOutliner.Available = true;
                    this.mmiMotionEditor.Available = true;
                    this.mmiMotionManager.Available = true;
                    this.mmiCharacterEditor.Available = true;
                    this.mmiActorRBEditor.Available = true;
                    this.mmiTerrainRBEditor.Available = false;
                    this.mmiShapeEditor.Available = true;
                    this.mmiJointEditor.Available = true;
                }
                else
                {
                    this.mmiMotionSep.Available = false;
                    this.mmiMotionOutliner.Available = false;
                    this.mmiMotionEditor.Available = false;
                    this.mmiMotionManager.Available = false;
                    this.mmiCharacterEditor.Available = false;
                    this.mmiActorRBEditor.Available = false;
                    this.mmiTerrainRBEditor.Available = true;
                    this.mmiShapeEditor.Available = false;
                    this.mmiJointEditor.Available = false;
                }

                this.mmiFile.Enabled = true;
                this.mmiView.Enabled = true;
                this.mmiHelp.Enabled = true;
                this.mmiTool.Enabled = true;
                this.mmiEditStyle.Enabled = true;

                this.mmiSaveProject.Enabled = true;
                this.mmiPropProject.Enabled = true;
            }
            else
            {
                this.scene.DrawObject = null;

                this.mmiFile.Enabled = true;
                this.mmiView.Enabled = false;
                this.mmiHelp.Enabled = true;

                this.mmiSaveProject.Enabled = false;
                this.mmiPropProject.Enabled = false;
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // エディットスタイルの設定
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            System.Collections.Generic.List<EditStyleItem> editStyleItems = new List<EditStyleItem>();

            if (this.projectMgr.DrawObject is Mix.Tool.Graphics.ObjectModel)
            {
                editStyleItems.Add(new EditStyleItem(Properties.Resources.ES_MATERIAL, Project.EditStyle.Material));
                editStyleItems.Add(new EditStyleItem(Properties.Resources.ES_MOTION, Project.EditStyle.Motion));
                editStyleItems.Add(new EditStyleItem(Properties.Resources.ES_PHYSICS, Project.EditStyle.Physics));
            }
            else if (this.projectMgr.DrawObject is Mix.Tool.Graphics.MapModel)
            {
                editStyleItems.Add(new EditStyleItem(Properties.Resources.ES_MATERIAL, Project.EditStyle.Material));
                editStyleItems.Add(new EditStyleItem(Properties.Resources.ES_PHYSICS, Project.EditStyle.Physics));
            }

            //イベントを解除( 選択を通知させないため )
            this.mmiEditStyle.ComboBox.SelectedValueChanged -= new EventHandler(this.mmiEditStyle_SelectedValueChanged);

            this.mmiEditStyle.ComboBox.DisplayMember = "Display";
            this.mmiEditStyle.ComboBox.ValueMember = "Value";
            this.mmiEditStyle.ComboBox.DataSource = editStyleItems;
            this.mmiEditStyle.ComboBox.SelectedValue = this.projectMgr.EditStyle;

            //イベントを再登録
            this.mmiEditStyle.ComboBox.SelectedValueChanged += new EventHandler(this.mmiEditStyle_SelectedValueChanged);

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // タイトルの設定
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            this.UpdateTitle();

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // クローズを有効にする
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            this.SetCloseEnabled(true);
        }

        #endregion

        #region Other Method

        private void UpdateTitle()
        {
            if ((this.projectMgr == null) ||
                (this.projectMgr.DrawObject == null))
            {
                this.Text = Properties.Resources.TITLE;
            }
            else
            {
                this.Text = this.projectMgr.Name + " - " + Properties.Resources.TITLE;
            }
        }

        private void SetCloseEnabled(bool state)
        {
            IntPtr hMenu = GetSystemMenu(this.Handle, 0);

            this.closeEnabled = state;

            if (this.closeEnabled == true)
            {
                EnableMenuItem(hMenu, SC_CLOSE, MF_BYCOMMAND | MF_ENABLED);
            }
            else
            {
                EnableMenuItem(hMenu, SC_CLOSE, MF_BYCOMMAND | MF_GRAYED);
            }
        }

        #endregion
    }
}
