﻿using System;
using System.Windows.Forms;
using System.Drawing;

namespace Mix.Tool.ModelEditor
{
    public partial class MaterialEditorForm : Mix.Tool.Docking.DockContent
    {
        #region Private Member

        private Mix.Tool.Graphics.Manager manager = null;
        private MaterialEditorSource src = null;

        private Mix.Tool.Graphics.MaterialSlot materialSlot = null;
        private Mix.Tool.Graphics.Material material = null;

        #endregion

        #region Public Control Method

        /// <summary>
        /// ソースの取得、または設定
        /// </summary>
        public MaterialEditorSource Source
        {
            get { return this.src; }
            set
            {
                this.src = value;

                if (this.src != null)
                {
                    this.materialSlot = this.src.MaterialSlot;
                    this.material = this.src.Material;
                    this.UpdateControls(true);
                }
                else
                {
                    this.materialSlot = null;
                    this.material = null;
                    this.UpdateControls(false);
                }
            }
        }

        public MaterialEditorForm(Mix.Tool.Graphics.Manager _manager)
        {
            InitializeComponent();

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // マネージャ
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            this.manager = _manager;

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // タブコントロールの初期化
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            this.tabCtrl.SetExMode();

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // cbGeneShader の初期化
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            Utility.ComboBoxItem<Mix.Tool.Graphics.ShaderTypes>[] dsGeneShader = new Utility.ComboBoxItem<Mix.Tool.Graphics.ShaderTypes>[3];

            dsGeneShader[0] = new Utility.ComboBoxItem<Mix.Tool.Graphics.ShaderTypes>(Properties.Resources.JP_DEFAULT, Mix.Tool.Graphics.ShaderTypes.Normal);
            dsGeneShader[1] = new Utility.ComboBoxItem<Mix.Tool.Graphics.ShaderTypes>(Properties.Resources.JP_WATER, Mix.Tool.Graphics.ShaderTypes.Water);
            dsGeneShader[2] = new Utility.ComboBoxItem<Mix.Tool.Graphics.ShaderTypes>(Properties.Resources.JP_SHADELESS, Mix.Tool.Graphics.ShaderTypes.Shadeless);

            this.cbShader.DataSource = dsGeneShader;
            this.cbShader.DisplayMember = "Text";
            this.cbShader.ValueMember = "Value";

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // cbBlendMode の初期化
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            Utility.ComboBoxItem<Mix.Tool.Graphics.BlendModes>[] dsGeneBlendMode = new Utility.ComboBoxItem<Mix.Tool.Graphics.BlendModes>[6];

            dsGeneBlendMode[0] = new Utility.ComboBoxItem<Mix.Tool.Graphics.BlendModes>(Properties.Resources.JP_COPY, Mix.Tool.Graphics.BlendModes.Copy);
            dsGeneBlendMode[1] = new Utility.ComboBoxItem<Mix.Tool.Graphics.BlendModes>(Properties.Resources.JP_NORMAL, Mix.Tool.Graphics.BlendModes.Normal);
            dsGeneBlendMode[2] = new Utility.ComboBoxItem<Mix.Tool.Graphics.BlendModes>(Properties.Resources.JP_ADD, Mix.Tool.Graphics.BlendModes.Add);
            dsGeneBlendMode[3] = new Utility.ComboBoxItem<Mix.Tool.Graphics.BlendModes>(Properties.Resources.JP_SUB, Mix.Tool.Graphics.BlendModes.Sub);
            dsGeneBlendMode[4] = new Utility.ComboBoxItem<Mix.Tool.Graphics.BlendModes>(Properties.Resources.JP_MUL, Mix.Tool.Graphics.BlendModes.Mul);
            dsGeneBlendMode[5] = new Utility.ComboBoxItem<Mix.Tool.Graphics.BlendModes>(Properties.Resources.JP_SCREEN, Mix.Tool.Graphics.BlendModes.Screen);

            this.cbBlendMode.DataSource = dsGeneBlendMode;
            this.cbBlendMode.DisplayMember = "Text";
            this.cbBlendMode.ValueMember = "Value";

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // cbTexMappingFiltering の初期化
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            Utility.ComboBoxItem<Mix.Tool.Graphics.TransparencyMethods>[] dsTransparencyMethod = new Utility.ComboBoxItem<Mix.Tool.Graphics.TransparencyMethods>[2];

            dsTransparencyMethod[0] = new Utility.ComboBoxItem<Mix.Tool.Graphics.TransparencyMethods>(Properties.Resources.JP_DRAW_LUMP, Mix.Tool.Graphics.TransparencyMethods.Lump);
            dsTransparencyMethod[1] = new Utility.ComboBoxItem<Mix.Tool.Graphics.TransparencyMethods>(Properties.Resources.JP_DRAW_POLYGON, Mix.Tool.Graphics.TransparencyMethods.Individual);

            this.cbTransparencyMethod.DataSource = dsTransparencyMethod;
            this.cbTransparencyMethod.DisplayMember = "Text";
            this.cbTransparencyMethod.ValueMember = "Value";

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // cbTexMappingFiltering の初期化
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            Utility.ComboBoxItem<Mix.Tool.Graphics.TextureFilterTypes>[] dsTexMappingFiltering = new Utility.ComboBoxItem<Mix.Tool.Graphics.TextureFilterTypes>[3];

            dsTexMappingFiltering[0] = new Utility.ComboBoxItem<Mix.Tool.Graphics.TextureFilterTypes>(Properties.Resources.JP_POINT, Mix.Tool.Graphics.TextureFilterTypes.Point);
            dsTexMappingFiltering[1] = new Utility.ComboBoxItem<Mix.Tool.Graphics.TextureFilterTypes>(Properties.Resources.JP_LINEAR, Mix.Tool.Graphics.TextureFilterTypes.Linear);
            dsTexMappingFiltering[2] = new Utility.ComboBoxItem<Mix.Tool.Graphics.TextureFilterTypes>(Properties.Resources.JP_ANISOTROPIC, Mix.Tool.Graphics.TextureFilterTypes.Anisotropic);

            this.cbTexMappingFiltering.DataSource = dsTexMappingFiltering;
            this.cbTexMappingFiltering.DisplayMember = "Text";
            this.cbTexMappingFiltering.ValueMember = "Value";

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // cbTexMappingAddressing の初期化
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            Utility.ComboBoxItem<Mix.Tool.Graphics.TextureAddressTypes>[] dsTexMappingAddressing = new Utility.ComboBoxItem<Mix.Tool.Graphics.TextureAddressTypes>[2];

            dsTexMappingAddressing[0] = new Utility.ComboBoxItem<Mix.Tool.Graphics.TextureAddressTypes>(Properties.Resources.JP_WRAP, Mix.Tool.Graphics.TextureAddressTypes.Wrap);
            dsTexMappingAddressing[1] = new Utility.ComboBoxItem<Mix.Tool.Graphics.TextureAddressTypes>(Properties.Resources.JP_CLAMP, Mix.Tool.Graphics.TextureAddressTypes.Clamp);

            this.cbTexMappingAddressing.DataSource = dsTexMappingAddressing;
            this.cbTexMappingAddressing.DisplayMember = "Text";
            this.cbTexMappingAddressing.ValueMember = "Value";

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // cbDiffuseType の初期化
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            Utility.ComboBoxItem<Mix.Tool.Graphics.DiffuseTypes>[] dsDiffuseType = new Utility.ComboBoxItem<Mix.Tool.Graphics.DiffuseTypes>[2];

            dsDiffuseType[0] = new Utility.ComboBoxItem<Mix.Tool.Graphics.DiffuseTypes>(Properties.Resources.JP_LAMBERT, Mix.Tool.Graphics.DiffuseTypes.Lambert);
            dsDiffuseType[1] = new Utility.ComboBoxItem<Mix.Tool.Graphics.DiffuseTypes>(Properties.Resources.JP_HALF_LAMBERT, Mix.Tool.Graphics.DiffuseTypes.HalfLambert);

            this.cbDiffuseType.DataSource = dsDiffuseType;
            this.cbDiffuseType.DisplayMember = "Text";
            this.cbDiffuseType.ValueMember = "Value";

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // cbSpecularType の初期化
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            Utility.ComboBoxItem<Mix.Tool.Graphics.SpecularTypes>[] dsSpecularType = new Utility.ComboBoxItem<Mix.Tool.Graphics.SpecularTypes>[4];

            dsSpecularType[0] = new Utility.ComboBoxItem<Mix.Tool.Graphics.SpecularTypes>(Properties.Resources.JP_NONE, Mix.Tool.Graphics.SpecularTypes.None);
            dsSpecularType[1] = new Utility.ComboBoxItem<Mix.Tool.Graphics.SpecularTypes>(Properties.Resources.JP_PHONG, Mix.Tool.Graphics.SpecularTypes.Phong);
            dsSpecularType[2] = new Utility.ComboBoxItem<Mix.Tool.Graphics.SpecularTypes>(Properties.Resources.JP_BLINN_PHONG, Mix.Tool.Graphics.SpecularTypes.BlinnPhong);
            dsSpecularType[3] = new Utility.ComboBoxItem<Mix.Tool.Graphics.SpecularTypes>(Properties.Resources.JP_COOK_TORRANCE, Mix.Tool.Graphics.SpecularTypes.CookTorrance);

            this.cbSpecularType.DataSource = dsSpecularType;
            this.cbSpecularType.DisplayMember = "Text";
            this.cbSpecularType.ValueMember = "Value";

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // cbReflect の初期化
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            Utility.ComboBoxItem<Mix.Tool.Graphics.ReflectTypes>[] dsReflectType = new Utility.ComboBoxItem<Mix.Tool.Graphics.ReflectTypes>[3];

            dsReflectType[0] = new Utility.ComboBoxItem<Mix.Tool.Graphics.ReflectTypes>(Properties.Resources.JP_NONE, Mix.Tool.Graphics.ReflectTypes.None);
            dsReflectType[1] = new Utility.ComboBoxItem<Mix.Tool.Graphics.ReflectTypes>(Properties.Resources.JP_NORMAL, Mix.Tool.Graphics.ReflectTypes.Normal);
            dsReflectType[2] = new Utility.ComboBoxItem<Mix.Tool.Graphics.ReflectTypes>(Properties.Resources.JP_FRESNEL, Mix.Tool.Graphics.ReflectTypes.Fresnel);

            this.cbReflectType.DataSource = dsReflectType;
            this.cbReflectType.DisplayMember = "Text";
            this.cbReflectType.ValueMember = "Value";

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // cbBumpWaveType の初期化
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            Utility.ComboBoxItem<Mix.Tool.Graphics.WaveTypes>[] dsWaveType = new Utility.ComboBoxItem<Mix.Tool.Graphics.WaveTypes>[2];

            dsWaveType[0] = new Utility.ComboBoxItem<Mix.Tool.Graphics.WaveTypes>(Properties.Resources.JP_NORMAL, Mix.Tool.Graphics.WaveTypes.Normal);
            dsWaveType[1] = new Utility.ComboBoxItem<Mix.Tool.Graphics.WaveTypes>(Properties.Resources.JP_REFRACTION, Mix.Tool.Graphics.WaveTypes.Refract);

            this.cbBumpWaveType.DataSource = dsWaveType;
            this.cbBumpWaveType.DisplayMember = "Text";
            this.cbBumpWaveType.ValueMember = "Value";

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // cbBumpWaveDir の初期化
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            Utility.ComboBoxItem<Mix.Tool.Graphics.WaveDirs>[] dsWaveDir = new Utility.ComboBoxItem<Mix.Tool.Graphics.WaveDirs>[4];

            dsWaveDir[0] = new Utility.ComboBoxItem<Mix.Tool.Graphics.WaveDirs>(Properties.Resources.JP_UV_UP, Mix.Tool.Graphics.WaveDirs.NegativeV);
            dsWaveDir[1] = new Utility.ComboBoxItem<Mix.Tool.Graphics.WaveDirs>(Properties.Resources.JP_UV_DOWN, Mix.Tool.Graphics.WaveDirs.PositiveV);
            dsWaveDir[2] = new Utility.ComboBoxItem<Mix.Tool.Graphics.WaveDirs>(Properties.Resources.JP_UV_LEFT, Mix.Tool.Graphics.WaveDirs.NegativeU);
            dsWaveDir[3] = new Utility.ComboBoxItem<Mix.Tool.Graphics.WaveDirs>(Properties.Resources.JP_UV_RIGHT, Mix.Tool.Graphics.WaveDirs.PositiveU);

            this.cbBumpWaveDir.DataSource = dsWaveDir;
            this.cbBumpWaveDir.DisplayMember = "Text";
            this.cbBumpWaveDir.ValueMember = "Value";

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // その他の初期化
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            this.UpdateControls(false);
        }

        #endregion

        #region General

        private void cbShader_SelectedValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.ShaderType = (Mix.Tool.Graphics.ShaderTypes)this.cbShader.SelectedValue;
                this.UpdateControls(true);
            }
        }

        private void cpAmbientColor_ChangedEditColor(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.AmbientColor = new Mix.Tool.Math.Vector(this.cpAmbientColor.EditColor);
            }
        }

        private void cbBothLighting_CheckedChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.BothLighting = this.cbBothLighting.Checked;
            }
        }

        private void cbBlendMode_SelectedValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.BlendMode = (Mix.Tool.Graphics.BlendModes)this.cbBlendMode.SelectedValue;
            }
        }

        private void cbTransparency_CheckedChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.TransparencyEnabled = this.cbTransparency.Checked;

                this.UpdateTransparencyControls();
                this.UpdateDiffuseControls();
                this.UpdateSpecularControls();
                this.UpdateBumpControls();
            }
        }

        private void cbTransparencyMethod_SelectedValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.TransparencyMethod = (Mix.Tool.Graphics.TransparencyMethods)this.cbTransparencyMethod.SelectedValue;
            }
        }

        private void cbZWrite_CheckedChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.ZWrite = this.cbZWrite.Checked;
            }
        }

        private void cbSoftEdge_CheckedChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.SoftEdgeEnabled = this.cbSoftEdge.Checked;
                this.UpdateTransparencyControls();
            }
        }

        private void nudSoftEdge_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.SoftEdgeTickness = (float)Convert.ToDouble(this.nudSoftEdge.Value);
            }
        }

        private void cbBackculling_CheckedChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Backculling = this.cbBackculling.Checked;
            }
        }

        private void cbTexMappingFiltering_SelectedValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.TextureMapping.FilterType = (Mix.Tool.Graphics.TextureFilterTypes)this.cbTexMappingFiltering.SelectedValue;
            }
        }

        private void cbTexMappingAddressing_SelectedValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.TextureMapping.AddressType = (Mix.Tool.Graphics.TextureAddressTypes)this.cbTexMappingAddressing.SelectedValue;
            }
        }

        private void cbShadowReceive_CheckedChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.ShadowMapping.Receive = this.cbShadowReceive.Checked;
            }
        }

        private void cbShadowCast_CheckedChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.ShadowMapping.Cast = this.cbShadowCast.Checked;
            }
        }

        private void tbOutput_TextChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.SaveFilePath = this.tbOutput.Text;
            }
        }

        private void btOutput_Click(object sender, EventArgs e)
        {
            using (SaveFileDialog dlg = new SaveFileDialog())
            {
                if (this.tbOutput.Text.Length > 0)
                {
                    try
                    {
                        dlg.InitialDirectory = System.IO.Path.GetDirectoryName(this.tbOutput.Text);
                    }
                    catch
                    {
                        dlg.InitialDirectory = System.IO.Path.GetPathRoot(this.tbOutput.Text);
                    }

                    dlg.FileName = System.IO.Path.GetFileName(this.tbOutput.Text);
                }
                else
                {
                    dlg.InitialDirectory = "";
                    dlg.FileName = this.material.Name + Properties.Resources.FILE_EXT_MTL;
                }

                dlg.Filter = Properties.Resources.FILE_FILTER_MTL;
                dlg.FilterIndex = 1;

                if (dlg.ShowDialog() == DialogResult.OK)
                {
                    this.tbOutput.Text = dlg.FileName;
                }
            }
        }

        #endregion

        #region Diffuse

        private void cbDiffuseType_SelectedValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Diffuse.Type = (Mix.Tool.Graphics.DiffuseTypes)this.cbDiffuseType.SelectedValue;
            }
        }

        private void cpDiffuseColor_ChangedEditColor(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Diffuse.Color = new Mix.Tool.Math.Vector(this.cpDiffuseColor.EditColor);
            }
        }

        private void tbDiffuseTex_TextChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Diffuse.LoadTexture(this.tbDiffuseTex.Text);
            }
        }

        private void btDiffuseTex_Click(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.SelectTextureFile(this.tbDiffuseTex);
            }
        }

        private void btDiffuseTex_Clear_Click(object sender, EventArgs e)
        {
            this.tbDiffuseTex.Text = "";
        }

        private void cbDiffuseFresnel_CheckedChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Diffuse.Fresnel = this.cbDiffuseFresnel.Checked;
                this.UpdateDiffuseControls();
            }
        }

        private void nudDiffuseFresnel_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Diffuse.FresnelRatio = NUDtoTB(this.nudDiffuseFresnel, this.tbDiffuseFresnel);
            }
        }

        private void tbDiffuseFresnel_ValueChanged(object sender, EventArgs e)
        {
            TBtoNUD(this.tbDiffuseFresnel, this.nudDiffuseFresnel);
        }

        private void cbDiffuseRL_CheckedChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Diffuse.RimLighting = this.cbDiffuseRL.Checked;
                this.UpdateDiffuseControls();
            }
        }

        private void nudDiffuseRLHardness_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Diffuse.RimLightingHardness = (float)Convert.ToDouble(this.nudDiffuseRLHardness.Value);
            }
        }

        private void nudDiffuseRLScale_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Diffuse.RimLightingScale = (float)Convert.ToDouble(this.nudDiffuseRLScale.Value);
            }
        }

        #endregion

        #region Specular

        private void cbSpecularType_SelectedValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Specular.Type = (Mix.Tool.Graphics.SpecularTypes)this.cbSpecularType.SelectedValue;
                this.UpdateSpecularControls();
            }
        }

        private void cpSpecularColor_ChangedEditColor(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Specular.Color = new Mix.Tool.Math.Vector(this.cpSpecularColor.EditColor);
            }
        }

        private void tbSpecularTex_TextChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Specular.LoadTexture(this.tbSpecularTex.Text);
            }
        }

        private void btSpecularTex_Click(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.SelectTextureFile(this.tbSpecularTex);
            }
        }

        private void btSpecularTex_Clear_Click(object sender, EventArgs e)
        {
            this.tbSpecularTex.Text = "";
        }

        private void nudSpecularHardness_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Specular.Hardness = (float)Convert.ToDouble(this.nudSpecularHardness.Value);
            }
        }

        private void nudSpecularScale_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Specular.Scale = (float)Convert.ToDouble(this.nudSpecularScale.Value);
            }
        }

        private void nudSpecularFresnel_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Specular.CookTorranceFresnel = NUDtoTB(this.nudSpecularFresnel, this.tbSpecularFresnel);
            }
        }

        private void tbSpecularFresnel_ValueChanged(object sender, EventArgs e)
        {
            TBtoNUD(this.tbSpecularFresnel, this.nudSpecularFresnel);
        }

        private void nudSpecularRoughness_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Specular.CookTorranceRoughness = NUDtoTB(this.nudSpecularRoughness, this.tbSpecularRoughness);
            }
        }

        private void tbSpecularRoughness_ValueChanged(object sender, EventArgs e)
        {
            TBtoNUD(this.tbSpecularRoughness, this.nudSpecularRoughness);
        }

        #endregion

        #region Specular ( Reflection )

        private void cbReflect_SelectedValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Specular.ReflectType = (Mix.Tool.Graphics.ReflectTypes)this.cbReflectType.SelectedValue;
                this.UpdateSpecularControls();
                this.UpdateBumpControls();
            }
        }

        private void nudReflectScale_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Specular.ReflectScale = (float)Convert.ToDouble(this.nudReflectScale.Value);
            }
        }

        private void nudReflectIntensity_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Specular.ReflectIntensity = NUDtoTB(this.nudReflectIntensity, this.tbReflectIntensity);
            }
        }

        private void tbReflectIntensity_ValueChanged(object sender, EventArgs e)
        {
            TBtoNUD(this.tbReflectIntensity, this.nudReflectIntensity);
        }

        private void nudReflectBias_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Specular.ReflectBias = NUDtoTB(this.nudReflectBias, this.tbReflectBias);
            }
        }

        private void tbReflectBias_ValueChanged(object sender, EventArgs e)
        {
            TBtoNUD(this.tbReflectBias, this.nudReflectBias);
        }

        private void nudReflectExprosure_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Specular.ReflectExprosure = (float)Convert.ToDouble(this.nudReflectExprosure.Value);
            }
        }

        private void nudReflectBump_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Specular.ReflectBumpRatio = NUDtoTB(this.nudReflectBump, this.tbReflectBump);
            }
        }

        private void tbReflectBump_ValueChanged(object sender, EventArgs e)
        {
            TBtoNUD(this.tbReflectBump, this.nudReflectBump);
        }

        #endregion

        #region Emissive

        private void cpEmissiveColor_ChangedEditColor(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Emissive.Color = new Mix.Tool.Math.Vector(this.cpEmissiveColor.EditColor);
            }
        }

        private void tbEmissiveTex_TextChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Emissive.LoadTexture(this.tbEmissiveTex.Text);
            }
        }

        private void btEmissiveTex_Click(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.SelectTextureFile(this.tbEmissiveTex);
            }
        }

        private void btEmissiveTex_Clear_Click(object sender, EventArgs e)
        {
            this.tbEmissiveTex.Text = "";
        }

        private void nudEmissiveScale_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Emissive.Scale = (float)Convert.ToDouble(this.nudEmissiveScale.Value);
            }
        }

        #endregion

        #region Bump

        private void tbBumpTex_TextChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Bump.LoadTexture(this.tbBumpTex.Text);

                this.UpdateSpecularControls();
                this.UpdateBumpControls();
            }
        }

        private void btBumpTex_Click(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.SelectTextureFile(this.tbBumpTex);
            }
        }

        private void btBumpTex_Clear_Click(object sender, EventArgs e)
        {
            this.tbBumpTex.Text = "";
        }

        private void cbBumpParallax_CheckedChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Bump.ParallaxEnabled = this.cbBumpParallax.Checked;
                this.UpdateBumpControls();
            }
        }

        private void nudBumpParallaxScale_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Bump.ParallaxHeightScale = NUDtoTB(this.nudBumpParallaxScale, this.tbBumpParallaxScale);
            }
        }

        private void tbBumpParallaxScale_ValueChanged(object sender, EventArgs e)
        {
            TBtoNUD(this.tbBumpParallaxScale, this.nudBumpParallaxScale);
        }

        private void nudBumpParallaxSample_ValueChanged(object sender, EventArgs e)
        {
            int value = Convert.ToInt32(this.nudBumpParallaxSample.Value);

            if (this.material != null)
            {
                this.material.Bump.ParallaxSample = value;
            }

            this.tbBumpParallaxSample.Value = value;
        }

        private void tbBumpParallaxSample_ValueChanged(object sender, EventArgs e)
        {
            this.nudBumpParallaxSample.Value = Convert.ToDecimal(this.tbBumpParallaxSample.Value);
        }

        private void cbBumpWaveType_SelectedValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Bump.WaveType = (Mix.Tool.Graphics.WaveTypes)this.cbBumpWaveType.SelectedValue;
                this.UpdateBumpControls();
            }
        }

        private void cbBumpWaveDir_SelectedValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Bump.WaveDir = (Mix.Tool.Graphics.WaveDirs)this.cbBumpWaveDir.SelectedValue;
            }
        }

        private void cbBumpWaveCross_CheckedChanged(object sender, EventArgs e)
        {

        }

        private void cbBumpWaveInvert_CheckedChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Bump.WaveInvert = this.cbBumpWaveInvert.Checked;
            }
        }

        private void nudBumpWaveVel_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Bump.WaveVelocity = NUDtoTB(this.nudBumpWaveAnimVel, this.tbBumpWaveAnimVel);
            }
        }

        private void tbBumpWaveVel_ValueChanged(object sender, EventArgs e)
        {
            TBtoNUD(this.tbBumpWaveAnimVel, this.nudBumpWaveAnimVel);
        }

        private void nudBumpWaveUV_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Bump.WaveUVOffset = NUDtoTB(this.nudBumpWaveAnimUV, this.tbBumpWaveAnimUV);
            }
        }

        private void tbBumpWaveUV_ValueChanged(object sender, EventArgs e)
        {
            TBtoNUD(this.tbBumpWaveAnimUV, this.nudBumpWaveAnimUV);
        }

        private void nudBumpWaveAnimGap_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Bump.WaveGap = (float)Convert.ToDouble(this.nudBumpWaveAnimGap.Value);
            }
        }

        private void nudBumpRefractIntensity_ValueChanged(object sender, EventArgs e)
        {
            if (this.material != null)
            {
                this.material.Bump.WaveRefraction = NUDtoTB(this.nudBumpRefractIntensity, this.tbBumpRefractIntensity);
            }
        }

        private void tbBumpRefractIntensity_ValueChanged(object sender, EventArgs e)
        {
            TBtoNUD(this.tbBumpRefractIntensity, this.nudBumpRefractIntensity);
        }

        #endregion

        #region Private Other Method

        private void UpdateControls(bool state)
        {
            if ((this.material != null) &&
                (state == true))
            {
                this.tabCtrl.Enabled = true;

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // タイトル
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                this.Text = string.Format("{0} ({1}-{2})", Properties.Resources.TITLE_MATERIAL_EDITOR, this.materialSlot.Name, this.material.Name);

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // 更新を開始
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                this.material.AutoUpdate = false;

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // 全般
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                bool isLighting = this.material.ShaderType != Mix.Tool.Graphics.ShaderTypes.Shadeless;

                // シェーダー //

                this.cbShader.SelectedValue = this.material.ShaderType;
                this.lbShader.Enabled = true;
                this.cbShader.Enabled = true;

                // アンビエントカラー //

                this.lbAmbientColor.Enabled = isLighting;
                this.cpAmbientColor.EditColor = this.material.AmbientColor.ToColor();
                this.cpAmbientColor.Enabled = isLighting;

                // 両面ライティング //

                this.cbBothLighting.Checked = this.material.BothLighting;
                this.cbBothLighting.Enabled = isLighting;

                // ブレンド //

                this.cbBlendMode.SelectedValue = this.material.BlendMode;
                this.lbBlend.Enabled = true;
                this.cbBlendMode.Enabled = true;

                // 半透明 //

                this.cbTransparency.Checked = this.material.TransparencyEnabled;
                this.cbTransparencyMethod.SelectedValue = this.material.TransparencyMethod;
                this.cbZWrite.Checked = this.material.ZWrite;
                this.cbSoftEdge.Checked = this.material.SoftEdgeEnabled;
                this.nudSoftEdge.Value = Convert.ToDecimal(this.material.SoftEdgeTickness);
                this.UpdateTransparencyControls();

                // 背面カリング //

                this.cbBackculling.Checked = this.material.Backculling;
                this.cbBackculling.Enabled = true;

                // テクスチャマッピング //

                this.cbTexMappingFiltering.SelectedValue = this.material.TextureMapping.FilterType;
                this.cbTexMappingAddressing.SelectedValue = this.material.TextureMapping.AddressType;

                this.partTexMapping.Enabled = true;
                this.lbTexMappingFiltering.Enabled = true;
                this.cbTexMappingFiltering.Enabled = true;
                this.lbTexMappingAddressing.Enabled = true;
                this.cbTexMappingAddressing.Enabled = true;

                // シャドウ //

                bool shadowEnabled = ((this.manager.EffectPackage.ProcessFlags & Mix.Tool.Graphics.ShaderProcessFlags.Shadow) == Mix.Tool.Graphics.ShaderProcessFlags.Shadow);

                this.cbShadowReceive.Checked = this.material.ShadowMapping.Receive;
                this.cbShadowCast.Checked = this.material.ShadowMapping.Cast;

                this.partShadow.Enabled = shadowEnabled;
                this.cbShadowReceive.Enabled = shadowEnabled;
                this.cbShadowCast.Enabled = shadowEnabled;

                // 出力 //

                bool outputEnabled = (this.material.IsDefault == false);

                this.tbOutput.Text = this.material.SaveFilePath;

                this.partOutput.Enabled = outputEnabled;
                this.lbOutput.Enabled = outputEnabled;
                this.tbOutput.Enabled = outputEnabled;
                this.btOutput.Enabled = outputEnabled;

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // ディフューズ
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                this.UpdateDiffuseControls();

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // スペキュラー
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                this.UpdateSpecularControls();

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // エミッシブ
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                this.UpdateEmissiveControls();

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // バンプ
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                this.UpdateBumpControls();

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // 更新を終了
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                this.material.AutoUpdate = true;
                this.material.Update();
            }
            else
            {
                foreach (Control ctrl in this.Controls)
                {
                    ctrl.Enabled = false;
                }
            }
        }

        private void UpdateTransparencyControls()
        {
            bool enabled = this.material.TransparencyEnabled;

            this.cbTransparency.Enabled = true;
            this.cbTransparencyMethod.Enabled = enabled;
            this.cbZWrite.Enabled = enabled;
            this.cbSoftEdge.Enabled = enabled;
            this.nudSoftEdge.Enabled = enabled && this.cbSoftEdge.Checked;
        }

        private void UpdateDiffuseControls()
        {
            bool lightingEnabled;
            bool texEnabled;
            bool fresnelEnabled;
            bool rimLightingEnabled;

            lightingEnabled = this.material.ShaderType != Mix.Tool.Graphics.ShaderTypes.Shadeless;
            texEnabled = this.material.ShaderType != Mix.Tool.Graphics.ShaderTypes.Water;
            fresnelEnabled = ((lightingEnabled == true) && (this.material.Diffuse.Fresnel == true));
            rimLightingEnabled = ((lightingEnabled == true) && (this.material.Diffuse.RimLighting == true));

            this.cbDiffuseType.SelectedValue = this.material.Diffuse.Type;
            this.cpDiffuseColor.EditColor = this.material.Diffuse.Color.ToColor();
            this.tbDiffuseTex.Text = this.material.Diffuse.TextureFileName;
            this.cbDiffuseFresnel.Checked = this.material.Diffuse.Fresnel;
            this.nudDiffuseFresnel.Value = Convert.ToDecimal(this.material.Diffuse.FresnelRatio);
            this.cbDiffuseRL.Checked = this.material.Diffuse.RimLighting;
            this.nudDiffuseRLHardness.Value = Convert.ToDecimal(this.material.Diffuse.RimLightingHardness);
            this.nudDiffuseRLScale.Value = Convert.ToDecimal(this.material.Diffuse.RimLightingScale);

            this.partDiffuseGene.Enabled = true;
            this.lbDiffuseType.Enabled = lightingEnabled;
            this.cbDiffuseType.Enabled = lightingEnabled;
            this.lbDiffuseColor.Enabled = true;
            this.cpDiffuseColor.Enabled = true;
            this.lbDiffuseTex.Enabled = texEnabled;
            this.tbDiffuseTex.Enabled = texEnabled;
            this.btDiffuseTex.Enabled = texEnabled;
            this.btDiffuseTex_Clear.Enabled = texEnabled;

            this.cbDiffuseFresnel.Enabled = lightingEnabled;
            this.nudDiffuseFresnel.Enabled = fresnelEnabled;
            this.tbDiffuseFresnel.Enabled = fresnelEnabled;
            this.cbDiffuseRL.Enabled = lightingEnabled;
            this.lbDiffuseRLHardness.Enabled = rimLightingEnabled;
            this.nudDiffuseRLHardness.Enabled = rimLightingEnabled;
            this.lbDiffuseRLScale.Enabled = rimLightingEnabled;
            this.nudDiffuseRLScale.Enabled = rimLightingEnabled;
        }

        private void UpdateSpecularControls()
        {
            bool lightingEnabled;
            bool lightingChildEnabled;
            bool texEnabled;
            bool cookTorranceEnabled;
            bool bumpEnabled;
            bool reflectEnabled;
            bool reflectFresnelEnabled;
            bool reflectBumpEnabled;

            this.cbSpecularType.SelectedValue = this.material.Specular.Type;
            this.cpSpecularColor.EditColor = this.material.Specular.Color.ToColor();
            this.tbSpecularTex.Text = this.material.Specular.TextureFileName;
            this.nudSpecularHardness.Value = Convert.ToDecimal(this.material.Specular.Hardness);
            this.nudSpecularScale.Value = Convert.ToDecimal(this.material.Specular.Scale);
            this.nudSpecularFresnel.Value = Convert.ToDecimal(this.material.Specular.CookTorranceFresnel);
            this.nudSpecularRoughness.Value = Convert.ToDecimal(this.material.Specular.CookTorranceRoughness);
            this.cbReflectType.SelectedValue = this.material.Specular.ReflectType;
            this.nudReflectIntensity.Value = Convert.ToDecimal(this.material.Specular.ReflectIntensity);
            this.nudReflectBias.Value = Convert.ToDecimal(this.material.Specular.ReflectBias);
            this.nudReflectExprosure.Value = Convert.ToDecimal(this.material.Specular.ReflectExprosure);
            this.nudReflectBump.Value = Convert.ToDecimal(this.material.Specular.ReflectBumpRatio);

            if (this.material.ShaderType != Mix.Tool.Graphics.ShaderTypes.Shadeless)
            {
                lightingEnabled = true;
                lightingChildEnabled = this.material.Specular.Type != Mix.Tool.Graphics.SpecularTypes.None;
                texEnabled = (lightingChildEnabled == true) && (this.material.ShaderType != Mix.Tool.Graphics.ShaderTypes.Water);
                cookTorranceEnabled = this.material.Specular.Type == Mix.Tool.Graphics.SpecularTypes.CookTorrance;
                bumpEnabled = (this.tbBumpTex.Text.Length > 0);
            }
            else
            {
                lightingEnabled = false;
                lightingChildEnabled = false;
                texEnabled = false;
                cookTorranceEnabled = false;
                bumpEnabled = false;
            }

            if (lightingChildEnabled == true)
            {
                reflectEnabled = (this.material.Specular.ReflectType != Mix.Tool.Graphics.ReflectTypes.None);
                reflectFresnelEnabled = (reflectEnabled == true) && (this.material.Specular.ReflectType == Mix.Tool.Graphics.ReflectTypes.Fresnel);
                reflectBumpEnabled = (reflectEnabled == true) && (bumpEnabled == true);
            }
            else
            {
                reflectEnabled = false;
                reflectFresnelEnabled = false;
                reflectBumpEnabled = false;
            }

            this.partSpecularGene.Enabled = lightingEnabled;
            this.lbSpecularType.Enabled = lightingEnabled;
            this.cbSpecularType.Enabled = lightingEnabled;
            this.lbSpecularColor.Enabled = lightingChildEnabled;
            this.cpSpecularColor.Enabled = lightingChildEnabled;
            this.lbSpecularTex.Enabled = texEnabled;
            this.tbSpecularTex.Enabled = texEnabled;
            this.btSpecularTex.Enabled = texEnabled;
            this.btSpecularTex_Clear.Enabled = texEnabled;
            this.lbSpecularHardness.Enabled = lightingChildEnabled;
            this.nudSpecularHardness.Enabled = lightingChildEnabled;
            this.lbSpecularScale.Enabled = lightingChildEnabled;
            this.nudSpecularScale.Enabled = lightingChildEnabled;
            this.lbSpecularFresnel.Enabled = cookTorranceEnabled;
            this.nudSpecularFresnel.Enabled = cookTorranceEnabled;
            this.tbSpecularFresnel.Enabled = cookTorranceEnabled;
            this.lbSpecularRoughness.Enabled = cookTorranceEnabled;
            this.nudSpecularRoughness.Enabled = cookTorranceEnabled;
            this.tbSpecularRoughness.Enabled = cookTorranceEnabled;

            this.partSpecularReflect.Enabled = lightingChildEnabled;
            this.lbReflectType.Enabled = lightingChildEnabled;
            this.cbReflectType.Enabled = lightingChildEnabled;
            this.lbReflectScale.Enabled = reflectEnabled;
            this.nudReflectScale.Enabled = reflectEnabled;
            this.lbReflectIntensity.Enabled = reflectEnabled;
            this.tbReflectIntensity.Enabled = reflectEnabled;
            this.nudReflectIntensity.Enabled = reflectEnabled;
            this.lbReflectBias.Enabled = reflectFresnelEnabled;
            this.tbReflectBias.Enabled = reflectFresnelEnabled;
            this.nudReflectBias.Enabled = reflectFresnelEnabled;
            this.lbReflectExprosure.Enabled = reflectFresnelEnabled;
            this.nudReflectExprosure.Enabled = reflectFresnelEnabled;
            this.lbReflectBump.Enabled = reflectBumpEnabled;
            this.nudReflectBump.Enabled = reflectBumpEnabled;
            this.tbReflectBump.Enabled = reflectBumpEnabled;
        }

        private void UpdateEmissiveControls()
        {
            bool texEnabled = this.material.ShaderType != Mix.Tool.Graphics.ShaderTypes.Water;

            this.cpEmissiveColor.EditColor = this.material.Emissive.Color.ToColor();
            this.tbEmissiveTex.Text = this.material.Emissive.TextureFileName;

            this.lbEmissiveColor.Enabled = true;
            this.cpEmissiveColor.Enabled = true;
            this.lbEmissiveTex.Enabled = texEnabled;
            this.tbEmissiveTex.Enabled = texEnabled;
            this.btEmissiveTex.Enabled = texEnabled;
            this.btEmissiveTex_Clear.Enabled = texEnabled;
        }

        private void UpdateBumpControls()
        {
            Mix.Tool.Graphics.ShaderProcessFlags procFlags = this.manager.EffectPackage.ProcessFlags;

            bool enabled;
            bool texEnabled;
            bool parallaxEnabled;
            bool reflectEnabled;
            bool waveEnabled;
            bool refractEnabled;

            this.tbBumpTex.Text = this.material.Bump.TextureFileName;
            this.cbBumpParallax.Checked = this.material.Bump.ParallaxEnabled;
            this.nudBumpParallaxScale.Value = Convert.ToDecimal(this.material.Bump.ParallaxHeightScale);
            this.nudBumpParallaxSample.Value = Convert.ToDecimal(this.material.Bump.ParallaxSample);

            this.cbBumpWaveType.SelectedValue = this.material.Bump.WaveType;
            this.cbBumpWaveDir.SelectedValue = this.material.Bump.WaveDir;
            this.cbBumpWaveInvert.Checked = this.material.Bump.WaveInvert;
            this.nudBumpWaveAnimVel.Value = Convert.ToDecimal(this.material.Bump.WaveVelocity);
            this.nudBumpWaveAnimUV.Value = Convert.ToDecimal(this.material.Bump.WaveUVOffset);
            this.nudBumpWaveAnimGap.Value = Convert.ToDecimal(this.material.Bump.WaveGap);

            this.nudBumpRefractIntensity.Value = Convert.ToDecimal(this.material.Bump.WaveRefraction);

            if (this.material.ShaderType != Mix.Tool.Graphics.ShaderTypes.Shadeless)
            {
                enabled = true;
                texEnabled = (this.tbBumpTex.Text.Length > 0);

                if (texEnabled == true)
                {
                    parallaxEnabled = this.material.Bump.ParallaxEnabled;

                    if ((this.material.ShaderType != Mix.Tool.Graphics.ShaderTypes.Shadeless) &&
                        (this.material.Specular.Type != Mix.Tool.Graphics.SpecularTypes.None))
                    {
                        reflectEnabled = (this.material.Specular.ReflectType != Mix.Tool.Graphics.ReflectTypes.None);
                    }
                    else
                    {
                        reflectEnabled = false;
                    }

                    if (this.material.ShaderType == Mix.Tool.Graphics.ShaderTypes.Water)
                    {
                        waveEnabled = true;

                        if ((this.material.TransparencyEnabled == true) &&
                            ((procFlags & Mix.Tool.Graphics.ShaderProcessFlags.Refract) == Mix.Tool.Graphics.ShaderProcessFlags.Refract) &&
                            (this.material.Bump.WaveType == Mix.Tool.Graphics.WaveTypes.Refract))
                        {
                            refractEnabled = true;
                        }
                        else
                        {
                            refractEnabled = false;
                        }
                    }
                    else
                    {
                        waveEnabled = false;
                        refractEnabled = false;
                    }
                }
                else
                {
                    parallaxEnabled = false;
                    reflectEnabled = false;
                    waveEnabled = false;
                    refractEnabled = false;
                }
            }
            else
            {
                enabled = false;
                texEnabled = false;
                parallaxEnabled = false;
                reflectEnabled = false;
                waveEnabled = false;
                refractEnabled = false;
            }

            this.partBumpGene.Enabled = enabled;
            this.lbBumpTex.Enabled = enabled;
            this.tbBumpTex.Enabled = enabled;
            this.btBumpTex.Enabled = enabled;
            this.btBumpTex_Clear.Enabled = enabled;
            this.cbBumpParallax.Enabled = texEnabled;
            this.lbBumpParallaxScale.Enabled = texEnabled;
            this.tbBumpParallaxScale.Enabled = parallaxEnabled;
            this.nudBumpParallaxScale.Enabled = parallaxEnabled;
            this.lbBumpParallaxSample.Enabled = texEnabled;
            this.tbBumpParallaxSample.Enabled = parallaxEnabled;
            this.nudBumpParallaxSample.Enabled = parallaxEnabled;

            this.partBumpWave.Enabled = waveEnabled;
            this.lbBumpWaveType.Enabled = waveEnabled;
            this.cbBumpWaveType.Enabled = waveEnabled;
            this.lbBumpWaveDir.Enabled = waveEnabled;
            this.cbBumpWaveDir.Enabled = waveEnabled;
            this.cbBumpWaveInvert.Enabled = waveEnabled;
            this.lbBumpWaveAnimVel.Enabled = waveEnabled;
            this.tbBumpWaveAnimVel.Enabled = waveEnabled;
            this.nudBumpWaveAnimVel.Enabled = waveEnabled;
            this.lbBumpWaveAnimUV.Enabled = waveEnabled;
            this.tbBumpWaveAnimUV.Enabled = waveEnabled;
            this.nudBumpWaveAnimUV.Enabled = waveEnabled;
            this.lbBumpWaveAnimGap.Enabled = waveEnabled;
            this.nudBumpWaveAnimGap.Enabled = waveEnabled;
            this.lbBumpRefractIntensity.Enabled = refractEnabled;
            this.nudBumpRefractIntensity.Enabled = refractEnabled;
            this.tbBumpRefractIntensity.Enabled = refractEnabled;
        }

        private void SelectTextureFile(TextBox textBox)
        {
            using (OpenFileDialog dlg = new OpenFileDialog())
            {
                dlg.Filter = Properties.Resources.FILE_FILTER_TEXTURE;
                dlg.FilterIndex = 6;
                dlg.FileName = textBox.Text;

                if (dlg.ShowDialog() == DialogResult.OK)
                {
                    textBox.Text = dlg.FileName;
                }
            }
        }

        private static decimal TBtoNUD(int value, decimal divValue, decimal minValue, decimal maxValue)
        {
            decimal ret;

            ret = Convert.ToDecimal(value) / divValue;

            if (minValue > ret)
            {
                ret = minValue;
            }
            else if (maxValue < ret)
            {
                ret = maxValue;
            }

            return ret;
        }

        private static void TBtoNUD(System.Windows.Forms.TrackBar tb, System.Windows.Forms.NumericUpDown nud)
        {
            decimal value;

            value = Convert.ToDecimal(tb.Value) / Convert.ToDecimal(tb.Maximum);

            if (nud.Minimum > value)
            {
                value = nud.Minimum;
            }
            else if (nud.Maximum < value)
            {
                value = nud.Maximum;
            }

            nud.Value = value;
        }

        private static int NUDtoTB(decimal value, decimal mulValue, int minValue, int maxValue)
        {
            int ret = Convert.ToInt32(value * mulValue);

            if (minValue > ret)
            {
                ret = minValue;
            }
            else if (maxValue < ret)
            {
                ret = maxValue;
            }

            return ret;
        }

        private static float NUDtoTB(System.Windows.Forms.NumericUpDown nud, System.Windows.Forms.TrackBar tb)
        {
            int value = Convert.ToInt32(nud.Value * Convert.ToDecimal(tb.Maximum));

            if (tb.Minimum > value)
            {
                value = tb.Minimum;
            }
            else if (tb.Maximum < value)
            {
                value = tb.Maximum;
            }

            tb.Value = value;

            return (float)Convert.ToDouble(nud.Value);
        }

        #endregion
    }
}
