﻿using System;
using System.Xml;
using System.Text;
using System.IO;
using System.Collections.Generic;
using System.Drawing;
using System.ComponentModel;

namespace Mix.Tool.ModelEditor.Project
{
    /// <summary>
    /// エディットスタイル列挙定数
    /// </summary>
    public enum EditStyle
    {
        /// <summary>
        /// マテリアル
        /// </summary>
        Material = 0,

        /// <summary>
        /// モーション
        /// </summary>
        Motion = 1,

        /// <summary>
        /// フィジクス
        /// </summary>
        Physics = 2,
    };

    /// <summary>
    /// マネージャークラス
    /// </summary>
    public class Manager : IDisposable
    {
        #region BackgorundTask

        /// <summary>
        /// マップコリジョンインポートタスククラス
        /// </summary>
        public class ImportMapCollisionTask : Mix.Tool.ModelEditor.Project.IBackgroundTask
        {
            private Mix.Tool.Dynamics.Design.Terrain terrain = null;
            private string filePath = "";
            private int serializeCapacity = 0;

            public ImportMapCollisionTask(Mix.Tool.Dynamics.Design.Terrain _terrain, string _filePath, int _serializeCapacity)
            {
                if (_terrain == null)
                {
                    throw new System.ArgumentNullException("_terrain");
                }

                if (_terrain.Status != Mix.Tool.Dynamics.Design.TerrainStatus.Destroyed)
                {
                    throw new System.Exception("破棄されている状態でないとインポートはできません");
                }

                if (_filePath == null)
                {
                    throw new System.ArgumentNullException("_filePath");
                }

                if (_serializeCapacity <= 0)
                {
                    throw new System.ArgumentOutOfRangeException("_serializeCapacity は 1以上でなければなりません");
                }

                this.terrain = _terrain;
                this.terrain.Initialize();

                this.filePath = _filePath;
                this.serializeCapacity = _serializeCapacity;
            }

            public void OnExecute()
            {
                try
                {
                    this.terrain.Import(this.filePath, this.serializeCapacity);
                }
                catch
                {
                    Mix.Tool.Logger.Print(Logger.Type.Info, "マップコリジョンをインポートに失敗しました(Import)");
                }
            }

            public void OnFinish()
            {
                try
                {
                    this.terrain.ImportFinalize();
                    Mix.Tool.Logger.Print(Logger.Type.Info, "マップコリジョンをインポートしました");
                }
                catch
                {
                    Mix.Tool.Logger.Print(Logger.Type.Info, "マップコリジョンをインポートに失敗しました(ImportFinalize)");
                }
            }
        }

        /// <summary>
        /// モデルのインポートタスククラス
        /// </summary>
        private abstract class ImportModelTask
        {
            #region Protected Member

            protected Mix.Tool.Dynamics.PhysicsMaterialCollection physicsMaterials = null;
            protected Mix.Tool.Graphics.Model model = null;
            protected XmlElement elmModel = null;
            protected List<XmlElement> elmMaterialSlots = null;

            #endregion

            #region Public Property

            /// <summary>
            /// モデルの取得
            /// </summary>
            public Mix.Tool.Graphics.Model Model
            {
                get { return this.model; }
            }

            #endregion

            #region Public Method

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="_physicsMaterials"></param>
            /// <param name="_model"></param>
            /// <param name="_elmModel"></param>
            /// <param name="_elmMaterialSlots"></param>
            public ImportModelTask(
                Mix.Tool.Dynamics.PhysicsMaterialCollection _physicsMaterials,
                Mix.Tool.Graphics.Model _model,
                XmlElement _elmModel,
                List<XmlElement> _elmMaterialSlots)
            {
                this.physicsMaterials = _physicsMaterials;
                this.model = _model;
                this.elmModel = _elmModel;
                this.elmMaterialSlots = _elmMaterialSlots;
            }

            #endregion

            #region Protected Method

            /// <summary>
            /// ダイナミクス : アクターを処理する際に呼び出されます
            /// </summary>
            /// <param name="elmActor"></param>
            protected virtual void OnDynamicsActor(
                XmlElement elmActor,
                Mix.Tool.Dynamics.Design.BoundingBoxCollection boundingBoxes,
                Mix.Tool.Dynamics.Design.Actor actor,
                string name) { }

            /// <summary>
            /// ダイナミクス : パートを処理する際に呼び出されます
            /// </summary>
            /// <param name="elmActor"></param>
            protected virtual void OnDynamicsPart(
                XmlElement elmPart,
                Mix.Tool.Dynamics.Design.BoundingBoxCollection boundingBoxes,
                Mix.Tool.Dynamics.Design.Part part,
                string nodeName ) { }

            /// <summary>
            /// ダイナミクス : テラインを処理する際に呼び出されます
            /// </summary>
            /// <param name="elmActor"></param>
            protected virtual void OnDynamicsTerrain(
                XmlElement elmTerrain,
                Mix.Tool.Dynamics.Design.Terrain terrain) { }

            /// <summary>
            /// モデルを適用
            /// </summary>
            protected void Execute_ApplyModel()
            {
                if (this.elmModel != null)
                {
                    foreach (XmlElement elmChild in this.elmModel.ChildNodes)
                    {
                        if (elmChild.Name.Equals(Properties.Resources.XML_PRJ_NODE) == true)
                        {
                            //ルートノード
                            try
                            {
                                XmlAttribute attrName = elmChild.Attributes[Properties.Resources.XML_PRJ_NAME];

                                if (this.model.RootNode.Name.Equals(attrName.Value) == true)
                                {
                                    this.Execute_ApplyNode(elmChild, this.model.RootNode);
                                }
                            }
                            catch
                            {
                            }
                        }
                        else if (elmChild.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS) == true)
                        {
                            //ダイナミクス
                            this.Execute_ApplyModel_Dynamics(elmChild);
                        }
                    }
                }
            }

            /// <summary>
            /// マテリアルスロットを適用
            /// </summary>
            protected void Finish_ApplyMaterialSlots()
            {
                if (this.elmMaterialSlots == null)
                {
                    return;
                }

                foreach (XmlElement elmMaterialSlot in this.elmMaterialSlots)
                {
                    string defaultName;
                    string name;
                    Mix.Tool.Graphics.MaterialSlot materialSlot;

                    defaultName = elmMaterialSlot.Attributes[Properties.Resources.XML_PRJ_DEFAULTNAME].Value;
                    name = elmMaterialSlot.Attributes[Properties.Resources.XML_PRJ_NAME].Value;

                    materialSlot = this.model.MaterialSlots[defaultName];
                    if (materialSlot != null)
                    {
                        materialSlot.Name = name;

                        foreach (XmlElement elmMaterial in elmMaterialSlot.ChildNodes)
                        {
                            if (elmMaterial.Name.Equals(Properties.Resources.XML_PRJ_MATERIAL) == false)
                            {
                                continue;
                            }

                            bool def;
                            bool actv;
                            Mix.Tool.Graphics.Material material;

                            //マテリアルの基本情報を取得
                            defaultName = elmMaterial.Attributes[Properties.Resources.XML_PRJ_DEFAULTNAME].Value;
                            name = elmMaterial.Attributes[Properties.Resources.XML_PRJ_NAME].Value;
                            def = bool.Parse(elmMaterial.Attributes[Properties.Resources.XML_PRJ_DEFAULT].Value);
                            actv = bool.Parse(elmMaterial.Attributes[Properties.Resources.XML_PRJ_ACTIVE].Value);

                            //デフォルトの設定
                            if (def == true)
                            {
                                material = materialSlot.Materials[defaultName];
                            }
                            else
                            {
                                material = materialSlot.Materials.Add();
                            }

                            //アクティブの設定
                            if (actv == true)
                            {
                                material.Activate();
                            }

                            //名前の設定
                            material.Name = name;

                            //シェーダーの設定
                            material.AutoUpdate = false;

                            foreach (XmlElement elmMP in elmMaterial.ChildNodes)
                            {
                                if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_SAVEFILEPATH) == true)
                                {
                                    try { material.SaveFilePath = elmMP.Attributes[Properties.Resources.XML_PRJ_VALUE].Value; }
                                    catch { }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_SHADERTYPE) == true)
                                {
                                    try { material.ShaderType = (Mix.Tool.Graphics.ShaderTypes)Enum.Parse(typeof(Mix.Tool.Graphics.ShaderTypes), elmMP.Attributes[Properties.Resources.XML_PRJ_VALUE].Value); }
                                    catch { }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_AMBIENT_COLOR) == true)
                                {
                                    try { material.AmbientColor = Mix.Tool.Math.Vector.Parse(elmMP.Attributes[Properties.Resources.XML_PRJ_VALUE].Value); }
                                    catch { }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_BOTH_LIGHTING) == true)
                                {
                                    try { material.BothLighting = bool.Parse(elmMP.Attributes[Properties.Resources.XML_PRJ_VALUE].Value); }
                                    catch { }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_BLENDMODE) == true)
                                {
                                    try { material.BlendMode = (Mix.Tool.Graphics.BlendModes)Enum.Parse(typeof(Mix.Tool.Graphics.BlendModes), elmMP.Attributes[Properties.Resources.XML_PRJ_VALUE].Value); }
                                    catch { }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_TRANSPARENCY) == true)
                                {
                                    try { material.TransparencyEnabled = bool.Parse(elmMP.Attributes[Properties.Resources.XML_PRJ_VALUE].Value); }
                                    catch { }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_TRANSPARENCY_METHOD) == true)
                                {
                                    try { material.TransparencyMethod = (Mix.Tool.Graphics.TransparencyMethods)Enum.Parse(typeof(Mix.Tool.Graphics.TransparencyMethods), elmMP.Attributes[Properties.Resources.XML_PRJ_VALUE].Value); }
                                    catch { }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_ZWRITE) == true)
                                {
                                    try { material.ZWrite = bool.Parse(elmMP.Attributes[Properties.Resources.XML_PRJ_VALUE].Value); }
                                    catch { }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_SOFTEDGE) == true)
                                {
                                    try { material.SoftEdgeEnabled = bool.Parse(elmMP.Attributes[Properties.Resources.XML_PRJ_VALUE].Value); }
                                    catch { }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_SOFTEDGE_TICKNESS) == true)
                                {
                                    try { material.SoftEdgeTickness = float.Parse(elmMP.Attributes[Properties.Resources.XML_PRJ_VALUE].Value); }
                                    catch { }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_BACKCULLING) == true)
                                {
                                    try { material.Backculling = bool.Parse(elmMP.Attributes[Properties.Resources.XML_PRJ_VALUE].Value); }
                                    catch { }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_TEXTUREMAPPING) == true)
                                {
                                    foreach (XmlElement elm in elmMP.ChildNodes)
                                    {
                                        try
                                        {
                                            if (elm.Name.Equals(Properties.Resources.XML_PRJ_FILTER) == true)
                                            {
                                                material.TextureMapping.FilterType = (Mix.Tool.Graphics.TextureFilterTypes)Enum.Parse(typeof(Mix.Tool.Graphics.TextureFilterTypes),
                                                    elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_ADDRESS) == true)
                                            {
                                                material.TextureMapping.AddressType = (Mix.Tool.Graphics.TextureAddressTypes)Enum.Parse(typeof(Mix.Tool.Graphics.TextureAddressTypes),
                                                    elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                        }
                                        catch
                                        {
                                        }
                                    }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_SHADOWMAPPING) == true)
                                {
                                    foreach (XmlElement elm in elmMP.ChildNodes)
                                    {
                                        try
                                        {
                                            if (elm.Name.Equals(Properties.Resources.XML_PRJ_RECEIVE) == true)
                                            {
                                                material.ShadowMapping.Receive = bool.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_CAST) == true)
                                            {
                                                material.ShadowMapping.Cast = bool.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                        }
                                        catch
                                        {
                                        }
                                    }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_DIFFUSE) == true)
                                {
                                    foreach (XmlElement elm in elmMP.ChildNodes)
                                    {
                                        try
                                        {
                                            if (elm.Name.Equals(Properties.Resources.XML_PRJ_TYPE) == true)
                                            {
                                                material.Diffuse.Type = (Mix.Tool.Graphics.DiffuseTypes)Enum.Parse(typeof(Mix.Tool.Graphics.DiffuseTypes),
                                                    elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_COLOR) == true)
                                            {
                                                material.Diffuse.Color = Mix.Tool.Math.Vector.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_TEXTUREFILENAME) == true)
                                            {
                                                material.Diffuse.LoadTexture(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_FRESNEL) == true)
                                            {
                                                material.Diffuse.Fresnel = bool.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_FRESNEL_INTENSITY) == true)
                                            {
                                                material.Diffuse.FresnelRatio = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_RL) == true)
                                            {
                                                material.Diffuse.RimLighting = bool.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_RL_HARDNESS) == true)
                                            {
                                                material.Diffuse.RimLightingHardness = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_RL_SCALE) == true)
                                            {
                                                material.Diffuse.RimLightingScale = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                        }
                                        catch
                                        {
                                        }
                                    }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_SPECULAR) == true)
                                {
                                    foreach (XmlElement elm in elmMP.ChildNodes)
                                    {
                                        try
                                        {
                                            if (elm.Name.Equals(Properties.Resources.XML_PRJ_TYPE) == true)
                                            {
                                                material.Specular.Type = (Mix.Tool.Graphics.SpecularTypes)Enum.Parse(typeof(Mix.Tool.Graphics.SpecularTypes),
                                                    elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_COLOR) == true)
                                            {
                                                material.Specular.Color = Mix.Tool.Math.Vector.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_TEXTUREFILENAME) == true)
                                            {
                                                material.Specular.LoadTexture(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_HARDNESS) == true)
                                            {
                                                material.Specular.Hardness = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_SCALE) == true)
                                            {
                                                material.Specular.Scale = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_FRESNEL) == true)
                                            {
                                                material.Specular.CookTorranceFresnel = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_ROUGHNESS) == true)
                                            {
                                                material.Specular.CookTorranceRoughness = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_REFLECT_TYPE) == true)
                                            {
                                                material.Specular.ReflectType = (Mix.Tool.Graphics.ReflectTypes)Enum.Parse(typeof(Mix.Tool.Graphics.ReflectTypes),
                                                    elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_REFLECT_SCALE) == true)
                                            {
                                                material.Specular.ReflectScale = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_REFLECT_INTENSITY) == true)
                                            {
                                                material.Specular.ReflectIntensity = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_REFLECT_BIAS) == true)
                                            {
                                                material.Specular.ReflectBias = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_REFLECT_EXP) == true)
                                            {
                                                material.Specular.ReflectExprosure = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_REFLECT_BUMP_RATIO) == true)
                                            {
                                                material.Specular.ReflectBumpRatio = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                        }
                                        catch
                                        {
                                        }
                                    }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_EMISSIVE) == true)
                                {
                                    foreach (XmlElement elm in elmMP.ChildNodes)
                                    {
                                        try
                                        {
                                            if (elm.Name.Equals(Properties.Resources.XML_PRJ_COLOR) == true)
                                            {
                                                material.Emissive.Color = Mix.Tool.Math.Vector.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_TEXTUREFILENAME) == true)
                                            {
                                                material.Emissive.LoadTexture(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_SCALE) == true)
                                            {
                                                material.Emissive.Scale = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                        }
                                        catch
                                        {
                                        }
                                    }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_BUMP) == true)
                                {
                                    foreach (XmlElement elm in elmMP.ChildNodes)
                                    {
                                        try
                                        {
                                            if (elm.Name.Equals(Properties.Resources.XML_PRJ_TEXTUREFILENAME) == true)
                                            {
                                                material.Bump.LoadTexture(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_PARALLAXMAPPING) == true)
                                            {
                                                material.Bump.ParallaxEnabled = bool.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_PARALLAXMAPPING_HS) == true)
                                            {
                                                material.Bump.ParallaxHeightScale = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_PARALLAXMAPPING_SAMPLE) == true)
                                            {
                                                material.Bump.ParallaxSample = int.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_WAVE_TYPE) == true)
                                            {
                                                material.Bump.WaveType = (Mix.Tool.Graphics.WaveTypes)Enum.Parse(typeof(Mix.Tool.Graphics.WaveTypes),
                                                    elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_WAVE_DIR) == true)
                                            {
                                                material.Bump.WaveDir = (Mix.Tool.Graphics.WaveDirs)Enum.Parse(typeof(Mix.Tool.Graphics.WaveDirs),
                                                    elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_WAVE_INVERT) == true)
                                            {
                                                material.Bump.WaveInvert = bool.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_WAVE_VEL) == true)
                                            {
                                                material.Bump.WaveVelocity = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_WAVE_UV) == true)
                                            {
                                                material.Bump.WaveUVOffset = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_WAVE_GAP) == true)
                                            {
                                                material.Bump.WaveGap = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_WAVE_REFRACTION) == true)
                                            {
                                                material.Bump.WaveRefraction = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_REFRACT_INTENSITY) == true)
                                            {
                                                //破棄予定( 名前の変更 )
                                                material.Bump.WaveRefraction = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_REFLECT_INTENSITY) == true)
                                            {
                                                //破棄予定( スペキュラーの移動 )
                                                material.Specular.ReflectBumpRatio = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                        }
                                        catch
                                        {
                                        }
                                    }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_SHADOW) == true)
                                {
                                    //破棄予定( 名前の変更 )
                                    foreach (XmlElement elm in elmMP.ChildNodes)
                                    {
                                        try
                                        {
                                            if (elm.Name.Equals(Properties.Resources.XML_PRJ_RECEIVE) == true)
                                            {
                                                material.ShadowMapping.Receive = bool.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_CAST) == true)
                                            {
                                                material.ShadowMapping.Cast = bool.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                        }
                                        catch
                                        {
                                        }
                                    }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_PARALLAXMAPPING_SAMPLE) == true)
                                {
                                    //破棄予定( バグ )
                                    foreach (XmlElement elm in elmMP.ChildNodes)
                                    {
                                        try
                                        {
                                            if (elm.Name.Equals(Properties.Resources.XML_PRJ_SOFT) == true)
                                            {
                                                material.SoftEdgeEnabled = bool.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_SOFT_THICKNESS) == true)
                                            {
                                                material.SoftEdgeTickness = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                        }
                                        catch
                                        {
                                        }
                                    }
                                }
                                else if (elmMP.Name.Equals(Properties.Resources.XML_PRJ_AMBIENT) == true)
                                {
                                    //破棄予定( 移動 )
                                    try
                                    {
                                        foreach (XmlElement elm in elmMP.ChildNodes)
                                        {
                                            if (elm.Name.Equals(Properties.Resources.XML_PRJ_COLOR) == true)
                                            {
                                                material.AmbientColor = Mix.Tool.Math.Vector.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                            }
                                        }
                                    }
                                    catch
                                    {
                                    }
                                }
                            }

                            material.AutoUpdate = true;
                            material.Update();
                        }
                    }
                    else
                    {
                        Mix.Tool.Logger.Print(Logger.Type.Warning, Properties.Resources.PROJECT_NOTFOUND_MATERIALSLOT_DM + " : DefaultName[" + defaultName + "] Name[" + name + "]");

                        foreach (XmlElement elmMaterial in elmMaterialSlot.ChildNodes)
                        {
                            if (elmMaterial.Name.Equals(Properties.Resources.XML_PRJ_MATERIAL) == false)
                            {
                                try
                                {
                                    defaultName = elmMaterial.Attributes[Properties.Resources.XML_PRJ_DEFAULTNAME].Value;
                                    name = elmMaterial.Attributes[Properties.Resources.XML_PRJ_NAME].Value;

                                    Mix.Tool.Logger.Print(Logger.Type.Warning, "  DefaultName[" + defaultName + "] Name[" + name + "]");
                                }
                                catch
                                {
                                }
                            }
                        }
                    }
                }
            }

            #endregion

            #region Private Method

            /// <summary>
            /// ノードを適用
            /// </summary>
            /// <param name="elmNode"></param>
            /// <param name="node"></param>
            private void Execute_ApplyNode(XmlElement elmNode, Mix.Tool.Graphics.Node node)
            {
                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // ノードを処理
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                try
                {
                    XmlAttribute attrType = elmNode.Attributes[Properties.Resources.XML_PRJ_TYPE];

                    if (attrType.Value.Equals(Properties.Resources.XML_PRJ_EMPTY))
                    {
                        //エンプティノード
                    }
                    else if (attrType.Value.Equals(Properties.Resources.XML_PRJ_MESH))
                    {
                        //メッシュノード
                        if (node is Mix.Tool.Graphics.MeshNode)
                        {
                            Mix.Tool.Graphics.MeshNode meshNode = node as Mix.Tool.Graphics.MeshNode;

                            try
                            {
                                XmlAttribute attrVisible = elmNode.Attributes[Properties.Resources.XML_PRJ_VISIBLE];
                                meshNode.Visible = bool.Parse(attrVisible.Value);
                            }
                            catch
                            {
                                meshNode.Visible = true;
                            }
                        }
                    }
                    else if (attrType.Value.Equals(Properties.Resources.XML_PRJ_BONE))
                    {
                        //ボーンノード
                    }
                }
                catch
                {
                }

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // 子を処理
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                foreach (XmlElement elmChild in elmNode.ChildNodes)
                {
                    if (elmChild.Name.Equals(Properties.Resources.XML_PRJ_NODE) == true)
                    {
                        //ノード
                        try
                        {
                            XmlAttribute attrName = elmChild.Attributes[Properties.Resources.XML_PRJ_NAME];
                            int childIndex = node.Childs.IndexOfName(attrName.Value);

                            if (childIndex >= 0)
                            {
                                this.Execute_ApplyNode(elmChild, node.Childs[childIndex]);
                            }
                        }
                        catch
                        {
                        }
                    }
                    else if (elmChild.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS) == true)
                    {
                        //ダイナミクス
                        this.Execute_ApplyNode_Dynamics(elmChild, node);
                    }
                }
            }

            /// <summary>
            /// モデルのダイナミクスを適用
            /// </summary>
            /// <param name="elmDynamics"></param>
            private void Execute_ApplyModel_Dynamics(XmlElement elmDynamics)
            {
                foreach (XmlElement elmChild in elmDynamics.ChildNodes)
                {
                    if (elmChild.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_ACTOR) == true)
                    {
                        if (this.model is Mix.Tool.Graphics.ObjectModel)
                        {
                            Mix.Tool.Graphics.ObjectModel objModel = this.model as Mix.Tool.Graphics.ObjectModel;
                            if (objModel.DynamicsDesigner is Mix.Tool.Dynamics.Design.Actor)
                            {
                                this.OnDynamicsActor(
                                    elmChild,
                                    objModel.BoundingBoxes,
                                    objModel.DynamicsDesigner as Mix.Tool.Dynamics.Design.Actor,
                                    "");
                            }
                        }
                    }
                    else if (elmChild.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_TERRAIN) == true)
                    {
                        if (this.model is Mix.Tool.Graphics.MapModel)
                        {
                            Mix.Tool.Graphics.MapModel mapModel = this.model as Mix.Tool.Graphics.MapModel;
                            if (mapModel.DynamicsDesigner is Mix.Tool.Dynamics.Design.Terrain)
                            {
                                this.OnDynamicsTerrain(
                                    elmChild,
                                    mapModel.DynamicsDesigner as Mix.Tool.Dynamics.Design.Terrain);
                            }
                        }
                    }
                }
            }

            /// <summary>
            /// ノードのダイナミクスを適用
            /// </summary>
            /// <param name="elmDynamics"></param>
            private void Execute_ApplyNode_Dynamics(XmlElement elmDynamics, Mix.Tool.Graphics.Node node)
            {
                foreach (XmlElement elmChild in elmDynamics.ChildNodes)
                {
                    if (elmChild.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_PART) == true)
                    {
                        if (node.DynamicsDesigner is Mix.Tool.Dynamics.Design.Part)
                        {
                            this.OnDynamicsPart(
                                elmChild,
                                node.BoundingBoxes,
                                node.DynamicsDesigner as Mix.Tool.Dynamics.Design.Part,
                                node.Name);
                        }
                    }
                }
            }

            #endregion
        };

        /// <summary>
        /// オブジェクトモデルのインポートタスククラス
        /// </summary>
        private class ImportObjectModelTask : ImportModelTask, Mix.Tool.ModelEditor.Project.IBackgroundTask
        {
            #region Private Member

            private Mix.Tool.Graphics.ObjectModel.ImportConfig importConfig = null;
            private Mix.Tool.Graphics.ObjectModel objectModel = null;
            private List<XmlElement> elmMotionControllers = null;
            private XmlElement elmUnassignedMotions = null;
            private XmlElement elmRootMotion = null;

            #endregion

            #region Public Method

            public ImportObjectModelTask(
                Mix.Tool.Dynamics.PhysicsMaterialCollection _physicsMaterials,
                Mix.Tool.Graphics.ObjectModel.ImportConfig _importConfig,
                Mix.Tool.Graphics.ObjectModel _objectModel,
                XmlElement _elmModel,
                List<XmlElement> _elmMaterialSlots,
                List<XmlElement> _elmMotionControllers,
                XmlElement _elmUnassignedMotions,
                XmlElement _elmRootMotion)
                : base(_physicsMaterials, _objectModel, _elmModel, _elmMaterialSlots)
            {
                this.importConfig = _importConfig;
                this.objectModel = _objectModel;
                this.elmMotionControllers = _elmMotionControllers;
                this.elmUnassignedMotions = _elmUnassignedMotions;
                this.elmRootMotion = _elmRootMotion;
            }

            #endregion

            #region Private Method

            /// <summary>
            /// モーションコントローラーの適用
            /// </summary>
            private void Finish_ApplyMotionControllers()
            {
                if (this.elmMotionControllers != null)
                {
                    try
                    {
                        Mix.Tool.Graphics.ObjectModel objModel = this.model as Mix.Tool.Graphics.ObjectModel;
                        Mix.Tool.Graphics.MotionController uaMotCtrl = objModel.UnassignedMotionController;

                        //分類済み
                        foreach (XmlElement elmMotCtrl in this.elmMotionControllers)
                        {
                            Mix.Tool.Graphics.MotionController motCtrl = objModel.MotionControllers.Add();

                            motCtrl.Name = elmMotCtrl.Attributes[Properties.Resources.XML_PRJ_NAME].Value;
                            motCtrl.Priority = uint.Parse(elmMotCtrl.Attributes[Properties.Resources.XML_PRJ_PRIORITY].Value);

                            foreach (XmlElement elmMot in elmMotCtrl.ChildNodes)
                            {
                                string defaultName;
                                string name;
                                Mix.Tool.Graphics.Motion mot;

                                //名前を取得
                                defaultName = elmMot.Attributes[Properties.Resources.XML_PRJ_DEFAULTNAME].Value;
                                name = elmMot.Attributes[Properties.Resources.XML_PRJ_NAME].Value;

                                //モーションを取得
                                mot = uaMotCtrl.Motions[defaultName];
                                if (mot != null)
                                {
                                    //未分類からモーションコントローラーへ移動
                                    uaMotCtrl.Motions.Remove(mot);
                                    motCtrl.Motions.Add(mot);

                                    //名前を設定
                                    mot.Name = name;

                                    //残りを設定
                                    this.Finish_ApplyMotionState(elmMot, mot);
                                }
                                else
                                {
                                    //見つからないので警告を出す
                                    Mix.Tool.Logger.Print(Logger.Type.Warning,
                                        Properties.Resources.PROJECT_NOTFOUND_MOTION_FAILED_MOVE + " : ControllerName[" + motCtrl.Name + "] DefaultName[" + defaultName + "] Name[" + name + "]");
                                }
                            }
                        }

                        objModel.MotionControllers.Sort();
                    }
                    catch
                    {
                        Mix.Tool.Logger.Print(Logger.Type.Error, " モーションコントローラーの設定に失敗 : プロジェクトファイルのフォーマットが異常です");
                    }
                }

                if (this.elmUnassignedMotions != null)
                {
                    Mix.Tool.Graphics.MotionController uaMotCtrl = this.objectModel.UnassignedMotionController;

                    try
                    {
                        foreach (XmlElement elmMot in this.elmUnassignedMotions.ChildNodes)
                        {
                            string defaultName;
                            string name;
                            Mix.Tool.Graphics.Motion mot;

                            //名前を取得
                            defaultName = elmMot.Attributes[Properties.Resources.XML_PRJ_DEFAULTNAME].Value;
                            name = elmMot.Attributes[Properties.Resources.XML_PRJ_NAME].Value;

                            //モーションを取得
                            mot = uaMotCtrl.Motions[defaultName];
                            if (mot != null)
                            {
                                //名前を設定
                                mot.Name = name;

                                //残りを設定
                                this.Finish_ApplyMotionState(elmMot, mot);
                            }
                            else
                            {
                                //未分類だからとくに警告はださなくていいかな、、、
                            }
                        }
                    }
                    catch
                    {
                        Mix.Tool.Logger.Print(Logger.Type.Error, "  未分類モーションの設定に失敗 : プロジェクトファイルのフォーマットが異常です");
                    }
                }
            }

            /// <summary>
            /// ルートモーションの適用
            /// </summary>
            private void Finish_ApplyRootMotion()
            {
                if (this.elmRootMotion != null)
                {
                    this.objectModel.RootTransform.BeginUpdate();

                    try
                    {
                        foreach (XmlElement elm in this.elmRootMotion.ChildNodes)
                        {
                            if (elm.Name.Equals(Properties.Resources.XML_PRJ_ACTIVE) == true)
                            {
                                this.objectModel.RootTransform.Active = (Mix.Tool.Graphics.RootTransformFlags)Enum.Parse(
                                    typeof(Mix.Tool.Graphics.RootTransformFlags),
                                    elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_NODE_NAME) == true)
                            {
                                this.objectModel.RootTransform.NodeName = elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value;
                            }
                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_THORIZ) == true)
                            {
                                this.objectModel.RootTransform.TranslationHorizonal = (Mix.Tool.Graphics.RootTransformHorizonal)Enum.Parse(
                                    typeof(Mix.Tool.Graphics.RootTransformHorizonal),
                                    elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_TVERT) == true)
                            {
                                this.objectModel.RootTransform.TranslationVertical = (Mix.Tool.Graphics.RootTransformVertical)Enum.Parse(
                                    typeof(Mix.Tool.Graphics.RootTransformVertical),
                                    elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_RAXIS) == true)
                            {
                                this.objectModel.RootTransform.RotationAxis = Mix.Tool.Graphics.RootTransformAxis.Parse(
                                    elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                            else if (elm.Name.Equals(Properties.Resources.XML_PRJ_TAXIS) == true)
                            {
                                this.objectModel.RootTransform.TranslationAxis = Mix.Tool.Graphics.RootTransformAxis.Parse(
                                    elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                        }
                    }
                    catch
                    {
                        Mix.Tool.Logger.Print(Logger.Type.Error, "  ルートモーションの設定に失敗 : プロジェクトファイルのフォーマットが異常です");
                    }

                    this.objectModel.RootTransform.EndUpdate();
                }
            }

            #endregion

            #region Private Misc Method

            /// <summary>
            /// モーションステートの適用
            /// </summary>
            /// <param name="elmMot"></param>
            /// <param name="mot"></param>
            private void Finish_ApplyMotionState(XmlElement elmMot, Mix.Tool.Graphics.Motion mot)
            {
                try
                {
                    mot.Default = bool.Parse(elmMot.Attributes[Properties.Resources.XML_PRJ_DEFAULT].Value);

                    foreach (XmlElement elmParam in elmMot.ChildNodes)
                    {
                        if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_MOTCURVE) == true)
                        {
                            string linkNodeName;
                            Mix.Tool.Graphics.MotionCurve motCurve;

                            linkNodeName = elmParam.Attributes[Properties.Resources.XML_PRJ_LINKNODENAME].Value;

                            motCurve = mot.Curves[linkNodeName];
                            if (motCurve == null)
                            {
                                continue;
                            }

                            motCurve.Enabled = bool.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_ENABLED].Value);
                        }
                        else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_LOOP) == true)
                        {
                            mot.Loop = bool.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_SPEED) == true)
                        {
                            mot.Speed = float.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_LOOPSTARTTIME) == true)
                        {
                            mot.LoopStartTime = int.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_LOOPENDTIME) == true)
                        {
                            mot.LoopEndTime = int.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_BLENDRATIO) == true)
                        {
                            mot.BlendRatio = float.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_ROOTTRANSFORM) == true)
                        {
                            foreach (XmlElement elm in elmParam.ChildNodes)
                            {
                                if (elm.Name.Equals(Properties.Resources.XML_PRJ_FLAGS) == true)
                                {
                                    mot.RootTransform = (Mix.Tool.Graphics.RootTransformFlags)Enum.Parse(
                                        typeof(Mix.Tool.Graphics.RootTransformFlags),
                                        elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                }
                            }
                        }
                        else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_SAVEFILEPATH) == true)
                        {
                            mot.SaveFilePath = elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value;
                        }
                    }
                }
                catch
                {
                    throw;
                }
            }

            #endregion

            #region ImportModelTask

            protected override void OnDynamicsActor(
                XmlElement elmActor,
                Mix.Tool.Dynamics.Design.BoundingBoxCollection boundingBoxes,
                Mix.Tool.Dynamics.Design.Actor actor,
                string name)
            {
                foreach (XmlElement elm in elmActor.ChildNodes)
                {
                    if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_BASIC) == true)
                    {
                        LoadDynamics_Basic(elm, boundingBoxes, actor, name);
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_KINEMATIC_CHARACTER) == true)
                    {
                        LoadDynamics_Actor_KinematicCharacter(elm, actor);
                    }
                }
            }

            protected override void OnDynamicsPart(
                XmlElement elmPart,
                Mix.Tool.Dynamics.Design.BoundingBoxCollection boundingBoxes,
                Mix.Tool.Dynamics.Design.Part part,
                string nodeName)
            {
                foreach (XmlElement elm in elmPart.ChildNodes)
                {
                    if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_BASIC) == true)
                    {
                        LoadDynamics_Basic(elm, boundingBoxes, part, nodeName);
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_JOINT) == true)
                    {
                        LoadDynamics_Part_Joint(elm, part, nodeName);
                    }
                }
            }

            private void LoadDynamics_Basic(
                XmlElement elmBasic,
                Mix.Tool.Dynamics.Design.BoundingBoxCollection boundingBoxes,
                Mix.Tool.Dynamics.Design.Basic basic,
                string name)
            {
                foreach (XmlElement elm in elmBasic.ChildNodes)
                {
                    if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_RIGID_BODY) == true)
                    {
                        LoadDynamics_Basic_RigidBody(elm, boundingBoxes, basic, name);
                    }
                }
            }

            private void LoadDynamics_Basic_RigidBody(
                XmlElement elmRigidBody,
                Mix.Tool.Dynamics.Design.BoundingBoxCollection boundingBoxes,
                Mix.Tool.Dynamics.Design.Basic basic,
                string nodeName)
            {
                Mix.Tool.Dynamics.Design.BasicChild basicChild = null;

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // ベーシックチャイルド : 追加
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                try
                {
                    Mix.Tool.Dynamics.Design.BasicChildType basicChildType;

                    basicChildType = (Mix.Tool.Dynamics.Design.BasicChildType)Enum.Parse(
                        typeof(Mix.Tool.Dynamics.Design.BasicChildType),
                        elmRigidBody.Attributes[Properties.Resources.XML_PRJ_TYPE].Value);

                    basicChild = basic.Childs.Add(basicChildType);
                }
                catch
                {
                    return;
                }

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // ベーシックチャイルド : パラメータの設定
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                foreach (XmlElement elm in elmRigidBody.ChildNodes)
                {
                    if (elm.Name.Equals(Properties.Resources.XML_PRJ_NAME) == true)
                    {
                        //名前
                        try
                        {
                            basicChild.Name = elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value;
                        }
                        catch { }
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_ROTATION) == true)
                    {
                        //中心 : 回転
                        try
                        {
                            basicChild.CenterRotation = Mix.Tool.Math.Vector.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        catch { }
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_POSITION) == true)
                    {
                        //中心 : 位置
                        try
                        {
                            basicChild.CenterPosition = Mix.Tool.Math.Vector.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        catch { }
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_PHYSICS_MATERIAL) == true)
                    {
                        //物理マテリアル
                        try
                        {
                            basicChild.PhysicsMaterial = this.physicsMaterials[elm.Attributes[Properties.Resources.XML_PRJ_NAME].Value];
                        }
                        catch { }
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_STATUS) == true)
                    {
                        //ステータス
                        try
                        {
                            basicChild.Status = (Mix.Tool.Dynamics.Design.BasicChildStatus)Enum.Parse(
                                typeof(Mix.Tool.Dynamics.Design.BasicChildStatus),
                                elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        catch { }
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_MASS) == true)
                    {
                        //質量
                        try
                        {
                            basicChild.Mass = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        catch { }
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_ROTATION_AXIS) == true)
                    {
                        //回転軸
                        try
                        {
                            basicChild.RotationAxisFlags = (Mix.Tool.Dynamics.Design.RotationAxisFlags)Enum.Parse(
                                typeof(Mix.Tool.Dynamics.Design.RotationAxisFlags),
                                elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        catch { }
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_ALWAYS_ACTIVE) == true)
                    {
                        //常に活動状態
                        try
                        {
                            basicChild.AlwaysActive = bool.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        catch { }
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_SHAPE) == true)
                    {
                        //シェイプ
                        try
                        {
                            int baseIndex = boundingBoxes.IndexOfName(elm.Attributes[Properties.Resources.XML_PRJ_BASE_NAME].Value);
                            int baseDepth = int.Parse(elm.Attributes[Properties.Resources.XML_PRJ_BASE_DEPTH].Value);
                            string shapeType = elm.Attributes[Properties.Resources.XML_PRJ_TYPE].Value;

                            if ((baseIndex >= 0) &&
                                (boundingBoxes.Count > baseIndex) &&
                                (baseDepth >= 0))
                            {
                                Mix.Tool.Dynamics.Design.BoundingBox boundingBox = null;// boundingBoxes[baseIndex].Descend(baseDepth);

                                //もとになるバウンディングボックスの取得
                                if (boundingBoxes[baseIndex].IsAvailableDescend == true)
                                {
                                    boundingBox = boundingBoxes[baseIndex].Descend(baseDepth);
                                }
                                else
                                {
                                    boundingBox = boundingBoxes[baseIndex];
                                }

                                //バウンディングボックスからシェイプを作成
                                if (shapeType.Equals(Properties.Resources.XML_PRJ_BOX) == true)
                                {
                                    this.LoadDynamics_Basic_BoxShape(elm, basicChild, boundingBox, nodeName);
                                }
                                else if (shapeType.Equals(Properties.Resources.XML_PRJ_CAPSULE) == true)
                                {
                                    this.LoadDynamics_Basic_CapsuleShape(elm, basicChild, boundingBox, nodeName);
                                }
                                else if (shapeType.Equals(Properties.Resources.XML_PRJ_SPHERE) == true)
                                {
                                    this.LoadDynamics_Basic_SphereShape(elm, basicChild, boundingBox, nodeName);
                                }
                                else
                                {
                                    //サポートされないシェイプ
                                    Mix.Tool.Logger.Print(Logger.Type.Warning,
                                        (nodeName.Length > 0) ? string.Format(Properties.Resources.WARNING_LOAD_SHAPE_NODE, nodeName) : Properties.Resources.WARNING_LOAD_SHAPE_MODEL);
                                }
                            }
                            else
                            {
                                //シェイプの基礎となるバウンディングボックスのパラメータが不足している
                                Mix.Tool.Logger.Print(Logger.Type.Warning,
                                    (nodeName.Length > 0) ? string.Format(Properties.Resources.WARNING_LOAD_SHAPE_NODE, nodeName) : Properties.Resources.WARNING_LOAD_SHAPE_MODEL);
                            }
                        }
                        catch { }
                    }
                }
            }

            private void LoadDynamics_Basic_BoxShape(
                XmlElement elmShape,
                Mix.Tool.Dynamics.Design.BasicChild basicChild,
                Mix.Tool.Dynamics.Design.BoundingBox boundingBox,
                string nodeName)
            {
                try
                {
                    Mix.Tool.Math.Vector halfExtents = null;
                    bool halfExtentsEnabled = false;

                    Mix.Tool.Math.Vector rot = null;
                    Mix.Tool.Math.Vector pos = null;

                    foreach (XmlElement elm in elmShape.ChildNodes)
                    {
                        if (elm.Name.Equals(Properties.Resources.XML_PRJ_HALF_EXTENTS) == true)
                        {
                            halfExtents = Mix.Tool.Math.Vector.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            halfExtentsEnabled = true;
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_ROTATION) == true)
                        {
                            rot = Mix.Tool.Math.Vector.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_POSITION) == true)
                        {
                            pos = Mix.Tool.Math.Vector.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                    }

                    if (halfExtentsEnabled == true)
                    {
                        basicChild.Shapes.Add(new Mix.Tool.Dynamics.Design.BoxShape(boundingBox, halfExtents, rot, pos));
                    }
                    else
                    {
                        Mix.Tool.Logger.Print(Logger.Type.Warning,
                            (nodeName.Length > 0) ? string.Format(Properties.Resources.WARNING_LOAD_SHAPE_NODE, nodeName) : Properties.Resources.WARNING_LOAD_SHAPE_MODEL);
                    }
                }
                catch
                {
                    throw;
                }
            }

            private void LoadDynamics_Basic_CapsuleShape(
                XmlElement elmShape,
                Mix.Tool.Dynamics.Design.BasicChild basicChild,
                Mix.Tool.Dynamics.Design.BoundingBox boundingBox,
                string nodeName)
            {
                try
                {
                    Mix.Tool.Dynamics.Axis axis = Mix.Tool.Dynamics.Axis.X;
                    float length = 0.0f;
                    float radius = 0.0f;

                    bool axisEnabled = false;
                    bool lengthEnabled = false;
                    bool radiusEnabled = false;

                    Mix.Tool.Math.Vector rot = null;
                    Mix.Tool.Math.Vector pos = null;

                    foreach (XmlElement elm in elmShape.ChildNodes)
                    {
                        if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_AXIS) == true)
                        {
                            axis = (Mix.Tool.Dynamics.Axis)Enum.Parse(typeof(Mix.Tool.Dynamics.Axis), elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            axisEnabled = true;
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_LENGTH) == true)
                        {
                            length = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            lengthEnabled = true;
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_RADIUS) == true)
                        {
                            radius = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            radiusEnabled = true;
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_ROTATION) == true)
                        {
                            rot = Mix.Tool.Math.Vector.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_POSITION) == true)
                        {
                            pos = Mix.Tool.Math.Vector.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                    }

                    if ((axisEnabled == true) &&
                        (lengthEnabled == true) &&
                        (radiusEnabled == true))
                    {
                        basicChild.Shapes.Add(new Mix.Tool.Dynamics.Design.CapsuleShape(boundingBox, axis, length, radius, rot, pos));
                    }
                    else
                    {
                        Mix.Tool.Logger.Print(Logger.Type.Warning,
                            (nodeName.Length > 0) ? string.Format(Properties.Resources.WARNING_LOAD_SHAPE_NODE, nodeName) : Properties.Resources.WARNING_LOAD_SHAPE_MODEL);
                    }
                }
                catch
                {
                    throw;
                }
            }

            private void LoadDynamics_Basic_SphereShape(
                XmlElement elmShape,
                Mix.Tool.Dynamics.Design.BasicChild basicChild,
                Mix.Tool.Dynamics.Design.BoundingBox boundingBox,
                string nodeName)
            {
                try
                {
                    float radius = 0.0f;
                    bool radiusEnabled = false;

                    Mix.Tool.Math.Vector rot = null;
                    Mix.Tool.Math.Vector pos = null;

                    foreach (XmlElement elm in elmShape.ChildNodes)
                    {
                        if (elm.Name.Equals(Properties.Resources.XML_PRJ_RADIUS) == true)
                        {
                            radius = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            radiusEnabled = true;
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_ROTATION) == true)
                        {
                            rot = Mix.Tool.Math.Vector.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_POSITION) == true)
                        {
                            pos = Mix.Tool.Math.Vector.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                    }

                    if (radiusEnabled == true)
                    {
                        basicChild.Shapes.Add(new Mix.Tool.Dynamics.Design.SphereShape(boundingBox, radius, rot, pos));
                    }
                    else
                    {
                        Mix.Tool.Logger.Print(Logger.Type.Warning,
                            (nodeName.Length > 0) ? string.Format(Properties.Resources.WARNING_LOAD_SHAPE_NODE, nodeName) : Properties.Resources.WARNING_LOAD_SHAPE_MODEL);
                    }
                }
                catch
                {
                    throw;
                }
            }

            private void LoadDynamics_Actor_KinematicCharacter(XmlElement elmKC, Mix.Tool.Dynamics.Design.Actor actor)
            {
                Mix.Tool.Dynamics.KinematicCharacter kinematicCharacter = new Mix.Tool.Dynamics.KinematicCharacter();

                try
                {
                    string physicsMaterialName = "";
                    float height = -1.0f;
                    float radius = -1.0f;
                    Mix.Tool.Math.Vector center = null;
                    float gravity = -1.0f;
                    float maxFallSpeed = -1.0f;
                    float initJumpSpeed = -1.0f;
                    float stepHeight = -1.0f;
                    float slopeLimit = -1.0f;

                    foreach (XmlElement elm in elmKC.ChildNodes)
                    {
                        if (elm.Name.Equals(Properties.Resources.XML_PRJ_PHYSICS_MATERIAL) == true)
                        {
                            physicsMaterialName = elm.Attributes[Properties.Resources.XML_PRJ_NAME].Value;
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_HEIGHT) == true)
                        {
                            height = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_RADIUS) == true)
                        {
                            radius = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_CENTER) == true)
                        {
                            center = Mix.Tool.Math.Vector.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_GRAVITY) == true)
                        {
                            gravity = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_MAX_FALL_SPEED) == true)
                        {
                            maxFallSpeed = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_INIT_JUMP_SPEED) == true)
                        {
                            initJumpSpeed = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_STEP_HEIGHT) == true)
                        {
                            stepHeight = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                        else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_SLOPE_LIMIT) == true)
                        {
                            slopeLimit = float.Parse(elm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                        }
                    }

                    if (height >= 0.0f) { kinematicCharacter.Height = height; }
                    if (radius >= 0.0f) { kinematicCharacter.Radius = radius; }
                    if (gravity >= 0.0f) { kinematicCharacter.Gravity = gravity; }
                    if (maxFallSpeed >= 0.0f) { kinematicCharacter.MaxFallSpeed = maxFallSpeed; }
                    if (initJumpSpeed >= 0.0f) { kinematicCharacter.InitJumpSpeed = initJumpSpeed; }
                    if (stepHeight >= 0.0f) { kinematicCharacter.StepOffset = stepHeight; }
                    if (slopeLimit >= 0.0f) { kinematicCharacter.SlopeLimit = slopeLimit; }
                    if (center != null) { actor.KinematicCharacterCenterPosition = center; }
                    if (physicsMaterialName.Length > 0) { kinematicCharacter.PhysicsMaterial = this.physicsMaterials[physicsMaterialName]; }
                }
                catch
                {
                    Mix.Tool.Logger.Print(Logger.Type.Warning, Properties.Resources.WARNING_LOAD_KINEMATIC_CHARACTER);
                }

                actor.KinematicCharacter = kinematicCharacter;
            }

            private static void LoadDynamics_Part_Joint(XmlElement elmJoint, Mix.Tool.Dynamics.Design.Part part, string nodeName)
            {
                string type = elmJoint.Attributes[Properties.Resources.XML_PRJ_TYPE].Value;
                Mix.Tool.Math.Vector pivot = new Mix.Tool.Math.Vector(0.0f, 0.0f, 0.0f, 1.0f);

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // 情報の取得、ジョイント固有の設定
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                if (type.Equals(Properties.Resources.XML_PRJ_DYNAMICS_POINT) == true)
                {
                    //ポイント
                    bool collisionDisabled = false;
                    float spring = 0.0f;
                    float damper = 0.0f;

                    try
                    {
                        foreach (XmlElement elmParam in elmJoint.ChildNodes)
                        {
                            if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_COLLISION_DISABLED) == true)
                            {
                                collisionDisabled = bool.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                            if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_SPRING) == true)
                            {
                                spring = float.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_DAMPER) == true)
                            {
                                damper = float.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_PIVOT) == true)
                            {
                                pivot = Mix.Tool.Math.Vector.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                        }

                        part.Joint = new Mix.Tool.Dynamics.PointJoint(collisionDisabled, spring, damper);
                    }
                    catch
                    {
                        part.Joint = new Mix.Tool.Dynamics.PointJoint();
                        Mix.Tool.Logger.Print(Logger.Type.Warning, string.Format(Properties.Resources.WARNING_LOAD_JOINT, Properties.Resources.EN_POINT_JOINT, nodeName));
                    }
                }
                else if (type.Equals(Properties.Resources.XML_PRJ_DYNAMICS_HINGE) == true)
                {
                    //ヒンジ
                    try
                    {
                        bool axisEnabled = false;
                        bool lowerLimitEnabled = false;
                        bool upperLimitEnabled = false;

                        Mix.Tool.Math.Vector axis = null;
                        float lowerLimit = 0.0f;
                        float upperLimit = 0.0f;

                        bool collisionDisabled = false;
                        float limitSpring = 0.0f;
                        float limitDamper = 0.0f;

                        foreach (XmlElement elmParam in elmJoint.ChildNodes)
                        {
                            if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_AXIS) == true)
                            {
                                axis = Mix.Tool.Math.Vector.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                axisEnabled = true;
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_LOWER_LIMIT) == true)
                            {
                                lowerLimit = float.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                lowerLimitEnabled = true;
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_UPPER_LIMIT) == true)
                            {
                                upperLimit = float.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                upperLimitEnabled = true;
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_COLLISION_DISABLED) == true)
                            {
                                collisionDisabled = bool.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_LIMIT_SPRING) == true)
                            {
                                limitSpring = float.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_LIMIT_DAMPER) == true)
                            {
                                limitDamper = float.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_PIVOT) == true)
                            {
                                pivot = Mix.Tool.Math.Vector.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                        }

                        if ((axisEnabled == true) &&
                            (lowerLimitEnabled == true) &&
                            (upperLimitEnabled == true))
                        {
                            part.Joint = new Mix.Tool.Dynamics.HingeJoint(axis, lowerLimit, upperLimit, collisionDisabled, limitSpring, limitDamper);
                        }
                        else
                        {
                            part.Joint = new Mix.Tool.Dynamics.HingeJoint();
                            Mix.Tool.Logger.Print(Logger.Type.Warning, string.Format(Properties.Resources.WARNING_LOAD_JOINT, Properties.Resources.EN_HINGE_JOINT, nodeName));
                        }
                    }
                    catch
                    {
                        part.Joint = new Mix.Tool.Dynamics.HingeJoint();
                        Mix.Tool.Logger.Print(Logger.Type.Warning, string.Format(Properties.Resources.WARNING_LOAD_JOINT, Properties.Resources.EN_HINGE_JOINT, nodeName));
                    }
                }
                else if (type.Equals(Properties.Resources.XML_PRJ_DYNAMICS_BALL) == true)
                {
                    //ボール
                    try
                    {
                        bool swingAxisEnabled = false;
                        bool twistAxisEnabled = false;
                        bool swingLimit1Enabled = false;
                        bool swingLimit2Enabled = false;
                        bool twistLimitEnabled = false;

                        Mix.Tool.Math.Vector swingAxis = null;
                        Mix.Tool.Math.Vector twistAxis = null;
                        float swingLimit1 = 0.0f;
                        float swingLimit2 = 0.0f;
                        float twistLimit = 0.0f;

                        bool collisionDisabled = false;
                        float spring = 0.0f;
                        float damper = 0.0f;
                        float limitSpring = 0.0f;
                        float limitDamper = 0.0f;

                        foreach (XmlElement elmParam in elmJoint.ChildNodes)
                        {
                            if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_SWING_AXIS) == true)
                            {
                                swingAxis = Mix.Tool.Math.Vector.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                swingAxisEnabled = true;
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_TWIST_AXIS) == true)
                            {
                                twistAxis = Mix.Tool.Math.Vector.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                twistAxisEnabled = true;
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_SWING_LIMIT1) == true)
                            {
                                swingLimit1 = float.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                swingLimit1Enabled = true;
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_SWING_LIMIT2) == true)
                            {
                                swingLimit2 = float.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                swingLimit2Enabled = true;
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_TWIST_LIMIT) == true)
                            {
                                twistLimit = float.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                                twistLimitEnabled = true;
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_COLLISION_DISABLED) == true)
                            {
                                collisionDisabled = bool.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_SPRING) == true)
                            {
                                spring = float.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_DAMPER) == true)
                            {
                                damper = float.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_LIMIT_SPRING) == true)
                            {
                                limitSpring = float.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_LIMIT_DAMPER) == true)
                            {
                                limitDamper = float.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                            else if (elmParam.Name.Equals(Properties.Resources.XML_PRJ_PIVOT) == true)
                            {
                                pivot = Mix.Tool.Math.Vector.Parse(elmParam.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                            }
                        }

                        if ((swingAxisEnabled == true) &&
                            (twistAxisEnabled == true) &&
                            (swingLimit1Enabled == true) &&
                            (swingLimit2Enabled == true) &&
                            (twistLimitEnabled == true))
                        {
                            part.Joint = new Mix.Tool.Dynamics.BallJoint(swingAxis, twistAxis, swingLimit1, swingLimit2, twistLimit, collisionDisabled, spring, damper, limitSpring, limitDamper);
                        }
                        else
                        {
                            part.Joint = new Mix.Tool.Dynamics.BallJoint();
                            Mix.Tool.Logger.Print(Logger.Type.Warning, string.Format(Properties.Resources.WARNING_LOAD_JOINT, Properties.Resources.EN_BALL_JOINT, nodeName));
                        }
                    }
                    catch
                    {
                        part.Joint = new Mix.Tool.Dynamics.BallJoint();
                        Mix.Tool.Logger.Print(Logger.Type.Warning, string.Format(Properties.Resources.WARNING_LOAD_JOINT, Properties.Resources.EN_BALL_JOINT, nodeName));
                    }
                }

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // ピボットの設定
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                part.JointPivot = pivot;
            }

            #endregion

            #region IBackgroundTask

            public void OnExecute()
            {
                //オブジェクトモデルのインポート
                this.objectModel.Import(this.importConfig);

                //モデルの適用
                this.Execute_ApplyModel();
            }

            public void OnFinish()
            {
                //インポートのファイナライズ
                this.objectModel.ImportFinalize();

                //マテリアルスロットを適用
                this.Finish_ApplyMaterialSlots();

                //モーションコントローラーを適用
                this.Finish_ApplyMotionControllers();

                //ルートモーションを適用
                this.Finish_ApplyRootMotion();
            }

            #endregion
        };

        /// <summary>
        /// マップモデルのインポートタスククラス
        /// </summary>
        private class ImportMapModelTask : ImportModelTask, Mix.Tool.ModelEditor.Project.IBackgroundTask
        {
            #region Private Member

            private Mix.Tool.Graphics.MapModel.ImportConfig importConfig = null;
            private Mix.Tool.Graphics.MapModel mapModel = null;
            private XmlElement elmDynamicsTerrain = null;

            #endregion

            #region Public Method

            public ImportMapModelTask(
                Mix.Tool.Dynamics.PhysicsMaterialCollection _physicsMaterials,
                Mix.Tool.Graphics.MapModel.ImportConfig _importConfig,
                Mix.Tool.Graphics.MapModel _mapModel,
                XmlElement _elmModel,
                List<XmlElement> _elmMaterialSlots)
                : base(_physicsMaterials, _mapModel, _elmModel, _elmMaterialSlots)
            {
                this.importConfig = _importConfig;
                this.mapModel = _mapModel;
            }

            #endregion

            #region Private Method

            private void Finish_ApplyDynamicsTerrain()
            {
                if (this.elmDynamicsTerrain == null)
                {
                    return;
                }

                try
                {
                    Mix.Tool.Dynamics.Design.Terrain terrain = this.mapModel.DynamicsDesigner;

                    //インポートのファイナライズ
                    terrain.ImportFinalize();

                    //パートの適用
                    foreach (XmlElement elmPart in this.elmDynamicsTerrain.ChildNodes)
                    {
                        if (elmPart.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS_PART) == true)
                        {
                            try
                            {
                                XmlAttribute attrName = elmPart.Attributes[Properties.Resources.XML_PRJ_NAME];
                                int partIndex = terrain.Parts.IndexOfName(attrName.Value);

                                if (partIndex >= 0)
                                {
                                    Mix.Tool.Dynamics.Design.TerrainPart part = terrain.Parts[partIndex];

                                    foreach (XmlElement elmPartChild in elmPart.ChildNodes)
                                    {
                                        if (elmPartChild.Name.Equals(Properties.Resources.XML_PRJ_PHYSICS_MATERIAL) == true)
                                        {
                                            attrName = elmPartChild.Attributes[Properties.Resources.XML_PRJ_NAME];
                                            part.PhysicsMaterial = this.physicsMaterials[attrName.Value];
                                        }
                                    }
                                }
                                else
                                {
                                }
                            }
                            catch
                            {
                            }
                        }
                    }
                }
                catch
                {
                }
            }

            #endregion

            #region ImportModelTask

            protected override void OnDynamicsTerrain(XmlElement elmTerrain, Mix.Tool.Dynamics.Design.Terrain terrain)
            {
                try
                {
                    XmlAttribute attrFilePath = elmTerrain.Attributes[Properties.Resources.XML_PRJ_FILEPATH];

                    //インポート
                    terrain.Initialize();
                    terrain.Import(attrFilePath.Value, 5);

                    //パートの設定は Finish(カレントスレッドで行う)
                    this.elmDynamicsTerrain = elmTerrain;
                }
                catch
                {
                }
            }

            #endregion

            #region IBackgroundTask

            public void OnExecute()
            {
                //インポート
                this.mapModel.Import(this.importConfig);

                //モデルの適用
                // OnDynamicsTerrainの呼び出し
                this.Execute_ApplyModel();
            }

            public void OnFinish()
            {
                //インポートののファイナライズ
                this.mapModel.ImportFinalize();

                //マテリアルスロットを適用
                this.Finish_ApplyMaterialSlots();

                //ダイナミクス : テラインのファイナライズ
                this.Finish_ApplyDynamicsTerrain();
            }

            #endregion
        };

        /// <summary>
        /// プロジェクトの読み込み完了タスククラス
        /// </summary>
        public class EndLoadTask : Mix.Tool.ModelEditor.Project.IBackgroundTask
        {
            /// <summary>
            /// タイプ列挙定数
            /// </summary>
            public enum Types
            {
                /// <summary>
                /// 新規
                /// </summary>
                New,

                /// <summary>
                /// 既存
                /// </summary>
                Existing,
            };

            private EndLoadTask.Types type;
            private Size mainWndSize;
            private System.Windows.Forms.FormWindowState mainWndState;
            private EditStyle mainWndStyle;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="_type">タイプ</param>
            /// <param name="_mainWndSize">メインウィンドウのサイズ</param>
            /// <param name="_mainWndState">メインウィンドウのステート</param>
            internal EndLoadTask(
                EndLoadTask.Types _type,
                Size _mainWndSize,
                System.Windows.Forms.FormWindowState _mainWndState,
                EditStyle _mainWndStyle)
            {
                this.type = _type;
                this.mainWndSize = _mainWndSize;
                this.mainWndState = _mainWndState;
                this.mainWndStyle = _mainWndStyle;
            }

            /// <summary>
            /// プロジェクトの開始タイプの取得
            /// </summary>
            public EndLoadTask.Types Type
            {
                get { return this.type; }
            }

            /// <summary>
            /// メインウィンドウのサイズの取得
            /// </summary>
            public Size MainWndSize
            {
                get { return this.mainWndSize; }
            }

            /// <summary>
            /// メインウィンドウのステートの取得
            /// </summary>
            public System.Windows.Forms.FormWindowState MainWndState
            {
                get { return this.mainWndState; }
            }

            /// <summary>
            /// メインエインドウのスタイルの取得
            /// </summary>
            public EditStyle MainWndStyle
            {
                get { return this.mainWndStyle; }
            }

            #region IBackgroundTask

            public void OnExecute() { }
            public void OnFinish() { }

            #endregion
        };

        #endregion

        #region Private Member

        private Project.BackgroundWorker backgroundWorker = null;
        private Mix.Tool.Graphics.Manager graphicsMgr = null;
        private Mix.Tool.Graphics.Scene scene = null;
        private System.Windows.Forms.Form mainWindow = null;
        private Mix.Tool.Docking.DockPanel dockPanel = null;
        private Mix.Tool.Docking.DeserializeDockContent deserializeDockContent = null;

        private string rootDirPath = "";
        private Template template = new Template();
        private Config config = new Config();
        private Dynamics dynamics = new Dynamics();
        private string name = "";
        private string dirPath = "";

        private Preview preview = new Preview();
        private Log log = new Log();

        private Mix.Tool.Graphics.Model drawObject = null;

        private EditStyle editStyle = EditStyle.Material;

        #endregion

        #region Public Event

        /// <summary>
        /// New、Laod のときに発生する作成開始イベント
        /// </summary>
        public event System.EventHandler BeginLoad = null;
        /// <summary>
        /// New、Load のときに発生する作成終了イベント
        /// </summary>
        public event System.EventHandler EndLoad = null;

        #endregion

        #region Public Property

        /// <summary>
        /// ルートディレクトリパスの取得、または設定
        /// </summary>
        public string RootDirectoryPath
        {
            get { return this.rootDirPath; }
            set { this.rootDirPath = value; }
        }

        /// <summary>
        /// テンプレートの取得
        /// </summary>
        public Template Template
        {
            get { return this.template; }
        }

        /// <summary>
        /// ダイナミクスの取得
        /// </summary>
        public Dynamics Dynamics
        {
            get { return this.dynamics; }
        }

        /// <summary>
        /// 設定の取得
        /// </summary>
        public Config Config
        {
            get { return this.config; }
        }

        /// <summary>
        /// プロジェクトの名前の取得、または設定
        /// </summary>
        public string Name
        {
            get { return this.name; }
            set { this.name = value; }
        }

        /// <summary>
        /// プロジェクトのディレクトリパスの取得、または設定
        /// </summary>
        public string DirectoryPath
        {
            get { return this.dirPath; }
            set { this.dirPath = value; }
        }

        /// <summary>
        /// プロジェクトのファイルパスの取得
        /// </summary>
        public string FilePath
        {
            get
            {
                string dirPath = System.IO.Path.Combine(this.dirPath, this.name);
                string filePath = System.IO.Path.Combine(dirPath, this.name + Properties.Resources.FILE_EXT_PRJ);

                return filePath;
            }
        }

        /// <summary>
        /// プレビューの取得
        /// </summary>
        public Preview Preview
        {
            get { return this.preview; }
        }

        /// <summary>
        /// ログの取得
        /// </summary>
        public Log Log
        {
            get { return this.log; }
        }

        /// <summary>
        /// 描画オブジェクトの取得
        /// </summary>
        public Mix.Tool.Graphics.Model DrawObject
        {
            get { return this.drawObject; }
        }

        /// <summary>
        /// エディットスタイルの取得、または設定
        /// </summary>
        public EditStyle EditStyle
        {
            get { return this.editStyle; }
            set
            {
                if (this.editStyle != value)
                {
                    //直前のエディットスタイルを保存
                    this.SaveEditStyle(this.editStyle); 

                    //エディットスタイルをロード
                    this.LoadEditStyle(value);

                    //新規エディットスタイル
                    this.editStyle = value;
                }
            }
        }

        #endregion

        #region Public Method

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="_graphicsMgr">グラフィックスマネージャ</param>
        /// <param name="_scene">シーン</param>
        /// <param name="mainForm">メインフォーム</param>
        /// <param name="_dockPanel">ドックパネル</param>
        /// <param name="_deserializeDockContent">ドックコンテントのデシリアライズデリゲート</param>
        public Manager( Mix.Tool.ModelEditor.Project.BackgroundWorker _backgroundWorker,
                        Mix.Tool.Graphics.Manager _graphicsMgr,
                        Mix.Tool.Graphics.Scene _scene,
                        System.Windows.Forms.Form _mainForm,
                        Mix.Tool.Docking.DockPanel _dockPanel,
                        Mix.Tool.Docking.DeserializeDockContent _deserializeDockContent)
        {
            this.backgroundWorker = _backgroundWorker;
            this.graphicsMgr = _graphicsMgr;
            this.scene = _scene;
            this.mainWindow = _mainForm;
            this.dockPanel = _dockPanel;
            this.deserializeDockContent = _deserializeDockContent;

            this.dynamics.ChangedState += new Dynamics.ChangedStateEventHandler(Dynamics_ChangedState);
        }

        /// <summary>
        /// プロジェクトのロード( 新規作成 )
        /// </summary>
        public void Load()
        {
            if (this.drawObject != null)
            {
                throw new System.Exception(Properties.Resources.PROJECT_NOT_DISPOSE_MES);
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // 初期状態のドキュメントを作成
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            string projectDirPath;
            string projectFilePath;

            try
            {
                projectDirPath = System.IO.Path.Combine(this.dirPath, this.name);
                projectFilePath = System.IO.Path.Combine(projectDirPath, this.name + Properties.Resources.FILE_EXT_PRJ);
            }
            catch
            {
                throw;
            }

            try
            {
                XmlDocument doc = new XmlDocument();
                XmlElement elmRoot;

                //プロジェクトディレクトリの作成
                System.IO.Directory.CreateDirectory(projectDirPath);

                //ルート
                elmRoot = SaveRoot(doc);

                //基本
                this.SaveBasic(doc, elmRoot);

                //コンフィグ
                this.SaveConfig(doc, elmRoot);

                //プロジェクト
                this.SaveProject(doc, elmRoot);

                //保存
                doc.Save(projectFilePath);
            }
            catch
            {
                this.log.Save(System.IO.Path.Combine(projectDirPath, Properties.Resources.FILENAME_LOGTEXT));
                throw;
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // 作成開始
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            this.NotifyBeginLoad();

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // グラフィックスマネージャのセットアップ
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            this.SetupGraphicsManager();

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // モデルをインポート
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            try
            {
                this.Load(
                    EndLoadTask.Types.New,
                    this.config.InputFilePath,
                    null,
                    null,
                    null,
                    null,
                    null,
                    Mix.Tool.Graphics.RootTransformFlags.All,
                    this.mainWindow.Size,
                    this.mainWindow.WindowState,
                    EditStyle.Material);
            }
            catch
            {
                this.NotifyEndLoad();
                throw;
            }
        }

        /// <summary>
        /// プロジェクトのロード
        /// </summary>
        public void Load(string filePath)
        {
            if (this.drawObject != null)
            {
                throw new System.Exception(Properties.Resources.PROJECT_NOT_DISPOSE_MES);
            }

            Size mainWndSize = this.mainWindow.Size;
            System.Windows.Forms.FormWindowState mainWndState = System.Windows.Forms.FormWindowState.Normal;
            EditStyle mainWndStyle = EditStyle.Material;

            List<XmlElement> elmMaterials = new List<XmlElement>();
            List<XmlElement> elmMotCtrls = new List<XmlElement>();
            XmlElement elmModel = null;
            XmlElement elmMotUA = null;
            XmlElement elmMotRoot = null;

            Mix.Tool.Graphics.RootTransformFlags motRTFlags = Mix.Tool.Graphics.RootTransformFlags.All;

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // 作成開始
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            this.NotifyBeginLoad();

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // プロジェクトのデシリアライズ
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            try
            {
                XmlDocument doc = new XmlDocument();
                XmlElement elmRoot;

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // ロード
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                doc.Load(filePath);

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // ルートノード
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                elmRoot = doc.FirstChild as XmlElement;

                if (elmRoot.Name.Equals(Properties.Resources.XML_PRJ_ROOT) == false)
                {
                    this.NotifyEndLoad();
                    throw new System.Exception(Properties.Resources.ILLEGAL_FILE_FORMAT);
                }

                if (elmRoot.Attributes[Properties.Resources.XML_PRJ_VERSION].Value.Equals(Properties.Resources.XML_PRJ_VERSION_VALUE) == false)
                {
                    this.NotifyEndLoad();
                    throw new System.Exception(Properties.Resources.NOT_SUPPORT_VERSION_STR);
                }

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                // 子ノード
                ////////////////////////////////////////////////////////////////////////////////////////////////////

                foreach (XmlElement elm in elmRoot.ChildNodes)
                {
                    if (elm.Name.Equals(Properties.Resources.XML_PRJ_BASIC) == true)
                    {
                        this.LoadBasic(elm);
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_CONFIG) == true)
                    {
                        this.LoadConfig(elm);
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_DYNAMICS) == true)
                    {
                        this.LoadDynamics(elm);
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_PROJECT) == true)
                    {
                        this.LoadProject(elm);
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_MAINWINDOW) == true)
                    {
                        this.LoadMainWnd(elm, out mainWndSize, out mainWndState, out mainWndStyle);
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_PREF) == true)
                    {
                        this.LoadPreference(elm);
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_PREVIEW) == true)
                    {
                        this.LoadPreview(elm, out motRTFlags);
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_MODEL) == true)
                    {
                        elmModel = elm; //モデルのロード後に処理
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_MATERIALSLOT) == true)
                    {
                        elmMaterials.Add(elm); //モデルのロード後に処理
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_MOTIONCTRL) == true)
                    {
                        elmMotCtrls.Add(elm); //モデルのロード後に処理
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_UAMOTIONS) == true)
                    {
                        elmMotUA = elm; //モデルのロード後に処理
                    }
                    else if (elm.Name.Equals(Properties.Resources.XML_PRJ_ROOT_MOTION) == true)
                    {
                        elmMotRoot = elm; //モデルのロード後に処理
                    }
                }
            }
            catch
            {
                this.NotifyEndLoad();
                throw;
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // テンプレートをロード
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            try
            {
                this.template.Load();
            }
            catch
            {
                this.NotifyEndLoad();
                throw;
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // グラフィックスマネージャのセットアップ
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            this.SetupGraphicsManager();

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ダイナミクスのセットアップ
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            try
            {
                this.dynamics.Load();
            }
            catch
            {
                this.NotifyEndLoad();
                throw;
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ロード
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            try
            {
                this.Load(
                    EndLoadTask.Types.Existing,
                    this.config.InputFilePath,
                    elmModel,
                    elmMaterials,
                    elmMotCtrls,
                    elmMotUA,
                    elmMotRoot,
                    motRTFlags,
                    mainWndSize,
                    mainWndState,
                    mainWndStyle);
            }
            catch
            {
                this.ReleaseModel();
                this.NotifyEndLoad();
                throw;
            }
        }

        /// <summary>
        /// プロジェクトを保存
        /// </summary>
        public void Save()
        {
            string projectDirPath;
            string projectFilePath;
            string logFilePath;

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // パスの作成
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            try
            {
                projectDirPath = System.IO.Path.Combine(this.dirPath, this.name);
                projectFilePath = System.IO.Path.Combine(projectDirPath, this.name + Properties.Resources.FILE_EXT_PRJ);
                logFilePath = System.IO.Path.Combine(projectDirPath, Properties.Resources.FILENAME_LOGTEXT);
            }
            catch
            {
                throw;
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // プロジェクトディレクトリの作成
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            try
            {
                System.IO.Directory.CreateDirectory(projectDirPath);
            }
            catch
            {
                this.log.Save(logFilePath);
                throw;
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // エディットスタイルのシリアライズ
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            try
            {
                this.SaveEditStyle(this.editStyle);
            }
            catch
            {
                this.log.Save(logFilePath);
                throw;
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // プロジェクトのシリアライズ
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            try
            {
                XmlDocument doc = new XmlDocument();
                XmlElement elmRoot;

                //ルート
                elmRoot = SaveRoot(doc);

                //基本
                this.SaveBasic(doc, elmRoot);

                //ダイナミクス
                this.SaveDynamics(doc, elmRoot);

                //コンフィグ
                this.SaveConfig(doc, elmRoot);

                //プロジェクト
                this.SaveProject(doc, elmRoot);

                //メインウィンドウ
                this.SaveMainWindow(doc, elmRoot);

                //プリファレンス
                this.SavePreference(doc, elmRoot);

                //プレビュー
                this.SavePreview(doc, elmRoot);

                //モデル
                SaveModel(doc, elmRoot, this.drawObject);

                //マテリアルスロット
                foreach (Mix.Tool.Graphics.MaterialSlot materialSlot in this.drawObject.MaterialSlots)
                {
                    SaveMaterialSlot(doc, elmRoot, materialSlot);
                }

                //ルートモーション
                if (this.drawObject is Mix.Tool.Graphics.ObjectModel)
                {
                    SaveRootMotion(doc, elmRoot, this.drawObject as Mix.Tool.Graphics.ObjectModel);
                }

                //モーション
                if (this.drawObject is Mix.Tool.Graphics.ObjectModel)
                {
                    SaveMotionControllers(doc, elmRoot, this.drawObject as Mix.Tool.Graphics.ObjectModel);
                }

                //保存
                doc.Save(projectFilePath);

                //ログ
                Mix.Tool.Logger.Print(Logger.Type.Info, "----------------------------------------------------------------------------------------------------");
                Mix.Tool.Logger.Print(Logger.Type.Info, "  プロジェクトを保存しました : FilePath[" + projectFilePath + "]");
            }
            catch
            {
                this.log.Save(logFilePath);
                throw;
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // モデルの保存
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            if (this.config.OutputFilePath.Length > 0)
            {
                this.drawObject.ModelFilePath = this.config.OutputFilePath;

                if (this.drawObject.SaveModel() == false)
                {
                    this.log.Save(logFilePath);
                }
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // マテリアルの保存
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            if (this.drawObject.SaveMaterials() == false)
            {
                this.log.Save(logFilePath);
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // モーションの保存
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            if (this.drawObject is Mix.Tool.Graphics.ObjectModel)
            {
                Mix.Tool.Graphics.ObjectModel objModel = this.DrawObject as Mix.Tool.Graphics.ObjectModel;

                if (objModel.SaveMotions() == false)
                {
                    this.log.Save(logFilePath);
                }
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // エフェクトパッケージの保存
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            this.graphicsMgr.EffectPackage.Save();

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ログの保存
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            try
            {
                this.log.Save(logFilePath);
            }
            catch
            {
                throw;
            }
        }

        /// <summary>
        /// プロジェクトをリフレッシュ
        /// </summary>
        public void Refresh()
        {
            this.SetupGraphicsManager();
        }

        #endregion

        #region Dynamics Method

        private void Dynamics_ChangedState(object sender, Dynamics.ChangedStateEventArgs e)
        {
            if (e.State == true)
            {
                this.Dynamics_AssignPhysicsMaterial();
            }
        }

        private void Dynamics_AssignPhysicsMaterial()
        {
            if (this.drawObject != null)
            {
                if (this.dynamics.PhysicsMaterials.Count > 0)
                {
                    if (this.drawObject is Mix.Tool.Graphics.ObjectModel)
                    {
                        this.Dynamics_AssignPhysicsMaterial(this.drawObject as Mix.Tool.Graphics.ObjectModel);
                    }
                    else if (drawObject is Mix.Tool.Graphics.MapModel)
                    {
                        this.Dynamics_AssignPhysicsMaterial(this.drawObject as Mix.Tool.Graphics.MapModel);
                    }
                }
            }
        }

        private void Dynamics_AssignPhysicsMaterial(Mix.Tool.Graphics.ObjectModel objModel)
        {
            if (objModel.DynamicsDesigner != null)
            {
                this.Dynamics_AssignPhysicsMaterial(objModel.DynamicsDesigner);
            }

            if (objModel.RootNode != null)
            {
                this.Dynamics_AssignPhysicsMaterial(objModel.RootNode);
            }
        }

        private void Dynamics_AssignPhysicsMaterial(Mix.Tool.Graphics.MapModel mapModel)
        {
            if (mapModel.DynamicsDesigner != null)
            {
                foreach (Mix.Tool.Dynamics.Design.TerrainPart part in mapModel.DynamicsDesigner.Parts)
                {
                    if (part.PhysicsMaterial != null)
                    {
                        part.PhysicsMaterial = this.dynamics.PhysicsMaterials[part.PhysicsMaterial.Name];
                    }
                    else
                    {
                        part.PhysicsMaterial = this.dynamics.PhysicsMaterials[0];
                    }
                }
            }
        }

        private void Dynamics_AssignPhysicsMaterial(Mix.Tool.Graphics.Node node)
        {
            if (node.DynamicsDesigner != null)
            {
                this.Dynamics_AssignPhysicsMaterial(node.DynamicsDesigner);
            }

            foreach (Mix.Tool.Graphics.Node childNode in node.Childs)
            {
                this.Dynamics_AssignPhysicsMaterial(childNode);
            }
        }

        private void Dynamics_AssignPhysicsMaterial(Mix.Tool.Dynamics.Design.Basic dynBasic)
        {
            foreach (Mix.Tool.Dynamics.Design.BasicChild dynBasicChild in dynBasic.Childs)
            {
                if (dynBasicChild.PhysicsMaterial != null)
                {
                    dynBasicChild.PhysicsMaterial = this.dynamics.PhysicsMaterials[dynBasicChild.PhysicsMaterial.Name];
                }
                else
                {
                    dynBasicChild.PhysicsMaterial = this.dynamics.PhysicsMaterials[0];
                }
            }
        }

        #endregion

        #region Load Method

        private void LoadBasic(XmlElement elm)
        {
            foreach (XmlElement childElm in elm.ChildNodes)
            {
                if (childElm.Name.Equals(Properties.Resources.XML_PRJ_ROOTDIRPATH) == true)
                {
                    this.rootDirPath = childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value;
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_TEMPLATEFILEPATH) == true)
                {
                    this.template.FilePath = childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value;
                }
            }
        }

        private void LoadConfig(XmlElement elm)
        {
            foreach (XmlElement childElm in elm.ChildNodes)
            {
                if (childElm.Name.Equals(Properties.Resources.XML_PRJ_INPUTFILEPATH) == true)
                {
                    this.config.InputFilePath = childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value;
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_OUTPUTFILEPATH) == true)
                {
                    this.config.OutputFilePath = childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value;
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_OPTION) == true)
                {
                    this.config.Option = (Mix.Tool.Graphics.ModelImportOption)Enum.Parse(typeof(Mix.Tool.Graphics.ModelImportOption),
                        childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_SCALING) == true)
                {
                    this.config.Transform.Scaling = float.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_ROTATION) == true)
                {
                    this.config.Transform.Rotation = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_TRANSLATION) == true)
                {
                    this.config.Transform.Translation = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_SMOOTING) == true)
                {
                    this.config.Smooting.Enabled = bool.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_SMOOTING_ANGLE) == true)
                {
                    this.config.Smooting.Angle = float.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_FORMAT) == true)
                {
                    this.config.Format = (ModelFormats)Enum.Parse(typeof(ModelFormats),
                        childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_MOTTIMEMODE) == true)
                {
                    this.config.OM_TimeMode = (Mix.Tool.Graphics.ObjectModel.TimeModes)Enum.Parse(typeof(Mix.Tool.Graphics.ObjectModel.TimeModes),
                        childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_CELLSIZE) == true)
                {
                    this.config.MM_CellSize = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_IMPORT_COLLISION_FILEPATH) == true)
                {
                    this.config.MM_InputCollisionFilePath = childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value;
                }
            }
        }

        private void LoadDynamics(XmlElement elm)
        {
            foreach (XmlElement childElm in elm.ChildNodes)
            {
                if (childElm.Name.Equals(Properties.Resources.XML_PRJ_FILEPATH) == true)
                {
                    this.dynamics.FilePath = childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value;
                }
            }
        }

        private void LoadProject(XmlElement elm)
        {
            foreach (XmlElement childElm in elm.ChildNodes)
            {
                if (childElm.Name.Equals(Properties.Resources.XML_PRJ_NAME) == true)
                {
                    this.name = childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value;
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_DIRPATH) == true)
                {
                    this.dirPath = childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value;
                }
            }
        }

        private void LoadMainWnd(XmlElement elm, out Size size, out System.Windows.Forms.FormWindowState state, out EditStyle style )
        {
            size = this.mainWindow.Size;
            state = this.mainWindow.WindowState;
            style = EditStyle.Material;

            foreach (XmlElement childElm in elm.ChildNodes)
            {
                if (childElm.Name.Equals(Properties.Resources.XML_PRJ_SIZE) == true)
                {
                    size = Parse_Size(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_STATE) == true)
                {
                    state = (System.Windows.Forms.FormWindowState)Enum.Parse(
                        typeof(System.Windows.Forms.FormWindowState),
                        childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_STYLE) == true)
                {
                    style = (EditStyle)Enum.Parse(
                        typeof(EditStyle),
                        childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
            }
        }

        private void LoadPreference(XmlElement elm)
        {
            foreach (XmlElement childElm in elm.ChildNodes)
            {
                if (childElm.Name.Equals(Properties.Resources.XML_PRJ_SCENE) == true)
                {
                    this.LoadPreference_Scene(childElm);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_CONTROL) == true)
                {
                    this.LoadPreference_Control(childElm);
                }
            }
        }

        private void LoadPreference_Scene(XmlElement elm)
        {
            foreach (XmlElement childElm in elm.ChildNodes)
            {
                if (childElm.Name.Equals(Properties.Resources.XML_PRJ_BG_COLOR) == true)
                {
                    this.scene.General.BackgroundColor = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_GRID_COLOR) == true)
                {
                    this.scene.Grid.Color = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_GRID_PITCH) == true)
                {
                    this.scene.Grid.Pitch = float.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_GRID_EXTENT) == true)
                {
                    this.scene.Grid.Extent = float.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_TEXT_COLOR) == true)
                {
                    this.scene.General.TextColor = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_LIGHT_AMBIENT_COLOR) == true)
                {
                    this.scene.Light.AmbientColor = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_LIGHT_DIFFUSE_COLOR) == true)
                {
                    this.scene.Light.DiffuseColor = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_LIGHT_SHADING) == true)
                {
                    this.scene.Light.Shading = float.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_CAMERA_FOVY) == true)
                {
                    this.graphicsMgr.Perspective.FovY = float.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_CAMERA_NC) == true)
                {
                    this.graphicsMgr.Perspective.NearZ = float.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_CAMERA_FC) == true)
                {
                    this.graphicsMgr.Perspective.FarZ = float.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_REFLECT_TEX) == true)
                {
                    this.scene.ReflectionTextureFileName = childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value;
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_VV_COLOR) == true)
                {
                    this.scene.General.ViewVolumeColor = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_COLL_COLOR) == true)
                {
                    this.scene.General.CollisionColor = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_BONE_COLOR) == true)
                {
                    this.scene.General.BoneColor = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_NODE_SELECTED_COLOR) == true)
                {
                    this.scene.General.SelectedColor = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_NODE_SELECTED_AD) == true)
                {
                    this.scene.General.SelectedAlwaysDisplay = bool.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_AXIS_SCALE) == true)
                {
                    this.scene.General.AxisScale = float.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_JOINT_SCALE) == true)
                {
                    this.scene.General.JointScale = float.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
            }
        }

        private void LoadPreference_Control(XmlElement elm)
        {
            foreach (XmlElement childElm in elm.ChildNodes)
            {
                if (childElm.Name.Equals(Properties.Resources.XML_PRJ_MOUSE_S) == true)
                {
                    this.graphicsMgr.MouseSensitivity.Scaling = float.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_MOUSE_R) == true)
                {
                    this.graphicsMgr.MouseSensitivity.Rotation = float.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_MOUSE_T) == true)
                {
                    this.graphicsMgr.MouseSensitivity.Translation = float.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
            }
        }

        private void LoadPreview(XmlElement elm, out Mix.Tool.Graphics.RootTransformFlags rtFlags)
        {
            rtFlags = Mix.Tool.Graphics.RootTransformFlags.All;

            foreach (XmlElement childElm in elm.ChildNodes)
            {
                if (childElm.Name.Equals(Properties.Resources.XML_PRJ_EDITMODE) == true)
                {
                    this.preview.EditMode = (Preview.EidtModes)Enum.Parse(
                       typeof(Preview.EidtModes),
                       childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_RT_NAME) == true)
                {
                    rtFlags = (Mix.Tool.Graphics.RootTransformFlags)Enum.Parse(
                       typeof(Mix.Tool.Graphics.RootTransformFlags),
                       childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_DISPLAY) == true)
                {
                    this.preview.Display = (Mix.Tool.Graphics.Camera.DrawTypes)Enum.Parse(
                       typeof(Mix.Tool.Graphics.Camera.DrawTypes),
                       childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_DRAW) == true)
                {
                    this.preview.FillMode = (Mix.Tool.Graphics.Camera.FillModes)Enum.Parse(
                       typeof(Mix.Tool.Graphics.Camera.FillModes),
                       childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_CAMERA) == true)
                {
                    this.LoadPreview_Camera(childElm);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_LIGHT_POSE) == true)
                {
                    this.scene.Light.Pose = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
            }
        }

        private void LoadPreview_Camera(XmlElement elm)
        {
            foreach (XmlElement childElm in elm.ChildNodes)
            {
                if (childElm.Name.Equals(Properties.Resources.XML_PRJ_LOCALPOS) == true)
                {
                    this.preview.LocalPosition = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_TARGET) == true)
                {
                    this.preview.Target = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
                else if (childElm.Name.Equals(Properties.Resources.XML_PRJ_POSE) == true)
                {
                    this.preview.Pose = Mix.Tool.Math.Vector.Parse(childElm.Attributes[Properties.Resources.XML_PRJ_VALUE].Value);
                }
            }
        }

        #endregion

        #region Save Method

        private XmlElement SaveRoot(XmlDocument doc)
        {
            XmlElement elmRoot;
            XmlAttribute attr;

            elmRoot = doc.CreateElement(Properties.Resources.XML_PRJ_ROOT);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VERSION);
            attr.Value = Properties.Resources.XML_PRJ_VERSION_VALUE;
            elmRoot.Attributes.Append(attr);
            doc.AppendChild(elmRoot);

            return elmRoot;
        }

        private void SaveBasic(XmlDocument doc, XmlElement elmRoot)
        {
            XmlElement elmBasic;
            XmlElement elm;
            XmlAttribute attr;

            elmBasic = doc.CreateElement(Properties.Resources.XML_PRJ_BASIC);
            elmRoot.AppendChild(elmBasic);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_ROOTDIRPATH);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.rootDirPath;
            elm.Attributes.Append(attr);
            elmBasic.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_TEMPLATEFILEPATH);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.template.FilePath;
            elm.Attributes.Append(attr);
            elmBasic.AppendChild(elm);
        }

        private void SaveDynamics(XmlDocument doc, XmlElement elmRoot)
        {
            XmlElement elmDynamics;
            XmlElement elm;
            XmlAttribute attr;

            elmDynamics = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS);
            elmRoot.AppendChild(elmDynamics);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_FILEPATH);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.dynamics.FilePath;
            elm.Attributes.Append(attr);
            elmDynamics.AppendChild(elm);
        }

        private void SaveConfig(XmlDocument doc, XmlElement elmRoot)
        {
            XmlElement elmConfig;
            XmlElement elm;
            XmlAttribute attr;

            elmConfig = doc.CreateElement(Properties.Resources.XML_PRJ_CONFIG);
            elmRoot.AppendChild(elmConfig);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_INPUTFILEPATH);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.config.InputFilePath;
            elm.Attributes.Append(attr);
            elmConfig.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_OUTPUTFILEPATH);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.config.OutputFilePath;
            elm.Attributes.Append(attr);
            elmConfig.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_OPTION);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.config.Option.ToString();
            elm.Attributes.Append(attr);
            elmConfig.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_SCALING);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.config.Transform.Scaling.ToString();
            elm.Attributes.Append(attr);
            elmConfig.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_ROTATION);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.config.Transform.Rotation.ToString();
            elm.Attributes.Append(attr);
            elmConfig.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_TRANSLATION);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.config.Transform.Translation.ToString();
            elm.Attributes.Append(attr);
            elmConfig.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_SMOOTING);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.config.Smooting.Enabled.ToString();
            elm.Attributes.Append(attr);
            elmConfig.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_SMOOTING_ANGLE);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.config.Smooting.Angle.ToString();
            elm.Attributes.Append(attr);
            elmConfig.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_FORMAT);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = Enum.GetName(typeof(ModelFormats), this.config.Format);
            elm.Attributes.Append(attr);
            elmConfig.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_CELLSIZE);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.config.MM_CellSize.ToString();
            elm.Attributes.Append(attr);
            elmConfig.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_MOTTIMEMODE);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = Enum.GetName(typeof(Mix.Tool.Graphics.ObjectModel.TimeModes), this.config.OM_TimeMode);
            elm.Attributes.Append(attr);
            elmConfig.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_IMPORT_COLLISION_FILEPATH);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.config.MM_InputCollisionFilePath;
            elm.Attributes.Append(attr);
            elmConfig.AppendChild(elm);
        }

        private void SaveProject(XmlDocument doc, XmlElement elmRoot)
        {
            XmlElement elmProject;
            XmlElement elm;
            XmlAttribute attr;

            elmProject = doc.CreateElement(Properties.Resources.XML_PRJ_PROJECT);
            elmRoot.AppendChild(elmProject);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_NAME);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.name;
            elm.Attributes.Append(attr);
            elmProject.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_DIRPATH);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.dirPath;
            elm.Attributes.Append(attr);
            elmProject.AppendChild(elm);
        }

        private void SaveMainWindow(XmlDocument doc, XmlElement elmRoot)
        {
            XmlElement elmMainWindow;
            XmlElement elm;
            XmlAttribute attr;

            elmMainWindow = doc.CreateElement(Properties.Resources.XML_PRJ_MAINWINDOW);
            elmRoot.AppendChild(elmMainWindow);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_SIZE);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = ToString_Size(this.mainWindow.Size);
            elm.Attributes.Append(attr);
            elmMainWindow.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_STATE);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = Enum.GetName(typeof(System.Windows.Forms.FormWindowState), this.mainWindow.WindowState);
            elm.Attributes.Append(attr);
            elmMainWindow.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_STYLE);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.editStyle.ToString();
            elm.Attributes.Append(attr);
            elmMainWindow.AppendChild(elm);
        }

        private void SavePreference(XmlDocument doc, XmlElement elmRoot)
        {
            XmlElement elmPref;
            XmlElement elmPrefScene;
            XmlElement elmPrefControl;
            XmlElement elm;
            XmlAttribute attr;

            elmPref = doc.CreateElement(Properties.Resources.XML_PRJ_PREF);
            elmRoot.AppendChild(elmPref);

            // シーン

            elmPrefScene = doc.CreateElement(Properties.Resources.XML_PRJ_SCENE);
            elmPref.AppendChild(elmPrefScene);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_BG_COLOR);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.General.BackgroundColor.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_GRID_COLOR);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.Grid.Color.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_GRID_PITCH);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.Grid.Pitch.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_GRID_EXTENT);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.Grid.Extent.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_TEXT_COLOR);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.General.TextColor.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_LIGHT_AMBIENT_COLOR);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.Light.AmbientColor.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_LIGHT_DIFFUSE_COLOR);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.Light.DiffuseColor.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_LIGHT_SHADING);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.Light.Shading.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_CAMERA_FOVY);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.graphicsMgr.Perspective.FovY.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_CAMERA_NC);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.graphicsMgr.Perspective.NearZ.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_CAMERA_FC);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.graphicsMgr.Perspective.FarZ.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_REFLECT_TEX);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.ReflectionTextureFileName;
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_VV_COLOR);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.General.ViewVolumeColor.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_COLL_COLOR);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.General.CollisionColor.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_BONE_COLOR);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.General.BoneColor.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_NODE_SELECTED_COLOR);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.General.SelectedColor.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_AXIS_SCALE);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.General.AxisScale.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_JOINT_SCALE);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.General.JointScale.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_NODE_SELECTED_AD);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.General.SelectedAlwaysDisplay.ToString();
            elm.Attributes.Append(attr);
            elmPrefScene.AppendChild(elm);

            // コントロール

            elmPrefControl = doc.CreateElement(Properties.Resources.XML_PRJ_CONTROL);
            elmPref.AppendChild(elmPrefControl);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_MOUSE_S);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.graphicsMgr.MouseSensitivity.Scaling.ToString();
            elm.Attributes.Append(attr);
            elmPrefControl.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_MOUSE_R);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.graphicsMgr.MouseSensitivity.Rotation.ToString();
            elm.Attributes.Append(attr);
            elmPrefControl.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_MOUSE_T);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.graphicsMgr.MouseSensitivity.Translation.ToString();
            elm.Attributes.Append(attr);
            elmPrefControl.AppendChild(elm);
        }

        private void SavePreview(XmlDocument doc, XmlElement elmRoot)
        {
            XmlElement elmPreview;
            XmlElement elmPreviewCamera;
            XmlElement elm;
            XmlAttribute attr;

            this.preview.Flush();

            elmPreview = doc.CreateElement(Properties.Resources.XML_PRJ_PREVIEW);
            elmRoot.AppendChild(elmPreview);

            // 編集モード
            elm = doc.CreateElement(Properties.Resources.XML_PRJ_EDITMODE);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.preview.EditMode.ToString();
            elm.Attributes.Append(attr);
            elmPreview.AppendChild(elm);

            // ルートトランスフォーム
            if (this.drawObject is Mix.Tool.Graphics.ObjectModel)
            {
                Mix.Tool.Graphics.ObjectModel objModel = this.drawObject as Mix.Tool.Graphics.ObjectModel;

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_RT_NAME);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = objModel.RootTransform.Active.ToString();
                elm.Attributes.Append(attr);
                elmPreview.AppendChild(elm);
            }

            // 表示
            elm = doc.CreateElement(Properties.Resources.XML_PRJ_DISPLAY);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.preview.Display.ToString();
            elm.Attributes.Append(attr);
            elmPreview.AppendChild(elm);

            // 描画
            elm = doc.CreateElement(Properties.Resources.XML_PRJ_DRAW);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.preview.FillMode.ToString();
            elm.Attributes.Append(attr);
            elmPreview.AppendChild(elm);

            // カメラ
            elmPreviewCamera = doc.CreateElement(Properties.Resources.XML_PRJ_CAMERA);
            elmPreview.AppendChild(elmPreviewCamera);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_LOCALPOS);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.preview.LocalPosition.ToString();
            elm.Attributes.Append(attr);
            elmPreviewCamera.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_TARGET);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.preview.Target.ToString();
            elm.Attributes.Append(attr);
            elmPreviewCamera.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_POSE);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.preview.Pose.ToString();
            elm.Attributes.Append(attr);
            elmPreviewCamera.AppendChild(elm);

            // ライトポーズ
            elm = doc.CreateElement(Properties.Resources.XML_PRJ_LIGHT_POSE);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = this.scene.Light.Pose.ToString();
            elm.Attributes.Append(attr);
            elmPreview.AppendChild(elm);
        }

        private static void SaveModel(XmlDocument doc, XmlElement elmRoot, Mix.Tool.Graphics.Model model)
        {
            XmlElement elmModel;

            elmModel = doc.CreateElement(Properties.Resources.XML_PRJ_MODEL);
            elmRoot.AppendChild(elmModel);

            if (model is Mix.Tool.Graphics.ObjectModel)
            {
                //オブジェクトモデル
                Mix.Tool.Graphics.ObjectModel objModel = model as Mix.Tool.Graphics.ObjectModel;

                if (objModel.DynamicsDesigner != null)
                {
                    //ダイナミクス : アクター
                    SaveDynamics_Actor(doc, elmModel, objModel.DynamicsDesigner as Mix.Tool.Dynamics.Design.Actor);
                }
            }
            else if (model is Mix.Tool.Graphics.MapModel)
            {
                //マップモデル
                Mix.Tool.Graphics.MapModel mapModel = model as Mix.Tool.Graphics.MapModel;

                if ((mapModel.DynamicsDesigner != null) &&
                    (mapModel.DynamicsDesigner.Status == Mix.Tool.Dynamics.Design.TerrainStatus.Available))
                {
                    //ダイナミクス : テライン
                    SaveDynamics_Terrain(doc, elmModel, mapModel.DynamicsDesigner);
                }
            }

            SaveNode(doc, elmModel, model.RootNode);
        }

        private static void SaveNode(XmlDocument doc, XmlElement elmParentNode, Mix.Tool.Graphics.Node node)
        {
            XmlElement elmNode = doc.CreateElement(Properties.Resources.XML_PRJ_NODE);
            XmlAttribute attr;

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ルート(ノード)
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            //名前
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_NAME);
            attr.Value = node.Name;
            elmNode.Attributes.Append(attr);

            if (node is Mix.Tool.Graphics.EmptyNode)
            {
                //エンプティ
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_TYPE);
                attr.Value = Properties.Resources.XML_PRJ_EMPTY;
                elmNode.Attributes.Append(attr);
            }
            else if (node is Mix.Tool.Graphics.MeshNode)
            {
                //メッシュ
                Mix.Tool.Graphics.MeshNode meshNode = node as Mix.Tool.Graphics.MeshNode;

                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_TYPE);
                attr.Value = Properties.Resources.XML_PRJ_MESH;
                elmNode.Attributes.Append(attr);

                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VISIBLE);
                attr.Value = meshNode.Visible.ToString();
                elmNode.Attributes.Append(attr);
            }
            else if (node is Mix.Tool.Graphics.BoneNode)
            {
                //ボーン
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_TYPE);
                attr.Value = Properties.Resources.XML_PRJ_BONE;
                elmNode.Attributes.Append(attr);
            }

            elmParentNode.AppendChild(elmNode);

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ダイナミクス
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            if (node.DynamicsDesigner != null)
            {
                SaveDynamics_Part(doc, elmNode, node.DynamicsDesigner as Mix.Tool.Dynamics.Design.Part);
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // 子を処理
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            foreach (Mix.Tool.Graphics.Node childNode in node.Childs)
            {
                SaveNode(doc, elmNode, childNode);
            }
        }

        private static void SaveDynamics_Actor(XmlDocument doc, XmlElement elmParent, Mix.Tool.Dynamics.Design.Actor actor)
        {
            if (actor == null)
            {
                throw new System.ArgumentNullException("actor");
            }

            if ((actor.Childs.Count == 0) &&
                (actor.KinematicCharacter == null))
            {
                //何も設定されていないので、書き込まない
                return;
            }

            XmlElement elmDynamics;
            XmlElement elmActor;

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ルート
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            elmDynamics = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS);
            elmParent.AppendChild(elmDynamics);

            elmActor = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_ACTOR);
            elmDynamics.AppendChild(elmActor);

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ベーシック
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            SaveDynamics_Basic(doc, elmActor, actor);

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // キネマティックキャラクター
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            if (actor.KinematicCharacter != null)
            {
                XmlElement elmKinematicCharacter;
                XmlElement elm;
                XmlAttribute attr;

                //ルート(kinematicCharacter)
                elmKinematicCharacter = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_KINEMATIC_CHARACTER);
                elmActor.AppendChild(elmKinematicCharacter);

                //height
                elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_HEIGHT);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = actor.KinematicCharacter.Height.ToString();
                elm.Attributes.Append(attr);
                elmKinematicCharacter.AppendChild(elm);

                //radius
                elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_RADIUS);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = actor.KinematicCharacter.Radius.ToString();
                elm.Attributes.Append(attr);
                elmKinematicCharacter.AppendChild(elm);

                //gravity
                elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_GRAVITY);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = actor.KinematicCharacter.Gravity.ToString();
                elm.Attributes.Append(attr);
                elmKinematicCharacter.AppendChild(elm);

                //maxFallSpeed
                elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_MAX_FALL_SPEED);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = actor.KinematicCharacter.MaxFallSpeed.ToString();
                elm.Attributes.Append(attr);
                elmKinematicCharacter.AppendChild(elm);

                //initJumpSpeed
                elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_INIT_JUMP_SPEED);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = actor.KinematicCharacter.InitJumpSpeed.ToString();
                elm.Attributes.Append(attr);
                elmKinematicCharacter.AppendChild(elm);

                //stepHeight
                elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_STEP_HEIGHT);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = actor.KinematicCharacter.StepOffset.ToString();
                elm.Attributes.Append(attr);
                elmKinematicCharacter.AppendChild(elm);

                //slopeLimit
                elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_SLOPE_LIMIT);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = actor.KinematicCharacter.SlopeLimit.ToString();
                elm.Attributes.Append(attr);
                elmKinematicCharacter.AppendChild(elm);

                //center
                elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_CENTER);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = actor.KinematicCharacterCenterPosition.ToString();
                elm.Attributes.Append(attr);
                elmKinematicCharacter.AppendChild(elm);

                //physicsMaterial
                if (actor.KinematicCharacter.PhysicsMaterial != null)
                {
                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_PHYSICS_MATERIAL);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_NAME);
                    attr.Value = actor.KinematicCharacter.PhysicsMaterial.Name;
                    elm.Attributes.Append(attr);
                    elmKinematicCharacter.AppendChild(elm);
                }
            }
        }

        private static void SaveDynamics_Part(XmlDocument doc, XmlElement elmParent, Mix.Tool.Dynamics.Design.Part part)
        {
            if (part == null)
            {
                throw new System.ArgumentNullException("part");
            }

            if ((part.Childs.Count == 0) &&
                (part.Joint == null))
            {
                //何も設定されていないので、書き込まない
                return;
            }

            XmlElement elmDynamics;
            XmlElement elmPart;

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ルート
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            elmDynamics = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS);
            elmParent.AppendChild(elmDynamics);

            elmPart = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_PART);
            elmDynamics.AppendChild(elmPart);

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ベーシック
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            SaveDynamics_Basic(doc, elmPart, part);

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ジョイント
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            if (part.Joint != null)
            {
                XmlElement elmJoint;
                XmlElement elm;
                XmlAttribute attr;

                //ルート
                elmJoint = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_JOINT);
                elmPart.AppendChild(elmJoint);

                //固有情報
                if (part.Joint is Mix.Tool.Dynamics.PointJoint)
                {
                    Mix.Tool.Dynamics.PointJoint joint = part.Joint as Mix.Tool.Dynamics.PointJoint;

                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_TYPE);
                    attr.Value = Properties.Resources.XML_PRJ_DYNAMICS_POINT;
                    elmJoint.Attributes.Append(attr);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_SPRING);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.Spring.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_DAMPER);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.Damper.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);
                }
                else if (part.Joint is Mix.Tool.Dynamics.HingeJoint)
                {
                    Mix.Tool.Dynamics.HingeJoint joint = part.Joint as Mix.Tool.Dynamics.HingeJoint;

                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_TYPE);
                    attr.Value = Properties.Resources.XML_PRJ_DYNAMICS_HINGE;
                    elmJoint.Attributes.Append(attr);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_AXIS);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.Axis.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_LOWER_LIMIT);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.LowerLimit.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_UPPER_LIMIT);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.UpperLimit.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_LIMIT_SPRING);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.LimitSpring.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_LIMIT_DAMPER);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.LimitDamper.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);
                }
                else if (part.Joint is Mix.Tool.Dynamics.BallJoint)
                {
                    Mix.Tool.Dynamics.BallJoint joint = part.Joint as Mix.Tool.Dynamics.BallJoint;

                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_TYPE);
                    attr.Value = Properties.Resources.XML_PRJ_DYNAMICS_BALL;
                    elmJoint.Attributes.Append(attr);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_SWING_AXIS);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.SwingAxis.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_TWIST_AXIS);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.TwistAxis.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_SWING_LIMIT1);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.SwingLimit1.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_SWING_LIMIT2);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.SwingLimit2.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_TWIST_LIMIT);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.TwistLimit.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_SPRING);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.Spring.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_DAMPER);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.Damper.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_LIMIT_SPRING);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.LimitSpring.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_LIMIT_DAMPER);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = joint.LimitDamper.ToString();
                    elm.Attributes.Append(attr);
                    elmJoint.AppendChild(elm);
                }

                //ピボット
                elm = doc.CreateElement(Properties.Resources.XML_PRJ_PIVOT);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = part.JointPivot.ToString();
                elm.Attributes.Append(attr);
                elmJoint.AppendChild(elm);

                //接続されたコライダー同士の衝突を無効
                elm = doc.CreateElement(Properties.Resources.XML_PRJ_COLLISION_DISABLED);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = part.Joint.CollisionDisabled.ToString();
                elm.Attributes.Append(attr);
                elmJoint.AppendChild(elm);
            }
        }

        private static void SaveDynamics_Basic(XmlDocument doc, XmlElement elmParent, Mix.Tool.Dynamics.Design.Basic basic)
        {
            if (basic == null)
            {
                throw new System.ArgumentNullException("basic");
            }

            if (basic.Childs.Count == 0)
            {
                return;
            }

            XmlElement elmBasic;

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ルート(ベーシック)
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            elmBasic = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_BASIC);
            elmParent.AppendChild(elmBasic);

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // リジッドボディ
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            foreach (Mix.Tool.Dynamics.Design.BasicChild child in basic.Childs)
            {
                SaveDynamics_RigidBody(doc, elmBasic, child);
            }
        }

        private static void SaveDynamics_RigidBody( XmlDocument doc, XmlElement elmBasic, Mix.Tool.Dynamics.Design.BasicChild child )
        {
            XmlElement elmRigidBody;
            XmlElement elm;
            XmlAttribute attr;

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ルート( リジッドボディ )
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            elmRigidBody = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_RIGID_BODY);

            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_TYPE);
            attr.Value = child.Type.ToString();
            elmRigidBody.Attributes.Append(attr);

            elmBasic.AppendChild(elmRigidBody);

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // 基本情報
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            //名前
            elm = doc.CreateElement(Properties.Resources.XML_PRJ_NAME);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = child.Name;
            elm.Attributes.Append(attr);
            elmRigidBody.AppendChild(elm);

            //中心 : 回転
            elm = doc.CreateElement(Properties.Resources.XML_PRJ_ROTATION);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = child.CenterRotation.ToString();
            elm.Attributes.Append(attr);
            elmRigidBody.AppendChild(elm);

            //中心 : 位置
            elm = doc.CreateElement(Properties.Resources.XML_PRJ_POSITION);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = child.CenterPosition.ToString();
            elm.Attributes.Append(attr);
            elmRigidBody.AppendChild(elm);

            //物理マテリアル
            if (child.PhysicsMaterial != null)
            {
                elm = doc.CreateElement(Properties.Resources.XML_PRJ_PHYSICS_MATERIAL);

                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_NAME);
                attr.Value = child.PhysicsMaterial.Name;
                elm.Attributes.Append(attr);

                elmRigidBody.AppendChild(elm);
            }

            //ステータス
            elm = doc.CreateElement(Properties.Resources.XML_PRJ_STATUS);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = child.Status.ToString();
            elm.Attributes.Append(attr);
            elmRigidBody.AppendChild(elm);

            //質量
            elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_MASS);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = child.Mass.ToString();
            elm.Attributes.Append(attr);
            elmRigidBody.AppendChild(elm);

            //回転軸
            elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_ROTATION_AXIS);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = child.RotationAxisFlags.ToString();
            elm.Attributes.Append(attr);
            elmRigidBody.AppendChild(elm);

            //常にアクティブ
            elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_ALWAYS_ACTIVE);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = child.AlwaysActive.ToString();
            elm.Attributes.Append(attr);
            elmRigidBody.AppendChild(elm);

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // シェイプ
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            foreach (Mix.Tool.Dynamics.Design.Shape shape in child.Shapes)
            {
                XmlElement elmShape;

                elmShape = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_SHAPE);

                //固有情報
                if (shape is Mix.Tool.Dynamics.Design.BoxShape)
                {
                    Mix.Tool.Dynamics.Design.BoxShape boxShape = shape as Mix.Tool.Dynamics.Design.BoxShape;

                    //タイプ
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_TYPE);
                    attr.Value = Properties.Resources.XML_PRJ_BOX;
                    elmShape.Attributes.Append(attr);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_HALF_EXTENTS);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = boxShape.HalfExtents.ToString();
                    elm.Attributes.Append(attr);
                    elmShape.AppendChild(elm);
                }
                else if (shape is Mix.Tool.Dynamics.Design.CapsuleShape)
                {
                    Mix.Tool.Dynamics.Design.CapsuleShape capsuleShape = shape as Mix.Tool.Dynamics.Design.CapsuleShape;

                    //タイプ
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_TYPE);
                    attr.Value = Properties.Resources.XML_PRJ_CAPSULE;
                    elmShape.Attributes.Append(attr);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_AXIS);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = capsuleShape.Axis.ToString();
                    elm.Attributes.Append(attr);
                    elmShape.AppendChild(elm);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_RADIUS);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = capsuleShape.Radius.ToString();
                    elm.Attributes.Append(attr);
                    elmShape.AppendChild(elm);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_LENGTH);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = capsuleShape.Length.ToString();
                    elm.Attributes.Append(attr);
                    elmShape.AppendChild(elm);
                }
                else if (shape is Mix.Tool.Dynamics.Design.SphereShape)
                {
                    Mix.Tool.Dynamics.Design.SphereShape sphereShape = shape as Mix.Tool.Dynamics.Design.SphereShape;

                    //タイプ
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_TYPE);
                    attr.Value = Properties.Resources.XML_PRJ_SPHERE;
                    elmShape.Attributes.Append(attr);

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_RADIUS);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                    attr.Value = sphereShape.Radius.ToString();
                    elm.Attributes.Append(attr);
                    elmShape.AppendChild(elm);
                }

                //ベース(バウンディングボックス) : 名前(属性)
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_BASE_NAME);
                attr.Value = shape.BoundingBox.Name;
                elmShape.Attributes.Append(attr);

                //ベース(バウンディングボックス) : 深度(属性)
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_BASE_DEPTH);
                attr.Value = shape.BoundingBox.Depth.ToString();
                elmShape.Attributes.Append(attr);

                //回転
                elm = doc.CreateElement(Properties.Resources.XML_PRJ_ROTATION);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = shape.Rotation.ToString();
                elm.Attributes.Append(attr);
                elmShape.AppendChild(elm);

                //位置
                elm = doc.CreateElement(Properties.Resources.XML_PRJ_POSITION);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = shape.Position.ToString();
                elm.Attributes.Append(attr);
                elmRigidBody.AppendChild(elmShape);
                elmShape.AppendChild(elm);
            }
        }

        private static void SaveDynamics_Terrain(XmlDocument doc, XmlElement elmParent, Mix.Tool.Dynamics.Design.Terrain terrain)
        {
            if ((terrain == null) ||
                (terrain.Status != Mix.Tool.Dynamics.Design.TerrainStatus.Available))
            {
                return;
            }

            XmlElement elmDynamics;
            XmlElement elmTerrain;
            XmlAttribute attr;

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ルート
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            //ダイナミクス
            elmDynamics = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS);
            elmParent.AppendChild(elmDynamics);

            //テライン
            elmTerrain = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_TERRAIN);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_FILEPATH);
            attr.Value = terrain.FilePath;
            elmTerrain.Attributes.Append(attr);
            elmDynamics.AppendChild(elmTerrain);

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // パート
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            foreach (Mix.Tool.Dynamics.Design.TerrainPart part in terrain.Parts)
            {
                XmlElement elmPart;

                //ルート
                elmPart = doc.CreateElement(Properties.Resources.XML_PRJ_DYNAMICS_TERRAIN_PART);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_NAME);
                attr.Value = part.Name;
                elmPart.Attributes.Append(attr);

                //物理マテリアル
                if (part.PhysicsMaterial != null)
                {
                    XmlElement elm;

                    elm = doc.CreateElement(Properties.Resources.XML_PRJ_PHYSICS_MATERIAL);
                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_NAME);
                    attr.Value = part.PhysicsMaterial.Name;
                    elm.Attributes.Append(attr);

                    elmPart.AppendChild(elm);
                }

                elmTerrain.AppendChild(elmPart);
            }
        }

        private static void SaveMaterialSlot(XmlDocument doc, XmlElement elmRoot, Mix.Tool.Graphics.MaterialSlot materialSlot)
        {
            XmlElement elmMaterialSlot;
            XmlElement elm;
            XmlAttribute attr;

            elmMaterialSlot = doc.CreateElement(Properties.Resources.XML_PRJ_MATERIALSLOT);
            elmRoot.AppendChild(elmMaterialSlot);

            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_DEFAULTNAME);
            attr.Value = materialSlot.DefaultName;
            elmMaterialSlot.Attributes.Append(attr);

            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_NAME);
            attr.Value = materialSlot.Name;
            elmMaterialSlot.Attributes.Append(attr);

            foreach (Mix.Tool.Graphics.Material material in materialSlot.Materials)
            {
                XmlElement elmMaterial = doc.CreateElement(Properties.Resources.XML_PRJ_MATERIAL);
                XmlElement elmChild;

                //////////////////////////////////////////////////

                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_DEFAULTNAME);
                attr.Value = material.DefaultName;
                elmMaterial.Attributes.Append(attr);

                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_NAME);
                attr.Value = material.Name;
                elmMaterial.Attributes.Append(attr);

                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_DEFAULT);
                attr.Value = material.IsDefault.ToString();
                elmMaterial.Attributes.Append(attr);

                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_ACTIVE);
                attr.Value = material.IsActive.ToString();
                elmMaterial.Attributes.Append(attr);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_SAVEFILEPATH);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.SaveFilePath;
                elm.Attributes.Append(attr);
                elmMaterial.AppendChild(elm);

                //////////////////////////////////////////////////
                // General
                //////////////////////////////////////////////////

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_SHADERTYPE);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = Enum.GetName(typeof(Mix.Tool.Graphics.ShaderTypes), material.ShaderType);
                elm.Attributes.Append(attr);
                elmMaterial.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_AMBIENT_COLOR);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.AmbientColor.ToString();
                elm.Attributes.Append(attr);
                elmMaterial.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_BOTH_LIGHTING);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.BothLighting.ToString();
                elm.Attributes.Append(attr);
                elmMaterial.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_BLENDMODE);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = Enum.GetName(typeof(Mix.Tool.Graphics.BlendModes), material.BlendMode);
                elm.Attributes.Append(attr);
                elmMaterial.AppendChild(elm);

                // Transparency

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_TRANSPARENCY);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.TransparencyEnabled.ToString();
                elm.Attributes.Append(attr);
                elmMaterial.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_TRANSPARENCY_METHOD);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.TransparencyMethod.ToString();
                elm.Attributes.Append(attr);
                elmMaterial.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_ZWRITE);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.ZWrite.ToString();
                elm.Attributes.Append(attr);
                elmMaterial.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_SOFTEDGE);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.SoftEdgeEnabled.ToString();
                elm.Attributes.Append(attr);
                elmMaterial.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_SOFTEDGE_TICKNESS);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.SoftEdgeTickness.ToString();
                elm.Attributes.Append(attr);
                elmMaterial.AppendChild(elm);

                // Misc

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_BACKCULLING);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Backculling.ToString();
                elm.Attributes.Append(attr);
                elmMaterial.AppendChild(elm);

                // TextureMapping

                elmChild = doc.CreateElement(Properties.Resources.XML_PRJ_TEXTUREMAPPING);
                elmMaterial.AppendChild(elmChild);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_FILTER);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = Enum.GetName(typeof(Mix.Tool.Graphics.TextureFilterTypes), material.TextureMapping.FilterType);
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_ADDRESS);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = Enum.GetName(typeof(Mix.Tool.Graphics.TextureAddressTypes), material.TextureMapping.AddressType);
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                // ShadowMapping

                elmChild = doc.CreateElement(Properties.Resources.XML_PRJ_SHADOWMAPPING);
                elmMaterial.AppendChild(elmChild);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_RECEIVE);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.ShadowMapping.Receive.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_CAST);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.ShadowMapping.Cast.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                //////////////////////////////////////////////////
                // Diffuse
                //////////////////////////////////////////////////

                elmChild = doc.CreateElement(Properties.Resources.XML_PRJ_DIFFUSE);
                elmMaterial.AppendChild(elmChild);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_TYPE);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Diffuse.Type.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_COLOR);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Diffuse.Color.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_TEXTUREFILENAME);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Diffuse.TextureFileName;
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_FRESNEL);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Diffuse.Fresnel.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_FRESNEL_INTENSITY);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Diffuse.FresnelRatio.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_RL);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Diffuse.RimLighting.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_RL_HARDNESS);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Diffuse.RimLightingHardness.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_RL_SCALE);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Diffuse.RimLightingScale.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                //////////////////////////////////////////////////
                // Specular
                //////////////////////////////////////////////////

                elmChild = doc.CreateElement(Properties.Resources.XML_PRJ_SPECULAR);
                elmMaterial.AppendChild(elmChild);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_TYPE);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Specular.Type.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_COLOR);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Specular.Color.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_TEXTUREFILENAME);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Specular.TextureFileName;
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_HARDNESS);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Specular.Hardness.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_SCALE);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Specular.Scale.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_FRESNEL);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Specular.CookTorranceFresnel.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_ROUGHNESS);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Specular.CookTorranceRoughness.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_REFLECT_TYPE);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Specular.ReflectType.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_REFLECT_SCALE);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Specular.ReflectScale.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_REFLECT_INTENSITY);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Specular.ReflectIntensity.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_REFLECT_BIAS);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Specular.ReflectBias.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_REFLECT_EXP);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Specular.ReflectExprosure.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_REFLECT_BUMP_RATIO);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Specular.ReflectBumpRatio.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                //////////////////////////////////////////////////
                // Emissive
                //////////////////////////////////////////////////

                elmChild = doc.CreateElement(Properties.Resources.XML_PRJ_EMISSIVE);
                elmMaterial.AppendChild(elmChild);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_COLOR);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Emissive.Color.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_TEXTUREFILENAME);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Emissive.TextureFileName;
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_SCALE);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Emissive.Scale.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                //////////////////////////////////////////////////
                // Bump
                //////////////////////////////////////////////////

                elmChild = doc.CreateElement(Properties.Resources.XML_PRJ_BUMP);
                elmMaterial.AppendChild(elmChild);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_TEXTUREFILENAME);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Bump.TextureFileName;
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_PARALLAXMAPPING);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Bump.ParallaxEnabled.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_PARALLAXMAPPING_HS);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Bump.ParallaxHeightScale.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_PARALLAXMAPPING_SAMPLE);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Bump.ParallaxSample.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_WAVE_TYPE);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = Enum.GetName(typeof(Mix.Tool.Graphics.WaveTypes), material.Bump.WaveType);
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_WAVE_DIR);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = Enum.GetName(typeof(Mix.Tool.Graphics.WaveDirs), material.Bump.WaveDir);
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_WAVE_INVERT);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Bump.WaveInvert.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_WAVE_VEL);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Bump.WaveVelocity.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_WAVE_UV);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Bump.WaveUVOffset.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_WAVE_GAP);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Bump.WaveGap.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                elm = doc.CreateElement(Properties.Resources.XML_PRJ_WAVE_REFRACTION);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = material.Bump.WaveRefraction.ToString();
                elm.Attributes.Append(attr);
                elmChild.AppendChild(elm);

                //////////////////////////////////////////////////

                elmMaterialSlot.AppendChild(elmMaterial);
            }
        }

        private static void SaveRootMotion(XmlDocument doc, XmlElement elmParent, Mix.Tool.Graphics.ObjectModel objModel)
        {
            XmlElement elmRM = doc.CreateElement(Properties.Resources.XML_PRJ_ROOT_MOTION);

            XmlElement elm;
            XmlAttribute attr;

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_ACTIVE);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = objModel.RootTransform.Active.ToString();
            elm.Attributes.Append(attr);
            elmRM.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_NODE_NAME);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = objModel.RootTransform.NodeName.ToString();
            elm.Attributes.Append(attr);
            elmRM.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_THORIZ);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = objModel.RootTransform.TranslationHorizonal.ToString();
            elm.Attributes.Append(attr);
            elmRM.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_TVERT);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = objModel.RootTransform.TranslationVertical.ToString();
            elm.Attributes.Append(attr);
            elmRM.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_RAXIS);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = objModel.RootTransform.RotationAxis.ToString();
            elm.Attributes.Append(attr);
            elmRM.AppendChild(elm);

            elm = doc.CreateElement(Properties.Resources.XML_PRJ_TAXIS);
            attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
            attr.Value = objModel.RootTransform.TranslationAxis.ToString();
            elm.Attributes.Append(attr);
            elmRM.AppendChild(elm);

            elmParent.AppendChild(elmRM);
        }

        private static void SaveMotionControllers(XmlDocument doc, XmlElement elmRoot, Mix.Tool.Graphics.ObjectModel objModel)
        {
            Mix.Tool.Graphics.MotionController unaMotCtrl = objModel.UnassignedMotionController;

            XmlElement elm;
            XmlAttribute attr;

            //分類済みのモーション
            foreach (Mix.Tool.Graphics.MotionController motCtrl in objModel.MotionControllers)
            {
                //ルート
                elm = doc.CreateElement(Properties.Resources.XML_PRJ_MOTIONCTRL);
                elmRoot.AppendChild(elm);

                //名前
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_NAME);
                attr.Value = motCtrl.Name;
                elm.Attributes.Append(attr);

                //プライオリティ
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_PRIORITY);
                attr.Value = motCtrl.Priority.ToString();
                elm.Attributes.Append(attr);

                SaveMotionController(doc, elm, motCtrl);
            }

            //未分類のモーション
            if (unaMotCtrl.Motions.Count > 0)
            {
                elm = doc.CreateElement(Properties.Resources.XML_PRJ_UAMOTIONS);
                elmRoot.AppendChild(elm);

                SaveMotionController(doc, elm, unaMotCtrl);
            }
        }

        private static void SaveMotionController(XmlDocument doc, XmlElement elmMotCtrl, Mix.Tool.Graphics.MotionController motCtrl)
        {
            XmlElement elmMot;
            XmlElement elmRT;
            XmlElement elmParam;
            XmlAttribute attr;

            foreach (Mix.Tool.Graphics.Motion mot in motCtrl.Motions)
            {
                elmMot = doc.CreateElement(Properties.Resources.XML_PRJ_MOTION);

                //////////////////////////////////////////////////

                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_DEFAULTNAME);
                attr.Value = mot.DefaultName;
                elmMot.Attributes.Append(attr);

                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_NAME);
                attr.Value = mot.Name;
                elmMot.Attributes.Append(attr);

                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_DEFAULT);
                attr.Value = mot.Default.ToString();
                elmMot.Attributes.Append(attr);

                //////////////////////////////////////////////////

                elmRT = doc.CreateElement(Properties.Resources.XML_PRJ_ROOTTRANSFORM);
                elmMot.AppendChild(elmRT);

                elmParam = doc.CreateElement(Properties.Resources.XML_PRJ_FLAGS);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = mot.RootTransform.ToString();
                elmParam.Attributes.Append(attr);
                elmRT.AppendChild(elmParam);

                //////////////////////////////////////////////////

                elmParam = doc.CreateElement(Properties.Resources.XML_PRJ_SPEED);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = mot.Speed.ToString();
                elmParam.Attributes.Append(attr);
                elmMot.AppendChild(elmParam);

                elmParam = doc.CreateElement(Properties.Resources.XML_PRJ_BLENDRATIO);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = mot.BlendRatio.ToString();
                elmParam.Attributes.Append(attr);
                elmMot.AppendChild(elmParam);

                elmParam = doc.CreateElement(Properties.Resources.XML_PRJ_LOOP);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = mot.Loop.ToString();
                elmParam.Attributes.Append(attr);
                elmMot.AppendChild(elmParam);

                elmParam = doc.CreateElement(Properties.Resources.XML_PRJ_LOOPSTARTTIME);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = mot.LoopStartTime.ToString();
                elmParam.Attributes.Append(attr);
                elmMot.AppendChild(elmParam);

                elmParam = doc.CreateElement(Properties.Resources.XML_PRJ_LOOPENDTIME);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = mot.LoopEndTime.ToString();
                elmParam.Attributes.Append(attr);
                elmMot.AppendChild(elmParam);

                elmParam = doc.CreateElement(Properties.Resources.XML_PRJ_SAVEFILEPATH);
                attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_VALUE);
                attr.Value = mot.SaveFilePath.ToString();
                elmParam.Attributes.Append(attr);
                elmMot.AppendChild(elmParam);

                //////////////////////////////////////////////////

                foreach (Mix.Tool.Graphics.MotionCurve motCurve in mot.Curves)
                {
                    elmParam = doc.CreateElement(Properties.Resources.XML_PRJ_MOTCURVE);

                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_LINKNODENAME);
                    attr.Value = motCurve.LinkNodeName;
                    elmParam.Attributes.Append(attr);

                    attr = doc.CreateAttribute(Properties.Resources.XML_PRJ_ENABLED);
                    attr.Value = motCurve.Enabled.ToString();
                    elmParam.Attributes.Append(attr);

                    elmMot.AppendChild(elmParam);
                }

                //////////////////////////////////////////////////

                elmMotCtrl.AppendChild(elmMot);
            }
        }

        #endregion

        #region EditStyle Method

        private void LoadEditStyle(EditStyle loadEditStyle)
        {
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // 直前のドックコンテンツを破棄
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            this.CloseDockContents();

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // ドックコンテンツを表示j
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            try
            {
                this.dockPanel.LoadFromXml(GetEditStyleFilePath(loadEditStyle), this.deserializeDockContent);
            }
            catch
            {
                string esText = "";

                if (this.drawObject is Mix.Tool.Graphics.ObjectModel)
                {
                    switch (loadEditStyle)
                    {
                        case EditStyle.Material:
                            esText = Properties.Resources.defMaterial_es;
                            break;
                        case EditStyle.Motion:
                            esText = Properties.Resources.defMotion_ObjectModel_es;
                            break;
                        case EditStyle.Physics:
                            esText = Properties.Resources.defPhysics_ObjectModel_es;
                            break;
                        default:
                            throw new System.Exception("サポートされないエディットスタイルをロードしようとしました(オブジェクトモデル)");
                    }
                }
                else if (this.drawObject is Mix.Tool.Graphics.MapModel)
                {
                    switch (loadEditStyle)
                    {
                        case EditStyle.Material:
                            esText = Properties.Resources.defMaterial_es;
                            break;
                        case EditStyle.Physics:
                            esText = Properties.Resources.defPhysics_MapModel_es;
                            break;
                        default:
                            throw new System.Exception("サポートされないエディットスタイルをロードしようとしました(マップモデル)");
                    }
                }

                using (System.IO.MemoryStream memStream = new System.IO.MemoryStream(Encoding.Unicode.GetBytes(esText)))
                {
                    this.dockPanel.LoadFromXml(memStream, this.deserializeDockContent);
                }
            }
        }

        private void SaveEditStyle(EditStyle saveEditStyle)
        {
            try
            {
                this.dockPanel.SaveAsXml(GetEditStyleFilePath(saveEditStyle));
            }
            catch
            {
                throw;
            }
        }

        private string GetEditStyleFilePath(EditStyle editStyle)
        {
            try
            {
                string filePath;

                switch (editStyle)
                {
                    case EditStyle.Material:
                        filePath = Path.Combine(this.dirPath, this.name);
                        filePath = Path.Combine(filePath, Properties.Resources.ES_FILE_MATERIAL);
                        break;
                    case EditStyle.Motion:
                        filePath = Path.Combine(this.dirPath, this.name);
                        filePath = Path.Combine(filePath, Properties.Resources.ES_FILE_MOTION);
                        break;
                    case EditStyle.Physics:
                        filePath = Path.Combine(this.dirPath, this.name);
                        filePath = Path.Combine(filePath, Properties.Resources.ES_FILE_PHYSICS);
                        break;
                    default:
                        throw new System.Exception("サポートされないエディットスタイルのファイルパスを要求されました");
                }

                return filePath;
            }
            catch
            {
                throw;
            }
        }

        #endregion

        #region Notify Method

        private void NotifyBeginLoad()
        {
            if (this.BeginLoad != null)
            {
                this.BeginLoad(this, new System.EventArgs());
            }

            this.CloseDockContents();

            using (System.IO.MemoryStream memStream = new System.IO.MemoryStream(Encoding.Unicode.GetBytes(Properties.Resources.startDocking)))
            {
                this.dockPanel.LoadFromXml(memStream, this.deserializeDockContent);
            }
        }

        private void NotifyEndLoad()
        {
            if (this.EndLoad != null)
            {
                this.EndLoad(this, new System.EventArgs());
            }
        }

        #endregion

        #region BackgroundTask Excuted Method

        /// <summary>
        /// モデルのインポート完了ハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Executed_ImportModelTask(object sender, System.EventArgs e)
        {
            if (sender is ImportModelTask)
            {
                ImportModelTask task = sender as ImportModelTask;
                this.drawObject = task.Model;
            }
        }

        /// <summary>
        /// ロード完了ハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Executed_EndLoadTask(object sender, System.EventArgs e)
        {
            if (sender is EndLoadTask)
            {
                EndLoadTask task = sender as EndLoadTask;

                //プロジェクト
                string projectDirPath = System.IO.Path.Combine(this.dirPath, this.name);
                string projectFilePath = System.IO.Path.Combine(projectDirPath, this.name + Properties.Resources.FILE_EXT_PRJ);

                //メインウィンドウ
                this.mainWindow.Size = task.MainWndSize;
                this.mainWindow.WindowState = task.MainWndState;

                //エディットスタイル
                this.LoadEditStyle(task.MainWndStyle);
                this.editStyle = task.MainWndStyle;

                //ロードの完了通知
                this.NotifyEndLoad();

                //ログ
                switch (task.Type)
                {
                    case Manager.EndLoadTask.Types.New:
                        Mix.Tool.Logger.Print(Logger.Type.Info, "----------------------------------------------------------------------------------------------------");
                        Mix.Tool.Logger.Print(Logger.Type.Info, "  プロジェクトを作成しました : FilePath[" + projectFilePath + "]");
                        break;
                    case Manager.EndLoadTask.Types.Existing:
                        Mix.Tool.Logger.Print(Logger.Type.Info, "----------------------------------------------------------------------------------------------------");
                        Mix.Tool.Logger.Print(Logger.Type.Info, "  プロジェクトを開きました : FilePath[" + projectFilePath + "]");
                        break;
                }
            }
        }

        #endregion

        #region Misc Method

        /// <summary>
        /// グラフィックスマネージャのセットアップ
        /// </summary>
        private void SetupGraphicsManager()
        {
            this.graphicsMgr.RootDirectoryPath = this.rootDirPath;

            this.graphicsMgr.EffectPackage.Load(
                this.template.ShaderModels,
                this.template.ShaderProcFlags,
                this.template.EffectPackageDirectoryPath,
                this.template.EffectPackageName);
        }

        /// <summary>
        /// ロード
        /// </summary>
        /// <param name="type"></param>
        /// <param name="filePath"></param>
        /// <param name="elmModel"></param>
        /// <param name="elmMaterials"></param>
        /// <param name="elmMotCtrls"></param>
        /// <param name="elmMotUA"></param>
        /// <param name="elmMotRoot"></param>
        /// <param name="motRootTransformFlags"></param>
        /// <param name="mainWndSize"></param>
        /// <param name="mainWndState"></param>
        private void Load(
            Manager.EndLoadTask.Types type,
            string filePath,
            XmlElement elmModel,
            List<XmlElement> elmMaterials,
            List<XmlElement> elmMotCtrls,
            XmlElement elmMotUA,
            XmlElement elmMotRoot,
            Mix.Tool.Graphics.RootTransformFlags motRootTransformFlags,
            Size mainWndSize,
            System.Windows.Forms.FormWindowState mainWndState,
            EditStyle mainWndStyle)
        {
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // モデルのインポートタスク
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            if (this.config.Format == ModelFormats.Actor)
            {
                Mix.Tool.Graphics.ObjectModel.ImportConfig importConfig = new Mix.Tool.Graphics.ObjectModel.ImportConfig();
                Mix.Tool.Graphics.ObjectModel objectModel = new Mix.Tool.Graphics.ObjectModel();

                importConfig.FilePath = filePath;
                importConfig.Option = this.config.Option;
                importConfig.Scaling = this.config.Transform.Scaling;
                importConfig.Rotation = this.config.Transform.Rotation;
                importConfig.Translation = this.config.Transform.Translation;
                importConfig.Smooting = this.config.Smooting.Enabled;
                importConfig.SmootingAngle = this.config.Smooting.Angle;
                importConfig.TimeMode = this.config.OM_TimeMode;

                objectModel.RootTransform.Active = motRootTransformFlags;

                this.backgroundWorker.AddTask(new ImportObjectModelTask(
                    this.Dynamics.PhysicsMaterials,
                    importConfig,
                    objectModel,
                    elmModel,
                    elmMaterials,
                    elmMotCtrls,
                    elmMotUA,
                    elmMotRoot),
                    this.Executed_ImportModelTask);
            }
            else if (this.config.Format == ModelFormats.Terrain)
            {
                Mix.Tool.Graphics.MapModel.ImportConfig importConfig = new Mix.Tool.Graphics.MapModel.ImportConfig();
                Mix.Tool.Graphics.MapModel mapModel = new Mix.Tool.Graphics.MapModel();

                importConfig.FilePath = filePath;
                importConfig.Option = this.config.Option;
                importConfig.Scaling = this.config.Transform.Scaling;
                importConfig.Rotation = this.config.Transform.Rotation;
                importConfig.Translation = this.config.Transform.Translation;
                importConfig.Smooting = this.config.Smooting.Enabled;
                importConfig.SmootingAngle = this.config.Smooting.Angle;
                importConfig.CellSize = this.config.MM_CellSize;

                this.backgroundWorker.AddTask(new ImportMapModelTask(
                    this.Dynamics.PhysicsMaterials,
                    importConfig,
                    mapModel,
                    elmModel,
                    elmMaterials),
                    this.Executed_ImportModelTask);
            }

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // プロジェクトの読み込み完了タスク
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            this.backgroundWorker.AddTask(new EndLoadTask(type, mainWndSize, mainWndState, mainWndStyle), this.Executed_EndLoadTask);
        }

        /// <summary>
        /// モデルの解放
        /// </summary>
        private void ReleaseModel()
        {
            if (this.drawObject != null)
            {
                this.drawObject.Dispose();
                this.drawObject = null;
            }
        }

        /// <summary>
        /// ドックコンテンツを閉じる
        /// </summary>
        private void CloseDockContents()
        {
            while (this.dockPanel.Contents.Count > 0)
            {
                Mix.Tool.Docking.DockContent dockContent = this.dockPanel.Contents[0] as Mix.Tool.Docking.DockContent;
                dockContent.Close();
            }
        }

        /// <summary>
        /// Size の文字列化
        /// </summary>
        /// <param name="size"></param>
        /// <returns></returns>
        private static string ToString_Size(Size size)
        {
            return size.Width.ToString() + "," + size.Height.ToString();
        }

        /// <summary>
        /// Size のパース
        /// </summary>
        /// <param name="str"></param>
        /// <returns></returns>
        private static Size Parse_Size(string str)
        {
            Size size = new Size();

            string[] sepStrings = str.Split(',');
            if (sepStrings.Length == 2)
            {
                size.Width = int.Parse(sepStrings[0]);
                size.Height = int.Parse(sepStrings[1]);
            }
            else
            {
                size.Width = 0;
                size.Height = 0;
            }

            return size;
        }

        #endregion

        #region IDisposable Method

        public void Dispose()
        {
            //ダイナミクスの保存
            this.Dynamics.Save();

            //解放
            this.CloseDockContents();
            this.ReleaseModel();
            this.log.Clear();
        }

        #endregion
    }
}
