﻿using System;
using System.ComponentModel;
using System.Windows.Forms;

namespace Mix.Tool.ModelEditor
{
    /// <summary>
    /// プロジェクトプロパティフォームクラス
    /// </summary>
    public partial class ProjectPropertyForm : Mix.Tool.Docking.DockContent
    {
        private Project.Manager project = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="_project">プロジェクト</param>
        public ProjectPropertyForm(Project.Manager _project)
        {
            InitializeComponent();

            this.project = _project;

            this.tbBasicRootDir.Text = this.project.RootDirectoryPath;
            this.tbBasicTemplateDesc.Text = this.project.Template.Desc;
            this.tbInputFile.Text = this.project.Config.InputFilePath;
            this.tbOutputFile.Text = this.project.Config.OutputFilePath;
            this.tbProjectName.Text = this.project.Name;
            this.tbProjectDirPath.Text = this.project.DirectoryPath;

            this.UpdateControlsEnabled();
        }

        private void btBasicRootDir_Click(object sender, EventArgs e)
        {
            using (Mix.Tool.DirectoryBrowseDialog dlg = new DirectoryBrowseDialog())
            {
                dlg.Root = DirectoryBrowseDialog.RootType.Specify;
                dlg.RootDirectoryName = "";
                dlg.DirectoryName = this.tbBasicRootDir.Text;

                if (dlg.ShowDialog() == DialogResult.OK)
                {
                    this.tbBasicRootDir.Text = dlg.DirectoryName;
                }
            }
        }

        private void btBasicTemplateOpen_Click(object sender, EventArgs e)
        {
            using (OpenFileDialog dlg = new OpenFileDialog())
            {
                if (this.project.Template.FilePath.Length > 0)
                {
                    try
                    {
                        dlg.InitialDirectory = System.IO.Path.GetDirectoryName(this.project.Template.FilePath);
                    }
                    catch
                    {
                        dlg.InitialDirectory = System.IO.Path.GetPathRoot(this.project.Template.FilePath);
                    }

                    dlg.FileName = System.IO.Path.GetFileName(this.project.Template.FilePath);
                }
                else
                {
                    dlg.InitialDirectory = "";
                    dlg.FileName = "";
                }

                dlg.Filter = Properties.Resources.FILE_FILTER_PTM;
                dlg.FilterIndex = 1;

                if (dlg.ShowDialog() == DialogResult.OK)
                {
                    this.project.Template.FilePath = dlg.FileName;

                    try
                    {
                        this.project.Template.Load();
                        this.tbBasicTemplateDesc.Text = this.project.Template.Desc;
                    }
                    catch (System.Exception excep)
                    {
                        Mix.Tool.MessageDialog.ShowOK(
                            Properties.Resources.PROJECT_TEMPLATE_OPEN_ERROR + excep.Message,
                            Properties.Resources.TITLE,
                            MessageDialogIcon.Error);
                    }
                }
            }
        }

        private void btBasicTemplateEdit_Click(object sender, EventArgs e)
        {
            using (ProjectTemplateForm form = new ProjectTemplateForm(this.tbBasicRootDir.Text, this.project.Template))
            {
                if (form.ShowDialog() == DialogResult.OK)
                {
                    try
                    {
                        this.project.Template.Save();
                        this.tbBasicTemplateDesc.Text = this.project.Template.Desc;
                    }
                    catch (System.Exception excep)
                    {
                        Mix.Tool.MessageDialog.ShowOK(
                            Properties.Resources.PROJECT_TEMPLATE_SAVE_ERROR + excep.Message,
                            Properties.Resources.TITLE,
                            MessageDialogIcon.Error);
                    }
                }
            }
        }

        private void btInputFile_Click(object sender, EventArgs e)
        {
            using (OpenFileDialog dlg = new OpenFileDialog())
            {
                dlg.Filter = Properties.Resources.FILE_FILTER_INPUTMODEL;
                dlg.FilterIndex = 1;

                if (this.tbInputFile.Text.Length > 0)
                {
                    try
                    {
                        dlg.InitialDirectory = System.IO.Path.GetDirectoryName(this.tbInputFile.Text);
                    }
                    catch
                    {
                        dlg.InitialDirectory = System.IO.Path.GetPathRoot(this.tbInputFile.Text);
                    }

                    dlg.FileName = System.IO.Path.GetFileName(this.tbInputFile.Text);
                }
                else
                {
                    dlg.InitialDirectory = "";
                    dlg.FileName = "";
                }

                if (dlg.ShowDialog() == DialogResult.OK)
                {
                    this.tbInputFile.Text = dlg.FileName;
                }
            }
        }

        private void btOutputFile_Click(object sender, EventArgs e)
        {
            using (SaveFileDialog dlg = new SaveFileDialog())
            {
                switch (this.project.Config.Format)
                {
                    case Project.ModelFormats.Actor:
                        dlg.Filter = Properties.Resources.FILE_FILTER_MAM;
                        break;
                    case Project.ModelFormats.Terrain:
                        dlg.Filter = Properties.Resources.FILE_FILTER_MTM;
                        break;
                }

                dlg.FilterIndex = 1;

                if (this.tbOutputFile.Text.Length > 0)
                {
                    try
                    {
                        dlg.InitialDirectory = System.IO.Path.GetDirectoryName(this.tbOutputFile.Text);
                    }
                    catch
                    {
                        dlg.InitialDirectory = System.IO.Path.GetPathRoot(this.tbOutputFile.Text);
                    }

                    dlg.FileName = System.IO.Path.GetFileName(this.tbOutputFile.Text);
                }
                else
                {
                    dlg.InitialDirectory = "";
                    dlg.FileName = "";
                }

                if (dlg.ShowDialog() == DialogResult.OK)
                {
                    this.tbOutputFile.Text = dlg.FileName;
                }
            }
        }

        private void btOutputFileClear_Click(object sender, EventArgs e)
        {
            this.tbOutputFile.Text = "";
        }

        private void tbProjectName_TextChanged(object sender, EventArgs e)
        {
            this.UpdateControlsEnabled();
        }

        private void btProjectDirPath_Click(object sender, EventArgs e)
        {
            using (Mix.Tool.DirectoryBrowseDialog dlg = new DirectoryBrowseDialog())
            {
                dlg.Root = DirectoryBrowseDialog.RootType.Specify;
                dlg.RootDirectoryName = "";
                dlg.DirectoryName = this.tbProjectDirPath.Text;

                if (dlg.ShowDialog() == DialogResult.OK)
                {
                    this.tbProjectDirPath.Text = dlg.DirectoryName;
                }
            }
        }

        private void btOK_Click(object sender, EventArgs e)
        {
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // 基本
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            //ルートディレクトリパス
            this.project.RootDirectoryPath = this.tbBasicRootDir.Text;

            //テンプレートファイルパスは設定済み
            ;

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // モデル
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            //入力ファイルパス
            this.project.Config.InputFilePath = this.tbInputFile.Text;

            //出力ファイル
            this.project.Config.OutputFilePath = this.tbOutputFile.Text;

            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // プロジェクト
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            //名前
            this.project.Name = this.tbProjectName.Text;

            //ディレクトリパス
            this.project.DirectoryPath = this.tbProjectDirPath.Text;
        }

        private void ProjectPropertyForm_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Escape)
            {
                this.DialogResult = DialogResult.Cancel;
                this.Close();
            }
        }

        private void UpdateControlsEnabled()
        {
            if ((this.tbBasicRootDir.Text.Length > 0) &&
                (this.project.Template.FilePath.Length > 0) &&
                (this.tbInputFile.Text.Length > 0) &&
                (this.tbProjectDirPath.Text.Length > 0) &&
                (this.tbProjectName.Text.Length > 0))
            {
                this.btOK.Enabled = true;
            }
            else
            {
                this.btOK.Enabled = false;
            }
        }
    }
}
