﻿using System;
using System.Windows.Forms;
using System.ComponentModel;
using System.Drawing;

namespace Mix.Tool.ModelEditor
{
    [ToolboxItem(true)]
    public class PhysicsMaterialComboBox : ComboBox
    {
        private int imageMargin = 1;
        private int imageMargin2 = 2;

        /// <summary>
        /// イメージを描画する際のマージンの取得、または設定
        /// </summary>
        [Browsable(true)]
        [Category("表示")]
        [Description("イメージを描画する際のマージンを指定します。")]
        public int ImageMargin
        {
            get { return this.imageMargin; }
            set
            {
                this.imageMargin = (value > 0) ? value : 1;
                this.imageMargin2 = this.imageMargin * 2;
            }
        }

        /// <summary>
        /// 選択されている物理マテリアルの取得、または設定
        /// </summary>
        [Browsable(false)]
        public Mix.Tool.Dynamics.PhysicsMaterial Selected
        {
            get
            {
                object item = this.SelectedItem;

                if (item == null)
                {
                    return null;
                }

                return item as Mix.Tool.Dynamics.PhysicsMaterial;
            }

            set
            {
                object nextSelectedItem = (this.Items.Count > 0) ? this.Items[0] : null;

                if (value != null)
                {
                    foreach (Mix.Tool.Dynamics.PhysicsMaterial item in this.Items)
                    {
                        if (item.Name.Equals(value.Name) == true)
                        {
                            nextSelectedItem = item;
                            break;
                        }
                    }
                }

                if (nextSelectedItem != null)
                {
                    this.SelectedItem = nextSelectedItem;
                }
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PhysicsMaterialComboBox()
        {
            base.DrawMode = DrawMode.OwnerDrawFixed;
        }

        protected override void OnDrawItem(DrawItemEventArgs e)
        {
            base.OnDrawItem(e);

            if ((e.Index < 0) ||
                (e.Index >= this.Items.Count))
            {
                return;
            }

            Mix.Tool.Dynamics.PhysicsMaterial pm = this.Items[e.Index] as Mix.Tool.Dynamics.PhysicsMaterial;
            if (pm == null)
            {
                return;
            }

            int imgWidth = this.ItemHeight - imageMargin2;
            Image img = (pm.Image != null) ? pm.Image : Properties.Resources.NoImage_16;
            Rectangle imgSrcRect = new Rectangle(0, 0, img.Width, img.Height);
            Rectangle imgDstRect = new Rectangle(e.Bounds.X + imageMargin, e.Bounds.Y + imageMargin, imgWidth, imgWidth);

            //背景の描画
            e.DrawBackground();

            //イメージの描画
            e.Graphics.DrawImage(img, imgDstRect, imgSrcRect, GraphicsUnit.Pixel);

            //テキストの描画
            if (pm.Name.Length > 0)
            {
                Rectangle txtRect = new Rectangle(imgDstRect.Right + imageMargin, e.Bounds.Y, e.Bounds.Width - imgDstRect.Right - imageMargin2, e.Bounds.Height);
                StringFormat txtFormat = new StringFormat();
                Brush txtBrush = null;

                txtFormat.Alignment = StringAlignment.Near;
                txtFormat.LineAlignment = StringAlignment.Center;

                if ((e.State & DrawItemState.Selected) == DrawItemState.Selected)
                {
                    txtBrush = SystemBrushes.HighlightText;
                }
                else
                {
                    txtBrush = SystemBrushes.WindowText;
                }

                e.Graphics.DrawString(pm.Name, this.Font, txtBrush, txtRect, txtFormat);
            }

            //フォーカス矩形の描画
            e.DrawFocusRectangle();
        }
    }
}
