﻿using System;
using System.Windows.Forms;
using System.Drawing;
using System.ComponentModel;

namespace Mix.Tool.ModelEditor
{
    /// <summary>
    /// 物理マテリアルリストボックス
    /// </summary>
    [ToolboxItem(true)]
    public class PhysicsMaterialListBox : ListBox
    {
        private int imageMargin = 2;
        private int imageMargin2 = 4;

        /// <summary>
        /// イメージを描画する際のマージンの取得、または設定
        /// </summary>
        [Browsable(true)]
        [Category("表示")]
        [Description("イメージを描画する際のマージンを指定します。")]
        public int ImageMargin
        {
            get { return this.imageMargin; }
            set
            {
                this.imageMargin = (value > 0) ? value : 1;
                this.imageMargin2 = this.imageMargin * 2;
            }
        }

        /// <summary>
        /// 選択されている物理マテリアルの取得、または設定
        /// </summary>
        [Browsable(false)]
        public Mix.Tool.Dynamics.PhysicsMaterial Selected
        {
            get
            {
                object item = this.SelectedItem;

                if (item == null)
                {
                    return null;
                }

                return item as Mix.Tool.Dynamics.PhysicsMaterial;
            }

            set
            {
                object item = null;

                if (value != null)
                {
                    foreach (Mix.Tool.Dynamics.PhysicsMaterial physicsMaterial in this.Items)
                    {
                        if (physicsMaterial.Name.Equals(value.Name) == true)
                        {
                            item = physicsMaterial;
                            break;
                        }
                    }
                }

                this.SelectedItem = item;
            }
        }

        /// <summary>
        /// 描画モードの取得、または設定
        /// </summary>
        [Browsable(false)]
        [EditorBrowsable(EditorBrowsableState.Never)]
        public new DrawMode DrawMode
        {
            get { return base.DrawMode; }
            set { base.DrawMode = DrawMode.OwnerDrawFixed; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PhysicsMaterialListBox()
            : base()
        {
            this.DrawMode = DrawMode.OwnerDrawFixed;

            this.DoubleBuffered = true;
            this.SetStyle(ControlStyles.OptimizedDoubleBuffer, true);
            this.UpdateStyles();
        }

        protected override void OnDrawItem(DrawItemEventArgs e)
        {
            base.OnDrawItem(e);

            if ((e.Index < 0) ||
                (this.Items.Count <= e.Index))
            {
                return;
            }

            Rectangle itemRect = this.GetItemRectangle(e.Index);
            Mix.Tool.Dynamics.PhysicsMaterial physicsMaterial = this.Items[e.Index] as Mix.Tool.Dynamics.PhysicsMaterial;

            Image img = (physicsMaterial.Image != null) ? physicsMaterial.Image : Mix.Tool.ModelEditor.Properties.Resources.NoImage_32;
            int imgWidth = itemRect.Height - imageMargin2;
            Rectangle imgSrcRect = new Rectangle(0, 0, img.Width, img.Height);
            Rectangle imgDstRect = new Rectangle(imageMargin, itemRect.Y + imageMargin, imgWidth, imgWidth);

            //背景の描画
            e.DrawBackground();

            //イメージの描画
            e.Graphics.DrawImage(img, imgDstRect, imgSrcRect, GraphicsUnit.Pixel);

            //名前の描画
            if (physicsMaterial.Name.Length > 0)
            {
                Brush brush = ((e.State & DrawItemState.Selected) == DrawItemState.Selected) ? SystemBrushes.HighlightText : SystemBrushes.WindowText;
                Rectangle dstRect = new Rectangle();
                StringFormat strFormat = new StringFormat();

                dstRect.X = imgDstRect.Right + imageMargin;
                dstRect.Y = itemRect.Y;
                dstRect.Width = itemRect.Width - imgDstRect.Right - imageMargin;
                dstRect.Height = itemRect.Height;

                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Center;

                e.Graphics.DrawString(physicsMaterial.Name, e.Font, brush, dstRect, strFormat);
            }

            //フォーカス矩形の描画
            e.DrawFocusRectangle();
        }
    }
}
