////////////////////////////////////////////////////////////////////////////////////////////////////
// CN[h
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"

#include "Mix/IO.h"
#include "Mix/HID.h"
#include "Mix/Dynamics.h"
#include "Mix/Graphics.h"

#include "Utility/Common.h"
#include "Utility/ViewCameraHelper.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
// 萔
////////////////////////////////////////////////////////////////////////////////////////////////////

#define SCREEN_WIDTH ( 1024 )
#define SCREEN_HEIGHT ( 768 )

static const Float32 BASE_HEIGHT = 2.0f;
static const Float32 BASE_RADIUS = 0.5f;
static const Float32 BASE_RADIUS2 = BASE_RADIUS * 2.0f;

static const Float32 HEAD_HEIGHT = 1.2f;
static const Float32 HEAD_HALF_HEIGHT = HEAD_HEIGHT * 0.5f;

static const Float32 JOINT_MARGIN = 0.03f;
static const Float32 JOINT_SWING_LIMIT1 = MIX_TO_RAD( 50.0f );
static const Float32 JOINT_SWING_LIMIT2 = MIX_TO_RAD( 40.0f );
static const Float32 JOINT_TWIST_LIMIT = MIX_TO_RAD( 30.0f );

static const Mix::Vector3 CAM_INIT_AT( 0.0f, 1.0f, 0.0f );
static const Mix::Vector2 CAM_INIT_ANGLE( MIX_TO_RAD( 20.0f ), MIX_TO_RAD( -30.0f ) );
static const Float32 CAM_INIT_DIST = 5.0f;

////////////////////////////////////////////////////////////////////////////////////////////////////
// ֐
////////////////////////////////////////////////////////////////////////////////////////////////////

int APIENTRY _tWinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance, LPTSTR lpCmdLine, int nCmdShow )
{
	Mix::ENGINE_CONFIG engineConfig;

	engineConfig.flags = Mix::EC_IO_NARROW | Mix::EC_KEYBOARD | Mix::EC_MOUSE | Mix::EC_GRAPHICS | Mix::EC_DYNAMICS;
	engineConfig.pRootDirectoryPath = Utility::GetRootDirectoryPath();
	engineConfig.pPluginDirectoryPath = Utility::GetPluginsDirectoryPath();
	engineConfig.pUserDirectoryPath = Utility::GetUserDirectoryPath( L"Dynamics\\Joint\\Ball" );
	engineConfig.pCaption = L"Dynamics - BallJoint";
	engineConfig.targetSize = Mix::Point( SCREEN_WIDTH, SCREEN_HEIGHT );
	engineConfig.shaderModel = Mix::Graphics::SHADER_MODEL_3;

	if( Mix::Initialize( engineConfig ) == True )
	{
		Mix::IEngine* pEngine = Mix::GetEnginePtr();

		Mix::IO::IManager* pIOMgr = Mix::IO::GetManagerPtr();

		Mix::HID::IManager* pInputMgr = Mix::HID::GetManagerPtr();
		Mix::HID::IKeyboard* pKeyboard = NULL;
		Mix::HID::IMouse* pMouse = NULL;

		Mix::Dynamics::IManager* pDynamicsMgr = Mix::Dynamics::GetManagerPtr();
		Mix::Dynamics::IWorld* pDynamicsWorld = NULL;
		Mix::Dynamics::IObjectDragger* pDynamicsObjDrg = NULL;

		Mix::Graphics::IManager* pGraphicsMgr = Mix::Graphics::GetManagerPtr();
		Mix::Graphics::IDevice* pGraphicsDev = NULL;
		Mix::Graphics::Utility::IPerspectiveRenderer* pPersRenderer = NULL;
		Mix::Graphics::Utility::ICanvasRenderer* pCanvasRenderer = NULL;
		Mix::Graphics::Utility::IFont* pFont = NULL;

		Utility::ViewCameraHelper viewCamHelper;
		Mix::Matrix4x4 viewMat;
		Mix::Matrix4x4 projMat;

		////////////////////////////////////////////////////////////////////////////////////////////////////

		pIOMgr->MountDirectory( L"Data" );

		////////////////////////////////////////////////////////////////////////////////////////////////////

		pInputMgr->GetKeyboard( &pKeyboard );
		pInputMgr->GetMouse( &pMouse );

		////////////////////////////////////////////////////////////////////////////////////////////////////

		if( pDynamicsMgr->CreateWorld( Mix::Dynamics::WORLD_CONFIG(), &pDynamicsWorld, MIX_DEBUGNAME ) == True )
		{
			Mix::Dynamics::IStaticPlane* pStaticPlane = NULL;
			Mix::Dynamics::ICapsuleShape* pCapusleShape = NULL;
			Mix::Dynamics::ICompoundShape* pCompoundShape = NULL;
			Mix::Dynamics::IRigidBody* pBaseRigidBody = NULL;
			Mix::Dynamics::IRigidBody* pHeadRigidBody = NULL;
			Mix::Dynamics::IBallJoint* pJoint = NULL;

			/*
				StaticPlane
			*/

			if( pDynamicsMgr->CreateStaticPlane( Mix::Vector3( 0.0f, 1.0f, 0.0f ), 0.0f, &pStaticPlane, MIX_DEBUGNAME ) == True )
			{
				pDynamicsWorld->AddObject( pStaticPlane );
				MIX_RELEASE( pStaticPlane );
			}

			/*
				BaseRigidBody
			*/

			if( pDynamicsMgr->CreateCapsuleShape( Mix::Dynamics::AXIS_X, BASE_HEIGHT, BASE_RADIUS, &pCapusleShape, MIX_DEBUGNAME ) == True )
			{
				if( pDynamicsMgr->CreateRigidBody( 0.0f, pCapusleShape, &pBaseRigidBody, MIX_DEBUGNAME ) == True )
				{
					pBaseRigidBody->SetWorldPosition( Mix::Vector3( 0.0f, BASE_RADIUS, 0.0f ) );
					pDynamicsWorld->AddObject( pBaseRigidBody );
				}

				MIX_RELEASE( pCapusleShape );
			}

			/*
				HeadRigidBody
			*/

			if( pDynamicsMgr->CreateCompoundShape( &pCompoundShape, MIX_DEBUGNAME ) == True )
			{
				if( pDynamicsMgr->CreateCapsuleShape( Mix::Dynamics::AXIS_Y, HEAD_HEIGHT, 0.5f, &pCapusleShape, MIX_DEBUGNAME ) == True )
				{
					pCompoundShape->AddChild( pCapusleShape, Mix::Quaternion::Identity(), Mix::Vector3::Zero() );
					MIX_RELEASE( pCapusleShape );

					if( pDynamicsMgr->CreateCapsuleShape( Mix::Dynamics::AXIS_Z, 0.5f, 0.05f, &pCapusleShape, MIX_DEBUGNAME ) == True )
					{
						pCompoundShape->AddChild( pCapusleShape, Mix::Quaternion::Identity(), Mix::Vector3( 0.0f, 0.0f, 0.7f ) );
						MIX_RELEASE( pCapusleShape );

						if( pDynamicsMgr->CreateRigidBody( 1.0f, pCompoundShape, &pHeadRigidBody, MIX_DEBUGNAME ) == True )
						{
							pHeadRigidBody->SetWorldPosition( Mix::Vector3( 0.0f, ( BASE_RADIUS2 + HEAD_HALF_HEIGHT + JOINT_MARGIN ), 0.0f ) );
							pDynamicsWorld->AddObject( pHeadRigidBody );
						}
					}
				}

				MIX_RELEASE( pCompoundShape );
			}

			/*
				Ball Joint
			*/

			if( pDynamicsMgr->CreateBallJoint(	pHeadRigidBody,
												pBaseRigidBody,
												Mix::Vector3( 0.0f, -( 0.6f + JOINT_MARGIN ), 0.0f ),
												Mix::Vector3( 0.0f, +( 0.5f ), 0.0f ),
												Mix::Vector3( 0.0f, 1.0f, 0.0f ),
												Mix::Vector3( 0.0f, 1.0f, 0.0f ),
												&pJoint,
												MIX_DEBUGNAME ) == True )
			{
				pJoint->SetSwingLimit1( JOINT_SWING_LIMIT1 );
				pJoint->SetSwingLimit2( JOINT_SWING_LIMIT2 );
				pJoint->SetTwistLimit( JOINT_TWIST_LIMIT );

				pJoint->Debug_SetDrawLimitScaling( 0.5f );
				pDynamicsWorld->AddJoint( pJoint );

				MIX_RELEASE( pJoint );
			}

			/*
				ObjectDragger
			*/

			pDynamicsMgr->CreateObjectDragger( pDynamicsWorld, &pDynamicsObjDrg, MIX_DEBUGNAME );

			MIX_RELEASE( pBaseRigidBody );
			MIX_RELEASE( pHeadRigidBody );
		}

		////////////////////////////////////////////////////////////////////////////////////////////////////

		pGraphicsMgr->GetDevice( &pGraphicsDev );

		pGraphicsMgr->CreatePerspectiveRenderer( &pPersRenderer, MIX_DEBUGNAME );

		if( pGraphicsMgr->CreateCanvasRenderer( &pCanvasRenderer, MIX_DEBUGNAME ) == True )
		{
			if( pGraphicsMgr->CreateFontFromFile( L"Data\\Font\\mspg_16_400.fnt", &pFont ) == True )
			{
				pCanvasRenderer->SetFont( pFont );
			}
		}

		////////////////////////////////////////////////////////////////////////////////////////////////////

		Float32 camAspect = static_cast<Float32>( SCREEN_WIDTH ) / static_cast<Float32>( SCREEN_HEIGHT );

		viewCamHelper.Initialize( pMouse, pKeyboard, CAM_INIT_AT, CAM_INIT_ANGLE, CAM_INIT_DIST );

		Mix::Matrix4x4::PerspectiveFovLH( MIX_TO_RAD( 60.0f ), camAspect, 0.1f, 1000.0f, projMat );

		////////////////////////////////////////////////////////////////////////////////////////////////////

		pEngine->ResetFPS();

		while( pEngine->Update() == True )
		{
			////////////////////////////////////////////////////////////////////////////////////////////////////

			pDynamicsWorld->Update( pEngine->GetDT() );

			////////////////////////////////////////////////////////////////////////////////////////////////////

			viewCamHelper.Update();

			Mix::Matrix4x4::LookAtLH( viewCamHelper.GetRotation(), viewCamHelper.GetAt(), viewCamHelper.GetDistance(), viewMat );

			////////////////////////////////////////////////////////////////////////////////////////////////////

			Mix::Vector3 eyePos = viewCamHelper.GetAt() + Mix::Matrix4x4( viewCamHelper.GetRotation() ) * Mix::Vector3( 0.0f, 0.0f, viewCamHelper.GetDistance() );
			Mix::Vector2 mousePos = pMouse->GetPos();
			UInt32 mlbState = pMouse->GetButtonState( 0 );

			if( MIX_TESTBIT( mlbState, Mix::HID::PRESSED ) == Mix::HID::PRESSED )
			{
				Mix::Vector2 screenSize = pGraphicsDev->GetScreenSize();
				Mix::Vector3 rayFrom = Mix::Unproject( viewMat, projMat, Mix::Vector3( mousePos.x, mousePos.y, 0.0f ), screenSize );
				Mix::Vector3 rayTo = Mix::Unproject( viewMat, projMat, Mix::Vector3( mousePos.x, mousePos.y, 1.0f ), screenSize );

				pDynamicsObjDrg->CatchStart( eyePos, rayFrom, rayTo );
			}
			else if( MIX_TESTBIT( mlbState, Mix::HID::DOWN ) == Mix::HID::DOWN )
			{
				if( pDynamicsObjDrg->IsCatch() == True )
				{
					Mix::Vector2 screenSize = pGraphicsDev->GetScreenSize();
					Mix::Vector3 rayTo = Mix::Unproject( viewMat, projMat, Mix::Vector3( mousePos.x, mousePos.y, 1.0f ), screenSize );

					pDynamicsObjDrg->CatchContinue( eyePos, rayTo );
				}
			}
			else if( MIX_TESTBIT( mlbState, Mix::HID::RELEASED ) == Mix::HID::RELEASED )
			{
				if( pDynamicsObjDrg->IsCatch() == True )
				{
					pDynamicsObjDrg->Free();
				}
			}

			////////////////////////////////////////////////////////////////////////////////////////////////////

			pDynamicsWorld->Debug_Draw( pPersRenderer );

			pPersRenderer->Update();

			////////////////////////////////////////////////////////////////////////////////////////////////////

			Mix::String tempStr;
			Mix::Vector2 txtSize;
			Mix::Vector2 txtPos;

			/*
				Mouse Cursor
			*/

			Utility::DrawMouseCursor( pCanvasRenderer, pMouse->GetPos() );

			/*
				FPS
			*/

			tempStr.Sprintf( L"FPS %f", pEngine->GetFPS() );

			pCanvasRenderer->SetColor( Mix::Vector4( 1.0f, 1.0f, 1.0f, 1.0f ) );
			pCanvasRenderer->AddString( 16.0f, 16.0f, tempStr.GetConstPtr() );

			/*
				Desription
			*/

			tempStr.Sprintf( L"%s",	L"Controls :\n"
									L"  Rotate camera  : Mouse middle button\n"
									L"  Move camera    : Mouse middle button + Shift key\n"
									L"  Distant camera : Mouse middle button + Ctrl key or Mouse wheel\n"
									L"  Drag object    : Mouse left button" );

			txtSize = Mix::Graphics::Utility::MeasureString( pFont, tempStr.GetConstPtr() );
			txtPos.x = 16.0f;
			txtPos.y = static_cast<Float32>( SCREEN_HEIGHT ) - txtSize.y - 16.0f;

			pCanvasRenderer->SetColor( Mix::Vector4( 1.0f, 1.0f, 1.0f, 1.0f ) );
			pCanvasRenderer->AddString( txtPos, tempStr.GetConstPtr() );

			/*
				Update
			*/

			pCanvasRenderer->Update();

			////////////////////////////////////////////////////////////////////////////////////////////////////

			pGraphicsDev->SetViewBounds( 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT );
			pGraphicsDev->Clear( True, False, Mix::Vector4( 0.1f, 0.1f, 0.1f, 1.0f ) );

			if( pGraphicsDev->Begin() == True )
			{
				pPersRenderer->Draw( viewMat * projMat );
				pCanvasRenderer->Draw();

				pGraphicsDev->End();
			}

			pGraphicsDev->Present();
		}

		////////////////////////////////////////////////////////////////////////////////////////////////////

		viewCamHelper.Dispose();

		MIX_RELEASE( pFont );
		MIX_RELEASE( pCanvasRenderer );
		MIX_RELEASE( pPersRenderer );
		MIX_RELEASE( pGraphicsDev );
		MIX_RELEASE( pDynamicsObjDrg );
		MIX_RELEASE( pDynamicsWorld );
		MIX_RELEASE( pMouse );
		MIX_RELEASE( pKeyboard );

		Mix::Finalize();
	}

	return 0;
}
