#include "hud.h"

#include "Mix/Scene/IRenderer.h"
#include "Mix/Scene/ICamera.h"

const Int32 HUD::SMOOTH_RANGE_TABLE[Mix::Scene::BLUR::TYPE_MAX][2] =
{
	{ 0, 0, },
	{ 1, 16, },
	{ 1, 16, },
	{ 1, 64, },
	{ 4, 64, },
};

HUD::HUD( void ) :
m_pSceneRenderer( NULL ),
m_pCamera( NULL )
{
}

HUD::~HUD( void )
{
}

void HUD::Open( Utility::UI::Manager* pManager, Mix::Scene::IRenderer* pSceneRenderer, Mix::Scene::ICamera* pCamera )
{
	MIX_ASSERT( pManager != NULL );
	MIX_ASSERT( pSceneRenderer != NULL );
	MIX_ASSERT( pCamera != NULL );

	const Mix::Point& screenSize = pCamera->GetConfig().targetSize;

	MIX_ADD_REF( pSceneRenderer );
	m_pSceneRenderer = pSceneRenderer;

	MIX_ADD_REF( pCamera );
	m_pCamera = pCamera;

#if 1
	SetHUDEnabled( True );
#else
	SetCaption( L"Configuration" );
#endif

	SetPosition( Mix::Point( screenSize.x - HUD::DLG_WIDTH, screenSize.y - HUD::DLG_HEIGHT ) );
	SetSize( Mix::Point( HUD::DLG_WIDTH, HUD::DLG_HEIGHT ) );

	Dialog::Open( pManager );
}

void HUD::OnOpen( void )
{
	Mix::Scene::CAMERA_CONFIG camConfig = m_pCamera->GetConfig();
	Mix::Scene::ICamera::AMBIENT_OCCLUSION_SETTINGS aoSettings = m_pCamera->GetAmbientOcclusionSettings();

	Int32 itemY = 16;

#ifdef _DEBUG
	UInt32 debDrawFlags = m_pCamera->Debug_GetDrawFlags();
#endif //_DEBUG

	//////////////////////////////////////////////////////////////////////////////////////////
	// ݒ
	//////////////////////////////////////////////////////////////////////////////////////////

	m_SmoothTable[Mix::Scene::BLUR::T_NONE] = Mix::Scene::BLUR::NONE();
	m_SmoothTable[Mix::Scene::BLUR::T_KS_3x3] = Mix::Scene::BLUR::KS_3x3( 1 );
	m_SmoothTable[Mix::Scene::BLUR::T_KS_5x5] = Mix::Scene::BLUR::KS_5x5( 1 );
	m_SmoothTable[Mix::Scene::BLUR::T_GAUSSIAN] = Mix::Scene::BLUR::GAUSSIAN( 4.0f );
	m_SmoothTable[Mix::Scene::BLUR::T_GAUSSIAN_EX] = Mix::Scene::BLUR::GAUSSIAN_EX( 16 );

	aoSettings.radius = 0.03f;
	aoSettings.threshold = 0.0001f;
	aoSettings.dist = 0.1f;
	aoSettings.intensity = 2.5f;
	aoSettings.blur = m_SmoothTable[Mix::Scene::BLUR::T_KS_5x5];
	m_pCamera->SetAmbientOcclusionEnabled( True );
	m_pCamera->SetAmbientOcclusionSettings( aoSettings );

	//////////////////////////////////////////////////////////////////////////////////////////
	// General
	//////////////////////////////////////////////////////////////////////////////////////////

	m_EnabledCB.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_EnabledCB.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_EnabledCB.SetText( L"Enabled" );
	m_EnabledCB.SetState( m_pCamera->IsAmbientOcclusionEnabled() );
	m_EnabledCB.Initialize( this, HUD::CTRL_ENABLED_CB );
	itemY += HUD::ITEM_Y_STEP;

	//////////////////////////////////////////////////////////////////////////////////////////
	// Smooth
	//////////////////////////////////////////////////////////////////////////////////////////

	m_SmoothL.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_SmoothL.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_SmoothL.SetText( L"Smooth :" );
	m_SmoothL.Initialize( this, HUD::CTRL_SMOOTH_L );
	itemY += HUD::ITEM_Y_STEP;

	m_SmplCB.Initialize( this, HUD::CTRL_SMPL_CB );
	m_SmplCB.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_SmplCB.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_SmplCB.AddItem( Mix::Scene::RSMPL_DEFAULT, L"Default( Copy )" );
	m_SmplCB.AddItem( Mix::Scene::RSMPL_DOWN2X, L"Down 2x" );
	m_SmplCB.AddItem( Mix::Scene::RSMPL_DOWN4X, L"Down 4x" );
	m_SmplCB.SetSelectedItem( camConfig.aoSmplRes, False );
	itemY += HUD::ITEM_Y_STEP;

	m_SmoothCB.Initialize( this, HUD::CTRL_SMOOTH_CB );
	m_SmoothCB.SetPosition( Mix::Point( HUD::ITEM_X1, itemY ) );
	m_SmoothCB.SetSize( Mix::Point( HUD::ITEM_WIDTH1, HUD::ITEM_HEIGHT1 ) );
	m_SmoothCB.AddItem( Mix::Scene::BLUR::T_NONE, L"NONE" );
	m_SmoothCB.AddItem( Mix::Scene::BLUR::T_KS_3x3, L"KERNEL_3x3" );
	m_SmoothCB.AddItem( Mix::Scene::BLUR::T_KS_5x5, L"KERNEL_5x5" );
	m_SmoothCB.AddItem( Mix::Scene::BLUR::T_GAUSSIAN, L"GAUSSIAN" );
	m_SmoothCB.AddItem( Mix::Scene::BLUR::T_GAUSSIAN_EX, L"GAUSSIAN_EX" );
	m_SmoothCB.SetSelectedItem( aoSettings.blur.GetType(), False );

	m_SmoothSI.SetPosition( Mix::Point( HUD::ITEM_X2, itemY ) );
	m_SmoothSI.SetSize( Mix::Point( HUD::ITEM_WIDTH2, HUD::ITEM_HEIGHT2 ) );
	m_SmoothSI.SetText( L"" );
	m_SmoothSI.SetRange( HUD::SMOOTH_RANGE_TABLE[aoSettings.blur.GetType()][0], HUD::SMOOTH_RANGE_TABLE[aoSettings.blur.GetType()][1] );
	m_SmoothSI.SetValue( HUD::GetSmoothValue( aoSettings.blur ) );
	m_SmoothSI.SetEanbled( aoSettings.blur.GetType() != Mix::Scene::BLUR::T_NONE );
	m_SmoothSI.SetStep( 1 );
	m_SmoothSI.Initialize( this, HUD::CTRL_SMOOTH_SI );
	itemY += HUD::ITEM_Y_STEP;
	
	//////////////////////////////////////////////////////////////////////////////////////////
	// Occlusion
	//////////////////////////////////////////////////////////////////////////////////////////

	m_OcclL.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_OcclL.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_OcclL.SetText( L"Occlusion :" );
	m_OcclL.Initialize( this, HUD::CTRL_OCCL_L );
	itemY += HUD::ITEM_Y_STEP;

	m_RadiusF.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_RadiusF.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_RadiusF.SetText( L"Radius" );
	m_RadiusF.SetRange( 0.0f, 1.0f );
	m_RadiusF.SetStep( 0.001f );
	m_RadiusF.SetValue( aoSettings.radius );
	m_RadiusF.Initialize( this, HUD::CTRL_RADIUS_SF );
	itemY += HUD::ITEM_Y_STEP;

	m_ThresholdF.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_ThresholdF.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_ThresholdF.SetText( L"Threshold" );
	m_ThresholdF.SetRange( 0.0f, 1.0f );
	m_ThresholdF.SetStep( 0.0001f );
	m_ThresholdF.SetValue( aoSettings.threshold );
	m_ThresholdF.Initialize( this, HUD::CTRL_THRESHOLD_SF );
	itemY += HUD::ITEM_Y_STEP;

	m_DistanceF.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_DistanceF.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_DistanceF.SetText( L"Distance" );
	m_DistanceF.SetRange( 0.0f, 1.0f );
	m_DistanceF.SetStep( 0.001f );
	m_DistanceF.SetValue( aoSettings.dist );
	m_DistanceF.Initialize( this, HUD::CTRL_DISTANCE_SF );
	itemY += HUD::ITEM_Y_STEP;

	m_IntensityF.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_IntensityF.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_IntensityF.SetText( L"Intensity" );
	m_IntensityF.SetRange( 1.0f, 10.0f );
	m_IntensityF.SetStep( 0.01f );
	m_IntensityF.SetValue( aoSettings.intensity );
	m_IntensityF.Initialize( this, HUD::CTRL_INTENSITY_SF );
	itemY += HUD::ITEM_Y_STEP;
}

void HUD::OnDispose( void )
{
	MIX_RELEASE( m_pCamera );
	MIX_RELEASE( m_pSceneRenderer );
}

void HUD::OnCheckBox_Click( Utility::UI::CheckBox* pCheckBox )
{
#ifdef _DEBUG
	UInt32 debDrawFlags = m_pCamera->Debug_GetDrawFlags();
#endif //_DEBUG

	switch( pCheckBox->GetID() )
	{
	case HUD::CTRL_ENABLED_CB:
		m_pCamera->SetAmbientOcclusionEnabled( pCheckBox->GetState() );
		break;
	}
}

void HUD::OnSlider_ValueChanged( Utility::UI::SliderI* pSlider )
{
	Mix::Scene::ICamera::AMBIENT_OCCLUSION_SETTINGS aoSettings = m_pCamera->GetAmbientOcclusionSettings();

	switch( pSlider->GetID() )
	{
	case HUD::CTRL_SMOOTH_SI:
		HUD::SetSmoothValue( aoSettings.blur, pSlider->GetValue() );
		m_SmoothTable[aoSettings.blur.GetType()] = aoSettings.blur;
		m_pCamera->SetAmbientOcclusionSettings( aoSettings );
		break;
	}
}

void HUD::OnSlider_ValueChanged( Utility::UI::SliderF* pSlider )
{
	Mix::Scene::ICamera::AMBIENT_OCCLUSION_SETTINGS aoSettings = m_pCamera->GetAmbientOcclusionSettings();

	switch( pSlider->GetID() )
	{
	case HUD::CTRL_RADIUS_SF:
		aoSettings.radius = pSlider->GetValue();
		m_pCamera->SetAmbientOcclusionSettings( aoSettings );
		break;

	case HUD::CTRL_INTENSITY_SF:
		aoSettings.intensity = pSlider->GetValue();
		m_pCamera->SetAmbientOcclusionSettings( aoSettings );
		break;

	case HUD::CTRL_THRESHOLD_SF:
		aoSettings.threshold = pSlider->GetValue();
		m_pCamera->SetAmbientOcclusionSettings( aoSettings );
		break;

	case HUD::CTRL_DISTANCE_SF:
		aoSettings.dist = pSlider->GetValue();
		m_pCamera->SetAmbientOcclusionSettings( aoSettings );
		break;
	}
}

void HUD::OnComboBox_SelectItemChanged( Utility::UI::ComboBox* pComboBox )
{
	Mix::Scene::CAMERA_CONFIG camConfig = m_pCamera->GetConfig();
	Mix::Scene::ICamera::AMBIENT_OCCLUSION_SETTINGS aoSettings = m_pCamera->GetAmbientOcclusionSettings();

	Mix::Scene::RENDERER_SAMPLING_RESOLUTION smplRes;

	switch( pComboBox->GetID() )
	{
	case HUD::CTRL_SMPL_CB:
		smplRes = static_cast<Mix::Scene::RENDERER_SAMPLING_RESOLUTION>( pComboBox->GetSelectedItem() );
		if( camConfig.aoSmplRes != smplRes )
		{
			camConfig.aoSmplRes = smplRes;
			m_pCamera->SetConfig( camConfig );
		}
		break;

	case HUD::CTRL_SMOOTH_CB:
		aoSettings.blur = m_SmoothTable[pComboBox->GetSelectedItem()];
		m_pCamera->SetAmbientOcclusionSettings( aoSettings );
		m_SmoothSI.SetRange( HUD::SMOOTH_RANGE_TABLE[aoSettings.blur.GetType()][0], HUD::SMOOTH_RANGE_TABLE[aoSettings.blur.GetType()][1] );
		m_SmoothSI.SetValue( HUD::GetSmoothValue( aoSettings.blur ) );
		m_SmoothSI.SetEanbled( aoSettings.blur.GetType() != Mix::Scene::BLUR::T_NONE );
		break;
	}
}

Int32 HUD::GetSmoothValue( const Mix::Scene::BLUR& blur )
{
	Int32 ret = 0;

	switch( blur.GetType() )
	{
	case Mix::Scene::BLUR::T_NONE:
		ret = 0;
		break;
	case Mix::Scene::BLUR::T_KS_3x3:
		ret = blur.GetIntValue();
		break;
	case Mix::Scene::BLUR::T_KS_5x5:
		ret = blur.GetIntValue();
		break;
	case Mix::Scene::BLUR::T_GAUSSIAN:
		ret = static_cast<Int32>( blur.GetFloatValue() );
		break;
	case Mix::Scene::BLUR::T_GAUSSIAN_EX:
		ret = blur.GetIntValue();
		break;
	}

	return ret;
}

void HUD::SetSmoothValue( Mix::Scene::BLUR& blur, Int32 value )
{
	switch( blur.GetType() )
	{
	case Mix::Scene::BLUR::T_KS_3x3:
		blur = Mix::Scene::BLUR::KS_3x3( value );
		break;
	case Mix::Scene::BLUR::T_KS_5x5:
		blur = Mix::Scene::BLUR::KS_5x5( value );
		break;
	case Mix::Scene::BLUR::T_GAUSSIAN:
		blur = Mix::Scene::BLUR::GAUSSIAN( static_cast<Float32>( value ) );
		break;
	case Mix::Scene::BLUR::T_GAUSSIAN_EX:
		blur = Mix::Scene::BLUR::GAUSSIAN_EX( value );
		break;
	}
}
