#include "stdafx.h"

#include "Mix/IO.h"
#include "Mix/HID.h"
#include "Mix/Graphics.h"
#include "Mix/Scene.h"

#include "Utility/Common.h"
#include "Utility/UniversalCameraHelper.h"
#include "Utility/UI/Manager.h"

#include "hud.h"

#ifdef _DEBUG
	#include <crtdbg.h>
#endif //_DEBUG

#define SCREEN_WIDTH 1024
#define SCREEN_HEIGHT 768

static const Mix::Vector3 SUN_POS( 0.5f, 1.0f, -0.5f );

int APIENTRY _tWinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance, LPTSTR lpCmdLine, int nCmdShow )
{
#ifdef _DEBUG
	_CrtSetDbgFlag( _CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF );
#endif //_DEBUG

	Mix::ENGINE_CONFIG engineConfig;

	engineConfig.flags = Mix::EC_IO_NARROW | Mix::EC_KEYBOARD | Mix::EC_MOUSE | Mix::EC_GRAPHICS | Mix::EC_SCENE;
	engineConfig.pRootDirectoryPath = Utility::GetRootDirectoryPath();
	engineConfig.pPluginDirectoryPath = Utility::GetPluginsDirectoryPath();
	engineConfig.pUserDirectoryPath = Utility::GetUserDirectoryPath( L"Scene\\PostEffect\\Chromatism" );
	engineConfig.pCaption = L"Scene - PostEffect ( Chromatism )";
	engineConfig.targetSize = Mix::Point( SCREEN_WIDTH, SCREEN_HEIGHT );
	engineConfig.shaderModel = Mix::Graphics::SHADER_MODEL_3;

	if( Mix::Initialize( engineConfig ) == True )
	{
		Mix::IEngine* pEngine = Mix::GetEnginePtr();

		Mix::IO::IManager* pIOMgr = Mix::IO::GetManagerPtr();
		Mix::HID::IManager* pInputMgr = Mix::HID::GetManagerPtr();
		Mix::Graphics::IManager* pGraphicsMgr = Mix::Graphics::GetManagerPtr();
		Mix::Scene::IManager* pSceneMgr = Mix::Scene::GetManagerPtr();

		Mix::HID::IMouse* pMouse = NULL;
		Mix::HID::IKeyboard* pKeyboard = NULL;

		Mix::Graphics::IDevice* pGraphicsDev = NULL;
		Mix::Graphics::Utility::IFont* pFontS = NULL;
		Mix::Graphics::Utility::IFont* pFontL = NULL;
		Mix::Graphics::Utility::ICanvasRenderer* pCanvasRenderer = NULL;

		Mix::Scene::IEffectPackage* pSceneEffectPackage = NULL;
		Mix::Scene::IRenderer* pSceneRenderer = NULL;
		Mix::Scene::IUniversalCamera* pCamera = NULL;

		Mix::Graphics::ITexture* pChrTex = NULL;
		Utility::UniversalCameraHelper camHelper;

		Utility::UI::Manager uiManager;
		HUD hud;

		Mix::StringW tempStr;

		////////////////////////////////////////////////////////////////////////////////////////////////////

		pIOMgr->MountDirectory( L"Data" );

		////////////////////////////////////////////////////////////////////////////////////////////////////

		pInputMgr->GetMouse( &pMouse );
		pInputMgr->GetKeyboard( &pKeyboard );

		////////////////////////////////////////////////////////////////////////////////////////////////////

		pGraphicsMgr->GetDevice( &pGraphicsDev );

		pGraphicsMgr->CreateFontFromFile( L"Data\\Font\\mspg_12_200.fnt", &pFontS );
		pGraphicsMgr->CreateFontFromFile( L"Data\\Font\\mspg_16_400.fnt", &pFontL );

		if( pGraphicsMgr->CreateCanvasRenderer( &pCanvasRenderer, L"Sample" ) == True )
		{
			pCanvasRenderer->SetFont( pFontL );
		}

		pGraphicsDev->CreateTextureFromFile( L"Data\\Texture\\Chromatism.dds", &pChrTex );

		////////////////////////////////////////////////////////////////////////////////////////////////////

		if( pSceneMgr->CreateEffectPackage( L"Data\\", L"effects", &pSceneEffectPackage ) == True )
		{
			Mix::Scene::RENDERER_CONFIG config;

			config.caps = Mix::Scene::RCAP_SHADOW_MAPPING | Mix::Scene::RCAP_CHROMATISM;

			if( pSceneMgr->CreateRenderer( pSceneEffectPackage, config, &pSceneRenderer, L"Sample" ) == True )
			{
				Mix::Scene::CAMERA_CONFIG camConfig;
				Mix::Scene::ITerrainModel* pRoomModel = NULL;
				Mix::Scene::IDirectionalLight* pDirLight = NULL;

				camConfig.caps = Mix::Scene::RCAP_SHADOW_MAPPING | Mix::Scene::RCAP_CHROMATISM;
				camConfig.targetSize.Set( SCREEN_WIDTH, SCREEN_HEIGHT );
				camConfig.colorTexFormat = Mix::Graphics::FMT_R8G8B8A8;
				camConfig.smDepthTexSize = 2048;
				camConfig.smSmplRes = Mix::Scene::RSMPL_DEFAULT;

				if( pSceneMgr->CreateUniversalCamera( camConfig, &pCamera ) == True )
				{
					Mix::Scene::ICamera::VISIBILITY_SETTINGS visiSettings = pCamera->GetVisiblitySettings();
					Mix::Scene::ICamera::SHADOW_MAPPING_SETTINGS smSettings = pCamera->GetShadowMappingSettings();
					Mix::Scene::ICamera::CHROMATISM_SETTINGS caSetting = pCamera->GetChromatismSettings();

					pCamera->SetBackgroundColor( Mix::Vector4( 0.2f, 0.7f, 0.95f, 1.0f ) );
					pCamera->SetProjection( MIX_TO_RAD( 60.0f ), ( Float32 )SCREEN_WIDTH / ( Float32 )SCREEN_HEIGHT, 1.0f, 2000.0f );

					visiSettings.sdwLimitDistance = 200.0f;
					visiSettings.sdwFalloffAmount = 100.0f;
					pCamera->SetVisiblitySettings( visiSettings );

					smSettings.depthBias = -0.69997531f;
					pCamera->SetShadowMappingSettings( smSettings );

					caSetting.distortion.k = +0.05f;
					caSetting.distortion.kCube = -0.05f;
					pCamera->SetChromatismSettings( caSetting );
					pCamera->SetChromatismEnabled( True );

					pCamera->SetShadowMappingEnabled( True );

					pSceneRenderer->AddCamera( pCamera );
				}

				if( pSceneMgr->CreateTerrainModelFromFile( pSceneEffectPackage, L"Data\\Model\\blocks.mtm", &pRoomModel ) == True )
				{
					pSceneRenderer->AddTerrainModel( pRoomModel );
					MIX_RELEASE( pRoomModel );
				}

				if( pSceneMgr->CreateDirectionalLight( &pDirLight ) == True )
				{
					pDirLight->SetEnabled( True );
					pDirLight->SetDirection( -SUN_POS.ToNormalize() );

					pSceneRenderer->SetDirectionalLight( pDirLight );

					MIX_RELEASE( pDirLight );
				}

#ifdef _DEBUG
				pSceneRenderer->Debug_SetCanvasRenderer( pCanvasRenderer );
#endif //_DEBUG
			}
		}

		////////////////////////////////////////////////////////////////////////////////////////////////////

		camHelper.Initialize(	pKeyboard,
								pMouse,
								Mix::Vector3( 0.0f, 2.0f, -99.0f ),
								Mix::Vector3( 0.0f, 2.0f, 0.0f ),
								0.2f );

		uiManager.Initialize( pMouse, pCanvasRenderer, pFontS );

		hud.Initialize( pGraphicsDev->GetScreenSize(), pCamera, pChrTex );
		hud.Open( &uiManager );

		////////////////////////////////////////////////////////////////////////////////////////////////////

		pEngine->ResetFPS();

		while( pEngine->Update() == True )
		{
			////////////////////////////////////////////////////////////////////////////////////////////////////
			// J
			////////////////////////////////////////////////////////////////////////////////////////////////////

			camHelper.Update();

			if( pCamera != NULL )
			{
				pCamera->SetView( camHelper.GetEye(), camHelper.GetAt(), camHelper.GetUp() );
			}

			////////////////////////////////////////////////////////////////////////////////////////////////////
			// XV
			////////////////////////////////////////////////////////////////////////////////////////////////////

			if( pCamera != NULL )
			{
				pCamera->Update();
			}

			if( pSceneRenderer != NULL )
			{
				pSceneRenderer->Update( pEngine->GetDT() );
			}

			////////////////////////////////////////////////////////////////////////////////////////////////////
			// tbV
			////////////////////////////////////////////////////////////////////////////////////////////////////

			if( pSceneRenderer != NULL )
			{
				pSceneRenderer->Refresh();
			}

			////////////////////////////////////////////////////////////////////////////////////////////////////
			// `
			////////////////////////////////////////////////////////////////////////////////////////////////////

			pGraphicsDev->SetViewBounds( 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT );

			if( pGraphicsDev->Begin() == True )
			{
				if( pCamera != NULL )
				{
					pCamera->Draw();
				}

				if( pCanvasRenderer != NULL )
				{
					const Mix::Point& screenSize = pGraphicsDev->GetScreenSize();

					uiManager.Update();

					tempStr.Sprintf(	L"FPS  [%f]\n"
										L"STPS [%f]\n"
										L"IF   [%d/%d]",	pEngine->GetFPS(),
															pEngine->GetSleepTimePerSec(),
															pEngine->GetInsomniaFrameCountPerSec(), pEngine->GetInsomniaFrameMax() );

					pCanvasRenderer->SetColor( Mix::Vector4( 1.0f, 1.0f, 1.0f, 1.0f ) );
					pCanvasRenderer->AddString(	16.0f, 16.0f, tempStr.GetConstPtr() );
					pCanvasRenderer->AddString(	16.0f, screenSize.y - 64.0f,
										L"Controls :\n"
										L"  Rotate camera : Mouse middle button\n"
										L"  Move camera   : W S A D key\n" );

					Utility::DrawMouseCursor( pCanvasRenderer, pMouse->GetPos() );

					pCanvasRenderer->Update();
					pCanvasRenderer->Draw();
				}

				pGraphicsDev->End();
			}

			pGraphicsDev->Present();
		}

		hud.Dispose();
		uiManager.Dispose();

		MIX_RELEASE( pChrTex );
		camHelper.Dispose();

		MIX_RELEASE( pCamera );
		MIX_RELEASE( pSceneRenderer );
		MIX_RELEASE( pSceneEffectPackage );
		MIX_RELEASE( pCanvasRenderer );
		MIX_RELEASE( pFontL );
		MIX_RELEASE( pFontS );
		MIX_RELEASE( pGraphicsDev );
		MIX_RELEASE( pKeyboard );
		MIX_RELEASE( pMouse );

		Mix::Finalize();
	}

	return 0;
}
