#include "hud.h"

#include "Mix/Scene/IRenderer.h"
#include "Mix/Scene/ICamera.h"

const Int32 HUD::SMOOTH_RANGE_TABLE[Mix::Scene::BLUR::TYPE_MAX][2] =
{
	{ 0, 0, },
	{ 1, 16, },
	{ 1, 16, },
	{ 1, 64, },
	{ 4, 64, },
};

HUD::HUD( void ) :
m_pSceneRenderer( NULL ),
m_pCamera( NULL )
{
}

HUD::~HUD( void )
{
}

void HUD::Open( Utility::UI::Manager* pManager, Mix::Scene::IRenderer* pSceneRenderer, Mix::Scene::ICamera* pCamera )
{
	MIX_ASSERT( pManager != NULL );
	MIX_ASSERT( pSceneRenderer != NULL );
	MIX_ASSERT( pCamera != NULL );

	const Mix::Point& screenSize = pCamera->GetConfig().targetSize;

	MIX_ADD_REF( pSceneRenderer );
	m_pSceneRenderer = pSceneRenderer;

	MIX_ADD_REF( pCamera );
	m_pCamera = pCamera;

#if 1
	SetHUDEnabled( True );
#else
	SetCaption( L"Configuration" );
#endif

	SetPosition( Mix::Point( screenSize.x - HUD::DLG_WIDTH, screenSize.y - HUD::DLG_HEIGHT ) );
	SetSize( Mix::Point( HUD::DLG_WIDTH, HUD::DLG_HEIGHT ) );

	Dialog::Open( pManager );
}

void HUD::OnOpen( void )
{
	Mix::Scene::CAMERA_CONFIG camConfig = m_pCamera->GetConfig();
	Mix::Scene::ICamera::DOF_SETTINGS dofSettings = m_pCamera->GetDofSettings();

	Int32 itemY = 16;

#ifdef _DEBUG
	UInt32 debDrawFlags = m_pCamera->Debug_GetDrawFlags();
#endif //_DEBUG

	//////////////////////////////////////////////////////////////////////////////////////////
	// ݒ
	//////////////////////////////////////////////////////////////////////////////////////////

	m_Blur1Table[Mix::Scene::BLUR::T_NONE] = Mix::Scene::BLUR::NONE();
	m_Blur1Table[Mix::Scene::BLUR::T_KS_3x3] = Mix::Scene::BLUR::KS_3x3( 1 );
	m_Blur1Table[Mix::Scene::BLUR::T_KS_5x5] = Mix::Scene::BLUR::KS_5x5( 1 );
	m_Blur1Table[Mix::Scene::BLUR::T_GAUSSIAN] = Mix::Scene::BLUR::GAUSSIAN( 4.0f );
	m_Blur1Table[Mix::Scene::BLUR::T_GAUSSIAN_EX] = Mix::Scene::BLUR::GAUSSIAN_EX( 16 );

	m_Blur2Table[Mix::Scene::BLUR::T_NONE] = Mix::Scene::BLUR::NONE();
	m_Blur2Table[Mix::Scene::BLUR::T_KS_3x3] = Mix::Scene::BLUR::KS_3x3( 1 );
	m_Blur2Table[Mix::Scene::BLUR::T_KS_5x5] = Mix::Scene::BLUR::KS_5x5( 1 );
	m_Blur2Table[Mix::Scene::BLUR::T_GAUSSIAN] = Mix::Scene::BLUR::GAUSSIAN( 4.0f );
	m_Blur2Table[Mix::Scene::BLUR::T_GAUSSIAN_EX] = Mix::Scene::BLUR::GAUSSIAN_EX( 16 );

	dofSettings.blur1 = m_Blur1Table[Mix::Scene::BLUR::T_KS_3x3];
	dofSettings.blur2 = m_Blur2Table[Mix::Scene::BLUR::T_KS_5x5];
	m_pCamera->SetDofSettings( dofSettings );

	//////////////////////////////////////////////////////////////////////////////////////////
	// Enabled
	//////////////////////////////////////////////////////////////////////////////////////////

	m_EnabledCB.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_EnabledCB.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_EnabledCB.SetText( L"Enabled" );
	m_EnabledCB.SetState( m_pCamera->IsDofEnabled() );
	m_EnabledCB.Initialize( this, HUD::CTRL_ENABLED_CB );
	itemY += HUD::ITEM_Y_STEP;

	//////////////////////////////////////////////////////////////////////////////////////////
	// Smooth
	//////////////////////////////////////////////////////////////////////////////////////////

	m_BokehL.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_BokehL.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_BokehL.SetText( L"Bokeh :" );
	m_BokehL.Initialize( this, HUD::CTRL_BOKEH_L );
	itemY += HUD::ITEM_Y_STEP;

	m_SmplCB.Initialize( this, HUD::CTRL_SMPL_CB );
	m_SmplCB.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_SmplCB.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_SmplCB.AddItem( Mix::Scene::RSMPL_DEFAULT, L"Default( Copy )" );
	m_SmplCB.AddItem( Mix::Scene::RSMPL_DOWN2X, L"Down2x" );
	m_SmplCB.AddItem( Mix::Scene::RSMPL_DOWN4X, L"Down4x" );
	m_SmplCB.SetSelectedItem( camConfig.dofSmplRes, False );
	itemY += HUD::ITEM_Y_STEP;

	// 1

	m_Blur1TypeCB.Initialize( this, HUD::CTRL_BLUR1_TYPE_CB );
	m_Blur1TypeCB.SetPosition( Mix::Point( HUD::ITEM_X1, itemY ) );
	m_Blur1TypeCB.SetSize( Mix::Point( HUD::ITEM_WIDTH1, HUD::ITEM_HEIGHT1 ) );
	m_Blur1TypeCB.AddItem( Mix::Scene::BLUR::T_NONE, L"NONE" );
	m_Blur1TypeCB.AddItem( Mix::Scene::BLUR::T_KS_3x3, L"KERNEL_3x3" );
	m_Blur1TypeCB.AddItem( Mix::Scene::BLUR::T_KS_5x5, L"KERNEL_5x5" );
	m_Blur1TypeCB.AddItem( Mix::Scene::BLUR::T_GAUSSIAN, L"GAUSSIAN" );
	m_Blur1TypeCB.AddItem( Mix::Scene::BLUR::T_GAUSSIAN_EX, L"GAUSSIAN_EX" );
	m_Blur1TypeCB.SetSelectedItem( dofSettings.blur1.GetType(), False );

	m_Blur1ValueSI.SetPosition( Mix::Point( HUD::ITEM_X2, itemY ) );
	m_Blur1ValueSI.SetSize( Mix::Point( HUD::ITEM_WIDTH2, HUD::ITEM_HEIGHT2 ) );
	m_Blur1ValueSI.SetText( L"" );
	m_Blur1ValueSI.SetRange( HUD::SMOOTH_RANGE_TABLE[dofSettings.blur1.GetType()][0], HUD::SMOOTH_RANGE_TABLE[dofSettings.blur1.GetType()][1] );
	m_Blur1ValueSI.SetValue( HUD::GetSmoothValue( dofSettings.blur1 ) );
	m_Blur1ValueSI.SetEanbled( dofSettings.blur1.GetType() != Mix::Scene::BLUR::T_NONE );
	m_Blur1ValueSI.SetStep( 1 );
	m_Blur1ValueSI.Initialize( this, HUD::CTRL_BLUR1_VALUE_SI );
	itemY += HUD::ITEM_Y_STEP;

	// 2

	m_Blur2TypeCB.Initialize( this, HUD::CTRL_BLUR2_TYPE_CB );
	m_Blur2TypeCB.SetPosition( Mix::Point( HUD::ITEM_X1, itemY ) );
	m_Blur2TypeCB.SetSize( Mix::Point( HUD::ITEM_WIDTH1, HUD::ITEM_HEIGHT1 ) );
	m_Blur2TypeCB.AddItem( Mix::Scene::BLUR::T_NONE, L"NONE" );
	m_Blur2TypeCB.AddItem( Mix::Scene::BLUR::T_KS_3x3, L"KERNEL_3x3" );
	m_Blur2TypeCB.AddItem( Mix::Scene::BLUR::T_KS_5x5, L"KERNEL_5x5" );
	m_Blur2TypeCB.AddItem( Mix::Scene::BLUR::T_GAUSSIAN, L"GAUSSIAN" );
	m_Blur2TypeCB.AddItem( Mix::Scene::BLUR::T_GAUSSIAN_EX, L"GAUSSIAN_EX" );
	m_Blur2TypeCB.SetSelectedItem( dofSettings.blur2.GetType(), False );

	m_Blur2ValueSI.SetPosition( Mix::Point( HUD::ITEM_X2, itemY ) );
	m_Blur2ValueSI.SetSize( Mix::Point( HUD::ITEM_WIDTH2, HUD::ITEM_HEIGHT2 ) );
	m_Blur2ValueSI.SetText( L"" );
	m_Blur2ValueSI.SetRange( HUD::SMOOTH_RANGE_TABLE[dofSettings.blur2.GetType()][0], HUD::SMOOTH_RANGE_TABLE[dofSettings.blur2.GetType()][1] );
	m_Blur2ValueSI.SetValue( HUD::GetSmoothValue( dofSettings.blur2 ) );
	m_Blur2ValueSI.SetEanbled( dofSettings.blur2.GetType() != Mix::Scene::BLUR::T_NONE );
	m_Blur2ValueSI.SetStep( 1 );
	m_Blur2ValueSI.Initialize( this, HUD::CTRL_BLUR2_VALUE_SI );
	itemY += HUD::ITEM_Y_STEP;

	// Threshold

	m_BlurThresholdSF.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_BlurThresholdSF.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_BlurThresholdSF.SetText( L"Threshold" );
	m_BlurThresholdSF.SetRange( 0.0f, 1.0f );
	m_BlurThresholdSF.SetStep( 0.0001f );
	m_BlurThresholdSF.SetValue( dofSettings.blurThreshold );
	m_BlurThresholdSF.Initialize( this, HUD::CTRL_BLUR_THRESHOLD_SF );
	itemY += HUD::ITEM_Y_STEP;

	//////////////////////////////////////////////////////////////////////////////////////////
	// Focus
	//////////////////////////////////////////////////////////////////////////////////////////

	m_FocusL.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_FocusL.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_FocusL.SetText( L"Focus :" );
	m_FocusL.Initialize( this, HUD::CTRL_FOCUS_L );
	itemY += HUD::ITEM_Y_STEP;

	m_PointSF.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_PointSF.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_PointSF.SetText( L"Point" );
	m_PointSF.SetRange( 0.0f, 100.0f );
	m_PointSF.SetStep( 0.1f );
	m_PointSF.SetValue( 20.0f );
	m_PointSF.Initialize( this, HUD::CTRL_POINT_SF );
	itemY += HUD::ITEM_Y_STEP;

	m_RangeSF.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_RangeSF.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_RangeSF.SetText( L"Range" );
	m_RangeSF.SetRange( 0.0f, 100.0f );
	m_RangeSF.SetStep( 0.1f );
	m_RangeSF.SetValue( 20.0f );
	m_RangeSF.Initialize( this, HUD::CTRL_RANGE_SF );
	itemY += HUD::ITEM_Y_STEP;

	m_NearSF.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_NearSF.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_NearSF.SetText( L"Near" );
	m_NearSF.SetRange( 0.0f, 100.0f );
	m_NearSF.SetStep( 0.1f );
	m_NearSF.SetValue( 4.0f );
	m_NearSF.Initialize( this, HUD::CTRL_NEAR_SF );
	itemY += HUD::ITEM_Y_STEP;

	m_FarSF.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_FarSF.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_FarSF.SetText( L"Far" );
	m_FarSF.SetRange( 0.0f, 100.0f );
	m_FarSF.SetStep( 0.1f );
	m_FarSF.SetValue( 50.0f );
	m_FarSF.Initialize( this, HUD::CTRL_NEAR_SF );
	itemY += HUD::ITEM_Y_STEP;

	UpdateFocus();

	//////////////////////////////////////////////////////////////////////////////////////////
}

void HUD::OnDispose( void )
{
	MIX_RELEASE( m_pCamera );
	MIX_RELEASE( m_pSceneRenderer );
}

void HUD::OnCheckBox_Click( Utility::UI::CheckBox* pCheckBox )
{
	switch( pCheckBox->GetID() )
	{
	case HUD::CTRL_ENABLED_CB:
		m_pCamera->SetDofEnabled( pCheckBox->GetState() );
		break;
	}
}

void HUD::OnSlider_ValueChanged( Utility::UI::SliderI* pSlider )
{
	Mix::Scene::ICamera::DOF_SETTINGS dofSettings = m_pCamera->GetDofSettings();

	switch( pSlider->GetID() )
	{
	case HUD::CTRL_BLUR1_VALUE_SI:
		HUD::SetSmoothValue( dofSettings.blur1, pSlider->GetValue() );
		m_Blur1Table[dofSettings.blur1.GetType()] = dofSettings.blur1;
		m_pCamera->SetDofSettings( dofSettings );
		break;

	case HUD::CTRL_BLUR2_VALUE_SI:
		HUD::SetSmoothValue( dofSettings.blur2, pSlider->GetValue() );
		m_Blur2Table[dofSettings.blur2.GetType()] = dofSettings.blur2;
		m_pCamera->SetDofSettings( dofSettings );
		break;
	}
}

void HUD::OnSlider_ValueChanged( Utility::UI::SliderF* pSlider )
{
	Mix::Scene::ICamera::DOF_SETTINGS dofSettings = m_pCamera->GetDofSettings();

	switch( pSlider->GetID() )
	{
	case HUD::CTRL_POINT_SF:
	case HUD::CTRL_RANGE_SF:
	case HUD::CTRL_NEAR_SF:
	case HUD::CTRL_FAR_SF:
		UpdateFocus();
		break;

	case HUD::CTRL_BLUR_THRESHOLD_SF:
		dofSettings.blurThreshold = pSlider->GetValue();
		m_pCamera->SetDofSettings( dofSettings );
		break;
	}
}

void HUD::OnComboBox_SelectItemChanged( Utility::UI::ComboBox* pComboBox )
{
	Mix::Scene::CAMERA_CONFIG camConfig = m_pCamera->GetConfig();
	Mix::Scene::ICamera::DOF_SETTINGS dofSettings = m_pCamera->GetDofSettings();

	Mix::Scene::RENDERER_SAMPLING_RESOLUTION smplRes;

	switch( pComboBox->GetID() )
	{
	case HUD::CTRL_SMPL_CB:
		smplRes = static_cast<Mix::Scene::RENDERER_SAMPLING_RESOLUTION>( pComboBox->GetSelectedItem() );
		if( camConfig.dofSmplRes != smplRes )
		{
			camConfig.dofSmplRes = smplRes;
			m_pCamera->SetConfig( camConfig );
		}
		break;
	case HUD::CTRL_BLUR1_TYPE_CB:
		dofSettings.blur1 = m_Blur1Table[pComboBox->GetSelectedItem()];
		m_pCamera->SetDofSettings( dofSettings );
		m_Blur1ValueSI.SetRange( HUD::SMOOTH_RANGE_TABLE[dofSettings.blur1.GetType()][0], HUD::SMOOTH_RANGE_TABLE[dofSettings.blur1.GetType()][1] );
		m_Blur1ValueSI.SetValue( HUD::GetSmoothValue( dofSettings.blur1 ) );
		m_Blur1ValueSI.SetEanbled( dofSettings.blur1.GetType() != Mix::Scene::BLUR::T_NONE );
		break;
	case HUD::CTRL_BLUR2_TYPE_CB:
		dofSettings.blur2 = m_Blur2Table[pComboBox->GetSelectedItem()];
		m_pCamera->SetDofSettings( dofSettings );
		m_Blur2ValueSI.SetRange( HUD::SMOOTH_RANGE_TABLE[dofSettings.blur2.GetType()][0], HUD::SMOOTH_RANGE_TABLE[dofSettings.blur2.GetType()][1] );
		m_Blur2ValueSI.SetValue( HUD::GetSmoothValue( dofSettings.blur2 ) );
		m_Blur2ValueSI.SetEanbled( dofSettings.blur2.GetType() != Mix::Scene::BLUR::T_NONE );
		break;
	}
}

void HUD::UpdateFocus( void )
{
	Mix::Scene::ICamera::DOF_SETTINGS dofSettings = m_pCamera->GetDofSettings();

	Float32 point = m_PointSF.GetValue();
	Float32 halfRange = m_RangeSF.GetValue() * 0.5f;

	dofSettings.nearZ = point - halfRange;
	dofSettings.nearDist = m_NearSF.GetValue();
	dofSettings.farZ = point + halfRange;
	dofSettings.farDist = m_FarSF.GetValue();

	m_pCamera->SetDofSettings( dofSettings );
}

Int32 HUD::GetSmoothValue( const Mix::Scene::BLUR& blur )
{
	Int32 ret = 0;

	switch( blur.GetType() )
	{
	case Mix::Scene::BLUR::T_NONE:
		ret = 0;
		break;
	case Mix::Scene::BLUR::T_KS_3x3:
		ret = blur.GetIntValue();
		break;
	case Mix::Scene::BLUR::T_KS_5x5:
		ret = blur.GetIntValue();
		break;
	case Mix::Scene::BLUR::T_GAUSSIAN:
		ret = static_cast<Int32>( blur.GetFloatValue() );
		break;
	case Mix::Scene::BLUR::T_GAUSSIAN_EX:
		ret = blur.GetIntValue();
		break;
	}

	return ret;
}

void HUD::SetSmoothValue( Mix::Scene::BLUR& blur, Int32 value )
{
	switch( blur.GetType() )
	{
	case Mix::Scene::BLUR::T_KS_3x3:
		blur = Mix::Scene::BLUR::KS_3x3( value );
		break;
	case Mix::Scene::BLUR::T_KS_5x5:
		blur = Mix::Scene::BLUR::KS_5x5( value );
		break;
	case Mix::Scene::BLUR::T_GAUSSIAN:
		blur = Mix::Scene::BLUR::GAUSSIAN( static_cast<Float32>( value ) );
		break;
	case Mix::Scene::BLUR::T_GAUSSIAN_EX:
		blur = Mix::Scene::BLUR::GAUSSIAN_EX( value );
		break;
	}
}
