#include "hud.h"

#include "Mix/Scene/IRenderer.h"
#include "Mix/Scene/ICamera.h"

const Int32 HUD::SMOOTH_RANGE_TABLE[Mix::Scene::BLUR::TYPE_MAX][2] =
{
	{ 0, 0, },
	{ 1, 16, },
	{ 1, 16, },
	{ 1, 64, },
	{ 4, 64, },
};

HUD::HUD( void ) :
m_pSceneRenderer( NULL ),
m_pCamera( NULL )
{
}

HUD::~HUD( void )
{
}

void HUD::Open( Utility::UI::Manager* pManager, Mix::Scene::IRenderer* pSceneRenderer, Mix::Scene::ICamera* pCamera )
{
	MIX_ASSERT( pManager != NULL );
	MIX_ASSERT( pSceneRenderer != NULL );
	MIX_ASSERT( pCamera != NULL );

	const Mix::Point& screenSize = pCamera->GetConfig().targetSize;

	MIX_ADD_REF( pSceneRenderer );
	m_pSceneRenderer = pSceneRenderer;

	MIX_ADD_REF( pCamera );
	m_pCamera = pCamera;

#if 1
	SetHUDEnabled( True );
#else
	SetCaption( L"Configuration" );
#endif

	SetPosition( Mix::Point( screenSize.x - HUD::DLG_WIDTH, screenSize.y - HUD::DLG_HEIGHT ) );
	SetSize( Mix::Point( HUD::DLG_WIDTH, HUD::DLG_HEIGHT ) );

	Dialog::Open( pManager );
}

void HUD::OnOpen( void )
{
	const Mix::Scene::CAMERA_CONFIG& camConfig = m_pCamera->GetConfig();
	Mix::Scene::ICamera::VISIBILITY_SETTINGS visiSettings = m_pCamera->GetVisiblitySettings();
	Mix::Scene::ICamera::SHADOW_MAPPING_SETTINGS smSetting = m_pCamera->GetShadowMappingSettings();

	Int32 itemY = 16;

#ifdef _DEBUG
	UInt32 debDrawFlags = m_pCamera->Debug_GetDrawFlags();
#endif //_DEBUG

	//////////////////////////////////////////////////////////////////////////////////////////
	// ݒ
	//////////////////////////////////////////////////////////////////////////////////////////

	m_ProjSmoothTable[Mix::Scene::BLUR::T_NONE] = Mix::Scene::BLUR::NONE();
	m_ProjSmoothTable[Mix::Scene::BLUR::T_KS_3x3] = Mix::Scene::BLUR::KS_3x3( 1 );
	m_ProjSmoothTable[Mix::Scene::BLUR::T_KS_5x5] = Mix::Scene::BLUR::KS_5x5( 1 );
	m_ProjSmoothTable[Mix::Scene::BLUR::T_GAUSSIAN] = Mix::Scene::BLUR::GAUSSIAN( 4.0f );
	m_ProjSmoothTable[Mix::Scene::BLUR::T_GAUSSIAN_EX] = Mix::Scene::BLUR::GAUSSIAN_EX( 16 );

	smSetting.depthBias = 0.6f;
	m_pCamera->SetShadowMappingSettings( smSetting );

	//////////////////////////////////////////////////////////////////////////////////////////

#ifndef _DEBUG
	itemY += HUD::ITEM_Y_STEP;
#endif //_DEBUG

	m_EnabledCB.SetPosition( Mix::Point( 0, itemY ) );
	m_EnabledCB.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_EnabledCB.SetText( L"Enabled" );
	m_EnabledCB.SetState( m_pCamera->IsShadowMappingEnabled() );
	m_EnabledCB.Initialize( this, HUD::CTRL_ENABLED );

#ifdef _DEBUG
	itemY += HUD::ITEM_Y_STEP;

	m_DebugCB.SetPosition( Mix::Point( 0, itemY ) );
	m_DebugCB.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_DebugCB.SetText( L"Debug" );
	m_DebugCB.SetState( MIX_TESTBIT( debDrawFlags, Mix::Scene::DDF_SHADOW ) != 0 );
	m_DebugCB.Initialize( this, HUD::CTRL_DEBUG_CB );
	itemY += HUD::ITEM_YY_STEP;
#else //_DEBUG
	itemY += HUD::ITEM_YY_STEP;
#endif //_DEBUG

	//////////////////////////////////////////////////////////////////////////////////////////

	m_DepthL.Initialize( this, HUD::CTRL_Z_L );
	m_DepthL.SetPosition( Mix::Point( 0, itemY ) );
	m_DepthL.SetSize( Mix::Point( HUD::LABEL_WIDTH, HUD::ITEM_HEIGHT ) );
	m_DepthL.SetText( L"Depth :" );

	m_DepthBiasSF.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_DepthBiasSF.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_DepthBiasSF.SetText( L"Bias" );
	m_DepthBiasSF.SetRange( -100.0f, +100.0f );
	m_DepthBiasSF.SetStep( 0.01f );
	m_DepthBiasSF.SetValue( smSetting.depthBias );
	m_DepthBiasSF.Initialize( this, HUD::CTRL_ZBIAS_SF );
	itemY += HUD::ITEM_Y_STEP;

	m_DepthSlopeSF.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_DepthSlopeSF.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_DepthSlopeSF.SetText( L"SlopeScale" );
	m_DepthSlopeSF.SetRange( 0.0f, +10.0f );
	m_DepthSlopeSF.SetStep( 0.001f );
	m_DepthSlopeSF.SetValue( smSetting.depthSlopeScale );
	m_DepthSlopeSF.Initialize( this, HUD::CTRL_ZSLOPE_SF );
	itemY += HUD::ITEM_Y_STEP;

	m_DepthClampSF.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_DepthClampSF.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_DepthClampSF.SetText( L"BiasClamp" );
	m_DepthClampSF.SetRange( 0.0f, +1.0f );
	m_DepthClampSF.SetStep( 0.001f );
	m_DepthClampSF.SetValue( smSetting.depthBiasClamp );
	m_DepthClampSF.Initialize( this, HUD::CTRL_ZCLAMP_SF );
	itemY += HUD::ITEM_YY_STEP;

	//////////////////////////////////////////////////////////////////////////////////////////

	m_TargetL.Initialize( this, HUD::CTRL_TARGET_L );
	m_TargetL.SetPosition( Mix::Point( 0, itemY ) );
	m_TargetL.SetSize( Mix::Point( HUD::LABEL_WIDTH, HUD::ITEM_HEIGHT ) );
	m_TargetL.SetText( L"Target :" );

	m_TargetSizeCB.Initialize( this, HUD::CTRL_TARGET_SIZE_CB );
	m_TargetSizeCB.SetPosition( Mix::Point( HUD::ITEM_X1, itemY ) );
	m_TargetSizeCB.SetSize( Mix::Point( HUD::ITEM_WIDTH1, HUD::ITEM_HEIGHT ) );
	m_TargetSizeCB.AddItem(  256, L"256x256" );
	m_TargetSizeCB.AddItem(  512, L"512x512" );
	m_TargetSizeCB.AddItem( 1024, L"1024x1024" );
	m_TargetSizeCB.AddItem( 2048, L"2048x2048" );
	m_TargetSizeCB.AddItem( 4096, L"4096x4096" );
	m_TargetSizeCB.SetSelectedItem( camConfig.smDepthTexSize, False );

	m_TargetFormatCB.Initialize( this, HUD::CTRL_TARGET_FORMAT_CB );
	m_TargetFormatCB.SetPosition( Mix::Point( HUD::ITEM_X2, itemY ) );
	m_TargetFormatCB.SetSize( Mix::Point( HUD::ITEM_WIDTH2, HUD::ITEM_HEIGHT ) );
	m_TargetFormatCB.AddItem( Mix::Graphics::FMT_R16G16B16A16F, L"16Bit" );
	m_TargetFormatCB.AddItem( Mix::Graphics::FMT_R32G32B32A32F, L"32Bit" );
	m_TargetFormatCB.SetSelectedItem( camConfig.smDepthTexFormat, False );
	itemY += HUD::ITEM_YY_STEP;

	//////////////////////////////////////////////////////////////////////////////////////////

	m_SmoothL.Initialize( this, HUD::CTRL_SMOOTH_L );
	m_SmoothL.SetPosition( Mix::Point( 0, itemY ) );
	m_SmoothL.SetSize( Mix::Point( HUD::LABEL_WIDTH, HUD::ITEM_HEIGHT ) );
	m_SmoothL.SetText( L"Smooth :" );

	m_PcfCB.Initialize( this, HUD::CTRL_PCF_CB );
	m_PcfCB.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_PcfCB.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_PcfCB.AddItem( Mix::Scene::ICamera::SMPS_3, L"PCF 3x3" );
	m_PcfCB.AddItem( Mix::Scene::ICamera::SMPS_5, L"PCF 5x5" );
	m_PcfCB.AddItem( Mix::Scene::ICamera::SMPS_7, L"PCF 7x7" );
	m_PcfCB.SetSelectedItem( smSetting.pcfSize, False );
	itemY += HUD::ITEM_Y_STEP;

	m_SmplCB.Initialize( this, HUD::CTRL_SMPL_CB );
	m_SmplCB.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_SmplCB.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_SmplCB.AddItem( Mix::Scene::ICamera::SMPS_3, L"Default ( Copy )" );
	m_SmplCB.AddItem( Mix::Scene::ICamera::SMPS_5, L"Down 2x" );
	m_SmplCB.AddItem( Mix::Scene::ICamera::SMPS_7, L"Down 4x" );
	m_SmplCB.SetSelectedItem( camConfig.smSmplRes, False );
	itemY += HUD::ITEM_Y_STEP;

	m_BlurCB.Initialize( this, HUD::CTRL_BLUR_CB );
	m_BlurCB.SetPosition( Mix::Point( HUD::ITEM_X1, itemY ) );
	m_BlurCB.SetSize( Mix::Point( HUD::ITEM_WIDTH1, HUD::ITEM_HEIGHT1 ) );
	m_BlurCB.AddItem( Mix::Scene::BLUR::T_NONE, L"NONE" );
	m_BlurCB.AddItem( Mix::Scene::BLUR::T_KS_3x3, L"KERNEL_3x3" );
	m_BlurCB.AddItem( Mix::Scene::BLUR::T_KS_5x5, L"KERNEL_5x5" );
	m_BlurCB.AddItem( Mix::Scene::BLUR::T_GAUSSIAN, L"GAUSSIAN" );
	m_BlurCB.AddItem( Mix::Scene::BLUR::T_GAUSSIAN_EX, L"GAUSSIAN_EX" );
	m_BlurCB.SetSelectedItem( smSetting.blur.GetType(), False );

	m_BlurSI.SetPosition( Mix::Point( HUD::ITEM_X2, itemY ) );
	m_BlurSI.SetSize( Mix::Point( HUD::ITEM_WIDTH2, HUD::ITEM_HEIGHT2 ) );
	m_BlurSI.SetText( L"" );
	m_BlurSI.SetRange( HUD::SMOOTH_RANGE_TABLE[smSetting.blur.GetType()][0], HUD::SMOOTH_RANGE_TABLE[smSetting.blur.GetType()][1] );
	m_BlurSI.SetValue( HUD::GetSmoothValue( smSetting.blur ) );
	m_BlurSI.SetStep( 1 );
	m_BlurSI.SetEanbled( smSetting.blur.GetType() != Mix::Scene::BLUR::T_NONE );
	m_BlurSI.Initialize( this, HUD::CTRL_BLUR_SI );
	itemY += HUD::ITEM_YY_STEP;

	//////////////////////////////////////////////////////////////////////////////////////////

	m_MiscL.Initialize( this, HUD::CTRL_MISC_L );
	m_MiscL.SetPosition( Mix::Point( 0, itemY ) );
	m_MiscL.SetSize( Mix::Point( HUD::LABEL_WIDTH, HUD::ITEM_HEIGHT ) );
	m_MiscL.SetText( L"Misc :" );

	m_DistSF.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_DistSF.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_DistSF.SetText( L"Distance" );
	m_DistSF.SetRange( 0.0f, 100.0f );
	m_DistSF.SetStep( 1.0f );
	m_DistSF.SetValue( visiSettings.sdwLimitDistance );
	m_DistSF.Initialize( this, HUD::CTRL_DISTANCE_SF );
	itemY += HUD::ITEM_Y_STEP;

	m_TintSF.SetPosition( Mix::Point( HUD::ITEM_X, itemY ) );
	m_TintSF.SetSize( Mix::Point( HUD::ITEM_WIDTH, HUD::ITEM_HEIGHT ) );
	m_TintSF.SetText( L"Tint" );
	m_TintSF.SetRange( 0.0f, 1.0f );
	m_TintSF.SetStep( 0.01f );
	m_TintSF.SetValue( smSetting.tint );
	m_TintSF.Initialize( this, HUD::CTRL_TINT_SF );
	itemY += HUD::ITEM_Y_STEP;
}

void HUD::OnDispose( void )
{
	MIX_RELEASE( m_pCamera );
	MIX_RELEASE( m_pSceneRenderer );
}

void HUD::OnCheckBox_Click( Utility::UI::CheckBox* pCheckBox )
{
#ifdef _DEBUG
	UInt32 debDrawFlags = m_pCamera->Debug_GetDrawFlags();
#endif //_DEBUG

	switch( pCheckBox->GetID() )
	{
	case HUD::CTRL_ENABLED:
		m_pCamera->SetShadowMappingEnabled( pCheckBox->GetState() );
		break;
#ifdef _DEBUG
	case HUD::CTRL_DEBUG_CB:
		if( pCheckBox->GetState() == True )
		{
			MIX_SETBIT( debDrawFlags, Mix::Scene::DDF_SHADOW );
		}
		else
		{
			MIX_RESETBIT( debDrawFlags, Mix::Scene::DDF_SHADOW );
		}
		m_pCamera->Debug_SetDrawFlags( debDrawFlags );
		break;
#endif //_DEBUG
	}
}

void HUD::OnSlider_ValueChanged( Utility::UI::SliderI* pSlider )
{
	Mix::Scene::ICamera::SHADOW_MAPPING_SETTINGS smSettings = m_pCamera->GetShadowMappingSettings();
	Mix::Scene::ICamera::AMBIENT_OCCLUSION_SETTINGS aoSettings = m_pCamera->GetAmbientOcclusionSettings();

	switch( pSlider->GetID() )
	{
	case HUD::CTRL_BLUR_SI:
		HUD::SetSmoothValue( smSettings.blur, pSlider->GetValue() );
		m_ProjSmoothTable[smSettings.blur.GetType()] = smSettings.blur;
		m_pCamera->SetShadowMappingSettings( smSettings );
		break;
	}
}

void HUD::OnSlider_ValueChanged( Utility::UI::SliderF* pSlider )
{
	Mix::Scene::ICamera::VISIBILITY_SETTINGS visiSettings = m_pCamera->GetVisiblitySettings();
	Mix::Scene::ICamera::SHADOW_MAPPING_SETTINGS smSettings = m_pCamera->GetShadowMappingSettings();
	Mix::Scene::ICamera::AMBIENT_OCCLUSION_SETTINGS aoSettings = m_pCamera->GetAmbientOcclusionSettings();

	switch( pSlider->GetID() )
	{
	case HUD::CTRL_ZBIAS_SF:
		smSettings.depthBias = pSlider->GetValue();
		m_pCamera->SetShadowMappingSettings( smSettings );
		break;
	case HUD::CTRL_ZSLOPE_SF:
		smSettings.depthSlopeScale = pSlider->GetValue();
		m_pCamera->SetShadowMappingSettings( smSettings );
		break;
	case HUD::CTRL_ZCLAMP_SF:
		smSettings.depthBiasClamp = pSlider->GetValue();
		m_pCamera->SetShadowMappingSettings( smSettings );
		break;

	case HUD::CTRL_DISTANCE_SF:
		visiSettings.sdwLimitDistance = pSlider->GetValue();
		visiSettings.sdwFalloffAmount = min( 10.0f, visiSettings.sdwLimitDistance );
		m_pCamera->SetVisiblitySettings( visiSettings );
		break;
	case HUD::CTRL_TINT_SF:
		smSettings.tint = pSlider->GetValue();
		m_pCamera->SetShadowMappingSettings( smSettings );
		break;
	}
}

void HUD::OnComboBox_SelectItemChanged( Utility::UI::ComboBox* pComboBox )
{
	Mix::Scene::CAMERA_CONFIG camConfig = m_pCamera->GetConfig();
	Mix::Scene::ICamera::SHADOW_MAPPING_SETTINGS smSetting = m_pCamera->GetShadowMappingSettings();
	Mix::Scene::ICamera::AMBIENT_OCCLUSION_SETTINGS aoSettings = m_pCamera->GetAmbientOcclusionSettings();

	Mix::Graphics::FORMAT texFormat;
	UInt32 texSize;

	Mix::Scene::RENDERER_SAMPLING_RESOLUTION smplRes;

	switch( pComboBox->GetID() )
	{
	case HUD::CTRL_TARGET_FORMAT_CB:
		texFormat = static_cast<Mix::Graphics::FORMAT>( pComboBox->GetSelectedItem() );
		if( camConfig.smDepthTexFormat != texFormat )
		{
			camConfig.smDepthTexFormat = texFormat;
			m_pCamera->SetConfig( camConfig );

			camConfig = m_pCamera->GetConfig();
			pComboBox->SetSelectedItem( camConfig.smDepthTexFormat, False );
		}
		break;

	case HUD::CTRL_TARGET_SIZE_CB:
		texSize = static_cast<UInt32>( pComboBox->GetSelectedItem() );
		if( camConfig.smDepthTexSize != texSize )
		{
			camConfig.smDepthTexSize = texSize;
			m_pCamera->SetConfig( camConfig );

			camConfig = m_pCamera->GetConfig();
			pComboBox->SetSelectedItem( camConfig.smDepthTexSize, False );
		}
		break;

	case HUD::CTRL_PCF_CB:
		smSetting.pcfSize = static_cast<Mix::Scene::ICamera::SHADOW_MAPPING_PCF_SIZE>( pComboBox->GetSelectedItem() );
		m_pCamera->SetShadowMappingSettings( smSetting );
		break;
	case HUD::CTRL_SMPL_CB:
		smplRes = static_cast<Mix::Scene::RENDERER_SAMPLING_RESOLUTION>( pComboBox->GetSelectedItem() );
		if( camConfig.smSmplRes != smplRes )
		{
			camConfig.smSmplRes = smplRes;
			m_pCamera->SetConfig( camConfig );

			camConfig = m_pCamera->GetConfig();
			pComboBox->SetSelectedItem( camConfig.smSmplRes, False );
		}
		break;
	case HUD::CTRL_BLUR_CB:
		smSetting.blur = m_ProjSmoothTable[pComboBox->GetSelectedItem()];
		m_pCamera->SetShadowMappingSettings( smSetting );
		m_BlurSI.SetRange( HUD::SMOOTH_RANGE_TABLE[smSetting.blur.GetType()][0], HUD::SMOOTH_RANGE_TABLE[smSetting.blur.GetType()][1] );
		m_BlurSI.SetValue( HUD::GetSmoothValue( smSetting.blur ) );
		m_BlurSI.SetEanbled( smSetting.blur.GetType() != Mix::Scene::BLUR::T_NONE );
		break;
	}
}

Int32 HUD::GetSmoothValue( const Mix::Scene::BLUR& blur )
{
	Int32 ret = 0;

	switch( blur.GetType() )
	{
	case Mix::Scene::BLUR::T_NONE:
		ret = 0;
		break;
	case Mix::Scene::BLUR::T_KS_3x3:
		ret = blur.GetIntValue();
		break;
	case Mix::Scene::BLUR::T_KS_5x5:
		ret = blur.GetIntValue();
		break;
	case Mix::Scene::BLUR::T_GAUSSIAN:
		ret = static_cast<Int32>( blur.GetFloatValue() );
		break;
	case Mix::Scene::BLUR::T_GAUSSIAN_EX:
		ret = blur.GetIntValue();
		break;
	}

	return ret;
}

void HUD::SetSmoothValue( Mix::Scene::BLUR& blur, Int32 value )
{
	switch( blur.GetType() )
	{
	case Mix::Scene::BLUR::T_KS_3x3:
		blur = Mix::Scene::BLUR::KS_3x3( value );
		break;
	case Mix::Scene::BLUR::T_KS_5x5:
		blur = Mix::Scene::BLUR::KS_5x5( value );
		break;
	case Mix::Scene::BLUR::T_GAUSSIAN:
		blur = Mix::Scene::BLUR::GAUSSIAN( static_cast<Float32>( value ) );
		break;
	case Mix::Scene::BLUR::T_GAUSSIAN_EX:
		blur = Mix::Scene::BLUR::GAUSSIAN_EX( value );
		break;
	}
}
