#include "stdafx.h"

#include "Mix/IO.h"
#include "Mix/HID.h"
#include "Mix/Graphics.h"
#include "Mix/Scene.h"

#include "SoulParticleSystem.h"

#include "Utility/Common.h"
#include "Utility/ViewCameraHelper.h"

#ifdef _DEBUG
	#include <crtdbg.h>
#endif //_DEBUG

#define SCREEN_WIDTH 1024
#define SCREEN_HEIGHT 768

static const Mix::Vector2 TEXT_PADDING( 16.0f, 16.0f );

static const Mix::Vector3 SOUL_EMIT_RANGE( 6.0f, 4.0f, 10.0f );
static const Float32 SOUL_EMIT_OFFSET_Y = 1.0f;
static const Float32 SOUL_EMIT_INTERVAL = 0.5f;
static const UInt32 SOUL_EMIT_NUM = 40;

int APIENTRY _tWinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance, LPTSTR lpCmdLine, int nCmdShow )
{
#ifdef _DEBUG
	_CrtSetDbgFlag(_CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF);
#endif //_DEBUG

	Mix::ENGINE_CONFIG engineConfig;

	engineConfig.flags = Mix::EC_IO_NARROW | Mix::EC_KEYBOARD | Mix::EC_MOUSE | Mix::EC_GRAPHICS | Mix::EC_SCENE;
	engineConfig.pRootDirectoryPath = Utility::GetRootDirectoryPath();
	engineConfig.pPluginDirectoryPath = Utility::GetPluginsDirectoryPath();
	engineConfig.pUserDirectoryPath = Utility::GetUserDirectoryPath( L"Scene\\Widget\\Particle\\Scatter" );
	engineConfig.pCaption = L"Scene - Widget ( Scatter Particle )";
	engineConfig.targetSize = Mix::Point( SCREEN_WIDTH, SCREEN_HEIGHT );
	engineConfig.shaderModel = Mix::Graphics::SHADER_MODEL_3;

	if( Mix::Initialize( engineConfig ) == True )
	{
		Mix::IEngine* pEngine = Mix::GetEnginePtr();

		Mix::IO::IManager* pIOMgr = Mix::IO::GetManagerPtr();
		Mix::HID::IManager* pInputMgr = Mix::HID::GetManagerPtr();
		Mix::Graphics::IManager* pGraphicsMgr = Mix::Graphics::GetManagerPtr();
		Mix::Scene::IManager* pSceneMgr = Mix::Scene::GetManagerPtr();

		Mix::HID::IMouse* pMouse = NULL;
		Mix::HID::IKeyboard* pKeyboard = NULL;

		Mix::Graphics::IDevice* pGraphicsDev = NULL;
		Mix::Graphics::Utility::ICanvasRenderer* pCanvasRenderer = NULL;
		Mix::Graphics::Utility::IFont* pFont = NULL;

		Mix::Scene::IEffectPackage* pSceneEffectPackage = NULL;
		Mix::Scene::IRenderer* pSceneRenderer = NULL;
		Mix::Scene::IViewCamera* pCamera = NULL;

		Mix::Scene::IMaterial* pSoulMaterial = NULL;
		SoulParticleSystem* pSoulParticleSystem = NULL;
		Mix::Scene::IScatterParticle* pSoulParticle = NULL;
		Float32 soulTime = SOUL_EMIT_INTERVAL;

		Utility::ViewCameraHelper viewCamHelper;

		Mix::StringW tempStr;

		////////////////////////////////////////////////////////////////////////////////////////////////////

		pIOMgr->MountDirectory( L"Data" );

		////////////////////////////////////////////////////////////////////////////////////////////////////

		pInputMgr->GetMouse( &pMouse );
		pInputMgr->GetKeyboard( &pKeyboard );

		////////////////////////////////////////////////////////////////////////////////////////////////////

		pGraphicsMgr->GetDevice( &pGraphicsDev );

		if( pGraphicsMgr->CreateCanvasRenderer( &pCanvasRenderer, L"Sample" ) == True )
		{
			if( pGraphicsMgr->CreateFontFromFile( L"Data\\Font\\mspg_16_400.fnt", &pFont ) == True )
			{
				pCanvasRenderer->SetFont( pFont );
			}
		}

		////////////////////////////////////////////////////////////////////////////////////////////////////

		if( pSceneMgr->CreateEffectPackage( L"Data\\", L"effects", &pSceneEffectPackage ) == True )
		{
			Mix::Scene::RENDERER_CONFIG config;

			config.caps = Mix::Scene::RCAP_WIDGET | Mix::Scene::RCAP_BLOOM | Mix::Scene::RCAP_LUMINOSITY;

			if( pSceneMgr->CreateRenderer( pSceneEffectPackage, config, &pSceneRenderer, L"Sample" ) == True )
			{
				Mix::Scene::CAMERA_CONFIG camConfig;
				Mix::Scene::IDirectionalLight* pDirLight = NULL;
				Mix::Scene::ITerrainModel* pLandModel = NULL;

				/*
					J
				*/

				camConfig.caps = config.caps;
				camConfig.targetSize.Set( SCREEN_WIDTH, SCREEN_HEIGHT );
//				camConfig.colorTexFormat = Mix::Graphics::FMT_R8G8B8A8;
				camConfig.colorTexFormat = Mix::Graphics::FMT_R16G16B16A16F;
//				camConfig.colorTexFormat = Mix::Graphics::FMT_R32G32B32A32F;
				camConfig.bloomSmplRes = Mix::Scene::RSMPL_DOWN4X;
				camConfig.bloomOverflowNum = 2;

				if( pSceneMgr->CreateViewCamera( camConfig, &pCamera ) == True )
				{
					Mix::Scene::ICamera::LUMINOSITY_SETTINGS lumSettings = pCamera->GetLuminositySettings();

					pCamera->SetBackgroundColor( Mix::Vector4( 0.01f, 0.01f, 0.01f, 1.0f ) );
					pCamera->SetProjection( MIX_TO_RAD( 60.0f ), 0.01f, 1000.0f );

					Mix::Scene::ICamera::BLOOM_SETTINGS bloomSettings = pCamera->GetBloomSettings();
					Mix::Scene::ICamera::BLOOM_OVERFLOW bloomOF;

					bloomSettings.brightPass.bLumInf = True;
					bloomSettings.brightPass.white = 0.1f;
					pCamera->SetBloomSettings( bloomSettings );
					pCamera->SetBloomEnabled( True );

					bloomOF.intensity = 0.5f;
					bloomOF.blur = Mix::Scene::BLUR::GAUSSIAN_EX( 16 );
					pCamera->SetBloomOverflow( 0, bloomOF );

					bloomOF.intensity = 1.0f;
					bloomOF.blur = Mix::Scene::BLUR::GAUSSIAN_EX( 8 );
					pCamera->SetBloomOverflow( 1, bloomOF );

					lumSettings.flags = Mix::Scene::ICamera::LUM_TONE_MAPPING;
					pCamera->SetLuminositySettings( lumSettings );

					pSceneRenderer->AddCamera( pCamera );
				}

				/*
					f
				*/

				if( pSceneMgr->CreateTerrainModelFromFile( pSceneEffectPackage, L"Data\\Model\\bonefire\\land.mtm", &pLandModel ) == True )
				{
					pSceneRenderer->AddTerrainModel( pLandModel );

					MIX_RELEASE( pLandModel );
				}

				/*
					fBNViCg
				*/

				if( pSceneMgr->CreateDirectionalLight( &pDirLight ) == True )
				{
					pDirLight->SetEnabled( True );
					pDirLight->SetDirection( Mix::Vector3( +0.7f, -0.4f, 0.6f ).ToNormalize() );
					pDirLight->SetColor( Mix::Vector4( 1.0f, 0.9f, 0.7f, 0.04f ) );

					pSceneRenderer->SetDirectionalLight( pDirLight );

					MIX_RELEASE( pDirLight );
				}

				/*
					\Ep[eBN
				*/

				if( pSceneMgr->CreateMaterialFromFile( pSceneEffectPackage, L"Data\\Material\\Fire.mtl", &pSoulMaterial ) == True )
				{
					Mix::Scene::IDefaultMaterial* pDefMaterial = static_cast<Mix::Scene::IDefaultMaterial*>( pSoulMaterial );
					pDefMaterial->SetEmissiveColor( Mix::Vector4( 0.08f, 0.08f, 0.08f, 0.0f ) );
				}

				pSoulParticleSystem = SoulParticleSystem::CreateInstance();
				if( pSoulParticleSystem != NULL )
				{
					if( pSceneMgr->CreateScatterParticle( pSoulMaterial, pSoulParticleSystem, &pSoulParticle, MIX_DEBUGNAME ) == True )
					{
						pSoulParticle->Debug_SetDrawAxisScaling( 0.2f );
						pSceneRenderer->AddScatterParticle( pSoulParticle );
					}
				}

#ifdef _DEBUG
				Mix::Graphics::Utility::IPerspectiveRenderer* pPersRenderer = NULL;

				if( pGraphicsMgr->CreatePerspectiveRenderer( &pPersRenderer, MIX_DEBUGNAME ) == True )
				{
					pSceneRenderer->Debug_SetPerspectiveRenderer( pPersRenderer );
					MIX_RELEASE( pPersRenderer );
				}
#endif //_DEBUG
			}
		}

		////////////////////////////////////////////////////////////////////////////////////////////////////

		viewCamHelper.Initialize(	pMouse,
									pKeyboard,
									Mix::Vector3( 0.0f, 1.5f, 0.0f ),
									Mix::Vector2( MIX_TO_RAD( 20.0f ), MIX_TO_RAD( 10.0f ) ),
									10.0f );

		////////////////////////////////////////////////////////////////////////////////////////////////////

		pEngine->ResetFPS();

		while( pEngine->Update() == True )
		{
			Float32 dt = pEngine->GetDT();

#ifdef _DEBUG

			////////////////////////////////////////////////////////////////////////////////////////////////////
			// fobO
			////////////////////////////////////////////////////////////////////////////////////////////////////

			if( pKeyboard->GetKeyState( Mix::HID::KEY_D ) & Mix::HID::PRESSED )
			{
				UInt32 debDrawFlags = pCamera->Debug_GetDrawFlags();

				if( MIX_TESTBIT( debDrawFlags, Mix::Scene::DDF_PARTICLE ) == 0 )
				{
					debDrawFlags |= Mix::Scene::DDF_PARTICLE;
				}
				else
				{
					debDrawFlags ^= Mix::Scene::DDF_PARTICLE;
				}

				pCamera->Debug_SetDrawFlags( debDrawFlags );
			}

#endif //_DEBUG

			////////////////////////////////////////////////////////////////////////////////////////////////////
			// J
			////////////////////////////////////////////////////////////////////////////////////////////////////

			viewCamHelper.Update();

			if( pCamera != NULL )
			{
				pCamera->SetRotation( viewCamHelper.GetRotation() );
				pCamera->SetAt( viewCamHelper.GetAt() );
				pCamera->SetDistance( viewCamHelper.GetDistance() );
			}

			////////////////////////////////////////////////////////////////////////////////////////////////////
			// LoX
			////////////////////////////////////////////////////////////////////////////////////////////////////

			if( pCanvasRenderer != NULL )
			{
				Mix::Vector2 txtSize;

				pCanvasRenderer->SetColor( Mix::Vector4( 1.0f, 1.0f, 1.0f, 1.0f ) );

				tempStr.Sprintf( L"FPS %f", pEngine->GetFPS() );
				pCanvasRenderer->AddString( TEXT_PADDING, tempStr.GetConstPtr() );
				tempStr.Sprintf( L"STps %f", pEngine->GetSleepTimePerSec() );
				pCanvasRenderer->AddString( TEXT_PADDING + Mix::Vector2( 0.0f, 16.0f ), tempStr.GetConstPtr() );
				tempStr.Sprintf( L"IFCps %d/%d", pEngine->GetInsomniaFrameCountPerSec(), pEngine->GetInsomniaFrameMax() );
				pCanvasRenderer->AddString( TEXT_PADDING + Mix::Vector2( 0.0f, 32.0f ), tempStr.GetConstPtr() );

#ifdef _DEBUG
				tempStr.Sprintf( L"%s", L"Controls :\n"
										L"  Rotate camera     : Mouse middle button\n"
										L"  Move camera       : Mouse middle button + Shift key\n"
										L"  Distant camera    : Mouse middle button + Ctrl key or Mouse wheel\n"
										L"  Toggle debug draw : D key" );
#else //_DEBUG
				tempStr.Sprintf( L"%s", L"Controls :\n"
										L"  Rotate camera  : Mouse middle button\n"
										L"  Move camera    : Mouse middle button + Shift key\n"
										L"  Distant camera : Mouse middle button + Ctrl key or Mouse wheel\n" );
#endif //_DEBUG

				txtSize = Mix::Graphics::Utility::MeasureString( pFont, tempStr.GetConstPtr() );
				pCanvasRenderer->AddString( TEXT_PADDING.x, static_cast<Float32>( SCREEN_HEIGHT ) - txtSize.y - TEXT_PADDING.y, tempStr.GetConstPtr() );

				Utility::DrawMouseCursor( pCanvasRenderer, pMouse->GetPos() );
			}

			////////////////////////////////////////////////////////////////////////////////////////////////////
			// \E
			////////////////////////////////////////////////////////////////////////////////////////////////////

			if( pSoulParticle != NULL )
			{
				if( soulTime > SOUL_EMIT_INTERVAL )
				{
					Mix::Vector3 emitWorldPos;
					Mix::Matrix4x4 emitWorldMat;

					emitWorldMat = Mix::Matrix4x4::Identity();

					for( UInt32 i = 0; i < SOUL_EMIT_NUM; i++ )
					{
						emitWorldPos = ( Mix::Vector3::Rand() * 2.0f - Mix::Vector3( 1.0f, 1.0f, 1.0f ) ) * SOUL_EMIT_RANGE;
						emitWorldPos.y += SOUL_EMIT_RANGE.y + SOUL_EMIT_OFFSET_Y;

						emitWorldMat.SetTranslation( emitWorldPos );

						pSoulParticle->Emit( emitWorldMat );
					}

					soulTime = 0.0f;
				}

				soulTime += dt;
			}

			////////////////////////////////////////////////////////////////////////////////////////////////////
			// XV
			////////////////////////////////////////////////////////////////////////////////////////////////////

			if( pSoulParticle != NULL )
			{
				pSoulParticle->Update( dt );
			}

			if( pCamera != NULL )
			{
				pCamera->Update();
			}

			if( pSceneRenderer != NULL )
			{
				pSceneRenderer->Update( dt, pEngine->GetBaseDT() );
			}

			if( pCanvasRenderer != NULL )
			{
				pCanvasRenderer->Update();
			}

			////////////////////////////////////////////////////////////////////////////////////////////////////
			// tbV
			////////////////////////////////////////////////////////////////////////////////////////////////////

			if( pSoulParticle != NULL )
			{
				pSoulParticle->Refresh();
			}

			if( pSceneRenderer != NULL )
			{
				pSceneRenderer->Refresh();
			}

			////////////////////////////////////////////////////////////////////////////////////////////////////
			// `
			////////////////////////////////////////////////////////////////////////////////////////////////////

			pGraphicsDev->SetViewBounds( 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT );

			if( pGraphicsDev->Begin() == True )
			{
				if( pCamera != NULL )
				{
					pCamera->Draw();
				}

				if( pCanvasRenderer != NULL )
				{
					pCanvasRenderer->Draw();
				}

				pGraphicsDev->End();
			}

			pGraphicsDev->Present();
		}

		viewCamHelper.Dispose();

		MIX_RELEASE( pSoulParticle );
		MIX_RELEASE( pSoulParticleSystem );
		MIX_RELEASE( pSoulMaterial );

		MIX_RELEASE( pCamera );
		MIX_RELEASE( pSceneRenderer );
		MIX_RELEASE( pSceneEffectPackage );
		MIX_RELEASE( pFont );
		MIX_RELEASE( pCanvasRenderer );
		MIX_RELEASE( pGraphicsDev );
		MIX_RELEASE( pKeyboard );
		MIX_RELEASE( pMouse );

		Mix::Finalize();
	}

	return 0;
}
