#include "UniversalCameraHelper.h"

#include "Mix/HID/IKeyboard.h"
#include "Mix/HID/IMouse.h"

namespace Utility{

const Mix::Vector3 UniversalCameraHelper::UP_VEC( 0.0f, 1.0f, 0.0f );

UniversalCameraHelper::UniversalCameraHelper( void ) :
m_pKeyboard( NULL ),
m_pMouse( NULL ),
m_LinCoiffe( 0.0f ),
m_RotCoiffe( 0.0f )
{
}

UniversalCameraHelper::~UniversalCameraHelper( void )
{
	Dispose();
}

void UniversalCameraHelper::Initialize(	Mix::HID::IKeyboard* pKeyboard,
										Mix::HID::IMouse* pMouse,
										const Mix::Vector3& pos,
										const Mix::Vector3& target,
										Float32 linCoiffe,
										Float32 rotCoiffe )
{
	MIX_ASSERT( pKeyboard != NULL );
	MIX_ASSERT( pMouse != NULL );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_ADD_REF( pKeyboard );
	m_pKeyboard = pKeyboard;

	MIX_ADD_REF( pMouse );
	m_pMouse = pMouse;

	m_LinCoiffe = linCoiffe;
	m_RotCoiffe = rotCoiffe;

	m_PrePos = pos;
	m_Pos = pos;
	m_Dir = ( target - pos ).ToNormalize();
	m_Vel.Set( 0.0f, 0.0f, 0.0f );
	m_At = m_Pos + m_Dir;
}

void UniversalCameraHelper::Update( void )
{
	Mix::Vector3 crossDir = Mix::Vector3::Cross( UniversalCameraHelper::UP_VEC, m_Dir );
	Mix::Vector2 mouseVel = m_pMouse->GetVelocity();

	if( ( m_pKeyboard->GetKeyState( Mix::HID::KEY_W ) & Mix::HID::DOWN ) == Mix::HID::DOWN )
	{
		m_Pos += m_Dir * m_LinCoiffe;
	}
	else if( ( m_pKeyboard->GetKeyState( Mix::HID::KEY_S ) & Mix::HID::DOWN ) == Mix::HID::DOWN )
	{
		m_Pos -= m_Dir * m_LinCoiffe;
	}

	if( ( m_pKeyboard->GetKeyState( Mix::HID::KEY_A ) & Mix::HID::DOWN ) == Mix::HID::DOWN )
	{
		m_Pos -= crossDir * m_LinCoiffe;
	}
	else if( ( m_pKeyboard->GetKeyState( Mix::HID::KEY_D ) & Mix::HID::DOWN ) == Mix::HID::DOWN )
	{
		m_Pos += crossDir * m_LinCoiffe;
	}

	if( mouseVel.GetLengthSqr() > 0.0f )
	{
		if( ( m_pMouse->GetButtonState( 2 ) & Mix::HID::DOWN ) == Mix::HID::DOWN )
		{
			Mix::Vector3 upDir = Mix::Vector3::Cross( m_Dir, crossDir );
			Mix::Vector3 yAxis( 0.0f, 1.0f, 0.0f );
			Mix::Vector3 xAxis( 1.0f, 0.0f, 0.0f );
			Mix::Matrix4x4 mat;
			Mix::Quaternion rot;

			mat.SetRow( 0, Mix::Vector4( crossDir.x, crossDir.y, crossDir.z, 0.0f ) );
			mat.SetRow( 1, Mix::Vector4( upDir.x, upDir.y, upDir.z, 0.0f ) );
			mat.SetRow( 2, Mix::Vector4( m_Dir.x, m_Dir.y, m_Dir.z, 0.0f ) );
			mat.SetRow( 3, Mix::Vector4( 0.0f, 0.0f, 0.0f, 1.0f ) );

			yAxis = mat * yAxis;
			xAxis = mat * xAxis;

			rot.SetRotationAxis( yAxis, mouseVel.x * m_RotCoiffe );
			rot.RotationAxis( xAxis, mouseVel.y * m_RotCoiffe );
			rot.Normalize();

			m_Dir = Mix::Matrix4x4( rot ) * m_Dir;
			m_Dir.Normalize();
		}
	}

	m_At = m_Pos + m_Dir;
	m_Vel = m_Pos - m_PrePos;
	m_PrePos = m_Pos;
}

void UniversalCameraHelper::Refresh( const Mix::Vector3& eye )
{
	m_PrePos = eye;
	m_Pos = eye;
	m_At = m_Pos + m_Dir;
	m_Vel = Mix::Vector3::Zero();
}

const Mix::Vector3& UniversalCameraHelper::GetEye( void ) const
{
	return m_Pos;
}

const Mix::Vector3& UniversalCameraHelper::GetAt( void ) const
{
	return m_At;
}

const Mix::Vector3& UniversalCameraHelper::GetUp( void ) const
{
	return UniversalCameraHelper::UP_VEC;
}

const Mix::Vector3& UniversalCameraHelper::GetVelocity( void ) const
{
	return m_Vel;
}

void UniversalCameraHelper::Dispose( void )
{
	MIX_RELEASE( m_pKeyboard );
	MIX_RELEASE( m_pMouse );
}

}
