/*
    avicore
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "avibase.h"


/******************************************************************************
*                                                                             *
* ja:編集関数                                                                 *
*                                                                             *
******************************************************************************/
/*  ja:AVIを削除する
    avi_edit,AVI編集ハンドル
       start,削除をはじめるサンプル番号
     samples,サンプル数
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_delete (AviEdit    *avi_edit,
            const gint  start,
            const gint  samples)
{
  gint i, j = 0, samples_n;
  AviFile *p, *q;

  /* ja:スタート位置とサンプル数の調整 */
  if (!avi_edit || start < 0)
    return FALSE;
  samples_n = MIN (avi_length (avi_edit) - start, samples);
  if (samples_n <= 0 || avi_length (avi_edit) <= samples_n)
    return FALSE;
  /* ja:キャッシュ */
  for (i = 0; i < AVI_EDIT_CACHE && avi_edit->buf[i].data; i++)
    if (start <= avi_edit->buf[i].pos
                                && avi_edit->buf[i].pos < start + samples_n)
      {
        g_free (avi_edit->buf[i].data);
        avi_edit->buf[i].data = NULL;
      }
    else if (start + samples_n <= avi_edit->buf[i].pos)
      {
        avi_edit->buf[i].pos -= samples_n;
      }
  for (i = 0; i < AVI_EDIT_CACHE; i++)
    if (avi_edit->buf[i].data)
      avi_edit->buf[j++] = avi_edit->buf[i];
  while (j < AVI_EDIT_CACHE)
    avi_edit->buf[j++].data = NULL;

  avi_split_file (avi_edit, start);/* ja:AVIファイルを分割する */
  p = avi_edit->file;
  if (start + samples_n < avi_length (avi_edit))
    {
      /* ja:削除する位置で分割する */
      avi_split_file (avi_edit, start + samples_n);
      q = avi_edit->file;
    }
  else
    {
      /* ja:末尾まで削除する */
      q = NULL;
    }
  /* ja:削除位置よりも前に移動する */
  while (start - 1 < avi_edit->offset && avi_edit->file->prev)
    {
      avi_edit->file = avi_edit->file->prev;
      avi_edit->offset -= avi_edit->file->length;
    }
  while (p != q)
    {
      AviFile *r;

      if (avi_edit->file == p)
        avi_edit->file = avi_edit->file->next;
      if (p->prev)
        p->prev->next = p->next;
      if (p->next)
        p->next->prev = p->prev;
      r = p->next;
      avi_release_file (p);
      p = r;
    }
  /* ja:サンプル数を減らす */
  avi_edit->length -= samples_n;
  return TRUE;
}


/*  ja:AVIをコピーする
    avi_edit,AVI編集ハンドル
       start,コピーをはじめるサンプル番号
     samples,サンプル数
         RET,AVI編集ハンドル,NULL:エラー                                    */
AviEdit *
avi_copy (AviEdit    *avi_edit,
          const gint  start,
          const gint  samples)
{
  gint start_n, samples_n;
  AviEdit *avi_copy;

  /* ja:スタート位置とサンプル数の調整 */
  if (!avi_edit || start < 0)
    return NULL;
  samples_n = MIN (avi_length (avi_edit) - start, samples);
  if (samples_n <= 0 || avi_length (avi_edit) < samples_n)
    return NULL;

  avi_get_file (avi_edit, start);/* ja:AVIファイルを求める */

  /* ja:メモリ確保 */
  avi_copy = g_malloc0 (sizeof (AviEdit));
  avi_copy->type = avi_type (avi_edit);
  avi_set_rate (avi_copy, avi_get_rate (avi_edit));
  avi_set_scale (avi_copy, avi_get_scale (avi_edit));
                                        /* ja:rate÷scale=1秒間のフレーム数 */
  avi_set_priority (avi_copy, avi_get_priority (avi_edit));
  avi_set_language (avi_copy, avi_get_language (avi_edit));
  avi_set_name (avi_copy, avi_get_name (avi_edit));
  /* ja:ビデオ固有 */                   /* ja:展開後のフォーマット */
  if (avi_edit->bmih)
    avi_copy->bmih = g_memdup (avi_edit->bmih,
                                            bm_header_bytes (avi_edit->bmih));
  /* ja:オーディオ固有 */               /* ja:展開後のフォーマット */
  if (avi_edit->wfx)
    avi_copy->wfx = g_memdup (avi_edit->wfx, wf_header_bytes (avi_edit->wfx));

  start_n = start;
  while (samples_n > 0)
    {
      AviFile *avi_file;

      /* ja:コピー */
      avi_file = g_malloc (sizeof (AviFile));

      avi_file->fio = avi_edit->file->fio
                    ? fileio_open (avi_edit->file->name, FILEIO_ACCESS_READ,
                        FILEIO_SHARE_READ, FILEIO_MODE_OPEN_EXISTING) : NULL;
      avi_file->data = avi_edit->file->data && avi_edit->file->bmih
                                ? g_memdup (avi_edit->file->data,
                                bm_image_bytes (avi_edit->file->bmih)) : NULL;
      avi_file->name = g_strdup (avi_edit->file->name);
      avi_file->start = start_n - avi_edit->offset + avi_edit->file->start;
      avi_file->length
                = MIN (avi_edit->file->length - (start_n - avi_edit->offset),
                                                                    samples_n);
      avi_file->entries = avi_edit->file->entries;
      avi_file->param = avi_edit->file->param;
      avi_file->index = g_memdup (avi_edit->file->index,
                                avi_edit->file->entries * sizeof (AviIndex));
      avi_file->handler = avi_edit->file->handler;
      avi_file->bmih = avi_edit->file->bmih ? g_memdup (avi_edit->file->bmih,
                                bm_header_bytes (avi_edit->file->bmih)) : NULL;
      avi_file->wfx = avi_edit->file->wfx ? g_memdup (avi_edit->file->wfx,
                                wf_header_bytes (avi_edit->file->wfx)) : NULL;
      /* ja:リストに挿入する */
      avi_copy->length += avi_file->length;
      avi_file->prev = avi_copy->file;
      avi_file->next = NULL;
      if (avi_copy->file)
        {
          avi_copy->file->next = avi_file;
          avi_copy->offset += avi_copy->file->length;
        }
      avi_copy->file = avi_file;
      /* ja:次へ */
      if (avi_edit->file->next)
        {
          avi_edit->offset += avi_edit->file->length;
          avi_edit->file = avi_edit->file->next;
        }
      start_n += avi_file->length;
      samples_n -= avi_file->length;
    }
  return avi_copy;
}


/*  ja:AVIを貼り付ける
    avi_dst,貼り付けられるAVI編集ハンドル
        pos,貼り付けるサンプル番号
    avi_src,貼り付けるAVI編集ハンドル
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
avi_paste (AviEdit    *avi_dst,
           const gint  pos,
           AviEdit    *avi_src)
{
  gint i;
  AviFile *p, *q;

  /* ja:違う種類または範囲外ならばエラー */
  if (!avi_dst || !avi_src || avi_type (avi_dst) != avi_type (avi_src)
                                    || pos < 0 || avi_length (avi_dst) < pos)
    return FALSE;
  /* ja:キャッシュ */
  for (i = 0; i < AVI_EDIT_CACHE && avi_dst->buf[i].data; i++)
    if (pos <= avi_dst->buf[i].pos)
      avi_dst->buf[i].pos += avi_length (avi_src);
  if (pos < avi_length (avi_dst))
    {
      /* ja:貼り付ける位置で分割する */
      avi_split_file (avi_dst, pos);
      p = avi_dst->file->prev;
      q = avi_dst->file;
    }
  else
    {
      /* ja:末尾に貼り付ける */
      while (avi_dst->file->next)
        {
          avi_dst->offset += avi_dst->file->length;
          avi_dst->file = avi_dst->file->next;
        }
      p = avi_dst->file;
      q = NULL;
    }
  /* ja:先頭へ移動する */
  while (avi_src->file->prev)
    avi_src->file = avi_src->file->prev;
  avi_src->file->prev = p;
  if (p)
    p->next = avi_src->file;
  /* ja:末尾へ移動する */
  while (avi_src->file->next)
    avi_src->file = avi_src->file->next;
  avi_src->file->next = q;
  if (q)
    q->prev = avi_src->file;
  /* ja:リストのオフセットを設定する */
  if (pos <= avi_dst->offset)
    avi_dst->offset += avi_length (avi_src);
  /* ja:サンプル数の増加と貼り付け元の破棄 */
  avi_dst->length += avi_length (avi_src);
  g_free (avi_src->name);
  g_free (avi_src->bmih);
  for (i = 0; i < AVI_EDIT_CACHE; i++)
    g_free (avi_src->buf[i].data);
  g_free (avi_src->wfx);
  g_free (avi_src);
  /* ja:最適化 */
  if (p && p->name && p->next->name
                                && g_strfilecmp (p->name, p->next->name) == 0
                                    && p->start + p->length == p->next->start)
    {
      p->length += p->next->length;
      p = p->next;
      if (avi_dst->file == p)
        {
          avi_dst->file = avi_dst->file->prev;
          avi_dst->offset -= avi_dst->file->length;
        }
      if (p->prev)
        p->prev->next = p->next;
      if (p->next)
        p->next->prev = p->prev;
      avi_release_file (p);
    }
  if (q && q->name && q->prev->name
                                && g_strfilecmp (q->name, q->prev->name) == 0
                            && q->prev->start + q->prev->length == q->start)
    {
      if (avi_dst->file == q)
        {
          avi_dst->file = avi_dst->file->prev;
          avi_dst->offset -= avi_dst->file->length;
        }
      q->prev->length += q->length;
      if (q->prev)
        q->prev->next = q->next;
      if (q->next)
        q->next->prev = q->prev;
      avi_release_file (q);
    }
  return TRUE;
}
