/*
    avicore
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "avibase.h"
#include "avifmt.h"
#include "chunk.h"


/******************************************************************************
*                                                                             *
* ja:AVI入力関数                                                              *
*                                                                             *
******************************************************************************/
/*  ja:ファイルを判別する
    file,ファイル名
     RET,AVI_FILE_*                                                         */
gint
avi_file_type (const gchar *file)
{
  FileIO *fio;
  guint8 buffer[12];            /* ja:ファイル判別用のバッファ */

  /* ja:ファイルを開く */
  if (!file || !(fio = fileio_open (file, FILEIO_ACCESS_READ,
                                FILEIO_SHARE_READ, FILEIO_MODE_OPEN_EXISTING)))
    return AVI_FILE_UNKNOWN;
  /* ja:ヘッタ部分の読み込み */
  if (fileio_read (fio, buffer, sizeof (guint8) * 12) != sizeof (guint8) * 12)
    {
      fileio_close (fio);
      return AVI_FILE_UNKNOWN;
    }
  if (!fileio_close (fio))
    return AVI_FILE_UNKNOWN;
  /* ja:ファイルの形式毎に分岐 */
  if (buffer[0] == 'R' && buffer[1] == 'I'
                                    && buffer[2] == 'F' && buffer[3] == 'F'
                                    && buffer[8] == 'A' && buffer[9] == 'V'
                                    && buffer[10] == 'I' && buffer[11] == ' ')
    return AVI_FILE_AVI;/* en:AVI */
  else if (buffer[0] == 'B' && buffer[1] == 'M')
    return AVI_FILE_BITMAP;/* ja:ビットマップ */
  else if (buffer[0] == 'R' && buffer[1] == 'I'
                                    && buffer[2] == 'F' && buffer[3] == 'F'
                                    && buffer[8] == 'W' && buffer[9] == 'A'
                                    && buffer[10 ]== 'V' && buffer[11] == 'E')
    return AVI_FILE_WAVE;/* ja:WAVEファイル */
  else if (buffer[0] == '#' && buffer[1] == 'V' && buffer[2] == 'M'
                                    && buffer[3] == 'A' && buffer[4] == 'I'
                                    && buffer[5] == 'D' && buffer[6] == '\n')
    return AVI_FILE_SCENARIO;/* ja:シナリオオブジェクト */
  return AVI_FILE_UNKNOWN;
}


/*  ja:AVIのストリームの数を取得する
    file,ファイル名
     RET,ストリームの数,0:エラー                                            */
gint
avi_main_streams (const gchar *file)
{
  FileIO *fio;
  guint8 buffer[12];            /* ja:ファイル判別用のバッファ */
  AviMainHeader amh;
  Chunk *ck;

  /* ja:ファイルを開く */
  if (!file || !(fio = fileio_open (file, FILEIO_ACCESS_READ,
                                FILEIO_SHARE_READ, FILEIO_MODE_OPEN_EXISTING)))
    return 0;
  /* ja:ヘッタ部分の読み込み */
  if (fileio_read (fio, buffer, sizeof (guint8) * 12) != sizeof (guint8) * 12)
    {
      fileio_close (fio);
      return 0;
    }
  if (buffer[0] != 'R' || buffer[1] != 'I'
                                    || buffer[2] != 'F' || buffer[3] != 'F'
                                    || buffer[8] != 'A' || buffer[9] != 'V'
                                    || buffer[10] != 'I' || buffer[11] != ' ')
    {
      fileio_close (fio);
      return 0;
    }
  /* ja:ファイルディスクリターからチャンクポインタを作る */
  ck = chunk_open (fio);
  /* ja:RIFFチャンクに入る */
  if (!chunk_in (fio, ck))
    {
      chunk_free (ck);
      fileio_close (fio);
      return 0;
    }
  /* ja:フォームがhdrlのLISTチャンクを探す */
  while (chunk_form (fio) != CK_LIST_AVIHEADER)
    if (!chunk_next (fio, ck))
      {
        chunk_free (ck);
        fileio_close (fio);
        return 0;
      }
  /* ja:フォームがhdrlのLISTチャンクに入る(RIFF'AVI ' -> LIST'hdrl') */
  if (!chunk_in (fio, ck))
    {
      chunk_free (ck);
      fileio_close (fio);
      return 0;
    }
  /* ja:avihチャンクを探す(AviMainHeader) */
  while (chunk_id (fio) != CK_ID_AVIMAINHDR)
    if (!chunk_next (fio, ck))
      {
        chunk_free (ck);
        fileio_close (fio);
        return 0;
      }
  /* ja:avihチャンクのサイズ(AviMainHeader) */
  /* ja:avihチャンクに入る(AviMainHeader) (RIFF'AVI ' -> LIST'hdrl' -> avih) */
  /* ja:AviMainHeaderを読み込む */
  if (chunk_size (fio) != AMH_SIZE || !chunk_in (fio, ck)
                            || fileio_read (fio, &amh, AMH_SIZE) != AMH_SIZE)
    {
      chunk_free (ck);
      fileio_close (fio);
      return 0;
    }
  /* ja:チャンクポインタを解放する */
  return chunk_free (ck) & fileio_close (fio) ? amh_get_streams (&amh) : 0;
}


/*  ja:AVIをファイルから開く
     file,ファイル名
    param,ストリーム番号
      RET,AVI編集ハンドル,NULL:エラー                                       */
AviEdit *
avi_open (const gchar *file,
          const gint   param)
{
  AviEdit *avi_edit;

  if (!file)
    return NULL;
  avi_edit = g_malloc0 (sizeof (AviEdit));      /* ja:メモリ確保 */
  avi_edit->file = avi_open_file (file, param,
                    &avi_edit->type, &avi_edit->rate, &avi_edit->scale,
                    &avi_edit->priority, &avi_edit->language, &avi_edit->name);
  if (!avi_edit->file)
    {
      g_free (avi_edit);
      return NULL;
    }
  /* ja:展開後のフォーマット */
  if (avi_type (avi_edit) == AVI_TYPE_VIDEO)
    {
      if (avi_edit->file->handler == AVI_STREAM_COMP_DIB
                            || avi_edit->file->handler == BI_COMP_RGB
                            || avi_edit->file->handler == BI_COMP_RLE8
                            || avi_edit->file->handler == BI_COMP_RLE4
                            || avi_edit->file->handler == BI_COMP_BITFIELDS)
        {
          avi_edit->file->handler = AVI_STREAM_COMP_DIB;
          avi_edit->bmih = g_memdup (avi_edit->file->bmih,
                                    bm_header_bytes (avi_edit->file->bmih));
          if (bmih_get_compression (avi_edit->bmih) == AVI_STREAM_COMP_DIB)
            bmih_set_compression (avi_edit->bmih, BI_COMP_RGB);
          if (bmih_get_size_image (avi_edit->bmih) <= 0)
            bmih_set_size_image (avi_edit->bmih,
                                            bm_image_bytes (avi_edit->bmih));
        }
      else
        {
          IcmObject *icm_object;

          /* ja:圧縮 */
          icm_object = icm_open (avi_edit->file->handler,
                                                        ICM_MODE_DECOMPRESS);
          if (!icm_object)
            {
              avi_release_file (avi_edit->file);
              g_free (avi_edit);
              return NULL;
            }
          avi_edit->bmih = g_malloc (icm_decompress_get_format_size
                                        (icm_object, avi_edit->file->bmih));
          if (!icm_decompress_get_format (icm_object,
                                        avi_edit->file->bmih, avi_edit->bmih)
                                                    | !icm_close (icm_object))
            {
              avi_release_file (avi_edit->file);
              g_free (avi_edit->bmih);
              g_free (avi_edit);
              return NULL;
            }
          avi_edit->bmih = g_realloc (avi_edit->bmih,
                                            bm_header_bytes (avi_edit->bmih));
        }
    }
  else
    {
      if (wfx_get_format_tag (avi_edit->file->wfx) == WAVE_FORMAT_PCM)
        {
          avi_edit->wfx = g_memdup (avi_edit->file->wfx,
                                        wf_header_bytes (avi_edit->file->wfx));
        }
      else
        {
          /* ja:圧縮 */
          avi_release_file (avi_edit->file);
          g_free (avi_edit);
          return NULL;
        }
    }
  /* ja:情報 */
  avi_edit->length = avi_edit->file->length;
  return avi_edit;
}


/*  ja:AVIを閉じる
    avi_edit,AVI編集ハンドル
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_release (AviEdit *avi_edit)
{
  gint i;

  if (!avi_edit)
    return FALSE;
  while (avi_edit->file)
    {
      AviFile *avi_file;

      avi_file = avi_edit->file->next;
      avi_release_file (avi_edit->file);
      avi_edit->file = avi_file;
    }
  g_free (avi_edit->name);
  g_free (avi_edit->bmih);
  for (i = 0; i < AVI_EDIT_CACHE; i++)
    g_free (avi_edit->buf[i].data);
  g_free (avi_edit->wfx);
  g_free (avi_edit);
  return TRUE;
}


#ifdef DEPRECATED
/*  ja:AVIのデータのバイト数を取得する
    avi_edit,AVI編集ハンドル
       start,読み込みをはじめるサンプル番号
     samples,サンプル数
         RET,バイト数                                                       */
gint
avi_sample_size (AviEdit    *avi_edit,
                 const gint  start,
                 const gint  samples)
{
  gint i, length, samples_n;

  /* ja:スタート位置とサンプル数の調整 */
  if (!avi_edit || start < 0)
    return 0;
  samples_n = MIN (avi_length (avi_edit) - start, samples);
  if (samples_n <= 0)
    return 0;
  switch (avi_type (avi_edit))
    {
      case AVI_TYPE_VIDEO:/* ja:ビデオ */
        avi_get_file (avi_edit, start);/* ja:AVIファイルを求める */
        length = 0;
        for (i = start; i < start + samples_n; i++)
          {
            if (i - avi_edit->offset >= avi_edit->file->length)
              {
                avi_edit->offset += avi_edit->file->length;
                avi_edit->file = avi_edit->file->next;
              }
            length += avi_edit->file->
                    index[i - avi_edit->offset + avi_edit->file->start].size;
          }
        return length;
      case AVI_TYPE_AUDIO:/* ja:オーディオ */
        return wfx_get_block_align (avi_edit->wfx) * samples_n;
    }
  return 0;
}


/*  ja:AVIのデータを読み込む
    avi_edit,AVI編集ハンドル
       start,読み込みをはじめるサンプル番号
     samples,サンプル数
      buffer,データを格納するバッファ
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_read (AviEdit    *avi_edit,
          const gint  start,
          const gint  samples,
          gpointer    buffer)
{
  gint i, dif, length, start_n, samples_n;
  guint32 size;
  goffset offset;

  /* ja:スタート位置とサンプル数の調整 */
  if (!avi_edit || samples <= 0 || start < 0
                                    || avi_length (avi_edit) < start + samples)
    return FALSE;
  avi_get_file (avi_edit, start);/* ja:AVIファイルを求める */
  switch (avi_type (avi_edit))
    {
      case AVI_TYPE_VIDEO:/* ja:ビデオ */
        for (i = start; i < start + samples; i++)
          {
            if (i - avi_edit->offset >= avi_edit->file->length)
              {
                avi_edit->offset += avi_edit->file->length;
                avi_edit->file = avi_edit->file->next;
              }
            offset = avi_edit->file->
                    index[i - avi_edit->offset + avi_edit->file->start].offset;
            size = avi_edit->file->
                    index[i - avi_edit->offset + avi_edit->file->start].size;
            if (!avi_edit->file->data)
              {
                /* ja:ファイル */
                if (fileio_seek (avi_edit->file->fio, offset,
                                                        FILEIO_SEEK_SET) == -1
                    || fileio_read (avi_edit->file->fio, buffer, size) != size)
                  return FALSE;
              }
            else
              {
                /* ja:メモリ */
                g_memmove (buffer, (guint8 *)avi_edit->file->data + offset,
                                                                        size);
              }
            (guint8 *)buffer += size;
          }
        return TRUE;
      case AVI_TYPE_AUDIO:/* ja:オーディオ */
        length = start - avi_edit->offset + avi_edit->file->start;
        dif = 0;
        for (i = 0; dif <= length; i++)
          dif += avi_edit->file->index[i].size
                                        / wfx_get_block_align (avi_edit->wfx);
        dif -= avi_edit->file->index[--i].size
                                        / wfx_get_block_align (avi_edit->wfx);
        dif = length - dif;
        start_n = start;
        samples_n = samples;
        while (samples_n > 0)
          {
            if (avi_edit->offset + avi_edit->file->length <= start_n)
              {
                avi_edit->offset += avi_edit->file->length;
                avi_edit->file = avi_edit->file->next;
                dif = 0;
                for (i = 0; dif <= avi_edit->file->start; i++)
                  dif += avi_edit->file->index[i].size
                                        / wfx_get_block_align (avi_edit->wfx);
                dif -= avi_edit->file->index[--i].size
                                        / wfx_get_block_align (avi_edit->wfx);
                dif = length - dif;
              }
            offset = (goffset)dif * wfx_get_block_align (avi_edit->wfx);
            size = MIN (wfx_get_block_align (avi_edit->wfx) * samples_n,
                                    avi_edit->file->index[i].size - offset);
            offset += avi_edit->file->index[i].offset;
            if (!avi_edit->file->data)
              {
                /* ja:ファイル */
                if (fileio_seek (avi_edit->file->fio, offset,
                                                        FILEIO_SEEK_SET) == -1
                    || fileio_read (avi_edit->file->fio, buffer, size) != size)
                  return FALSE;
              }
            else
              {
                /* ja:メモリ */
                g_memmove (buffer, (guint8 *)avi_edit->file->data + offset,
                                                                        size);
              }
            (guint8 *)buffer += size;
            start_n += size / wfx_get_block_align (avi_edit->wfx);
            samples_n -= size / wfx_get_block_align (avi_edit->wfx);
            dif = 0;
            i++;
          }
        return TRUE;
    }
  return FALSE;
}


/*  ja:AVIのフォーマットのサイズを取得する
    avi_edit,AVI編集ハンドル
         RET,フォーマットのバイト数,0:エラー                                */
gint
avi_format_size (AviEdit *avi_edit)
{
  if (avi_edit)
    switch (avi_type (avi_edit))
      {
        case AVI_TYPE_VIDEO:/* ja:ビデオ */
          return bm_header_bytes (avi_edit->bmih);
        case AVI_TYPE_AUDIO:/* ja:オーディオ */
          return wf_header_bytes (avi_edit->wfx);
      }
  return 0;
}


/*  ja:AVIのフォーマットを読み込む
    avi_edit,AVI編集ハンドル
      format,バッファへのポインタ
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_read_format (AviEdit  *avi_edit,
                 gpointer  format)
{
  if (avi_edit)
    switch (avi_type (avi_edit))
      {
        case AVI_TYPE_VIDEO:/* ja:ビデオ */
          g_memmove (format, avi_edit->bmih, bm_header_bytes (avi_edit->bmih));
          return TRUE;
        case AVI_TYPE_AUDIO:/* ja:オーディオ */
          g_memmove (format, avi_edit->wfx, wf_header_bytes (avi_edit->wfx));
          return TRUE;
      }
  return FALSE;
}
#endif /* DEPRECATED */
