/*
    avicore
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "avibase.h"


struct _AviPcm
{
  gpointer tmp_buf;
  gint tmp_samples;
  gint pos;             /* ja:前回の終了点 */
  gint marker;          /* ja:読み込みの終了点 */
  AviEdit *avi_edit;
  WaveConv *wcv;
  WaveFormatEx *wfx;
};


/******************************************************************************
*                                                                             *
* ja:PCM関数                                                                  *
*                                                                             *
******************************************************************************/
/*  ja:PCMに展開されたオーディオを開く
    avi_edit,AVI編集ハンドル
         RET,AVIPCM構造体,NULL:エラー                                       */
AviPcm *
avi_get_pcm_open (AviEdit *avi_edit)
{
  AviPcm *avi_pcm;

  if (!avi_edit || avi_type (avi_edit) != AVI_TYPE_AUDIO)
    return NULL;
  avi_pcm = g_malloc0 (sizeof (AviPcm));
  avi_pcm->avi_edit = avi_edit;
  avi_pcm->pos = -1;
  avi_pcm->marker = -1;
  return avi_pcm;
}


/*  ja:PCMに展開されたオーディオを閉じる
    avi_pcm,AVIPCM構造体
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
avi_get_pcm_close (AviPcm *avi_pcm)
{
  gboolean result;

  if (!avi_pcm)
    return FALSE;
  result = !avi_pcm->wcv || wave_convert_end (avi_pcm->wcv, NULL, NULL);
  g_free (avi_pcm->tmp_buf);
  g_free (avi_pcm->wfx);
  g_free (avi_pcm);
  return result;
}


/*  ja:オーディオデータを取得する
       avi_file,AVIファイル構造体
          start,取得するサンプル番号
        samples,サンプル数
        out_buf,オーディオデータ
    out_samples,オーディオデータの実サンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
static gboolean
avi_get_pcm_raw_data (AviFile    *avi_file,
                      const gint  start,
                      const gint  samples,
                      gpointer   *out_buf,
                      gint       *out_samples)
{
  gpointer tmp_buf;
  gint i, dif, real_start, real_samples, tmp_bytes = 0, tmp_samples = 0;

  if (start < 0 || avi_file->length < start + samples || samples <= 0)
    return FALSE;
  real_start = (glonglong)(avi_file->start + start)
        * wfx_get_samples_per_sec (avi_file->wfx) / AVI_PCM_SAMPLES_PER_SEC;
  real_samples = (glonglong)samples
        * wfx_get_samples_per_sec (avi_file->wfx) / AVI_PCM_SAMPLES_PER_SEC;
  if (real_samples < 1)
    real_samples = 1;
  /* ja:データの先頭があるエントリーを求める */
  dif = 0;/* ja:先頭のエントリーの読み飛ばすサンプル数 */
  for (i = 0; i < avi_file->entries; i++)
    {
      gint n;

      n = avi_file->index[i].size / wfx_get_block_align (avi_file->wfx);
      if (real_start < dif + n)
        break;
      dif += n;
    }
  dif = real_start - dif;
  /* ja:バッファにファイルのサンプルを確保する */
  tmp_buf = g_malloc (real_samples * wfx_get_block_align (avi_file->wfx));
  while (tmp_samples < real_samples && i < avi_file->entries)
    {
      goffset off;
      gint read_samples, read_bytes;

      read_samples = MIN (real_samples - tmp_samples,
        (avi_file->index[i].size / wfx_get_block_align (avi_file->wfx)) - dif);
      read_bytes = read_samples * wfx_get_block_align (avi_file->wfx);
      off = (goffset)dif * wfx_get_block_align (avi_file->wfx)
                                                + avi_file->index[i].offset;
      if (!avi_file->data)
        {
          /* ja:ファイル */
          if (fileio_seek (avi_file->fio, off, FILEIO_SEEK_SET) == -1
                    || fileio_read (avi_file->fio,
                    (guint8 *)tmp_buf + tmp_bytes, read_bytes) != read_bytes)
            {
              g_free (tmp_buf);
              return FALSE;
            }
        }
      else
        {
          /* ja:メモリ */
          g_memmove ((guint8 *)tmp_buf + tmp_bytes,
                                (guint8 *)avi_file->data + off, read_bytes);
        }
      tmp_bytes += read_bytes;
      tmp_samples += read_samples;
      dif = 0;
      i++;
    }
  if (tmp_samples <= 0)
    {
      g_free (tmp_buf);
      return FALSE;
    }
  if (out_buf)
    *out_buf = tmp_buf;
  else
    g_free (tmp_buf);
  if (out_samples)
    *out_samples = tmp_samples;
  return TRUE;
}


/*  ja:バッファに追加する
        avi_pcm,AVIPCM構造体
            buf,追加するデータ
        samples,追加するデータの実サンプル数
    block_align,1サンプルのバイト数                                         */
static void
avi_get_pcm_add_buffer (AviPcm     *avi_pcm,
                        gpointer    buf,
                        const gint  samples,
                        guint16     block_align)
{
  avi_pcm->tmp_buf = g_realloc (avi_pcm->tmp_buf,
                            (avi_pcm->tmp_samples + samples) * block_align);
  g_memmove ((guint8 *)avi_pcm->tmp_buf + avi_pcm->tmp_samples * block_align,
                                                buf, samples * block_align);
  avi_pcm->tmp_samples += samples;
}


/*  ja:PCMに展開されたオーディオを取得する
        avi_pcm,AVIPCM構造体
          start,取得するサンプル番号
        samples,サンプル数
        out_buf,PCMデータ
    out_samples,PCMデータの実サンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
avi_get_pcm (AviPcm     *avi_pcm,
             const gint  start,
             const gint  samples,
             gpointer   *out_buf,
             gint       *out_samples)
{
  gpointer result_buf;
  gint real_samples, virtual_samples, result_samples, result_bytes, tmp_bytes;
  AviEdit *avi_edit;

  if (!avi_pcm)
    return FALSE;
  avi_edit = avi_pcm->avi_edit;
  if (start < 0 || avi_length (avi_edit) < start + samples || samples <= 0)
    return FALSE;                   /* ja:範囲外 */
  real_samples = (glonglong)samples * wfx_get_samples_per_sec (avi_edit->wfx)
                                                    / AVI_PCM_SAMPLES_PER_SEC;
  virtual_samples = samples;
  if (avi_pcm->pos == start)
    {
      virtual_samples -= (glonglong)avi_pcm->tmp_samples
        * AVI_PCM_SAMPLES_PER_SEC / wfx_get_samples_per_sec (avi_edit->wfx);
    }
  else
    {
      gboolean result = TRUE;

      if (avi_pcm->wcv)
        {
          result = wave_convert_end (avi_pcm->wcv, NULL, NULL);
          avi_pcm->wcv = NULL;
        }
      g_free (avi_pcm->tmp_buf);
      avi_pcm->tmp_buf = NULL;
      avi_pcm->tmp_samples = 0;
      avi_pcm->marker = start;
      if (!result)
        return FALSE;
    }
  while (avi_pcm->tmp_samples < real_samples
                                    && avi_pcm->marker < avi_length (avi_edit))
    {
      gpointer o_buf, conv_buf;
      gint file_start, file_samples, o_samples, conv_samples;

      avi_get_file (avi_edit, avi_pcm->marker);/* ja:AVIファイルを求める */
      file_start = avi_pcm->marker - avi_edit->offset;
      file_samples = MIN (avi_edit->file->length - file_start,
                                                            virtual_samples);
      if (!avi_get_pcm_raw_data (avi_edit->file, file_start, file_samples,
                                                        &o_buf, &o_samples))
        {
          wave_convert_end (avi_pcm->wcv, NULL, NULL);
          avi_pcm->wcv = NULL;
          g_free (avi_pcm->tmp_buf);
          avi_pcm->tmp_buf = NULL;
          avi_pcm->tmp_samples = 0;
          avi_pcm->marker = -1;
          return FALSE;
        }
      avi_pcm->marker += file_samples;
      virtual_samples -= file_samples;
      if (virtual_samples <= 0)
        virtual_samples = AVI_PCM_SAMPLES_PER_SEC;
      /* ja:PCMを変換する */
      if (!avi_pcm->wcv
            || wf_header_bytes (avi_pcm->wfx)
                                    != wf_header_bytes (avi_edit->file->wfx)
            || g_memcmp (avi_pcm->wfx, avi_edit->file->wfx,
                                        wf_header_bytes (avi_pcm->wfx)) != 0)
        {
          if (avi_pcm->wcv)
            {
              gpointer e_buf;
              gint e_samples;

              if (!wave_convert_end (avi_pcm->wcv, &e_buf, &e_samples))
                {
                  avi_pcm->wcv = NULL;
                  g_free (avi_pcm->tmp_buf);
                  avi_pcm->tmp_buf = NULL;
                  avi_pcm->tmp_samples = 0;
                  avi_pcm->marker = -1;
                  return FALSE;
                }
              avi_get_pcm_add_buffer (avi_pcm, e_buf, e_samples,
                                        wfx_get_block_align (avi_edit->wfx));
            }
          g_free (avi_pcm->wfx);
          avi_pcm->wfx = g_memdup (avi_edit->file->wfx,
                                        wf_header_bytes (avi_edit->file->wfx));
          avi_pcm->wcv = wave_convert_begin (avi_edit->file->wfx,
                                                                avi_edit->wfx);
          if (!avi_pcm->wcv)
            {
              g_free (avi_pcm->tmp_buf);
              avi_pcm->tmp_buf = NULL;
              avi_pcm->tmp_samples = 0;
              avi_pcm->marker = -1;
              return FALSE;
            }
        }
      if (!wave_convert (avi_pcm->wcv,
                                o_buf, o_samples, &conv_buf, &conv_samples))
        {
          wave_convert_end (avi_pcm->wcv, NULL, NULL);
          avi_pcm->wcv = NULL;
          g_free (avi_pcm->tmp_buf);
          avi_pcm->tmp_buf = NULL;
          avi_pcm->tmp_samples = 0;
          avi_pcm->marker = -1;
          return FALSE;
        }
      avi_get_pcm_add_buffer (avi_pcm, conv_buf, conv_samples,
                                        wfx_get_block_align (avi_edit->wfx));
      /* ja:ファイルの末尾まできたとき */
      if (avi_length (avi_edit) <= avi_pcm->marker)
        {
          gpointer e_buf;
          gint e_samples;

          if (!wave_convert_end (avi_pcm->wcv, &e_buf, &e_samples))
            {
              avi_pcm->wcv = NULL;
              g_free (avi_pcm->tmp_buf);
              avi_pcm->tmp_buf = NULL;
              avi_pcm->tmp_samples = 0;
              avi_pcm->marker = -1;
              return FALSE;
            }
          avi_get_pcm_add_buffer (avi_pcm, e_buf, e_samples,
                                        wfx_get_block_align (avi_edit->wfx));
          avi_pcm->wcv = NULL;
          g_free (avi_pcm->wfx);
          avi_pcm->wfx = NULL;
          break;
        }
    }
  avi_pcm->pos = start + samples;
  result_samples = MIN (avi_pcm->tmp_samples, real_samples);
  result_bytes = result_samples * wfx_get_block_align (avi_edit->wfx);
  result_buf = g_malloc (result_bytes);
  g_memmove (result_buf, avi_pcm->tmp_buf, result_bytes);
  avi_pcm->tmp_samples -= result_samples;
  tmp_bytes = avi_pcm->tmp_samples * wfx_get_block_align (avi_edit->wfx);
  g_memmove (avi_pcm->tmp_buf,
                        (guint8 *)avi_pcm->tmp_buf + result_bytes, tmp_bytes);
  avi_pcm->tmp_buf = g_realloc (avi_pcm->tmp_buf, tmp_bytes);
  if (!result_buf)
    return FALSE;
  if (out_buf)
    *out_buf = result_buf;
  else
    g_free (result_buf);
  if (out_samples)
    *out_samples = result_samples;
  return TRUE;
}
