/*
    dllloader
    (C)Copyright 2000 by Hiroshi Takekawa
    copyright (c) 2002 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "advapi32.h"
#include "misc/profile.h"


/******************************************************************************
*                                                                             *
* 쥸ȥ                                                                  *
*                                                                             *
******************************************************************************/
#define ERROR_SUCCESS			0
#define ERROR_NO_MORE_ITEMS		259
#define ERROR_UNKNOWN			99999
#define HKEY_CLASSES_ROOT		0x80000000
#define HKEY_CURRENT_USER		0x80000001
#define HKEY_LOCAL_MACHINE		0x80000002
#define HKEY_USERS				0x80000003
#define HKEY_PERFORMANCE_DATA	0x80000004
#define HKEY_CURRENT_CONFIG		0x80000005
#define HKEY_DYN_DATA			0x80000006


typedef HANDLE HKEY,*PHKEY,*LPHKEY;
typedef DWORD REGSAM;


static const gchar *reg_name(HKEY hKey)
{
	gint i;
	static const struct {
		const guint32 key;
		const gchar *name;
	} reg_key[]={	{HKEY_CLASSES_ROOT,		"HKEY_CLASSES_ROOT"},
					{HKEY_CURRENT_USER,		"HKEY_CURRENT_USER"},
					{HKEY_LOCAL_MACHINE,	"HKEY_LOCAL_MACHINE"},
					{HKEY_USERS,			"HKEY_USERS"},
					{HKEY_PERFORMANCE_DATA,	"HKEY_PERFORMANCE_DATA"},
					{HKEY_CURRENT_CONFIG,	"HKEY_CURRENT_CONFIG"},
					{HKEY_DYN_DATA,			"HKEY_DYN_DATA"},
					{0,						NULL}};
	for (i=0;reg_key[i].name!=NULL;i++)
		if ((guint)hKey==reg_key[i].key)
			return reg_key[i].name;
	return hKey;
}


static LONG WINAPI RegCloseKey(HKEY hKey)
{
	g_debug("RegCloseKey");
	g_free(hKey);
	return ERROR_SUCCESS;
}


static LONG WINAPI RegCreateKeyA(HKEY hKey,LPCSTR lpSubKey,PHKEY phkResult)
{
	gchar *key;

	g_debug("RegCreateKeyA");
	if (hKey==NULL || phkResult==NULL)
		return ERROR_UNKNOWN;
	key=lpSubKey==NULL || lpSubKey[0]=='\0'
										?NULL:g_strconcat("\\",lpSubKey,NULL);
	*phkResult=g_strconcat(reg_name(hKey),key,NULL);
	g_free(key);
	return ERROR_SUCCESS;
}


static LONG WINAPI RegCreateKeyExA(HKEY hKey,LPCSTR lpSubKey,DWORD Reserved,
							LPSTR lpClass,DWORD dwOptions,REGSAM samDesired,
									LPSECURITY_ATTRIBUTES lpSecurityAttributes,
									PHKEY phkResult,LPDWORD lpdwDisposition)
{
	g_debug("RegCreateKeyExA");
	return RegCreateKeyA(hKey,lpSubKey,phkResult);
}


static LONG WINAPI RegDeleteKeyA(HKEY hKey,LPCSTR lpSubKey)
{
	gboolean result=FALSE;
	gchar *key,*name;
	gint i;
	GList *glist;
	Profile *profile;

	g_debug("RegDeleteKeyA");
	if (hKey==NULL)
		return ERROR_UNKNOWN;
	if ((profile=profile_open(NULL))!=NULL) {
		key=g_strconcat(hKey,"\\",lpSubKey,NULL);
		glist=profile_enum_key(profile,"registry");
		if (glist!=NULL) {
			for (i=g_list_length(glist)-1;i>=0;i--) {
				name=g_list_nth_data(glist,i);
				if (g_strncasecmp(name,key,g_strlen(key))==0)
					result=profile_delete_key(profile,"registry",name);
			}
			g_list_free(glist);
		}
		g_free(key);
		profile_close(profile);
	}
	return result?ERROR_SUCCESS:ERROR_UNKNOWN;
}


static LONG WINAPI RegDeleteValueA(HKEY hKey,LPCSTR lpValueName)
{
	gboolean result=FALSE;
	gchar *key;
	Profile *profile;

	g_debug("RegDeleteValueA");
	if (hKey==NULL)
		return ERROR_UNKNOWN;
	if ((profile=profile_open(NULL))!=NULL) {
		key=g_strconcat(reg_name(hKey),"\\",lpValueName,NULL);
		result=profile_delete_key(profile,"registry",key);
		g_free(key);
		profile_close(profile);
	}
	return result?ERROR_SUCCESS:ERROR_UNKNOWN;
}


static LONG WINAPI RegEnumKeyA(HKEY hKey,DWORD dwIndex,LPSTR lpName,
																DWORD cchName)
{
	return ERROR_UNKNOWN;
}


static LONG WINAPI RegEnumKeyExA(HKEY hKey,DWORD dwIndex,LPSTR lpName,
			LPDWORD lpcName,LPDWORD lpReserved,LPSTR lpClass,LPDWORD lpcClass,
												PFILETIME lpftLastWriteTime)
{
	return RegEnumKeyA(hKey,dwIndex,lpName,*lpcName);
}


static LONG WINAPI RegOpenKeyExA(HKEY hKey,LPCSTR lpSubKey,
							DWORD ulOptions,REGSAM samDesired,PHKEY phkResult)
{
	g_debug("RegOpenKeyExA");
	return RegCreateKeyA(hKey,lpSubKey,phkResult);
}


static LONG WINAPI RegQueryValueExA(HKEY hKey,LPCSTR lpValueName,
			LPDWORD lpReserved,LPDWORD lpType,LPBYTE lpData,LPDWORD lpcbData)
{
	gboolean result=FALSE;
	gchar *key;
	guint8 *data;
	gint size;
	Profile *profile;

	g_debug("RegQueryValueExA");
	if (hKey==NULL)
		return ERROR_UNKNOWN;
	if (lpcbData==NULL)
		return lpData==NULL?ERROR_SUCCESS:ERROR_UNKNOWN;
	if ((profile=profile_open(NULL))!=NULL) {
		key=g_strconcat(reg_name(hKey),"\\",lpValueName,NULL);
		if ((size=profile_get_size(profile,"registry",key,
									PROFILE_VALUE_TYPE_ARRAY))>sizeof(DWORD)) {
			if (lpData==NULL) {
				result=TRUE;
			} else if (size-sizeof(DWORD)<=*lpcbData) {
				data=g_malloc(size);
				result=profile_get_value(profile,"registry",key,
										data,size,PROFILE_VALUE_TYPE_ARRAY);
				g_memmove(lpData,data+sizeof(DWORD),size-sizeof(DWORD));
				if (lpType!=NULL)
					*lpType=*(LPDWORD)data;
				g_free(data);
			}
			*lpcbData=size-sizeof(DWORD);
		}
		g_free(key);
		profile_close(profile);
	}
	return result?ERROR_SUCCESS:ERROR_UNKNOWN;
}


static LONG WINAPI RegQueryValueA(HKEY hKey,LPCSTR lpSubKey,
												LPSTR lpValue,PLONG lpcbValue)
{
	g_debug("RegQueryValueA");
	return RegQueryValueExA(hKey,lpSubKey,NULL,NULL,lpValue,lpcbValue);
}


static LONG WINAPI RegSetValueExA(HKEY hKey,LPCSTR lpValueName,DWORD Reserved,
								DWORD dwType,CONST BYTE *lpData,DWORD cbData)
{
	gboolean result=FALSE;
	gchar *key;
	guint8 *data;
	Profile *profile;

	g_debug("RegSetValueExA");
	if (hKey==NULL)
		return ERROR_UNKNOWN;
	if ((profile=profile_open(NULL))!=NULL) {
		key=g_strconcat(reg_name(hKey),"\\",lpValueName,NULL);
		data=g_malloc(cbData+sizeof(DWORD));
		*(LPDWORD)data=dwType;
		g_memmove(data+sizeof(DWORD),lpData,cbData);
		result=profile_set_value(profile,"registry",key,
						data,cbData+sizeof(DWORD),PROFILE_VALUE_TYPE_ARRAY);
		g_free(data);
		g_free(key);
		profile_close(profile);
	}
	return result?ERROR_SUCCESS:ERROR_UNKNOWN;
}


static LONG WINAPI RegSetValueA(HKEY hKey,LPCSTR lpSubKey,DWORD dwType,
													LPCSTR lpData,DWORD cbData)
{
	g_debug("RegSetValueA");
	return RegSetValueExA(hKey,lpSubKey,0,0,lpData,cbData);
}


/******************************************************************************
*                                                                             *
* Ͽ                                                                        *
*                                                                             *
******************************************************************************/
static VOID WINAPI UnknownSymbol(VOID)
{
	g_message("unknown symbol in advapi32 called");
}


static SymbolInfo symbol_infos[]={
	{"RegCloseKey",RegCloseKey},
	{"RegCreateKeyA",RegCreateKeyA},
	{"RegCreateKeyExA",RegCreateKeyExA},
	{"RegDeleteKeyA",RegDeleteKeyA},
	{"RegDeleteValueA",RegDeleteValueA},
	{"RegEnumKeyA",RegEnumKeyA},
	{"RegEnumKeyExA",RegEnumKeyExA},
	{"RegOpenKeyA",RegCreateKeyA},
	{"RegOpenKeyExA",RegOpenKeyExA},
	{"RegQueryValueA",RegQueryValueA},
	{"RegQueryValueExA",RegQueryValueExA},
	{"RegSetValueA",RegSetValueA},
	{"RegSetValueExA",RegSetValueExA},
	{NULL,UnknownSymbol}
};


SymbolInfo *advapi32_get_export_symbols(void)
{
	return symbol_infos;
}
