/*
    Video maid
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <gdk/gdkkeysyms.h>
#include "command.h"
#include "file.h"
#include "general.h"
#include "sigfile.h"
#include "misc/misc.h"
#include "orz/orzmdi.h"


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(タイマ)                                           *
*                                                                             *
******************************************************************************/
gboolean
signal_timeout (gpointer data)
{
  gint page;

  caret =! caret;
#ifdef USE_THREAD
  gdk_threads_enter ();
#endif /* USE_THREAD */
  if ((page = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi))) >= 0)
    {
      gint sx;
      GdkRectangle rc;
      VmaidWindow *vmaid;

      vmaid = orz_mdi_get_data (ORZ_MDI (mdi), page);
      sx = MAX ((vmaid->drawing->allocation.width - 4) / vmaid->width - 2, 1);
      if (vmaid->top <= vmaid->cursor.frame
                                    && vmaid->cursor.frame <= vmaid->top + sx)
        {
          rc.x = (vmaid->cursor.frame - vmaid->top + 1) * vmaid->width + 1;
          rc.y = system_font_height * 2
                            + (vmaid->avi_edit[0] && vmaid->cursor.stream > 0
                                                        ? vmaid->height : 0);
          rc.width = 2;
          rc.height = vmaid->height;
          gtk_widget_draw (vmaid->drawing, &rc);
        }
    }
#ifdef USE_THREAD
  gdk_threads_leave ();
#endif /* USE_THREAD */
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(フレーム)                                        *
*                                                                             *
******************************************************************************/
gboolean
signal_focus_in (GtkWidget     *widget,
                 GdkEventFocus *event,
                 VmaidWindow   *vmaid)
{
  GTK_WIDGET_SET_FLAGS (widget, GTK_HAS_FOCUS);
  draw_caret (vmaid, NULL);
  return FALSE;
}


gboolean
signal_focus_out (GtkWidget     *widget,
                  GdkEventFocus *event,
                  VmaidWindow   *vmaid)
{
  GTK_WIDGET_UNSET_FLAGS (widget, GTK_HAS_FOCUS);
  draw_caret (vmaid, NULL);
  return FALSE;
}


void
signal_realize (GtkWidget *widget,
                gpointer   user_data)
{
  /* ja:カーソル */
  gdk_window_set_cursor (widget->window, gdk_cursor_new (GDK_XTERM));
}


void
signal_value_changed_hscroll (GtkAdjustment *adjust,
                              VmaidWindow   *vmaid)
{
  gint max, sx, top;

  top = vmaid->top;
  vmaid->top = adjust->value + 0.5;
  sx = MAX ((vmaid->drawing->allocation.width - 4) / vmaid->width - 2, 1);
  max = get_max_frame (vmaid, -1) + 1;
  if (vmaid->top>max - sx)
    vmaid->top = MAX (max - sx, 0);
  clear_sel (vmaid, &vmaid->cursor, &vmaid->select, top);
}


gboolean
signal_config (GtkWidget         *widget,
               GdkEventConfigure *config,
               VmaidWindow       *vmaid)
{
  gint sx, max;

  max = get_max_frame (vmaid, -1) + 1;
  sx = MAX ((config->width - 4) / vmaid->width - 2, 1);
  if (vmaid->top > max - sx)
    vmaid->top = MAX (max - sx, 0);
  draw_caret (vmaid, NULL);
  misc_set_scroll_bar (vmaid->hscroll,
                                GTK_SIGNAL_FUNC (signal_value_changed_hscroll),
                                                vmaid, 0, max, sx, vmaid->top);
  return TRUE;
}


void
signal_destroy_draw (GtkWidget   *widget,
                     VmaidWindow *vmaid)
{
  gint i;

  /* ja:メニュー */
  if (orz_mdi_get_n_pages (ORZ_MDI (mdi)) <= 1)
    {
      set_menu_bar (NULL);
      gtk_statusbar_pop (GTK_STATUSBAR (status),
            gtk_statusbar_get_context_id (GTK_STATUSBAR (status), "Status"));
    }
  /* ja:メモリを解放する */
  if (vmaid->timer_id != 0)
    {
      gtk_timeout_remove (vmaid->timer_id);
      vmaid->timer_id = 0;
    }
  for (i = 0; i < 2; i++)
    if (vmaid->avi_edit[i])
      avi_release (vmaid->avi_edit[i]);
  delete_list (&vmaid->undo);
  delete_list (&vmaid->redo);
  g_free (vmaid);
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(描画)                                            *
*                                                                             *
******************************************************************************/
static void
signal_expose_text (GtkWidget   *widget,
                    GdkGC       *gc,
                    VmaidWindow *vmaid,
                    gint         pos,
                    gint         x,
                    gint         y)
{
  gchar *text;
  gint t;

  text = g_strdup_printf (vmaid->avi_edit[0]
            && avi_is_keyframe (vmaid->avi_edit[0],pos) ? "%d*" : "%d", pos);
  if (gdk_string_width (system_font,text) <= vmaid->width)
    gdk_draw_string (widget->window, system_font, gc,
                                                x, system_font_ascent, text);
  g_free (text);
  t = (glonglong)pos * vmaid->scale * 1000 / vmaid->rate;
  text = g_strdup_printf ("%02d:%02d:%02d",
                                t / 60000 % 100, t / 1000 % 60, t / 10 % 100);
  if (gdk_string_width (system_font, text) <= vmaid->width)
    gdk_draw_string (widget->window, system_font,gc,
                        x, y + system_font_ascent + system_font_height, text);
  g_free (text);
}


static void
signal_expose_bitmap (GtkWidget *widget,
                      GdkGC     *gc,
                      AviFrame  *avi_frame,
                      gint       pos,
                      gint       x,
                      gint       y,
                      gint       width,
                      gint       height)
{
  guchar *buf;

  buf = avi_get_frame32 (avi_frame, pos, width, height);
  if (buf)
    gdk_draw_rgb_32_image (widget->window, gc, x, y, width, height,
                                        GDK_RGB_DITHER_NORMAL, buf, width * 4);
  else
    gdk_draw_rectangle (widget->window, gc, TRUE, x, y, width, height);
}


static void
signal_expose_wave (GtkWidget  *widget,
                    GdkGC      *gc,
                    AviEdit    *avi_edit,
                    AviPcm     *avi_pcm,
                    const gint  current_time,
                    const gint  next_time,
                    const gint  x,
                    const gint  y,
                    const gint  width,
                    const gint  height)
{
  gpointer out_buf = NULL;
  gint start, end, count, out_samples = 0;

  start = 0;
  end = avi_length_time (avi_edit);
  if (avi_pcm)
    {
      gint pos, samples;

      /* ja:範囲内のPCMを読み込む */
      pos = avi_time_to_sample (avi_edit, MAX (start, current_time));
      samples = avi_time_to_sample (avi_edit, MIN (end, next_time)) - pos;
      if (samples > avi_length (avi_edit) - pos)
        samples = avi_length (avi_edit) - pos;
      avi_get_pcm (avi_pcm, pos, samples, &out_buf, &out_samples);
    }
  if (out_samples <= 0)
    {
      GdkPoint pt[5];

      /* ja:PCMが読み込めない */
      if (start <= current_time && next_time <= end)
        {
          /* ja:通常の四角 */
          pt[0].x = pt[3].x = x + 1;
          pt[0].y = pt[1].y = y + 1;
          pt[1].x = pt[2].x = x + width - 2;
          pt[2].y = pt[3].y = y + height - 2;
          count = 4;
        }
      else if (current_time < start && end < next_time)
        {
          /* ja:フレームの範囲内にオーディオが収まる(菱形) */
          pt[0].x = pt[2].x = x + width / 2;
          pt[0].y = y + 1;
          pt[1].x = x + width - 2;
          pt[1].y = pt[3].y = y + height / 2;
          pt[2].y = y + height - 2;
          pt[3].x = x + 1;
          count = 4;
        }
      else if (start <= current_time && end < next_time)
        {
          /* ja:右側が欠ける */
          pt[0].x = pt[4].x = x + 1;
          pt[0].y = pt[1].y = y + 1;
          pt[1].x = pt[3].x = x + width / 2;
          pt[2].x = x + width - 2;
          pt[2].y = y + height / 2;
          pt[3].y = pt[4].y = y + height - 2;
          count = 5;
        }
      else if (current_time < start && next_time <= end)
        {
          /* ja:左側が欠ける */
          pt[0].x = x;
          pt[0].y = y + height / 2;
          pt[1].x = pt[4].x = x + width / 2;
          pt[1].y = pt[2].y = y + 1;
          pt[2].x = pt[3].x = x + width - 2;
          pt[3].y = pt[4].y = y + height - 2;
          count = 5;
        }
      else
        {
          count = 0;
        }
      if (count > 0)
        {
          gdk_gc_set_foreground (gc, system_color + 2);
          gdk_gc_set_fill (gc, GDK_STIPPLED);
          gdk_draw_polygon (widget->window, gc, TRUE, pt, count);
          gdk_gc_set_foreground (gc, system_color);
          gdk_gc_set_fill (gc, GDK_SOLID);
          gdk_draw_polygon (widget->window, gc, FALSE, pt, count);
        }
    }
  else
    {
      gint i, x0, x1, y0, y1;

      /* ja:波形表示 */
      gdk_gc_set_foreground (gc, system_color);
      x0 = MAX (width * (start - current_time)
                                / (next_time - current_time) + x, x);
      x1 = MIN (width * (end - current_time)
                                / (next_time - current_time) + x, x + width);
      count = x1 - x0;
      for (i = 0; i < count; i++)
        {
          gint j;

          /* ja:音量 */
          j = i * out_samples / count;
          if (avi_get_bits_per_sample (avi_edit) == 8)
            {
              guint8 *p;

              p = out_buf;
              if (avi_get_channels (avi_edit) == 1)
                {
                  y0 = y1 = ABS (p[j] - 128) * height / 256;
                }
              else
                {
                  j *= 2;
                  y0 = ABS (p[j] - 128) * height / 256;
                  y1 = ABS (p[j + 1] - 128) * height / 256;
                }
            }
          else
            {
              gint16 *p;

              p = out_buf;
              if (avi_get_channels (avi_edit) == 1)
                {
                  y0 = y1 = ABS (GINT16_FROM_LE (p[j])) * height / 65536;
                }
              else
                {
                  j *= 2;
                  y0 = ABS (GINT16_FROM_LE (p[j])) * height / 65536;
                  y1 = ABS (GINT16_FROM_LE (p[j + 1])) * height / 65536;
                }
            }
          gdk_gc_set_foreground (gc, system_color + 2);
          gdk_draw_line (widget->window, gc,
                    x0 + i, y + height / 2 - y0, x0 + i, y + height / 2 + y1);
        }
      /* ja:横線 */
      gdk_gc_set_foreground (gc, system_color);
      gdk_draw_line (widget->window, gc,
                                    x0, y + height / 2, x1, y + height / 2);
    }
  g_free (out_buf);
}


gboolean
signal_expose (GtkWidget      *widget,
               GdkEventExpose *event,
               VmaidWindow    *vmaid)
{
  gint i, sx, max, frame;
  GdkGC *gc;
  GdkRectangle rc0, rc1;

  gc = gdk_gc_new (widget->window);
  gdk_gc_set_line_attributes (gc, 1,
                                GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER);
  gdk_gc_set_stipple (gc, bitmap0);
  for (i = 0; i < 4; i++)
    gdk_color_alloc (gdk_colormap_get_system (), system_color + i);
  gdk_gc_set_clip_rectangle (gc, &event->area);
  /* ja:フレーム数、時間の背景 */
  if (system_font_height * 2 - event->area.y > 0)
    {
      gdk_gc_set_foreground (gc,system_color + 1);
      gdk_draw_rectangle (widget->window, gc, TRUE,
                    event->area.x, event->area.y,
                    event->area.width, system_font_height * 2 - event->area.y);
    }
  /* ja:フレーム数 */
  sx = MAX ((widget->allocation.width - 4) / vmaid->width - 2, 1);
  max = MIN (get_max_frame (vmaid, -1), vmaid->top + sx - 1);
  frame = vmaid->select.stream >= 0 ? vmaid->select.frame
                - (vmaid->cursor. frame < vmaid->select.frame ? 1 : 0) : -1;
  gdk_gc_set_foreground (gc, system_color);
  if (frame >= 0 && frame < vmaid->top) /* ja:選択開始位置が前のとき */
    signal_expose_text (widget, gc, vmaid, frame, 0, 0);
  for (i = vmaid->top, rc0.x = vmaid->width + 2; i <= max;
                                                i++, rc0. x += vmaid->width)
    signal_expose_text (widget, gc, vmaid, i, rc0. x, 0);
  if (vmaid->top + sx <= frame)         /* ja:選択開始位置が後のとき */
    signal_expose_text (widget, gc, vmaid, frame,
                                            vmaid->width * (sx + 1) + 4, 0);
  rc0.y = system_font_height * 2;
  rc0.height = vmaid->height;
  if (vmaid->avi_edit[0])
    {
      AviFrame *avi_frame;

      avi_frame = avi_get_frame_open (vmaid->avi_edit[0]);
      max = MIN (get_max_frame (vmaid, 0), vmaid->top + sx - 1);
      rc0.x = 0;
      if (frame >= 0 && frame < vmaid->top)
        {
          /* ja:選択開始位置が前のとき */
          rc0.width = vmaid->width;
          if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
            {
              signal_expose_bitmap (widget, gc, avi_frame, frame,
                                        rc0.x, rc0.y, rc0.width, rc0.height);
            }
          rc0.x = vmaid->width;
          rc0.width = 2;
        }
      else
        {
          rc0.width = vmaid->width + 2;
        }
      if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        {
          gdk_gc_set_foreground (gc,system_color + 1);
          gdk_draw_rectangle (widget->window, gc, TRUE, rc1.x, rc1.y,
                                                        rc1.width, rc1.height);
        }
      rc0.width = vmaid->width;
      for (i = vmaid->top,rc0.x = vmaid->width + 2; i <= max;
                                                    i++, rc0.x += vmaid->width)
        if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
          signal_expose_bitmap (widget, gc, avi_frame, i,
                                        rc0.x, rc0.y, rc0.width, rc0.height);
      rc0.width = vmaid->width * (sx + 1) + 4 - rc0.x;
      if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        {
          gdk_gc_set_foreground (gc,system_color + 1);
          gdk_draw_rectangle (widget->window, gc, TRUE, rc1.x, rc1.y,
                                                        rc1.width, rc1.height);
        }
      rc0.x = vmaid->width * (sx + 1) + 4;
      rc0.width = vmaid->width;
      if (vmaid->top + sx <= frame
                        && gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        {
          /* ja:選択開始位置が後のとき */
          signal_expose_bitmap (widget, gc, avi_frame, frame,
                                    rc0.x, rc0.y, vmaid->width, rc0.height);
          rc0.x += vmaid->width;
        }
      rc0.width = widget->allocation.width - rc0.x;
      if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        {
          gdk_gc_set_foreground (gc, system_color + 1);
          gdk_draw_rectangle (widget->window, gc, TRUE,
                                        rc1.x, rc1.y, rc1.width, rc1.height);
        }
      rc0.y += vmaid->height;
      avi_get_frame_close (avi_frame);
    }
  if (vmaid->avi_edit[1])
    {
      AviPcm *avi_pcm;

      /* ja:背景 */
      rc0.x = event->area.x;
      rc0.width = event->area.width;
      if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        {
          gdk_gc_set_foreground (gc, system_color + 1);
          gdk_draw_rectangle (widget->window, gc, TRUE,
                                        rc1.x, rc1.y, rc1.width, rc1.height);
        }
      avi_pcm = avi_get_pcm_open (vmaid->avi_edit[1]);
      max = MIN (get_max_frame (vmaid, 1), vmaid->top + sx - 1);
      rc0.x = 0;
      rc0.width = vmaid->width;
      if (frame >= 0 && frame < vmaid->top
                        && gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        /* ja:選択開始位置が前のとき */
        signal_expose_wave (widget, gc, vmaid->avi_edit[1], avi_pcm,
            (gint)((glonglong)frame * vmaid->scale * 1000 / vmaid->rate),
            (gint)((glonglong)(frame + 1) * vmaid->scale * 1000 / vmaid->rate),
                                        rc0.x, rc0.y, rc0.width, rc0.height);
      for (i = vmaid->top, rc0.x = vmaid->width + 2; i <= max;
                                                i++, rc0. x += vmaid->width)
        if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
          signal_expose_wave (widget, gc, vmaid->avi_edit[1], avi_pcm,
                (gint)((glonglong)i * vmaid->scale * 1000 / vmaid->rate),
                (gint)((glonglong)(i + 1)*vmaid->scale * 1000 / vmaid->rate),
                                        rc0.x, rc0.y, rc0.width, rc0.height);
      rc0.x = vmaid->width * (sx + 1) + 4;
      if (vmaid->top + sx <= frame
                        && gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        /* ja:選択開始位置が後のとき */
        signal_expose_wave (widget, gc, vmaid->avi_edit[1], avi_pcm,
            (gint)((glonglong)frame * vmaid->scale * 1000 / vmaid->rate),
            (gint)((glonglong)(frame + 1) * vmaid->scale * 1000 / vmaid->rate),
                                        rc0.x, rc0.y, rc0.width, rc0.height);
      avi_get_pcm_close (avi_pcm);
    }
  /* ja:選択範囲 */
  if (vmaid->select.stream >= 0)
    {
      /* ja:選択範囲があるとき */
      rc0.x = MAX (vmaid->top,
                        MIN (vmaid->cursor. frame, vmaid->select.frame));
      rc0.width = MIN (vmaid->top + sx,
                        MAX (vmaid->cursor. frame, vmaid->select.frame)) - 1;
      if (rc0.x <= rc0.width)
        {
          rc0.width = (rc0.width - rc0.x + 1) * vmaid->width;
          rc0.x = (rc0.x - vmaid->top + 1) * vmaid->width + 2;
          rc0.y = MIN (vmaid->cursor.stream, vmaid->select.stream);
          rc0.height = MAX (vmaid->cursor. stream, vmaid->select.stream)
                                                                - rc0.y + 1;
          if (!vmaid->avi_edit[0])
            rc0.y--;
          rc0.y = rc0.y * vmaid->height + system_font_height * 2;
          rc0.height *= vmaid->height;
          if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
            {
              gdk_gc_set_foreground (gc, system_color + 3);
              gdk_gc_set_fill (gc, GDK_STIPPLED);
              gdk_gc_set_stipple (gc, bitmap1);
              gdk_draw_rectangle (widget->window, gc, TRUE,
                                        rc1.x, rc1.y, rc1.width, rc1.height);
              gdk_gc_set_fill (gc, GDK_SOLID);
            }
        }
    }
  /* ja:キャレットの描画 */
  if (caret && GTK_WIDGET_HAS_FOCUS(widget)
                                    && vmaid->top <= vmaid->cursor.frame
                                    && vmaid->cursor.frame <= vmaid->top + sx)
    {
      rc0.x = (vmaid->cursor.frame - vmaid->top + 1) * vmaid->width + 1;
      rc0.y = system_font_height * 2
                            + (vmaid->avi_edit[0] && vmaid->cursor.stream > 0
                                                        ? vmaid->height : 0);
      rc0.width = 2;
      rc0.height = vmaid->height;
      if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        {
          gdk_gc_set_function (gc, GDK_INVERT);
          gdk_draw_rectangle (widget->window, gc, TRUE,
                                        rc1.x, rc1.y, rc1.width, rc1.height);
        }
    }
  gdk_gc_destroy (gc);
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(マウス)                                          *
*                                                                             *
******************************************************************************/
gboolean
signal_button_press_draw (GtkWidget      *widget,
                          GdkEventButton *event,
                          VmaidWindow    *vmaid)
{
  if (!GTK_WIDGET_HAS_FOCUS (widget))
    gtk_widget_grab_focus (widget);
  switch (event->type)
    {
      case GDK_BUTTON_PRESS:
        switch (event->button)
          {
            case 1:/* ja:左クリック */
              {
                gboolean shift;
                gint max, sx;
                VmaidCursor cursor, select;

                if (!vmaid->avi_edit[0] && !vmaid->avi_edit[1])
                  break;
                cursor = vmaid->cursor;
                select = vmaid->select;
                shift = event->state & GDK_SHIFT_MASK;
                max = get_max_frame (vmaid, -1) + 1;
                sx = MAX ((widget->allocation.width - 4) / vmaid->width - 2,
                                                                            1);
                /* ja:クリックした位置にキャレットを移動する */
                vmaid->cursor.frame = vmaid->top
                                + (event->x - vmaid->width - 2) / vmaid->width;
                if (vmaid->cursor.frame < vmaid->top)
                  vmaid->cursor.frame = vmaid->top;
                else if (vmaid->top + sx < vmaid->cursor.frame)
                  vmaid->cursor.frame = vmaid->top + sx;
                if (vmaid->cursor.frame > max)
                  vmaid->cursor.frame = max;
                if (vmaid->avi_edit[0] && vmaid->avi_edit[1])
                  vmaid->cursor.stream = event->y < system_font_height * 2
                                                    + vmaid->height ? 0 : 1;
                if (vmaid->select.stream >= 0 && (!shift
                                || vmaid->cursor.frame == vmaid->select.frame))
                  vmaid->select.stream = -1;/* ja:選択を解除する */
                else if (vmaid->select.stream < 0
                            && shift && vmaid->cursor.frame != cursor.frame)
                  vmaid->select = cursor;/* ja:新規の選択 */
                if (vmaid->cursor.stream == cursor.stream
                                    && vmaid->cursor.frame == cursor.frame
                                    && vmaid->select.stream == select.stream
                                    && vmaid->select.frame == select.frame)
                  break;/* ja:変化なし */
                /* ja:メニュー */
                if (vmaid->select.stream != select.stream)
                  set_menu_bar (vmaid);
                clear_sel (vmaid, &cursor, &select, vmaid->top);
                draw_caret (vmaid, &cursor);
              }
              break;
            case 2:/* ja:中クリック */
              break;
            case 3:/* ja:右クリック */
              {
                GtkWidget *menu_shell;

                gtk_widget_set_sensitive (misc_find_menu (popup_entries,
                                        "/undo"), vmaid->undo != NULL);
                gtk_widget_set_sensitive (misc_find_menu (popup_entries,
                                        "/cut"), vmaid->select.stream >= 0);
                gtk_widget_set_sensitive (misc_find_menu (popup_entries,
                                        "/copy"), vmaid->select.stream >= 0);
                gtk_widget_set_sensitive (misc_find_menu (popup_entries,
                                        "/delete"), vmaid->select.stream >= 0);
                menu_shell = misc_find_menu (popup_entries, "/");
                gtk_widget_show_all (menu_shell);
                gtk_menu_popup (GTK_MENU (menu_shell),
                        NULL, NULL, NULL, NULL, event->button, event->time);
              }
          }
        break;
      default:
        return FALSE;
    }
  return TRUE;
}


/*  ja:マウスの移動に関する処理
    vmaid,テキスト情報
        x,X座標
        y,Y座標
    state,ステータス
      RET,TRUE:スクロールあり,FALSE:スクロールなし                          */
static gboolean
signal_motion_notify_draw (VmaidWindow     *vmaid,
                           gint             x,
                           gint             y,
                           GdkModifierType  state)
{
  gint max, sx, top;
  VmaidCursor cursor, select;
  GdkPoint ps;

  cursor = vmaid->cursor;
  select = vmaid->select;
  top = vmaid->top;
  sx = MAX ((vmaid->drawing->allocation.width - 4) / vmaid->width - 2, 1);
  max = get_max_frame (vmaid, -1) + 1;
  ps.x = x;
  ps.y = y;
  if (ps.x < vmaid->width + 2)
    ps.x = vmaid->width + 2;
  else if (ps.x > (sx + 1) * vmaid->width + 2)
    ps.x = (sx + 1) * vmaid->width + 2;
  /* ja:カーソルが外にあるときスクロールする */
  if (x < vmaid->width + 2 && vmaid->top > 0)
    vmaid->top--;
  else if (x > (sx + 1) * vmaid->width + 2 && vmaid->top < max - sx)
    vmaid->top++;
  /* ja:カーソルの位置にキャレットを移動する */
  vmaid->cursor.frame = vmaid->top + (ps.x - vmaid->width - 2) / vmaid->width;
  if (vmaid->cursor.frame < 0)
    vmaid->cursor.frame = 0;
  else if (vmaid->cursor.frame > max)
    vmaid->cursor.frame = max;
  if (vmaid->avi_edit[0] && vmaid->avi_edit[1])
    vmaid->cursor.stream = ps.y < system_font_height * 2 + vmaid->height
                                                                    ? 0 : 1;
  if (vmaid->cursor.frame == cursor.frame
                                    && vmaid->cursor.stream == cursor.stream)
    return FALSE;/* ja:変化なし */
  if (vmaid->select.stream >= 0 && vmaid->cursor.frame == vmaid->select.frame)
    vmaid->select.stream = -1;/* ja:選択を解除する */
  else if (vmaid->select.stream < 0 && vmaid->cursor.frame != cursor.frame)
    vmaid->select = cursor;/* ja:新規の選択 */
  /* ja:メニュー */
  if (vmaid->select.stream != select.stream)
    set_menu_bar (vmaid);
  if (vmaid->top != top)
    misc_set_scroll_bar (vmaid->hscroll,
                                GTK_SIGNAL_FUNC (signal_value_changed_hscroll),
                                                vmaid, 0, max, sx, vmaid->top);
  clear_sel (vmaid, &cursor, &select, top);
  draw_caret (vmaid, &cursor);
  return vmaid->top != top;
}


static
gboolean signal_timeout_draw (VmaidWindow *vmaid)
{
  gboolean result;
  gint x, y;
  GdkModifierType state;

  gdk_window_get_pointer (vmaid->drawing->window, &x, &y, &state);
  if (state & GDK_BUTTON1_MASK)
    {
      result = signal_motion_notify_draw (vmaid, x, y, state);
      if (!result)
        {
          /* ja:スクロールが終わったらタイマは削除 */
          gtk_timeout_remove (vmaid->timer_id);
          vmaid->timer_id = 0;
        }
    }
  else
    {
      result = FALSE;
    }
  return result;
}


gboolean
signal_motion_notify (GtkWidget      *widget,
                      GdkEventMotion *event,
                      VmaidWindow    *vmaid)
{
  gint x, y;
  GdkModifierType state;

  if (event->is_hint)
    {
      gdk_window_get_pointer(event->window,&x,&y,&state);
    }
  else
    {
      x = event->x;
      y = event->y;
      state = event->state;
    }
  if ((state&GDK_BUTTON1_MASK)
                            && signal_motion_notify_draw (vmaid, x, y, state)
                                                    && vmaid->timer_id == 0)
    /* ja:はじめてスクロールしたとき */
    vmaid->timer_id
                = gtk_timeout_add (1, (GtkFunction)signal_timeout_draw, vmaid);
  return TRUE;
}


gboolean
signal_button_release (GtkWidget      *widget,
                       GdkEventButton *event,
                       VmaidWindow    *vmaid)
{
  if (vmaid->timer_id != 0)
    {
      gtk_timeout_remove (vmaid->timer_id);
      vmaid->timer_id = 0;
    }
  return TRUE;
}


gboolean
signal_scroll (GtkWidget      *widget,
               GdkEventScroll *event,
               VmaidWindow    *vmaid)
{
  gint max, sx, top;

  top = vmaid->top;
  sx = MAX ((widget->allocation.width - 4) / vmaid->width - 2, 1);
  max = get_max_frame (vmaid, -1) + 1;
  vmaid->top += event->direction == GDK_SCROLL_UP
                            || event->direction == GDK_SCROLL_LEFT ? -4 : 4;
  if (vmaid->top < 0)
    vmaid->top = 0;
  else if (vmaid->top > max - sx)
    vmaid->top = MAX (max - sx, 0);
  if (vmaid->top != top)
    misc_set_scroll_bar (vmaid->hscroll,
                                GTK_SIGNAL_FUNC (signal_value_changed_hscroll),
                                                vmaid, 0, max, sx, vmaid->top);
  clear_sel (vmaid, &vmaid->cursor, &vmaid->select, top);
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(キー)                                            *
*                                                                             *
******************************************************************************/
gboolean
signal_key_press (GtkWidget   *widget,
                  GdkEventKey *event,
                  VmaidWindow *vmaid)
{
  gboolean result = FALSE;/* ja:TRUE:処理終了,FALSE:処理継続 */

  switch (event->keyval)
    {
      case GDK_J:
      case GDK_K:
      case GDK_j:
      case GDK_k:
        if (!(event->state & GDK_MOD1_MASK))
          break;
      case GDK_Up:
      case GDK_Down:
        result = TRUE;
        {
          gint sx, max, top;
          VmaidCursor cursor, select;

          if (!vmaid->avi_edit[0] || !vmaid->avi_edit[1])
            break;
          cursor = vmaid->cursor;
          select = vmaid->select;
          top = vmaid->top;
          max = get_max_frame (vmaid, -1) + 1;
          sx = MAX ((widget->allocation.width - 4) / vmaid->width - 2, 1);
          vmaid->cursor.stream = vmaid->cursor.stream == 0 ? 1 : 0;
          /* ja:キャレットが表示されるように調整 */
          if (vmaid->top > vmaid->cursor.frame)
            vmaid->top = MIN (vmaid->cursor.frame, max - sx);
          if (vmaid->top < vmaid->cursor.frame - sx)
            vmaid->top = MAX (vmaid->cursor.frame - sx, 0);
          if (vmaid->select.stream >= 0 && !(event->state & GDK_SHIFT_MASK))
            {
              vmaid->select.stream = -1;/* ja:選択解除 */
              set_menu_bar (vmaid);/* ja:メニュー */
            }
          if (vmaid->top != top)
            misc_set_scroll_bar (vmaid->hscroll,
                                GTK_SIGNAL_FUNC (signal_value_changed_hscroll),
                                                vmaid, 0, max, sx, vmaid->top);
          clear_sel (vmaid, &cursor, &select, top);
          draw_caret (vmaid, &cursor);
        }
        break;
      case GDK_Home:
      case GDK_Left:
      case GDK_Right:
      case GDK_Page_Up:
      case GDK_Page_Down:
      case GDK_End:
      case GDK_comma:
      case GDK_period:
      case GDK_less:
      case GDK_greater:
      case GDK_H:
      case GDK_L:
      case GDK_h:
      case GDK_l:
        result=TRUE;
        {
          gboolean alt, ctrl;
          gint sx, max, top;
          VmaidCursor cursor, select;

          if (!vmaid->avi_edit[0] && !vmaid->avi_edit[1])
            break;
          cursor = vmaid->cursor;
          select = vmaid->select;
          top = vmaid->top;
          max = get_max_frame (vmaid, -1) + 1;
          sx = MAX ((widget->allocation.width - 4) / vmaid->width - 2, 1);
          alt = event->state & GDK_MOD1_MASK;
          ctrl = event->state & GDK_CONTROL_MASK;
          switch (event->keyval)
            {
              case GDK_Home:
                vmaid->cursor.frame = 0;
                break;
              case GDK_End:
                vmaid->cursor.frame
                            = get_max_frame (vmaid, vmaid->cursor.stream) + 1;
                break;
              case GDK_H:
              case GDK_h:
                if (!alt)
                  goto loop;
              case GDK_Left:
                if (!ctrl)
                  {
                    vmaid->cursor.frame--;
                  }
                else if (vmaid->top > 0)
                  {
                    vmaid->top--;
                    if (vmaid->cursor.frame-sx > vmaid->top)
                      vmaid->cursor.frame--;
                  }
                break;
              case GDK_L:
              case GDK_l:
                if (!alt)
                  goto loop;
              case GDK_Right:
                if (!ctrl)
                  {
                    vmaid->cursor.frame++;
                  }
                else if (vmaid->top < max - sx)
                 {
                    vmaid->top++;
                    if (vmaid->cursor.frame < vmaid->top)
                      vmaid->cursor.frame++;
                  }
                break;
              case GDK_Page_Up:
                if (ctrl)
                  {
                    vmaid->cursor.frame = vmaid->top;
                  }
                else
                  {
                    vmaid->cursor.frame -= sx;
                    vmaid->top -= sx;
                  }
                break;
              case GDK_Page_Down:
                if (ctrl)
                  {
                    vmaid->cursor.frame = vmaid->top+sx;
                  }
                else
                  {
                    vmaid->cursor.frame += sx;
                    vmaid->top += sx;
                  }
            }
          if (vmaid->cursor.frame < 0)
            vmaid->cursor.frame=0;
          else if (vmaid->cursor.frame > max)
            vmaid->cursor.frame = max;
          /* ja:キャレットが表示されるように調整 */
          if (vmaid->top > vmaid->cursor.frame)
            vmaid->top = MIN (vmaid->cursor.frame, max - sx);
          if (vmaid->top < vmaid->cursor.frame - sx)
            vmaid->top = MAX (vmaid->cursor.frame - sx, 0);
          if (vmaid->cursor.frame == cursor.frame && vmaid->top == top)
            break;/* ja:変化なし */
          if (event->state & GDK_SHIFT_MASK)
            {
              if (vmaid->select.stream < 0)
                vmaid->select = cursor;/* ja:新規の選択 */
              if (vmaid->cursor.frame == vmaid->select.frame)
                vmaid->select.stream = -1;/* ja:選択を解除する */
            }
          else
            {
              vmaid->select.stream = -1;
            }
            /* ja:メニュー */
          if ((vmaid->select.stream < 0 && select.stream >= 0)
                        || (vmaid->select.stream >= 0 && select.stream < 0))
            set_menu_bar (vmaid);
          if (vmaid->top != top)
            misc_set_scroll_bar (vmaid->hscroll,
                                GTK_SIGNAL_FUNC (signal_value_changed_hscroll),
                                                vmaid, 0, max, sx, vmaid->top);
          clear_sel (vmaid, &cursor, &select, top);
          draw_caret (vmaid, &cursor);
        }
        break;
      case GDK_Escape:
      case GDK_Cancel:
        result=TRUE;
        {
          gint sx, top;
          VmaidCursor cursor, select;

          if (vmaid->select.stream < 0)
            break;
          cursor = vmaid->cursor;
          select = vmaid->select;
          top = vmaid->top;
          sx = MAX ((widget->allocation.width - 4) / vmaid->width - 2, 1);
          /* ja:キャレットが表示されるように調整 */
          if (vmaid->top > vmaid->cursor.frame)
            vmaid->top = MIN (vmaid->cursor.frame,
                                        get_max_frame (vmaid, -1) + 1 - sx);
          if (vmaid->top < vmaid->cursor.frame - sx)
            vmaid->top = MAX (vmaid->cursor.frame - sx, 0);
          vmaid->select.stream = -1;/* ja:選択解除 */
          /* ja:メニュー */
          set_menu_bar (vmaid);
          if (vmaid->top != top)
            misc_set_scroll_bar (vmaid->hscroll,
                    GTK_SIGNAL_FUNC (signal_value_changed_hscroll),
                    vmaid, 0, get_max_frame (vmaid, -1) + 1, sx, vmaid->top);
          clear_sel (vmaid, &cursor, &select, top);
          draw_caret (vmaid, &cursor);
        }
        break;
      case GDK_BackSpace:
      case GDK_Delete:
      case GDK_KP_Delete:
      case GDK_3270_DeleteWord:
        result = TRUE;
        {
          gboolean ctrl;
          gint max;

          if (!vmaid->avi_edit[0] && !vmaid->avi_edit[1])
            break;
          ctrl = event->state & GDK_CONTROL_MASK;
          if (vmaid->select.stream < 0)
            {
              if (event->keyval == GDK_BackSpace)
                {
                  if (vmaid->cursor.frame <= 0)
                    break;
                  vmaid->select.frame = ctrl ? 0 : vmaid->cursor.frame - 1;
                }
              else
                {
                  max = get_max_frame (vmaid, vmaid->cursor.stream) + 1;
                  if (vmaid->cursor.frame >= max)
                    break;
                  vmaid->select.frame = ctrl ? max : vmaid->cursor.frame + 1;
                }
              vmaid->select.stream = vmaid->cursor.stream;
            }
          command_delete (widget, NULL);
        }
        break;
      case GDK_F10:
        {
          gint sx, top;

          if (event->state & GDK_SHIFT_MASK)
            break;
          top = vmaid->top;
          sx = MAX ((widget->allocation.width - 4) / vmaid->width - 2, 1);
          /* ja:キャレットが表示されるように調整 */
          if (vmaid->top > vmaid->cursor.frame)
            vmaid->top = MIN (vmaid->cursor.frame,
                                        get_max_frame (vmaid, -1) + 1 - sx);
          if (vmaid->top < vmaid->cursor.frame - sx)
            vmaid->top = MAX (vmaid->cursor.frame - sx, 0);
          if (vmaid->top != top)
            misc_set_scroll_bar (vmaid->hscroll,
                    GTK_SIGNAL_FUNC (signal_value_changed_hscroll),
                    vmaid, 0, get_max_frame (vmaid, -1) + 1, sx, vmaid->top);
          clear_sel (vmaid, &vmaid->cursor, &vmaid->select, top);
          gtk_widget_set_sensitive (misc_find_menu (popup_entries, "/undo"),
                                                    vmaid->undo != NULL);
          gtk_widget_set_sensitive (misc_find_menu (popup_entries, "/cut"),
                                                    vmaid->select.stream >= 0);
          gtk_widget_set_sensitive (misc_find_menu (popup_entries, "/copy"),
                                                    vmaid->select.stream >= 0);
          gtk_widget_set_sensitive (misc_find_menu (popup_entries, "/delete"),
                                                    vmaid->select.stream >= 0);
          gtk_menu_popup (GTK_MENU (misc_find_menu (popup_entries, "/")),
                                    NULL, NULL, NULL, NULL, 3, event->time);
        }
        result=TRUE;
        break;
      case GDK_Tab:
      case GDK_KP_Tab:
        result=TRUE;
        {
          gint page;
          guint length;

          length = orz_mdi_get_n_pages (ORZ_MDI (mdi));
          page = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
          if (event->state & GDK_SHIFT_MASK)
            {
              if (--page < 0)
                page = length - 1;
            }
          else
            {
              if (++page >= length)
                page = 0;
            }
          gtk_notebook_set_page (GTK_NOTEBOOK (mdi), page);
          gtk_widget_grab_focus (((VmaidWindow *)
                            orz_mdi_get_data (ORZ_MDI (mdi), page))->drawing);
        }
    }
  loop:
  return result;
}
