/*
    avicore
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "avibase.h"


struct _AviPcm
{
  gpointer tmp_buf;
  gint tmp_samples;     /* ja:テンポラリの実サンプル数 */
  gint in_samples;      /* ja:累積入力仮想サンプル数 */
  gint out_samples;     /* ja:累積出力実サンプル数 */
  gint pos;             /* ja:前回の終了点 */
  gint marker;          /* ja:読み込みの終了点 */
  AviEdit *avi_edit;
  AviFile *file;
  AcmObject *acm_object;
  WaveConv *wcv;
  WaveFormatEx *wfx;
};


/******************************************************************************
*                                                                             *
* ja:PCM関数                                                                  *
*                                                                             *
******************************************************************************/
/*  ja:PCMに展開されたオーディオを開く
    avi_edit,AVI編集ハンドル
         RET,AVIPCM構造体,NULL:エラー                                       */
AviPcm *
avi_pcm_open (AviEdit *avi_edit)
{
  AviPcm *avi_pcm;

  if (!avi_edit || avi_edit_type (avi_edit) != AVI_TYPE_AUDIO)
    return NULL;
  avi_pcm = g_malloc0 (sizeof (AviPcm));
  avi_pcm->avi_edit = avi_edit;
  avi_pcm->pos = -1;
  avi_pcm->marker = -1;
  return avi_pcm;
}


/*  ja:PCMに展開されたオーディオを閉じる
    avi_pcm,AVIPCM構造体
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
avi_pcm_close (AviPcm *avi_pcm)
{
  gboolean result;

  if (!avi_pcm)
    return FALSE;
  result = (!avi_pcm->wcv || wave_convert_end (avi_pcm->wcv, NULL, NULL))
         & (!avi_pcm->acm_object || acm_close (avi_pcm->acm_object));
  g_free (avi_pcm->wfx);
  g_free (avi_pcm->tmp_buf);
  g_free (avi_pcm);
  return result;
}


/*  ja:バッファに追加する
        avi_pcm,AVIPCM構造体
            buf,追加するデータ
        samples,追加するデータの実サンプル数                                */
static void
avi_pcm_add_buffer (AviPcm        *avi_pcm,
                    gconstpointer  buf,
                    const gsize    samples)
{
  avi_pcm->tmp_buf = g_realloc (avi_pcm->tmp_buf,
                            (avi_pcm->tmp_samples + samples)
                            * wfx_get_block_align (avi_pcm->avi_edit->wfx));
  g_memmove ((guint8 *)avi_pcm->tmp_buf + avi_pcm->tmp_samples
                                * wfx_get_block_align (avi_pcm->avi_edit->wfx),
                buf, samples * wfx_get_block_align (avi_pcm->avi_edit->wfx));
  avi_pcm->tmp_samples += samples;
}


/*  ja:PCMに展開されたオーディオを取得する
        avi_pcm,AVIPCM構造体
          start,取得する仮想サンプル番号
        samples,仮想サンプル数
        out_buf,PCMデータ
    out_samples,PCMデータの実サンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
avi_pcm_get_raw (AviPcm      *avi_pcm,
                 const gint   start,
                 const gsize  samples,
                 gpointer    *out_buf,
                 gsize       *out_samples)
{
  gpointer result_buf;
  gint pcm_samples, virtual_samples, result_samples, result_bytes, tmp_bytes;
  AviEdit *avi_edit;

  if (!avi_pcm)
    return FALSE;
  avi_edit = avi_pcm->avi_edit;
  if (start < 0 || avi_edit_length (avi_edit) < start + samples
                                                            || samples <= 0)
    return FALSE;                   /* ja:範囲外 */
  if (avi_pcm->pos == start)
    {
      pcm_samples = avi_audio_real_from_virtual_roundup
                                (avi_pcm->in_samples + samples, avi_edit->wfx)
                                                        - avi_pcm->out_samples;
      if (avi_pcm->marker < 0)
        {
          if (out_buf)
            *out_buf = wave_fill_silent_pcm (avi_edit->wfx, pcm_samples);
          if (out_samples)
            *out_samples = pcm_samples;
          avi_pcm->in_samples += samples;
          avi_pcm->out_samples += pcm_samples;
          avi_pcm->pos = start + samples;
          return TRUE;
        }
      virtual_samples = avi_audio_real_to_virtual
                        (pcm_samples - avi_pcm->tmp_samples, avi_edit->wfx);
    }
  else
    {
      gboolean result = TRUE;

      if (avi_pcm->acm_object)
        {
          if (!acm_close (avi_pcm->acm_object))
            result = FALSE;
          avi_pcm->acm_object = NULL;
        }
      if (avi_pcm->wcv)
        {
          if (!wave_convert_end (avi_pcm->wcv, NULL, NULL))
            result = FALSE;
          avi_pcm->wcv = NULL;
        }
      g_free (avi_pcm->wfx);
      avi_pcm->wfx = NULL;
      g_free (avi_pcm->tmp_buf);
      avi_pcm->tmp_buf = NULL;
      avi_pcm->tmp_samples = 0;
      avi_pcm->in_samples = 0;
      avi_pcm->out_samples = 0;
      avi_pcm->pos = -1;
      if (!result)
        {
          avi_pcm->file = NULL;
          avi_pcm->marker = -1;
          return FALSE;
        }
      avi_base_get_file (avi_edit, start);/* ja:AVIファイルを求める */
      avi_pcm->file = avi_edit->file;
      avi_pcm->marker = avi_audio_real_from_virtual
                                (start - avi_edit->offset, avi_pcm->file->wfx);
      pcm_samples = avi_audio_real_from_virtual_roundup
                                                    (samples, avi_edit->wfx);
      virtual_samples = samples;
    }
  while (avi_pcm->tmp_samples < pcm_samples)
    {
      gpointer file_buf, conv_buf;
      gsize file_length, file_samples, conv_samples;

      file_length = CLAMP (avi_audio_real_from_virtual_roundup
                                (avi_pcm->file->length, avi_pcm->file->wfx),
                                                    1, avi_pcm->file->samples);
      if (avi_pcm->marker >= file_length)
        {
          avi_pcm->file = avi_pcm->file->next;
          if (!avi_pcm->file)
            { /* ja:ファイルの末尾まできたとき */
              avi_pcm->file = NULL;
              avi_pcm->marker = -1;
              if (avi_pcm->acm_object)
                {
                  gpointer e_buf = NULL;
                  gsize e_samples;

                  if ((!acm_decompress_end (avi_pcm->acm_object,
                                                            &e_buf, &e_samples)
                                            | !acm_close (avi_pcm->acm_object))
                        || !wave_convert (avi_pcm->wcv,
                                e_buf, e_samples, &conv_buf, &conv_samples))
                    {
                      g_free (e_buf);
                      avi_pcm->acm_object = NULL;
                      wave_convert_end (avi_pcm->wcv, NULL, NULL);
                      avi_pcm->wcv = NULL;
                      g_free (avi_pcm->wfx);
                      avi_pcm->wfx = NULL;
                      g_free (avi_pcm->tmp_buf);
                      avi_pcm->tmp_buf = NULL;
                      avi_pcm->tmp_samples = 0;
                      avi_pcm->in_samples = 0;
                      avi_pcm->out_samples = 0;
                      avi_pcm->pos = -1;
                      avi_pcm->file = NULL;
                      avi_pcm->marker = -1;
                      return FALSE;
                    }
                  g_free (e_buf);
                  avi_pcm->acm_object = NULL;
                  avi_pcm_add_buffer (avi_pcm, conv_buf, conv_samples);
                  g_free (conv_buf);
                }
              if (avi_pcm->wcv
                && !wave_convert_end (avi_pcm->wcv, &conv_buf, &conv_samples))
                {
                  avi_pcm->wcv = NULL;
                  g_free (avi_pcm->wfx);
                  avi_pcm->wfx = NULL;
                  g_free (avi_pcm->tmp_buf);
                  avi_pcm->tmp_buf = NULL;
                  avi_pcm->tmp_samples = 0;
                  avi_pcm->in_samples = 0;
                  avi_pcm->out_samples = 0;
                  avi_pcm->pos = -1;
                  avi_pcm->file = NULL;
                  avi_pcm->marker = -1;
                  return FALSE;
                }
              avi_pcm_add_buffer (avi_pcm, conv_buf, conv_samples);
              avi_pcm->wcv = NULL;
              g_free (avi_pcm->wfx);
              avi_pcm->wfx = NULL;
              break;
            }
          file_length = CLAMP (avi_audio_real_from_virtual_roundup
                                (avi_pcm->file->length, avi_pcm->file->wfx),
                                                    1, avi_pcm->file->samples);
          avi_pcm->marker = 0;
        }
      file_samples = CLAMP (avi_audio_real_from_virtual (virtual_samples,
                        avi_pcm->file->wfx), 1, file_length - avi_pcm->marker);
      file_buf = g_malloc (avi_read_audio_size (avi_pcm->file,
                                            avi_pcm->marker, file_samples));
      if (!avi_read_audio (avi_pcm->file,
                                    avi_pcm->marker, file_samples, file_buf))
        {
          g_free (file_buf);
          acm_close (avi_pcm->acm_object);
          avi_pcm->acm_object = NULL;
          if (avi_pcm->wcv)
            {
              wave_convert_end (avi_pcm->wcv, NULL, NULL);
              avi_pcm->wcv = NULL;
            }
          g_free (avi_pcm->wfx);
          avi_pcm->wfx = NULL;
          g_free (avi_pcm->tmp_buf);
          avi_pcm->tmp_buf = NULL;
          avi_pcm->tmp_samples = 0;
          avi_pcm->in_samples = 0;
          avi_pcm->out_samples = 0;
          avi_pcm->pos = -1;
          avi_pcm->file = NULL;
          avi_pcm->marker = -1;
          return FALSE;
        }
      /* ja:PCMを変換する */
      if (!avi_pcm->wfx || !wf_header_cmp (avi_pcm->wfx, avi_pcm->file->wfx))
        { /* ja:新規またはフォーマットが変化したとき */
          WaveFormatEx *wfx = NULL;

          if (avi_pcm->acm_object)
            {
              gpointer e_buf = NULL;
              gsize e_samples;

              if ((!acm_decompress_end (avi_pcm->acm_object,
                                                            &e_buf, &e_samples)
                                            | !acm_close (avi_pcm->acm_object))
                    || !wave_convert (avi_pcm->wcv,
                                e_buf, e_samples, &conv_buf, &conv_samples))
                {
                  g_free (e_buf);
                  g_free (file_buf);
                  avi_pcm->acm_object = NULL;
                  wave_convert_end (avi_pcm->wcv, NULL, NULL);
                  avi_pcm->wcv = NULL;
                  g_free (avi_pcm->wfx);
                  avi_pcm->wfx = NULL;
                  g_free (avi_pcm->tmp_buf);
                  avi_pcm->tmp_buf = NULL;
                  avi_pcm->tmp_samples = 0;
                  avi_pcm->in_samples = 0;
                  avi_pcm->out_samples = 0;
                  avi_pcm->pos = -1;
                  avi_pcm->file = NULL;
                  avi_pcm->marker = -1;
                  return FALSE;
                }
              g_free (e_buf);
              avi_pcm->acm_object = NULL;
              avi_pcm_add_buffer (avi_pcm, conv_buf, conv_samples);
              g_free (conv_buf);
            }
          if (wfx_get_format_tag (avi_pcm->file->wfx) != WAVE_FMT_PCM)
            {
              gint header_bytes;

              avi_pcm->acm_object
                        = acm_open (wfx_get_format_tag (avi_pcm->file->wfx));
              if (!avi_pcm->acm_object
                    || (header_bytes = acm_decompress_get_format_size
                                (avi_pcm->acm_object, avi_pcm->file->wfx)) <= 0
                    || !(wfx = g_malloc (header_bytes))
                    || !acm_decompress_get_format (avi_pcm->acm_object,
                                                    avi_pcm->file->wfx, wfx)
                    || !acm_decompress_begin (avi_pcm->acm_object,
                                                        avi_pcm->file->wfx))
                {
                  g_free (wfx);
                  g_free (file_buf);
                  acm_close (avi_pcm->acm_object);
                  avi_pcm->acm_object = NULL;
                  if (avi_pcm->wcv)
                    {
                      wave_convert_end (avi_pcm->wcv, NULL, NULL);
                      avi_pcm->wcv = NULL;
                    }
                  g_free (avi_pcm->wfx);
                  avi_pcm->wfx = NULL;
                  g_free (avi_pcm->tmp_buf);
                  avi_pcm->tmp_buf = NULL;
                  avi_pcm->tmp_samples = 0;
                  avi_pcm->in_samples = 0;
                  avi_pcm->out_samples = 0;
                  avi_pcm->pos = -1;
                  avi_pcm->file = NULL;
                  avi_pcm->marker = -1;
                  return FALSE;
                }
            }
          if (!avi_pcm->wcv || !wf_header_cmp (avi_pcm->wfx,
                                            wfx ? wfx : avi_pcm->file->wfx))
            {
              if (avi_pcm->wcv)
                {
                  if (!wave_convert_end (avi_pcm->wcv,
                                                    &conv_buf, &conv_samples))
                    {
                      g_free (wfx);
                      g_free (file_buf);
                      avi_pcm->wcv = NULL;
                      g_free (avi_pcm->wfx);
                      avi_pcm->wfx = NULL;
                      g_free (avi_pcm->tmp_buf);
                      avi_pcm->tmp_buf = NULL;
                      avi_pcm->tmp_samples = 0;
                      avi_pcm->in_samples = 0;
                      avi_pcm->out_samples = 0;
                      avi_pcm->pos = -1;
                      avi_pcm->file = NULL;
                      avi_pcm->marker = -1;
                      return FALSE;
                    }
                  avi_pcm->wcv = NULL;
                  avi_pcm_add_buffer (avi_pcm, conv_buf, conv_samples);
                }
              avi_pcm->wcv = wave_convert_begin
                            (wfx ? wfx : avi_pcm->file->wfx, avi_edit->wfx);
              if (!avi_pcm->wcv)
                {
                  g_free (wfx);
                  g_free (file_buf);
                  acm_close (avi_pcm->acm_object);
                  avi_pcm->acm_object = NULL;
                  g_free (avi_pcm->wfx);
                  avi_pcm->wfx = NULL;
                  g_free (avi_pcm->tmp_buf);
                  avi_pcm->tmp_buf = NULL;
                  avi_pcm->tmp_samples = 0;
                  avi_pcm->in_samples = 0;
                  avi_pcm->out_samples = 0;
                  avi_pcm->pos = -1;
                  avi_pcm->file = NULL;
                  avi_pcm->marker = -1;
                  return FALSE;
                }
            }
          g_free (wfx);
          g_free (avi_pcm->wfx);
          avi_pcm->wfx = g_memdup (avi_pcm->file->wfx,
                                        wf_header_bytes (avi_pcm->file->wfx));
        }
      if (avi_pcm->acm_object)
        {
          gpointer e_buf = NULL;
          gsize e_samples;

          if (!acm_decompress (avi_pcm->acm_object,
                                    file_buf, file_samples, &e_buf, &e_samples)
                || !wave_convert (avi_pcm->wcv,
                                e_buf, e_samples, &conv_buf, &conv_samples))
            {
              g_free (e_buf);
              g_free (file_buf);
              acm_close (avi_pcm->acm_object);
              avi_pcm->acm_object = NULL;
              wave_convert_end (avi_pcm->wcv, NULL, NULL);
              avi_pcm->wcv = NULL;
              g_free (avi_pcm->wfx);
              avi_pcm->wfx = NULL;
              g_free (avi_pcm->tmp_buf);
              avi_pcm->tmp_buf = NULL;
              avi_pcm->tmp_samples = 0;
              avi_pcm->in_samples = 0;
              avi_pcm->out_samples = 0;
              avi_pcm->pos = -1;
              avi_pcm->file = NULL;
              avi_pcm->marker = -1;
              return FALSE;
            }
          g_free (e_buf);
        }
      else if (!wave_convert (avi_pcm->wcv,
                            file_buf, file_samples, &conv_buf, &conv_samples))
        {
          g_free (file_buf);
          wave_convert_end (avi_pcm->wcv, NULL, NULL);
          avi_pcm->wcv = NULL;
          g_free (avi_pcm->wfx);
          avi_pcm->wfx = NULL;
          g_free (avi_pcm->tmp_buf);
          avi_pcm->tmp_buf = NULL;
          avi_pcm->tmp_samples = 0;
          avi_pcm->in_samples = 0;
          avi_pcm->out_samples = 0;
          avi_pcm->pos = -1;
          avi_pcm->file = NULL;
          avi_pcm->marker = -1;
          return FALSE;
        }
      g_free (file_buf);
      avi_pcm_add_buffer (avi_pcm, conv_buf, conv_samples);
      g_free (conv_buf);
      avi_pcm->marker += file_samples;
      virtual_samples -= file_samples;
      if (virtual_samples <= 0)
        virtual_samples = MAX (avi_audio_real_to_virtual (pcm_samples
                                    - avi_pcm->tmp_samples, avi_edit->wfx), 1);
    }
  avi_pcm->in_samples += samples;
  avi_pcm->out_samples += pcm_samples;
  avi_pcm->pos = start + samples;
  if (avi_pcm->tmp_samples < pcm_samples)
    {
      gpointer silent_buf;
      gsize silent_samples;

      silent_samples = pcm_samples - avi_pcm->tmp_samples;
      silent_buf = wave_fill_silent_pcm (avi_edit->wfx, silent_samples);
      avi_pcm_add_buffer (avi_pcm, silent_buf, silent_samples);
      g_free (silent_buf);
    }
  result_samples = MIN (avi_pcm->tmp_samples, pcm_samples);
  result_bytes = result_samples * wfx_get_block_align (avi_edit->wfx);
  result_buf = g_malloc (result_bytes);
  g_memmove (result_buf, avi_pcm->tmp_buf, result_bytes);
  avi_pcm->tmp_samples -= result_samples;
  tmp_bytes = avi_pcm->tmp_samples * wfx_get_block_align (avi_edit->wfx);
  g_memmove (avi_pcm->tmp_buf,
                        (guint8 *)avi_pcm->tmp_buf + result_bytes, tmp_bytes);
  avi_pcm->tmp_buf = g_realloc (avi_pcm->tmp_buf, tmp_bytes);
  if (!result_buf)
    return FALSE;
  if (out_buf)
    *out_buf = result_buf;
  else
    g_free (result_buf);
  if (out_samples)
    *out_samples = result_samples;
  return TRUE;
}
