/*
    avicore
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "avibase.h"
#include "aviplay.h"
#ifdef USE_LIBAO
# include <ao/ao.h>
#endif /* USE_LIBAO */
#ifdef USE_ALSA
# include <alsa/asoundlib.h>
#endif /* USE_ALSA */
#ifdef G_OS_WIN32
# include <windows.h>
#endif /* G_OS_WIN32 */


static void avi_play_class_init (AviPlayClass *klass);
static void avi_play_init       (AviPlay      *play);
static void avi_play_destroy    (GtkObject    *object);


static GtkVBoxClass *parent_class = NULL;


enum
{
  STATUS_SIGNAL,
  LAST_SIGNAL
};


static gint avi_play_signals[LAST_SIGNAL] = {0};


/******************************************************************************
*                                                                             *
******************************************************************************/
GtkType
avi_play_get_type (void)
{
  static GType play_type = 0;

  if (!play_type)
    {
      const static GTypeInfo play_info =
      {
        sizeof (AviPlayClass),
        NULL,               /* base_init */
        NULL,               /* base_finalize */
        (GClassInitFunc)avi_play_class_init,
        NULL,               /* class_finalize */
        NULL,               /* class_data */
        sizeof (AviPlay),
        0,                  /* n_preallocs */
        (GInstanceInitFunc)avi_play_init,
      };

      play_type = g_type_register_static (GTK_TYPE_VBOX, "AviPlay",
                                                                &play_info, 0);
    }

  return play_type;
}


static void
avi_play_class_init (AviPlayClass *klass)
{
  GObjectClass *gobject_class;
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  gobject_class = (GObjectClass *)klass;
  object_class = (GtkObjectClass *)klass;
  widget_class = (GtkWidgetClass *)klass;

  parent_class = g_type_class_peek_parent (klass);

  object_class->destroy = avi_play_destroy;

  avi_play_signals[STATUS_SIGNAL]
        = g_signal_new ("status",
                G_TYPE_FROM_CLASS (klass),
                G_SIGNAL_RUN_FIRST | G_SIGNAL_DETAILED,
                G_STRUCT_OFFSET (AviPlayClass, status),
                NULL, NULL,
                g_cclosure_marshal_VOID__STRING,
                G_TYPE_NONE, 1,
                G_TYPE_STRING);
}


/*  ja:AVIの位置を設定する
         play,ウィジェット
    posisiton,位置                                                          */
static void
avi_play_set_position_internal (AviPlay    *play,
                                const gint  position)
{
  gboolean redraw;
  gchar *text;
  gint t;

  redraw = play->avi_frame && play->position
                    != MIN (avi_edit_length (play->avi_edit[0]) - 1, position);
  play->position = position;
  if (redraw)
    {
      GdkRectangle rc;

      rc.width = MIN (avi_edit_get_width (play->avi_edit[0])
                                    * play->drawing->allocation.height
                                    / avi_edit_get_height (play->avi_edit[0]),
                                            play->drawing->allocation.width);
      rc.height = MIN (avi_edit_get_height (play->avi_edit[0])
                                    * play->drawing->allocation.width
                                    / avi_edit_get_width (play->avi_edit[0]),
                                            play->drawing->allocation.height);
      rc.x = (play->drawing->allocation.width - rc.width) / 2;
      rc.y = (play->drawing->allocation.height - rc.height) / 2;
      gtk_widget_draw (play->drawing, &rc);
    }
  if (play->hbox)
    {
      /* ja:ボタン */
      gtk_widget_set_sensitive (play->play_button,
        !play->playing && (play->position < play->length - 1 || play->gap));
      gtk_widget_set_sensitive (play->stop_button, play->playing);
      gtk_widget_set_sensitive (play->previous_button,
                                        !play->playing && play->position > 0);
      gtk_widget_set_sensitive (play->rewind_button,
                                        !play->playing && play->position > 0);
      gtk_widget_set_sensitive (play->forward_button,
                        !play->playing && play->position < play->length - 1);
      gtk_widget_set_sensitive (play->next_button,
                        !play->playing && play->position < play->length - 1);
    }
  if (play->avi_edit[0])
    t = avi_time_from_sample (play->avi_edit[0], play->position);
  else if (play->avi_edit[1])
    t = avi_time_from_sample (play->avi_edit[1], play->position);
  else
    t = 0;
#ifdef USE_ALSA
  text = play->err
        ? g_strdup_printf (_("Frame %d, Time %02d:%02d:%02d %s"),
                    play->position,
                    t / 60000 % 100, t / 1000 % 60, t / 10 % 100, play->err)
        : g_strdup_printf (_("Frame %d, Time %02d:%02d:%02d"),
                    play->position,
                    t / 60000 % 100, t / 1000 % 60, t / 10 % 100);
#else /* not USE_ALSA */
  text = g_strdup_printf (_("Frame %d, Time %02d:%02d:%02d"),
                                play->position,
                                t / 60000 % 100, t / 1000 % 60, t / 10 % 100);
#endif /* not USE_ALSA */
  g_signal_emit_by_name (G_OBJECT (play), "status", text);
  if (play->label)
    gtk_label_set_text (GTK_LABEL (play->label), text);
  g_free (text);
}


/*  ja:AVIの位置を設定する
         play,ウィジェット
    posisiton,位置                                                          */
static void
avi_play_set_position_real (AviPlay    *play,
                            const gint  position)
{
  if (play)
    {
      if (play->hscale)
        gtk_range_set_value (GTK_RANGE (play->hscale), position);
      else if (play->position != position)
        avi_play_set_position_internal (play, position);
    }
}


/* ja:描画 */
static gboolean
avi_play_expose (GtkWidget      *widget,
                 GdkEventExpose *event,
                 AviPlay        *play)
{
  gint x, y, width, height;
  GdkGC *gc;
  GdkRectangle rc0, rc1;

  if (play->avi_frame)
    {
      width = MIN (avi_edit_get_width (play->avi_edit[0])
                                                    * widget->allocation.height
                                    / avi_edit_get_height (play->avi_edit[0]),
                                                    widget->allocation.width);
      height = MIN (avi_edit_get_height (play->avi_edit[0])
                                                    * widget->allocation.width
                                    / avi_edit_get_width (play->avi_edit[0]),
                                                    widget->allocation.height);
    }
  else
    {
      width = 0;
      height = 0;
    }
  x = (widget->allocation.width - width) / 2;
  y = (widget->allocation.height - height) / 2;

  gc = gdk_gc_new (widget->window);
  gdk_gc_set_clip_rectangle (gc, &event->area);
  /* ja:背景 */
  gdk_gc_set_foreground (gc, &play->color);
  if (width < widget->allocation.width)
    {
      rc0.x = rc0.y = 0;
      rc0.width = x;
      rc0.height = widget->allocation.height;
      if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        gdk_draw_rectangle (widget->window, gc, TRUE, rc1.x, rc1.y,
                                                        rc1.width, rc1.height);
      rc0.x = x + width;
      rc0.width = widget->allocation.width - rc0.x;
      if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        gdk_draw_rectangle (widget->window, gc, TRUE, rc1.x, rc1.y,
                                                        rc1.width, rc1.height);
    }
  if (height < widget->allocation.height)
    {
      rc0.x = rc0.y = 0;
      rc0.width = widget->allocation.width;
      rc0.height = y;
      if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        gdk_draw_rectangle (widget->window, gc, TRUE, rc1.x, rc1.y,
                                                        rc1.width, rc1.height);
      rc0.y = y + height;
      rc0.height = widget->allocation.height - rc0.y;
      if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        gdk_draw_rectangle (widget->window, gc, TRUE, rc1.x, rc1.y,
                                                        rc1.width, rc1.height);
    }
  /* ja:フレーム */
  if (play->avi_frame)
    {
      guchar *buf;

      buf = avi_frame_get_raw (play->avi_frame,
                MIN (play->position, avi_edit_length (play->avi_edit[0]) - 1),
                                                                width, height);
      if (buf)
        gdk_draw_rgb_32_image (widget->window, gc, x, y, width, height,
                                        GDK_RGB_DITHER_NORMAL, buf, width * 4);
    }
  gdk_gc_destroy (gc);
  return TRUE;
}


/* ja:タイマ */
static gboolean
avi_play_timeout (AviPlay *play)
{
  gint position, t;

#ifdef G_OS_WIN32
  t = timeGetTime () - play->dwTime;
#else /* not G_OS_WIN32 */
  t = g_timer_elapsed (play->gtimer, NULL) * 1000;
#endif /* not G_OS_WIN32 */
  if (play->avi_edit[0])
    position = avi_time_to_sample (play->avi_edit[0], t) + play->start;
  else if (play->avi_edit[1])
    position = avi_time_to_sample (play->avi_edit[1], t) + play->start;
  else
    position = 0;
  if (position >= play->length)
    {
      position = play->length - 1;
      avi_play_stop (play);
    }
  if (play->position != position)
    avi_play_set_position_real (play, position);
  return TRUE;
}


#ifdef USE_LIBAO
/* ja:スレッド */
static gpointer
avi_play_libao (AviPlay *play)
{
  ao_sample_format format;
  ao_device *device;

  format.bits = avi_edit_get_bits_per_sample (play->avi_edit[1]);
  format.rate = avi_edit_get_samples_per_sec (play->avi_edit[1]);
  format.channels = avi_edit_get_channels (play->avi_edit[1]);
  format.byte_format = AO_FMT_LITTLE;
  device = ao_open_live (play->driver_id, &format, NULL);
  if (device)
    {
      gint start, length;

      start = play->avi_edit[0] ? avi_time_sync_sample (play->avi_edit[0],
                                play->start, play->avi_edit[1]) : play->start;
      length = avi_edit_length (play->avi_edit[1]);
      while (play->playing && play->pos < length)
        {
          gint pos;
          gsize samples;

          pos = MIN (length, g_timer_elapsed (play->gtimer, NULL)
                                            * AVI_PCM_SAMPLES_PER_SEC + start);
          if (play->pos < pos)
            play->pos = pos;
          samples = pos + AVI_PCM_BUFFER_LENGTH * AVI_PCM_BUFFER_NUM
                                                                - play->pos;
          if (samples > length - play->pos)
            samples = length - play->pos;
          else if (samples < AVI_PCM_BUFFER_LENGTH)
            samples = 0;
          if (samples > 0)
            {
              gpointer out_buf;
              gsize out_samples;

              if (!avi_pcm_get_raw (play->avi_pcm,
                                play->pos, samples, &out_buf, &out_samples))
                break;
              if (out_samples > 0)
                {
                  gboolean result;
                  gsize bytes;

                  bytes = out_samples * play->block_align;
                  if (format.bits == 8)
                    {
                      gint i;

                      for (i = 0; i < bytes; i++)
                        ((gint8 *)out_buf)[i] = ((guint8 *)out_buf)[i] - 128;
                    }
                  result = ao_play (device, out_buf, bytes);
                  g_free (out_buf);
                  play->pos += samples;
                  if (!result)
                    break;
                }
            }
        }
      ao_close (device);
    }
  return NULL;
}
#endif /* USE_LIBAO */


#ifdef USE_ALSA
/* ja:スレッド */
static gpointer
avi_play_alsa (AviPlay *play)
{
  int e = 0;
  snd_pcm_uframes_t uframes = 0;
  snd_pcm_t *handle;

  if (snd_pcm_open (&handle, "default",
                            SND_PCM_STREAM_PLAYBACK, SND_PCM_NONBLOCK) == 0)
    {
      snd_pcm_hw_params_t *hw_params;

      if ((e = snd_pcm_nonblock (handle, FALSE)) == 0
                        && (e = snd_pcm_hw_params_malloc (&hw_params)) == 0)
        {
          unsigned buffer_time = 500 * 1000, period_time = 50 * 1000;

          if ((e = snd_pcm_hw_params_any (handle, hw_params)) == 0
                && (e = snd_pcm_hw_params_set_access (handle,
                                hw_params, SND_PCM_ACCESS_RW_INTERLEAVED)) == 0
                && (e = snd_pcm_hw_params_set_format (handle, hw_params,
                        avi_edit_get_bits_per_sample (play->avi_edit[1]) == 8
                            ? SND_PCM_FORMAT_U8 : SND_PCM_FORMAT_S16_LE)) == 0
                && (e = snd_pcm_hw_params_set_channels (handle, hw_params,
                            avi_edit_get_channels (play->avi_edit[1]))) == 0
                && (e = snd_pcm_hw_params_set_rate (handle, hw_params,
                    avi_edit_get_samples_per_sec (play->avi_edit[1]), 0)) == 0
                && (e = snd_pcm_hw_params_set_buffer_time_near (handle,
                                            hw_params, &buffer_time, 0)) == 0
                && (e = snd_pcm_hw_params_set_period_time_near (handle,
                                            hw_params, &period_time, 0)) == 0
                && (e = snd_pcm_hw_params (handle, hw_params)) == 0)
            {
              snd_pcm_uframes_t buffer_size, period_size;
              snd_pcm_sw_params_t *sw_params;

              if ((e = snd_pcm_hw_params_get_buffer_size (hw_params,
                                                            &buffer_size)) == 0
                    && (e = snd_pcm_hw_params_get_period_size (hw_params,
                                                        &period_size, 0)) == 0
                    && (e = snd_pcm_sw_params_malloc (&sw_params)) == 0)
                {
                  uframes = (glonglong)
                            avi_edit_get_samples_per_sec (play->avi_edit[1])
                            * AVI_PCM_BUFFER_LENGTH * (AVI_PCM_BUFFER_NUM + 1)
                                                    / AVI_PCM_SAMPLES_PER_SEC;
                  if ((e = snd_pcm_sw_params_current (handle, sw_params)) != 0
                        || (e = snd_pcm_sw_params_set_avail_min (handle,
                                                    sw_params, uframes)) != 0
                        || (e = snd_pcm_sw_params_set_start_threshold (handle,
                            sw_params, /* buffer_size - period_size */ 0)) != 0
                        || (e = snd_pcm_sw_params (handle, sw_params)) != 0
                        || (e = snd_pcm_prepare (handle)) != 0)
                    uframes = 0;
                  snd_pcm_sw_params_free (sw_params);
                }
            }
          snd_pcm_hw_params_free (hw_params);
        }
      if (uframes <= 0)
        snd_pcm_close (handle);
    }
  if (uframes > 0)
    {
      gpointer c_buf = NULL;
      gsize c_samples = 0;
      gint start, length;
      guint32 samples_per_sec;

      start = play->avi_edit[0] ? avi_time_sync_sample (play->avi_edit[0],
                                play->start, play->avi_edit[1]) : play->start;
      length = avi_edit_length (play->avi_edit[1]);
      samples_per_sec = avi_edit_get_samples_per_sec (play->avi_edit[1]);
      while (play->playing && play->pos < length)
        {
          gint pos;
          gsize samples;
          snd_pcm_sframes_t frames;

          e = frames = snd_pcm_avail_update (handle);
          if (e == -EPIPE && snd_pcm_state (handle) == SND_PCM_STATE_XRUN
                                        && (e = snd_pcm_prepare (handle)) == 0)
            e = frames = snd_pcm_avail_update (handle);
          if (e < 0)
            break;
          if (frames > uframes)
            frames = uframes;
          pos = MIN (length, g_timer_elapsed (play->gtimer, NULL)
                                            * AVI_PCM_SAMPLES_PER_SEC + start);
          if (play->pos < pos)
            play->pos = pos;
          samples = MIN (length - play->pos, (glonglong)frames
                                * AVI_PCM_SAMPLES_PER_SEC / samples_per_sec);
          while (play->playing && samples > 0 && c_samples < frames)
            {
              gpointer out_buf;
              gsize out_samples;

              if (avi_pcm_get_raw (play->avi_pcm,
                                play->pos, samples, &out_buf, &out_samples))
                {
                  c_buf = g_realloc (c_buf,
                                (c_samples + out_samples) * play->block_align);
                  g_memmove ((guint8 *)c_buf + c_samples * play->block_align,
                                    out_buf, out_samples * play->block_align);
                  c_samples += out_samples;
                  g_free (out_buf);
                  play->pos += samples;
                }
              samples = MIN (length - play->pos, AVI_PCM_SAMPLES_PER_SEC);
            }
          samples = MIN (c_samples, frames);
          if (samples > 0)
            {
              e = snd_pcm_writei (handle, c_buf, samples);
              if (e >= 0)
                {
                  c_samples -= samples;
                  g_memmove (c_buf,
                                (guint8 *)c_buf + samples * play->block_align,
                                                c_samples * play->block_align);
                  c_buf = g_realloc (c_buf, c_samples * play->block_align);
                }
              else if (e != -EPIPE
                                || snd_pcm_state (handle) != SND_PCM_STATE_XRUN
                                || (e = snd_pcm_prepare (handle)) != 0)
                {
                  break;
                }
            }
        }
      g_free (c_buf);
      snd_pcm_close (handle);
    }
  if (e < 0 && !play->err)
    play->err = g_locale_to_utf8 (snd_strerror (e), -1, NULL, NULL, NULL);
  play->thread = NULL;
  return NULL;
}
#endif /* USE_ALSA */


#ifdef G_OS_WIN32
static LRESULT CALLBACK
WndProc (HWND   hWnd,
         UINT   uMsg,
         WPARAM wParam,
         LPARAM lParam)
{
  if (uMsg == WM_USER)
    {
      AviPlay *play;
      LPWAVEHDR lpwhrOut;

      play = GUINT_TO_POINTER (wParam);
      lpwhrOut = GUINT_TO_POINTER (lParam);
      waveOutUnprepareHeader (play->hWaveOut, lpwhrOut, sizeof (WAVEHDR));
      g_free (lpwhrOut->lpData);
      lpwhrOut->lpData = NULL;
      lpwhrOut->dwBufferLength = 0;
      if (play->pos >= 0)
        {
          gpointer out_buf;
          gsize samples, out_samples;

          samples = MIN (avi_edit_length (play->avi_edit[1]) - play->pos,
                                                        AVI_PCM_BUFFER_LENGTH);
          if (samples > 0 && avi_pcm_get_raw (play->avi_pcm,
                                play->pos, samples, &out_buf, &out_samples))
            {
              lpwhrOut->lpData = out_buf;
              lpwhrOut->dwBufferLength = out_samples * play->block_align;
              if (waveOutPrepareHeader (play->hWaveOut,
                            lpwhrOut, sizeof (WAVEHDR)) == MMSYSERR_NOERROR)
                {
                  waveOutWrite (play->hWaveOut, lpwhrOut, sizeof (WAVEHDR));
                  play->pos += samples;
                }
              else
                {
                  g_free (lpwhrOut->lpData);
                  lpwhrOut->lpData = NULL;
                  lpwhrOut->dwBufferLength = 0;
                }
            }
        }
    }
  return DefWindowProc (hWnd, uMsg, wParam, lParam);
}


static VOID CALLBACK
waveOutProc (HWAVEOUT hWaveOut,
             UINT     uMsg,
             DWORD    dwInstance,
             DWORD    dwParam1,
             DWORD    dwParam2)
{
  if (uMsg == WOM_DONE)
    {
      AviPlay *play;

      play = GUINT_TO_POINTER (dwInstance);
      if (play->hWnd)
        PostMessage (play->hWnd, WM_USER, dwInstance, dwParam1);
    }
}
#endif /* G_OS_WIN32 */


/* ja:再生ボタンが押された */
static void
avi_play_clicked_button_play (GtkWidget *widget,
                              AviPlay   *play)
{
  avi_play_play (play);
}


/* ja:停止ボタンが押された */
static void
avi_play_clicked_button_stop (GtkWidget *widget,
                              AviPlay   *play)
{
  avi_play_stop (play);
}


/* ja:最初に巻戻しボタンが押された */
static void
avi_play_clicked_button_previous (GtkWidget *widget,
                                  AviPlay   *play)
{
  avi_play_set_position_real (play, 0);
}


/* ja:巻戻しボタンが押された */
static void
avi_play_clicked_button_rewind (GtkWidget *widget,
                                AviPlay   *play)
{
  avi_play_set_position_real (play, play->position - 1);
}


/* ja:早送りボタンが押された */
static void
avi_play_clicked_button_forward (GtkWidget *widget,
                                 AviPlay   *play)
{
  avi_play_set_position_real (play, play->position + 1);
}


/* ja:最後に早送りボタンが押された */
static void
avi_play_clicked_button_next (GtkWidget *widget,
                              AviPlay   *play)
{
  avi_play_set_position_real (play, play->length);
}


/* ja:スケールの値が変更された */
static void
avi_play_value_changed (GtkWidget *widget,
                        AviPlay   *play)
{
  avi_play_set_position_internal (play,
                                    gtk_range_get_value (GTK_RANGE (widget)));
}


/* ja:破棄 */
static void
avi_play_destroy (GtkObject *object)
{
  AviPlay *play;

  play = AVI_PLAY (object);
  avi_play_stop (play);
  if (play->avi_frame)
    {
      avi_frame_close (play->avi_frame);
      play->avi_frame = NULL;
    }
  avi_edit_close (play->avi_edit[0]);
  avi_edit_close (play->avi_edit[1]);
  play->avi_edit[0] = play->avi_edit[1] = NULL;
#ifdef USE_LIBAO
  ao_shutdown ();
#endif /* USE_LIBAO */
#ifdef G_OS_WIN32
  if (play->hWnd)
    {
      DestroyWindow (play->hWnd);
      play->hWnd = NULL;
    }
#endif /* G_OS_WIN32 */
  GTK_OBJECT_CLASS (parent_class)->destroy (object);
}


static void
avi_play_init (AviPlay *play)
{
  GtkStyle *style;
#ifdef G_OS_WIN32
  static gboolean register_class = FALSE;
#endif /* G_OS_WIN32 */

  play->shrink = FALSE;
  play->panel = FALSE;
  play->playing = FALSE;
  play->gap = FALSE;
  play->length = 0;
  play->position = 0;
  play->pos = -1;
  play->start = -1;
  play->timer_id = 0;
  play->block_align = 0;
  play->avi_edit[0] = play->avi_edit[1] = NULL;
  play->avi_frame = NULL;
  play->avi_pcm = NULL;
#ifdef USE_LIBAO
  ao_initialize ();
  play->driver_id = ao_default_driver_id ();
  play->thread = NULL;
#endif /* USE_LIBAO */
#ifdef USE_ALSA
  play->err = NULL;
  play->thread = NULL;
#endif /* USE_ALSA */
#ifdef G_OS_WIN32
  play->dwTime = 0;
  if (!register_class)
    {
      WNDCLASSEX wcxOut;

      g_memset (&wcxOut, 0, sizeof (WNDCLASSEX));
      wcxOut.cbSize = sizeof (WNDCLASSEX);
      wcxOut.lpfnWndProc = WndProc;
      wcxOut.hInstance = GetModuleHandle (NULL);
      wcxOut.lpszClassName = _T("Video maid Preview Window");
      register_class = RegisterClassEx (&wcxOut) != 0;
    }
  play->hWnd = CreateWindow (_T("Video maid Preview Window"),
                             NULL,
                             WS_DISABLED | WS_POPUP,
                             CW_USEDEFAULT,
                             CW_USEDEFAULT,
                             CW_USEDEFAULT,
                             CW_USEDEFAULT,
                             NULL,
                             NULL,
                             GetModuleHandle (NULL),
                             NULL);
  play->hWaveOut = NULL;
  g_memset (play->whrOut, 0, AVI_PCM_BUFFER_NUM * sizeof (WAVEHDR));
#else /* not G_OS_WIN32 */
  play->gtimer = NULL;
#endif /* not G_OS_WIN32 */

  style = gtk_widget_get_style (GTK_WIDGET (play));
  play->color = style->base[0];
  gdk_colormap_alloc_color (gdk_colormap_get_system (),
                                                    &play->color, FALSE, TRUE);

  /* ja:描画領域 */
  play->drawing = gtk_drawing_area_new ();
  g_signal_connect (G_OBJECT (play->drawing), "expose-event",
                                        G_CALLBACK (avi_play_expose), play);
  gtk_widget_show (play->drawing);
  gtk_box_pack_start (GTK_BOX (play), play->drawing, TRUE, TRUE, 0);

  play->play_button = NULL;
  play->stop_button = NULL;
  play->previous_button = NULL;
  play->rewind_button = NULL;
  play->forward_button = NULL;
  play->next_button = NULL;
  play->hscale = NULL;
  play->label = NULL;
  play->hbox = NULL;
}


/******************************************************************************
*                                                                             *
* ja:ダイアログ関数                                                           *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    RET,ウィジェット                                                        */
GtkWidget *
avi_play_new (void)
{
  return GTK_WIDGET (g_object_new (AVI_TYPE_PLAY, NULL));
}


/*  ja:パネルを表示する
    play,ウィジェット                                                       */
void
avi_play_show_panel (AviPlay *play)
{
  gchar *text;
  gint t;
  GtkRequisition req;
  GtkWidget *hbox0, *hbox1, *hbox2, *vbox;
#if ! GTK_CHECK_VERSION(2,6,0)
  GdkPixbuf *pixbuf;
/* XPM */
  const static gchar *media_forward16_xpm[] = {
"16 16 28 1",
"   c None",
".  c #000000",
"+  c #727170",
"@  c #7B7B7A",
"#  c #6C6B6A",
"$  c #7F7E7D",
"%  c #999996",
"&  c #7D7C7B",
"*  c #828180",
"=  c #9D9C9A",
"-  c #B0AFAC",
";  c #908F8D",
">  c #8C8B89",
",  c #A9A8A5",
"'  c #BDBCB9",
")  c #C5C3C0",
"!  c #D0CECC",
"~  c #FFFFFF",
"{  c #8F8E8C",
"]  c #B5B3B1",
"^  c #C7C6C3",
"/  c #D2D1CE",
"(  c #989898",
"_  c #B9B6B4",
":  c #D8D7D5",
"<  c #A19F9D",
"[  c #C7C6C4",
"}  c #A7A6A3",
"                ",
" .      .       ",
" ..     ..      ",
" .+.    .+.     ",
" .@#.   .@#.    ",
" .$%&.  .$%&.   ",
" .*=-;...*=-;.  ",
" .>,')!..>,')!.~",
" .{]^/.(.{]^/.~ ",
" .=_:.~ .=_:.~  ",
" .<[.~  .<[.~   ",
" .}.~   .}.~    ",
" ..~    ..~     ",
" .~     .~      ",
" ~      ~       ",
"                "};


/* XPM */
  const static gchar *media_next16_xpm[] = {
"16 16 55 1",
"   c None",
".  c #000000",
"+  c #FFFFFF",
"@  c #737372",
"#  c #848483",
"$  c #888887",
"%  c #727170",
"&  c #7E7E7D",
"*  c #A8A7A6",
"=  c #A4A4A3",
"-  c #7B7B7A",
";  c #6C6B6A",
">  c #A6A5A4",
",  c #B5B4B3",
"'  c #7F7E7D",
")  c #999996",
"!  c #7D7C7B",
"~  c #858483",
"{  c #AAA9A7",
"]  c #C0BFBE",
"^  c #828180",
"/  c #9D9C9A",
"(  c #B0AFAC",
"_  c #908F8D",
":  c #91908E",
"<  c #AFAEAC",
"[  c #CACAC8",
"}  c #8C8B89",
"|  c #A9A8A5",
"1  c #BDBCB9",
"2  c #C5C3C0",
"3  c #D0CECC",
"4  c #CDCDCD",
"5  c #8F8E8C",
"6  c #B5B3B1",
"7  c #C7C6C3",
"8  c #D2D1CE",
"9  c #B9B6B4",
"0  c #D8D7D5",
"a  c #A6A5A2",
"b  c #BBBAB7",
"c  c #D4D3D2",
"d  c #A19F9D",
"e  c #C7C6C4",
"f  c #AEADA9",
"g  c #BFBDBA",
"h  c #D7D5D3",
"i  c #A7A6A3",
"j  c #B2B0AD",
"k  c #C3C1BE",
"l  c #D9D8D6",
"m  c #B5B3B0",
"n  c #D1CFCC",
"o  c #DEDDDB",
"p  c #F3F3F3",
"                ",
"  .       .....+",
"  ..      .@#$.+",
"  .%.     .&*=.+",
"  .-;.    .&>,.+",
"  .')!.   .~{].+",
"  .^/(_.  .:<[.+",
"  .}|123.4.:<[.+",
"  .5678.+ .:<[.+",
"  ./90.+  .abc.+",
"  .de.+   .fgh.+",
"  .i.+    .jkl.+",
"  ..+     .mno.+",
"  .+      .....+",
"  +       +++++p",
"                "};


/* XPM */
  const static gchar *media_play16_xpm[] = {
"16 16 27 1",
"   c None",
".  c #000000",
"+  c #727170",
"@  c #7B7B7A",
"#  c #6C6B6A",
"$  c #7F7E7D",
"%  c #999996",
"&  c #7D7C7B",
"*  c #828180",
"=  c #9D9C9A",
"-  c #B0AFAC",
";  c #908F8D",
">  c #8C8B89",
",  c #A9A8A5",
"'  c #BDBCB9",
")  c #C5C3C0",
"!  c #D0CECC",
"~  c #FFFFFF",
"{  c #8F8E8C",
"]  c #B5B3B1",
"^  c #C7C6C3",
"/  c #D2D1CE",
"(  c #B9B6B4",
"_  c #D8D7D5",
":  c #A19F9D",
"<  c #C7C6C4",
"[  c #A7A6A3",
"                ",
"     .          ",
"     ..         ",
"     .+.        ",
"     .@#.       ",
"     .$%&.      ",
"     .*=-;.     ",
"     .>,')!.~   ",
"     .{]^/.~    ",
"     .=(_.~     ",
"     .:<.~      ",
"     .[.~       ",
"     ..~        ",
"     .~         ",
"     ~          ",
"                "};


/* XPM */
  const static gchar *media_previous16_xpm[] = {
"16 16 56 1",
"   c None",
".  c #000000",
"+  c #FFFFFF",
"@  c #D9D9D9",
"#  c #737372",
"$  c #848483",
"%  c #888887",
"&  c #7E7E7D",
"*  c #A8A7A6",
"=  c #A4A4A3",
"-  c #696867",
";  c #A6A5A4",
">  c #B5B4B3",
",  c #5B5A59",
"'  c #9C9C9A",
")  c #858483",
"!  c #AAA9A7",
"~  c #C0BFBE",
"{  c #636261",
"]  c #908F8D",
"^  c #C5C4C3",
"/  c #91908E",
"(  c #AFAEAC",
"_  c #CACAC8",
":  c #6F6E6C",
"<  c #959491",
"[  c #C0BFBD",
"}  c #D3D3D1",
"|  c #E4E4E4",
"1  c #A2A09F",
"2  c #A6A4A2",
"3  c #BDBCB9",
"4  c #C6C5C3",
"5  c #D4D3D2",
"6  c #A3A2A0",
"7  c #B7B6B3",
"8  c #D0CFCD",
"9  c #D4D4D3",
"0  c #A6A5A2",
"a  c #BBBAB7",
"b  c #A8A7A5",
"c  c #D5D5D3",
"d  c #DFDFDD",
"e  c #AEADA9",
"f  c #BFBDBA",
"g  c #D7D5D3",
"h  c #C4C4C2",
"i  c #DFDFDE",
"j  c #B2B0AD",
"k  c #C3C1BE",
"l  c #D9D8D6",
"m  c #D4D3D1",
"n  c #B5B3B0",
"o  c #D1CFCC",
"p  c #DEDDDB",
"q  c #F3F3F3",
"                ",
"  .....+      .@",
"  .#$%.+     ..+",
"  .&*=.+    .-.+",
"  .&;>.+   .,'.+",
"  .)!~.+  .{]^.+",
"  ./(_.+ .:<[}.+",
"  ./(_.|.12345.+",
"  ./(_.+ .6789.+",
"  .0a5.+  .bcd.+",
"  .efg.+  +.hi.+",
"  .jkl.+   +.m.+",
"  .nop.+     ..+",
"  .....+     +.+",
"  +++++q        ",
"                "};


/* XPM */
  const static gchar *media_rewind16_xpm[] = {
"16 16 30 1",
"   c None",
".  c #000000",
"+  c #D9D9D9",
"@  c #FFFFFF",
"#  c #696867",
"$  c #5B5A59",
"%  c #9C9C9A",
"&  c #636261",
"*  c #908F8D",
"=  c #C5C4C3",
"-  c #6F6E6C",
";  c #959491",
">  c #C0BFBD",
",  c #D3D3D1",
"'  c #BBBBBB",
")  c #A2A09F",
"!  c #A6A4A2",
"~  c #BDBCB9",
"{  c #C6C5C3",
"]  c #D4D3D2",
"^  c #A3A2A0",
"/  c #B7B6B3",
"(  c #D0CFCD",
"_  c #D4D4D3",
":  c #A8A7A5",
"<  c #D5D5D3",
"[  c #DFDFDD",
"}  c #C4C4C2",
"|  c #DFDFDE",
"1  c #D4D3D1",
"                ",
"       .+     .+",
"      ..@    ..@",
"     .#.@   .#.@",
"    .$%.@  .$%.@",
"   .&*=.@ .&*=.@",
"  .-;>,.'.-;>,.@",
" .)!~{]..)!~{].@",
"  .^/(_.@.^/(_.@",
"   .:<[.@ .:<[.@",
"   @.}|.@ @.}|.@",
"    @.1.@  @.1.@",
"      ..@    ..@",
"      @.@    @.@",
"                ",
"                "};


/* XPM */
  const static gchar *media_stop16_xpm[] = {
"16 16 29 1",
"   c None",
".  c #000000",
"+  c #FFFFFF",
"@  c #737372",
"#  c #848483",
"$  c #888887",
"%  c #7E7E7D",
"&  c #A8A7A6",
"*  c #A4A4A3",
"=  c #A6A5A4",
"-  c #B5B4B3",
";  c #858483",
">  c #AAA9A7",
",  c #C0BFBE",
"'  c #91908E",
")  c #AFAEAC",
"!  c #CACAC8",
"~  c #A6A5A2",
"{  c #BBBAB7",
"]  c #D4D3D2",
"^  c #AEADA9",
"/  c #BFBDBA",
"(  c #D7D5D3",
"_  c #B2B0AD",
":  c #C3C1BE",
"<  c #D9D8D6",
"[  c #B5B3B0",
"}  c #D1CFCC",
"|  c #DEDDDB",
"                ",
" .............+ ",
" .@#$$$$$$$$$.+ ",
" .%&*********.+ ",
" .%=---------.+ ",
" .;>,,,,,,,,,.+ ",
" .')!!!!!!!!!.+ ",
" .')!!!!!!!!!.+ ",
" .~{]]]]]]]]].+ ",
" .~{]]]]]]]]].+ ",
" .^/(((((((((.+ ",
" ._:<<<<<<<<<.+ ",
" .[}|||||||||.+ ",
" .............+ ",
" ++++++++++++++ ",
"                "};
#endif /* not GTK_CHECK_VERSION(2,6,0) */

  if (!play || play->label || play->hbox)
    return;
  /* ja:ボタン */
  play->play_button = gtk_button_new ();
  play->stop_button = gtk_button_new ();
  play->previous_button = gtk_button_new ();
  play->rewind_button = gtk_button_new ();
  play->forward_button = gtk_button_new ();
  play->next_button = gtk_button_new ();
#if GTK_CHECK_VERSION(2,6,0)
  gtk_container_add (GTK_CONTAINER (play->play_button),
    gtk_image_new_from_stock (GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_BUTTON));
  gtk_container_add (GTK_CONTAINER (play->stop_button),
    gtk_image_new_from_stock (GTK_STOCK_MEDIA_STOP, GTK_ICON_SIZE_BUTTON));
  gtk_container_add (GTK_CONTAINER (play->previous_button),
    gtk_image_new_from_stock (GTK_STOCK_MEDIA_PREVIOUS, GTK_ICON_SIZE_BUTTON));
  gtk_container_add (GTK_CONTAINER (play->rewind_button),
    gtk_image_new_from_stock (GTK_STOCK_MEDIA_REWIND, GTK_ICON_SIZE_BUTTON));
  gtk_container_add (GTK_CONTAINER (play->forward_button),
    gtk_image_new_from_stock (GTK_STOCK_MEDIA_FORWARD, GTK_ICON_SIZE_BUTTON));
  gtk_container_add (GTK_CONTAINER (play->next_button),
    gtk_image_new_from_stock (GTK_STOCK_MEDIA_NEXT, GTK_ICON_SIZE_BUTTON));
#else /* not GTK_CHECK_VERSION(2,6,0) */
  pixbuf = gdk_pixbuf_new_from_xpm_data (media_play16_xpm);
  gtk_container_add (GTK_CONTAINER (play->play_button),
                                        gtk_image_new_from_pixbuf (pixbuf));
  g_object_unref (pixbuf);
  pixbuf = gdk_pixbuf_new_from_xpm_data (media_stop16_xpm);
  gtk_container_add (GTK_CONTAINER (play->stop_button),
                                        gtk_image_new_from_pixbuf (pixbuf));
  g_object_unref (pixbuf);
  pixbuf = gdk_pixbuf_new_from_xpm_data (media_previous16_xpm);
  gtk_container_add (GTK_CONTAINER (play->previous_button),
                                        gtk_image_new_from_pixbuf (pixbuf));
  g_object_unref (pixbuf);
  pixbuf = gdk_pixbuf_new_from_xpm_data (media_rewind16_xpm);
  gtk_container_add (GTK_CONTAINER (play->rewind_button),
                                        gtk_image_new_from_pixbuf (pixbuf));
  g_object_unref (pixbuf);
  pixbuf = gdk_pixbuf_new_from_xpm_data (media_forward16_xpm);
  gtk_container_add (GTK_CONTAINER (play->forward_button),
                                        gtk_image_new_from_pixbuf (pixbuf));
  g_object_unref (pixbuf);
  pixbuf = gdk_pixbuf_new_from_xpm_data (media_next16_xpm);
  gtk_container_add (GTK_CONTAINER (play->next_button),
                                        gtk_image_new_from_pixbuf (pixbuf));
  g_object_unref (pixbuf);
#endif /* not GTK_CHECK_VERSION(2,6,0) */
  g_signal_connect (G_OBJECT (play->play_button), "clicked",
                        G_CALLBACK (avi_play_clicked_button_play), play);
  g_signal_connect (G_OBJECT (play->stop_button), "clicked",
                        G_CALLBACK (avi_play_clicked_button_stop), play);
  g_signal_connect (G_OBJECT (play->previous_button), "clicked",
                        G_CALLBACK (avi_play_clicked_button_previous), play);
  g_signal_connect (G_OBJECT (play->rewind_button), "clicked",
                        G_CALLBACK (avi_play_clicked_button_rewind), play);
  g_signal_connect (G_OBJECT (play->forward_button), "clicked",
                        G_CALLBACK (avi_play_clicked_button_forward), play);
  g_signal_connect (G_OBJECT (play->next_button), "clicked",
                        G_CALLBACK (avi_play_clicked_button_next), play);
  gtk_widget_set_sensitive (play->play_button,
        !play->playing && (play->position < play->length - 1 || play->gap));
  gtk_widget_set_sensitive (play->stop_button, play->playing);
  gtk_widget_set_sensitive (play->previous_button,
                                        !play->playing && play->position > 0);
  gtk_widget_set_sensitive (play->rewind_button,
                                        !play->playing && play->position > 0);
  gtk_widget_set_sensitive (play->forward_button,
                        !play->playing && play->position < play->length - 1);
  gtk_widget_set_sensitive (play->next_button,
                        !play->playing && play->position < play->length - 1);
  /* ja:スケール */
  play->hscale = gtk_hscale_new_with_range (0, MAX (play->length - 1, 1), 1);
  gtk_scale_set_draw_value (GTK_SCALE (play->hscale), FALSE);
  gtk_range_set_value (GTK_RANGE (play->hscale), play->position);
  gtk_widget_size_request (play->hscale, &req);
  gtk_widget_set_size_request (play->hscale, req.width * 2, req.height);
  g_signal_connect (G_OBJECT (play->hscale), "value-changed",
                                    G_CALLBACK (avi_play_value_changed), play);
  gtk_widget_set_sensitive (play->hscale, !play->playing && play->length > 1);
  /* ja:ラベル */
  if (play->avi_edit[0])
    t = avi_time_from_sample (play->avi_edit[0], play->position);
  else if (play->avi_edit[1])
    t = avi_time_from_sample (play->avi_edit[1], play->position);
  else
    t = 0;
  text = g_strdup_printf (_("Frame %d, Time %02d:%02d:%02d"), play->position,
                                t / 60000 % 100, t / 1000 % 60, t / 10 % 100);
  play->label = gtk_label_new (text);
  g_free (text);

  /* ja:フレームとボックス */
  vbox = gtk_vbox_new (FALSE, 0);
  play->hbox = gtk_hbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (play->hbox), SPACING);
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  hbox1 = gtk_hbox_new (FALSE, 0);
  hbox2 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), play->play_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), play->stop_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox2), play->previous_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox2), play->rewind_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox2), play->forward_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox2), play->next_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox2, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (play->hbox), vbox, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (play->hbox), play->hscale, TRUE, TRUE, 0);
  gtk_box_pack_end (GTK_BOX (play), play->label, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (play), play->hbox, FALSE, FALSE, 0);
  /* ja:表示 */
  gtk_widget_show (play->label);
  gtk_widget_show_all (play->hbox);
}


/*  ja:パネルを表示しない
    play,ウィジェット                                                       */
void
avi_play_hide_panel (AviPlay *play)
{
  if (play)
    {
      if (play->label)
        gtk_container_remove (GTK_CONTAINER (play), play->label);
      if (play->hbox)
        gtk_container_remove (GTK_CONTAINER (play), play->hbox);
      play->play_button = NULL;
      play->stop_button = NULL;
      play->previous_button = NULL;
      play->rewind_button = NULL;
      play->forward_button = NULL;
      play->next_button = NULL;
      play->hscale = NULL;
      play->label = NULL;
      play->hbox = NULL;
    }
}


/*  ja:ウィジェットの縮小を取得する
    play,ウィジェット
     RET,TRUE:小さくなる,FALSE:小さくならない                               */
gboolean
avi_play_get_shrink (AviPlay *play)
{
  return play ? play->shrink : FALSE;
}


/*  ja:ウィジェットの縮小を設定する
      play,ウィジェット
    shrink,TRUE:小さくなる,FALSE:小さくならない                             */
void
avi_play_set_shrink (AviPlay        *play,
                     const gboolean  shrink)
{
  if (play && play->shrink != shrink)
    {
      play->shrink = shrink;
      if (play->avi_edit[0])
        {
          if (!play->shrink)
            gtk_widget_set_size_request (play->drawing,
                                    avi_edit_get_width (play->avi_edit[0]),
                                    avi_edit_get_height (play->avi_edit[0]));
          else
            gtk_widget_set_size_request (play->drawing, 0, 0);
        }
    }
}


/*  ja:AVI編集ハンドルを設定する
        play,ウィジェット
    avi_edit,AVI編集ハンドル,NULL:設定なし                                  */
void
avi_play_set_handle (AviPlay  *play,
                     AviEdit **avi_edit)
{
  gint i;

  if (!play)
    return;
  if (play->playing)
    avi_play_stop (play);
  if (play->avi_frame)
    {
      avi_frame_close (play->avi_frame);
      play->avi_frame = NULL;
    }
  avi_edit_close (play->avi_edit[0]);
  avi_edit_close (play->avi_edit[1]);
  play->avi_edit[0] = play->avi_edit[1] = NULL;
  if (avi_edit)
    for (i = 0; avi_edit[i]; i++)
      switch (avi_edit_type (avi_edit[i]))
        {
          case AVI_TYPE_VIDEO:
            if (!play->avi_edit[0])
              play->avi_edit[0] = avi_edit_dup (avi_edit[i]);
            break;
          case AVI_TYPE_AUDIO:
            if (!play->avi_edit[1])
              play->avi_edit[1] = avi_edit_dup (avi_edit[i]);
        }
  if (play->avi_edit[0])
    {
      play->avi_frame = avi_frame_open (play->avi_edit[0]);
      play->rate = avi_edit_get_rate (play->avi_edit[0]);
      play->scale = avi_edit_get_scale (play->avi_edit[0]);
      play->length = avi_edit_length (play->avi_edit[0]);
      if (play->avi_edit[1])
        {
          gint length;

          length = ((glonglong)avi_time_length (play->avi_edit[1])
                                        * play->rate + play->scale * 1000 - 1)
                                        / ((glonglong)play->scale * 1000);
          play->gap = play->length < length;
          if (play->gap)
            play->length = length;
        }
      else
        {
          play->gap = FALSE;
        }
    }
  else
    {
      play->rate = play->scale = 0;
      play->length = play->avi_edit[1] ? avi_edit_length (play->avi_edit[1])
                                       : 0;
      play->gap = TRUE;
    }
  play->position = 0;
  play->block_align = play->avi_edit[1]
                    ? avi_edit_get_channels (play->avi_edit[1])
                    * avi_edit_get_bits_per_sample (play->avi_edit[1]) / 8 : 0;
  if (!play->shrink && play->avi_edit[0])
    gtk_widget_set_size_request (play->drawing,
                                    avi_edit_get_width (play->avi_edit[0]),
                                    avi_edit_get_height (play->avi_edit[0]));
  else
    gtk_widget_set_size_request (play->drawing, 0, 0);
  /* ja:スケール */
  gtk_range_set_value (GTK_RANGE (play->hscale), 0);
  gtk_range_set_range (GTK_RANGE (play->hscale), 0, MAX (play->length - 1, 1));
  gtk_widget_set_sensitive (play->hscale, !play->playing && play->length > 1);
  avi_play_set_position_internal (play, 0);
  gtk_widget_draw (play->drawing, NULL);
}


/*  ja:AVIの長さを取得する
    play,ウィジェット
     RET,長さ                                                               */
gint
avi_play_get_length (AviPlay *play)
{
  return play ? play->length : 0;
}


/*  ja:AVIを再生する
    play,ウィジェット
     RET,TRUE:正常終了,FALSE:エラー                                         */
void
avi_play_play (AviPlay *play)
{
  if (!play || play->playing || (!play->avi_edit[0] && !play->avi_edit[1])
                        || !(play->position < play->length - 1 || play->gap))
    return;
  /* ja:ボタン */
  if (play->hbox)
    {
      gtk_widget_set_sensitive (play->play_button, FALSE);
      gtk_widget_set_sensitive (play->stop_button, TRUE);
      gtk_widget_set_sensitive (play->previous_button, FALSE);
      gtk_widget_set_sensitive (play->rewind_button, FALSE);
      gtk_widget_set_sensitive (play->forward_button, FALSE);
      gtk_widget_set_sensitive (play->next_button, FALSE);
      /* ja:スケール */
      gtk_widget_set_sensitive (play->hscale, FALSE);
    }
  play->playing = TRUE;
  play->start = play->position;
  if (play->rate > 0 && play->scale > 0)
    play->timer_id = g_timeout_add_full (G_PRIORITY_LOW,
            (guint)MAX ((glonglong)play->scale * 1000 / play->rate, 1),
                                    (GSourceFunc)avi_play_timeout, play, NULL);
  else
    play->timer_id = g_timeout_add_full (G_PRIORITY_LOW, 100,
                                    (GSourceFunc)avi_play_timeout, play, NULL);
#ifdef USE_LIBAO
  if (play->driver_id >= 0 && play->avi_edit[1]
                        && (play->avi_pcm = avi_pcm_open (play->avi_edit[1])))
    {
      play->pos = play->avi_edit[0] ? avi_time_sync_sample (play->avi_edit[0],
                                            play->position, play->avi_edit[1])
                                                            : play->position;
      play->thread = g_thread_create ((GThreadFunc)avi_play_libao, play,
                                                                TRUE, NULL);
    }
#endif /* USE_LIBAO */
#ifdef USE_ALSA
  if (play->avi_edit[1] && (play->avi_pcm = avi_pcm_open (play->avi_edit[1])))
    {
      play->pos = play->avi_edit[0] ? avi_time_sync_sample (play->avi_edit[0],
                                            play->position, play->avi_edit[1])
                                                            : play->position;
      g_free (play->err);
      play->err = NULL;
      play->thread = g_thread_create ((GThreadFunc)avi_play_alsa, play,
                                                                TRUE, NULL);
    }
#endif /* USE_ALSA */
#ifdef G_OS_WIN32
  if (waveOutGetNumDevs() > 0 && play->avi_edit[1]
                        && (play->avi_pcm = avi_pcm_open (play->avi_edit[1])))
    {
      WAVEFORMATEX wfxOut;

      wfxOut.wFormatTag = WAVE_FORMAT_PCM;
      wfxOut.nChannels = avi_edit_get_channels (play->avi_edit[1]);
      wfxOut.nSamplesPerSec = avi_edit_get_samples_per_sec (play->avi_edit[1]);
      wfxOut.wBitsPerSample = avi_edit_get_bits_per_sample (play->avi_edit[1]);
      wfxOut.nBlockAlign = wfxOut.nChannels * wfxOut.wBitsPerSample / 8;
      wfxOut.nAvgBytesPerSec = wfxOut.nSamplesPerSec * wfxOut.nBlockAlign;
      wfxOut.cbSize = 0;
      if (waveOutOpen (&play->hWaveOut, WAVE_MAPPER, &wfxOut,
                                        (DWORD)GPOINTER_TO_UINT (waveOutProc),
                                        (DWORD)GPOINTER_TO_UINT (play),
                                        CALLBACK_FUNCTION) == MMSYSERR_NOERROR)
        {
          gint i;

          play->pos = play->avi_edit[0]
                                    ? avi_time_sync_sample (play->avi_edit[0],
                                            play->position, play->avi_edit[1])
                                                            : play->position;
          for (i = 0; i < AVI_PCM_BUFFER_NUM; i++)
            {
              gpointer out_buf;
              gsize samples, out_samples;

              samples = MIN (avi_edit_length (play->avi_edit[1]) - play->pos,
                                                        AVI_PCM_BUFFER_NUM);
              if (samples > 0 && avi_pcm_get_raw (play->avi_pcm,
                                play->pos, samples, &out_buf, &out_samples))
                {
                  play->whrOut[i].lpData = out_buf;
                  play->whrOut[i].dwBufferLength = out_samples
                                                        * play->block_align;
                  if (waveOutPrepareHeader (play->hWaveOut,
                                        &play->whrOut[i],
                                        sizeof (WAVEHDR)) == MMSYSERR_NOERROR)
                    {
                      waveOutWrite (play->hWaveOut,
                                        &play->whrOut[i], sizeof (WAVEHDR));
                      play->pos += samples;
                    }
                  else
                    {
                      g_free (play->whrOut[i].lpData);
                      play->whrOut[i].lpData = NULL;
                      play->whrOut[i].dwBufferLength = 0;
                    }
                }
            }
        }
      else
        {
          play->hWaveOut = NULL;
        }
    }
  play->dwTime = timeGetTime ();
#else /* not G_OS_WIN32 */
  play->gtimer = g_timer_new ();
#endif /* not G_OS_WIN32 */
}


/*  ja:AVIを停止する
    play,ウィジェット
     RET,TRUE:正常終了,FALSE:エラー                                         */
void
avi_play_stop (AviPlay *play)
{
  if (!play || !play->playing)
    return;
  play->playing = FALSE;
  if (play->timer_id != 0)
    {
      gtk_timeout_remove (play->timer_id);
      play->timer_id = 0;
    }
#ifdef USE_LIBAO
  if (play->thread)
    {
      g_thread_join (play->thread);
      play->thread = NULL;
      play->pos = -1;
    }
#endif /* USE_LIBAO */
#ifdef USE_ALSA
  if (play->thread)
    {
      g_thread_join (play->thread);
      g_free (play->err);
      play->err = NULL;
      play->thread = NULL;
      play->pos = -1;
    }
#endif /* USE_ALSA */
#ifdef G_OS_WIN32
  if (play->hWaveOut)
    {
      play->pos = -1;
      waveOutReset (play->hWaveOut);
      waveOutClose (play->hWaveOut);
      play->hWaveOut = NULL;
    }
#else /* not G_OS_WIN32 */
  if (play->gtimer)
    {
      g_timer_destroy (play->gtimer);
      play->gtimer = NULL;
    }
#endif /* not G_OS_WIN32 */
  if (play->avi_pcm)
    {
      avi_pcm_close (play->avi_pcm);
      play->avi_pcm = NULL;
    }
  if (play->hbox)
    {
      /* ja:ボタン */
      gtk_widget_set_sensitive (play->play_button,
                            play->position < play->length - 1 || play->gap);
      gtk_widget_set_sensitive (play->stop_button, FALSE);
      gtk_widget_set_sensitive (play->previous_button, play->position > 0);
      gtk_widget_set_sensitive (play->rewind_button, play->position > 0);
      gtk_widget_set_sensitive (play->forward_button,
                                            play->position < play->length - 1);
      gtk_widget_set_sensitive (play->next_button,
                                            play->position < play->length - 1);
      /* ja:スケール */
      gtk_widget_set_sensitive (play->hscale, play->length > 1);
    }
}


/*  ja:AVIの状態を取得する
    play,ウィジェット
     RET,TRUE:再生中,FALSE:停止                                             */
gboolean
avi_play_is_playing (AviPlay *play)
{
  return play ? play->playing : FALSE;
}


/*  ja:AVIの位置を取得する
    play,ウィジェット
     RET,位置                                                               */
gint
avi_play_get_position (AviPlay *play)
{
  return play ? play->position : 0;
}


/*  ja:AVIの位置を設定する
        play,ウィジェット
    position,位置(-1:最後)
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_play_set_position (AviPlay    *play,
                       const gint  position)
{
  if (!play || play->length < position || play->playing)
    return FALSE;
  avi_play_set_position_real (play, position >= 0 ? position : play->length);
  return TRUE;
}
