/* vim: set tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2007 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.sourceforge.webframe.springframework.util;

import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;

/**
 * SpringBeanHolder
 *
 * @author Yomei Komiya
 * @since 0.9.2
 *
 * @version $Id$
 */
public class SpringBeanHolder {
	
	/**
	 * Bean name
	 */
	private String beanName = "";
	
	/**
	 * Bean aliases
	 */
	private String[] beanAliases = null;
	
	/**
	 * Bean type
	 */
	private Class beanType = null;
	
	/**
	 * BeanDefinition object
	 */
	private BeanDefinition beanDefinition = null;

	/**
	 * Actual singleton object
	 */
	private Object actualSingleton = null;
	
	
	public SpringBeanHolder() {
		super();
	}


	/**
	 * Constructor
	 * 
	 * @param beanFactory ConfigurableListableBeanFactory object
	 * @param beanName Bean name
	 * @throws NoSuchBeanDefinitionException
	 */
	public SpringBeanHolder(
			ConfigurableListableBeanFactory beanFactory,
			String beanName) 
		throws NoSuchBeanDefinitionException {

		this.beanName = beanName;
		this.beanAliases = beanFactory.getAliases(beanName);
		this.beanType = beanFactory.getType(beanName);
		this.beanDefinition = beanFactory.getBeanDefinition(beanName);
		
		if (beanFactory.containsSingleton(beanName)) {
			this.actualSingleton = beanFactory.getBean(beanName);
		}
		else {
			this.actualSingleton = null;
		}
	}

	
	public String getBeanName() {
		return beanName;
	}

	
	public void setBeanName(String beanName) {
		this.beanName = beanName;
	}


	public String[] getBeanAliases() {
		return beanAliases;
	}


	public void setBeanAliases(String[] beanAliases) {
		this.beanAliases = beanAliases;
	}


	public Class getBeanType() {
		return beanType;
	}


	public void setBeanType(Class beanType) {
		this.beanType = beanType;
	}


	public BeanDefinition getBeanDefinition() {
		return beanDefinition;
	}

	
	public void setBeanDefinition(BeanDefinition beanDefinition) {
		this.beanDefinition = beanDefinition;
	}


	public boolean isSingleton() {
		return this.beanDefinition.isSingleton();
	}
	
	
	public Object getActualSingleton() {
		return actualSingleton;
	}


	public void setActualSingleton(Object actualSingleton) {
		this.actualSingleton = actualSingleton;
	}
	
	
	/**
	 * Get actual singleton object ID (hex string)
	 * 
	 * @return Object ID
	 */
	public String getActualSingletonObjectId() {
		if (this.actualSingleton != null) {
			return Integer.toHexString(this.actualSingleton.hashCode());
		}
		else {
			return null;
		}
	}

	
	public String getActualSingletonTypeName() {
		if (this.actualSingleton != null) {
			return this.actualSingleton.getClass().getName();
		}
		else {
			return null;
		}
	}

}
