/*
    GLmol - Molecular Viewer on WebGL/Javascript 

   (C) Copyright 2011, biochem_fan

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

    This program uses
      Three.js 
         https://github.com/mrdoob/three.js
         Copyright (c) 2010-2011 three.js Authors. All rights reserved.
      jQuery
         http://jquery.org/
         Copyright (c) 2011 John Resig
 */

//  BETA version (0.25)

// TODO:
//   zoom speed, zoom adjustment, zoom into
//   get_view, set_view
//   Fog, slab
//   True cartoon representation (with thickness, with arrowheads) -- duplicate vertex?
//   Show helix as cylinder -- make and use selectBySS
//   Improve bond detection (performance, accuracy)
//   Improve handling of alternative locations
//   SDF reader -- double bond (find plane? but what about H2SO4?)
//   Show/Hide representation without reloading everything
//   Show nucleotides bases --- line between N1(AG), N3(CT) and where? or fill rings? selectAtomAtResi
//   Support Three.js r46
//   Faster symmetry operation -> Use Object3D.clone in Three.js > r46
//   Labels (study textures and billboards)
//   Selection by mouse (DIFFICULT. use Ray casting?)
//   Improve stick performance -- HOW???
// Less important
//   Detect secondary structures
//   Support ionic radii and covalent radii
//   xul version
//   Fill unit cell -- necessary?
//   Fix mesh Normals -- workaround found (double sided lighting)
//   check if THREE.Ribbon is useful

// for old THREE.js (<r42)
if (!THREE.Color.prototype.getHex) {
   THREE.Color.prototype.getHex = function () {
      return this.hex;
   };
}

THREE.Geometry.prototype.colorAll = function (color) {
   for (var i = 0; i < this.faces.length; i++) {
      this.faces[i].color = color;
   }
};

THREE.Matrix4.prototype.isIdentity = function() {
   for (var i = 1; i <= 4; i++) {
      for (var j = 1; j <= 4; j++) {
         var shouldBe = (i == j) ? 1 : 0;
         if (this['n' + i + j] != shouldBe) return false;
      }
   }
   return true;
};

// for new THREE.js (>r45)
//console.warn = function() {};

var GLmol = (function() {
    function GLmol(id, suppressAutoload) {
   this.ElementColors = {"H": 0xCCCCCC, "C": 0xAAAAAA, "O": 0xCC0000, "N": 0x0000CC, "S": 0xCCCC00, "P": 0x6622CC,
                         "F": 0x00CC00, "CL": 0x00CC00, "BR": 0x882200, "I": 0x6600AA,
                         "FE": 0xCC6600, "CA": 0x8888AA};
// Reference: A. Bondi, J. Phys. Chem., 1964, 68, 441.
   this.vdwRadii = {"H": 1.2, "Li": 1.82, "Na": 2.27, "K": 2.75, "C": 1.7, "N": 1.55, "O": 1.52,
                   "F": 1.47, "P": 1.80, "S": 1.80, "CL": 1.75, "BR": 1.85, "SE": 1.90,
                   "ZN": 1.39, "CU": 1.4, "NI": 1.63};


   this.id = id;  

   // TODO: How can I set background color? 
   this.container = $('#' + this.id);
   this.WIDTH = this.container.width(), this.HEIGHT = this.container.height();
   this.VIEW_ANGLE = 30;
   this.ASPECT = this.WIDTH / this.HEIGHT;
   this.NEAR = 0.1, FAR = 300;
    
   this.renderer = new THREE.WebGLRenderer(); // TODO: {antialias: true} will work?
   this.renderer.setSize(this.WIDTH, this.HEIGHT);
   this.container.append(this.renderer.domElement);

   this.camera = new THREE.PerspectiveCamera(this.VIEW_ANGLE, this.ASPECT, this.NEAR, this.FAR);
   this.camera.position = new THREE.Vector3(0, 0, -150);
   this.camera.lookAt(new THREE.Vector3(0, 0, 0));

   this.scene = null;
   this.rotationGroup = null; // which contains modelGroup
   this.modelGroup = null;

   // Default values
   this.sphereRadius = 1.5; 
   this.cylinderRadius = 0.2;
   this.lineWidth = 0.5;
   this.curveWidth = 3.0;
   this.defaultColor = 0xCCCCCC;
   this.sphereQuality = 16;
   this.cylinderQuality = 32;
 
   // UI variables
   this.cq = new THREE.Quaternion(1, 0, 0, 0);
   this.dq = new THREE.Quaternion(1, 0, 0, 0);
   this.isDragging = false;
   this.mouseStartX = 0;
   this.mouseStartY = 0;
   this.currentModelPos = 0;
   this.cz = 0;
   this.enableMouse();

   // Run!
   if (suppressAutoload) return;
   this.rebuildScene();
}

GLmol.prototype.setupLights = function(scene) {
   // TODO: Improve lighting!
   // doubleSided is not working with LambertMaterial and PhongMaterial
   //   https://github.com/mrdoob/three.js/issues/388 
   //  now I light objects from both sides.
 
   var directionalLight = new THREE.DirectionalLight(0xFFFFFF);

/*   directionalLight.position = new THREE.Vector3(1, 1, 1);
   directionalLight.intensity = 1;
   scene.add(directionalLight);
   directionalLight = new THREE.DirectionalLight(0xFFFFFF);
   directionalLight.position = new THREE.Vector3(-1, -1, -1);
   directionalLight.intensity = 1;
   scene.add(directionalLight);*/

   directionalLight = new THREE.DirectionalLight(0xFFFFFF);
   directionalLight.position = this.camera.position;
   directionalLight.intensity = 0.8;
   scene.add(directionalLight);
   directionalLight = new THREE.DirectionalLight(0xFFFFFF);
   directionalLight.position = new THREE.Vector3().copy(this.camera.position).negate();
   directionalLight.intensity = 0.8;
   scene.add(directionalLight);
   var ambientLight = new THREE.AmbientLight(0x808080);
   scene.add(ambientLight);

   scene.add(directionalLight);  
};

GLmol.prototype.parseSDF = function(str) {
   var atoms = this.atoms;
   var protein = this.protein;

   var lines = str.split("\n");
   if (lines.length < 4) return;
   var atomCount = parseInt(lines[3].substr(0, 3));
   if (atomCount == 0) return; // might be a PDB file.
   var bondCount = parseInt(lines[3].substr(3, 3));
   var offset = 4;
   if (lines.length < 4 + atomCount + bondCount) return;
//   console.log([atomCount, bondCount]);
   for (var i = 1; i <= atomCount; i++) {
      var line = lines[offset];
      offset++;
      var atom = {};
      atom.serial = i;
      atom.x = parseFloat(line.substr(0, 10));
      atom.y = parseFloat(line.substr(10, 10));
      atom.z = parseFloat(line.substr(20, 10));
      atom.hetflag = true;
      atom.atom = atom.elem = line.substr(30, 3).replace(/ /g, "");
      atom.bonds = [];
      atoms[i] = atom;
   }
   for (i = i; i <= bondCount; i++) {
      var line = lines[offset];
      offset;
      var from = parseInt(line.substr(0, 3));
      var to = parseInt(line.substr(0, 3));
      atoms[from].bonds.push(to);
      atoms[to].bonds.push(from);
   }

   return protein;
};

GLmol.prototype.parsePDB2 = function(str) {
   var atoms = this.atoms;
   var protein = this.protein;

   var atoms_cnt = 0;
   lines = str.split("\n");
   for (var i = 0; i < lines.length; i++) {
      line = lines[i].replace(/^\s*/, ''); // remove indent
      var recordName = line.substr(0, 6);
      if (recordName == 'ATOM  ' || recordName == 'HETATM') {
              var atom, resn, chain, resi, x, y, z, hetflag, elem, serial, altLoc;
         serial = parseInt(line.substr(6, 5));
         atom = line.substr(12, 4).replace(/ /g, "");
         altLoc = line.substr(16, 1);
         if (altLoc != ' ' && altLoc != 'A') continue; // FIXME: ad hoc
         resn = line.substr(17, 3);
         chain = line.substr(21, 1);
         resi = parseInt(line.substr(22, 5)); 
         x = parseFloat(line.substr(30, 8));
         y = parseFloat(line.substr(38, 8));
         z = parseFloat(line.substr(46, 8));
         elem = line.substr(76, 2).replace(/ /g, "");
         if (elem == '') { // for some incorrect PDB files
            elem = line.substr(12, 4).replace(/ /g,"");
         }
         if (line[0] == 'H') hetflag = true;
         else hetflag = false;
         if (!protein[chain]) protein[chain] = {'residues':[]};
         if (!protein[chain].residues[resi]) protein[chain].residues[resi] = {atoms: {}};
         var residue = protein[chain].residues[resi];
         residue.resn = resn;
         atoms[serial] = {'resn': resn, 'x': x, 'y': y, 'z': z, 'elem': elem, 'hetflag': hetflag, 'chain': chain, 'resi': resi, 'serial': serial, 'atom': atom, 'bonds': [], 'ss': 'c', 'color': 0xFFFFFF, 'bonds': [], /*'altLoc': altLoc*/};
         residue.atoms[atom] = serial;
      } else if (recordName == 'SHEET ') {
         var startChain = line.substr(21, 1);
         var startResi = parseInt(line.substr(22, 4));
         var endChain = line.substr(32, 1);
         var endResi = parseInt(line.substr(33, 4));
         protein.sheet.push([startChain, startResi, endChain, endResi]);
     } else if (recordName == 'CONECT') {
// MEMO: We don't have to parse SSBOND, LINK because both are also 
// described in CONECT. But what about 2JYT???
         var from = parseInt(line.substr(6, 5));
         for (var j = 0; j < 4; j++) {
            var to = parseInt(line.substr([11, 16, 21, 26][j], 5));
            if (!isNaN(to)) atoms[from].bonds.push(to);
         }
     } else if (recordName == 'HELIX ') {
         var startChain = line.substr(19, 1);
         var startResi = parseInt(line.substr(21, 4));
         var endChain = line.substr(31, 1);
         var endResi = parseInt(line.substr(33, 4));
         protein.helix.push([startChain, startResi, endChain, endResi]);
     } else if (recordName == 'CRYST1') {
         protein.a = parseFloat(line.substr(6, 9));
         protein.b = parseFloat(line.substr(15, 9));
         protein.c = parseFloat(line.substr(24, 9));
         protein.alpha = parseFloat(line.substr(33, 7));
         protein.beta = parseFloat(line.substr(40, 7));
         protein.gamma = parseFloat(line.substr(47, 7));
         protein.spacegroup = line.substr(55, 11);
         this.defineCell();
      }  else if (recordName.substr(0, 5) == 'SCALE') {
         var n = parseInt(recordName.substr(5, 1));
         protein.scaleMatrix['n' + n + '1'] = parseFloat(line.substr(10, 10));
         protein.scaleMatrix['n' + n + '2'] = parseFloat(line.substr(20, 10));
         protein.scaleMatrix['n' + n + '3'] = parseFloat(line.substr(30, 10));
         protein.scaleMatrix['n' + n + '4'] = parseFloat(line.substr(45, 10));
      } else if (recordName == 'REMARK') {
         if (line.substr(13, 5) == 'BIOMT') {
            var n = parseInt(line[18]);
            var m = parseInt(line.substr(21, 2));
            if (protein.biomtMatrices[m] == undefined) protein.biomtMatrices[m] = new THREE.Matrix4().identity();
            protein.biomtMatrices[m]['n' + n + '1'] = parseFloat(line.substr(24, 9));
            protein.biomtMatrices[m]['n' + n + '2'] = parseFloat(line.substr(34, 9));
            protein.biomtMatrices[m]['n' + n + '3'] = parseFloat(line.substr(44, 9));
            protein.biomtMatrices[m]['n' + n + '4'] = parseFloat(line.substr(54, 10));

         } else if (line.substr(13, 5) == 'SMTRY') {
            var n = parseInt(line[18]);
            var m = parseInt(line.substr(21, 2));
            if (protein.symmetryMatrices[m] == undefined) protein.symmetryMatrices[m] = new THREE.Matrix4().identity();
            protein.symmetryMatrices[m]['n' + n + '1'] = parseFloat(line.substr(24, 9));
            protein.symmetryMatrices[m]['n' + n + '2'] = parseFloat(line.substr(34, 9));
            protein.symmetryMatrices[m]['n' + n + '3'] = parseFloat(line.substr(44, 9));
            protein.symmetryMatrices[m]['n' + n + '4'] = parseFloat(line.substr(54, 10));
         }
      }
   }

   // Assign secondary structures
 
   for (i = 0; i < atoms.length; i++) {
      atom = atoms[i]; if (atom == undefined) continue;

      var found = false;
      // MEMO: Can start chain and end chain differ?
      for (j = 0; j < protein.sheet.length; j++) {
         if (atom.chain != protein.sheet[j][0]) continue;
         if (atom.resi < protein.sheet[j][1]) continue;
         if (atom.resi > protein.sheet[j][3]) continue;
         atom.ss = 's';
         if (atom.resi == protein.sheet[j][1]) atom.ssbegin = true;
         if (atom.resi == protein.sheet[j][3]) atom.ssend = true;
      }
      for (j = 0; j < protein.helix.length; j++) {
         if (atom.chain != protein.helix[j][0]) continue;
         if (atom.resi < protein.helix[j][1]) continue;
         if (atom.resi > protein.helix[j][3]) continue;
         atom.ss = 'h';
         if (atom.resi == protein.helix[j][1]) atom.ssbegin = true;
         else if (atom.resi == protein.helix[j][3]) atom.ssend = true;
      }
   } 

   return protein;
};

// Catmull-Rom subdivision
GLmol.prototype.subdivide = function(points, DIV) { // points as Vector3
   var ret = [];

   for (var i = -1, size = points.length; i <= size - 3; i++) {
      var p0 = points[(i == -1) ? 0 : i];
      var p1 = points[i + 1], p2 = points[i + 2];
      var  p3 = points[(i == size - 3) ? size - 1 : i + 3];
      var v0 = new THREE.Vector3().sub(p2, p0).multiplyScalar(0.5);
      var v1 = new THREE.Vector3().sub(p3, p1).multiplyScalar(0.5);
      for (var j = 0; j < DIV; j++) {
         var t = 1.0 / DIV * j;
         var x = p1.x + t * v0.x 
                  + t * t * (-3 * p1.x + 3 * p2.x - 2 * v0.x - v1.x)
                  + t * t * t * (2 * p1.x - 2 * p2.x + v0.x + v1.x);
         var y = p1.y + t * v0.y 
                  + t * t * (-3 * p1.y + 3 * p2.y - 2 * v0.y - v1.y)
                  + t * t * t * (2 * p1.y - 2 * p2.y + v0.y + v1.y);
         var z = p1.z + t * v0.z 
                  + t * t * (-3 * p1.z + 3 * p2.z - 2 * v0.z - v1.z)
                  + t * t * t * (2 * p1.z - 2 * p2.z + v0.z + v1.z);
         ret.push(new THREE.Vector3(x, y, z));
      }
   }
   ret.push(points[points.length - 1]);
   return ret;
};

GLmol.prototype.drawAtomsAsSphere = function(group, atomlist, defaultRadius) {
   for (var i = 0; i < atomlist.length; i++) {
      var atom = this.atoms[atomlist[i]];
      if (atom == undefined) continue;

      var sphereGeometry = new THREE.SphereGeometry((this.vdwRadii[atom.elem] != undefined) ? this.vdwRadii[atom.elem] : defaultRadius, this.sphereQuality, this.sphereQuality); // radius, seg, ring
      var sphereMaterial = new THREE.MeshLambertMaterial({color: atom.color});
      var sphere = new THREE.Mesh(sphereGeometry, sphereMaterial);
      group.add(sphere);
      sphere.position.x = atom.x;
      sphere.position.y = atom.y;
      sphere.position.z = atom.z;
   }
};

GLmol.prototype.drawAtomsAsIcosahedron = function(group, atomlist, defaultRadius) {
   var geo = this.IcosahedronGeometry();
   console.log(geo);
   for (var i = 0; i < atomlist.length; i++) {
      var atom = this.atoms[atomlist[i]];
      if (atom == undefined) continue;

      var mat = new THREE.MeshPhongMaterial({color: atom.color});
      var sphere = new THREE.Mesh(geo, mat);
      sphere.doubleSided = true;
      sphere.scale.x = sphere.scale.y = sphere.scale.z = (this.vdwRadii[atom.elem] != undefined) ? this.vdwRadii[atom.elem] : defaultRadius;
      group.add(sphere);
      sphere.position.x = atom.x;
      sphere.position.y = atom.y;
      sphere.position.z = atom.z;
   }
};

GLmol.prototype.isConnected = function(atom1, atom2) {
   if (atom1.bonds.indexOf(atom2.serial) != -1) return true;

   // sqrt is slow, to be avoided if possible
   var distSquared = (atom1.x - atom2.x) * (atom1.x - atom2.x) + 
                     (atom1.y - atom2.y) * (atom1.y - atom2.y) + 
                     (atom1.z - atom2.z) * (atom1.z - atom2.z);

//   if (atom1.altLoc != atom2.altLoc) return false;
   if (isNaN(distSquared)) return false;
   if (distSquared < 0.5) return false; // FIXME: duplicate position? Is this treatment correct?

   if (distSquared > 1.3 && (atom1.elem == 'H' || atom2.elem == 'H')) return false;
   if (distSquared < 3.42 && (atom1.elem == 'S' || atom2.elem == 'S')) return true;
   if (distSquared > 2.78) return false;
   return true;
};

GLmol.prototype.drawBondsAsStick = function(group, atomlist, bondR, atomR) {
   var sphereGeometry = new THREE.SphereGeometry(atomR, this.sphereQuality, this.sphereQuality); // radius, seg, ring

   for (var _i in atomlist) {
      var i = atomlist[_i];
      var atom1 = this.atoms[i];
      for (var _j in atomlist) {
         var j = atomlist[_j];
         if (i == j) continue;
         var atom2 = this.atoms[j];
         if (atom1 == undefined || atom2 == undefined) continue;
         if (!this.isConnected(atom1, atom2)) continue;

         this.drawCylinder(group, new THREE.Vector3(atom1.x, atom1.y, atom1.z),
                           new THREE.Vector3((atom1.x + atom2.x) / 2, (atom1.y + atom2.y) / 2, (atom1.z + atom2.z) / 2), bondR, atom1.color);
       }
       var sphereMaterial = new THREE.MeshLambertMaterial({color: atom1.color});
       var sphere = new THREE.Mesh(sphereGeometry, sphereMaterial);
       group.add(sphere);
       sphere.position.x = atom1.x;
       sphere.position.y = atom1.y;
       sphere.position.z = atom1.z;
    }
};

GLmol.prototype.defineCell = function() {
    var protein = this.protein;
    if (protein.a == undefined) return;

    protein.ax = protein.a;
    protein.ay = 0;
    protein.az = 0;
    protein.bx = protein.b * Math.cos(Math.PI / 180.0 * protein.gamma);
    protein.by = protein.b * Math.sin(Math.PI / 180.0 * protein.gamma);
    protein.bz = 0;
    protein.cx = protein.c * Math.cos(Math.PI / 180.0 * protein.beta);
    protein.cy = protein.c * (Math.cos(Math.PI / 180.0 * protein.alpha) - 
               Math.cos(Math.PI / 180.0 * protein.gamma) 
             * Math.cos(Math.PI / 180.0 * protein.beta)
             / Math.sin(Math.PI / 180.0 * protein.gamma));
    protein.cz = Math.sqrt(protein.c * protein.c * Math.sin(Math.PI / 180.0 * protein.beta)
               * Math.sin(Math.PI / 180.0 * protein.beta) - protein.cy * protein.cy);
};

GLmol.prototype.drawUnitcell = function(group) {
    var protein = this.protein;
    if (protein.a == undefined) return;

    var vertices = [[0, 0, 0], [protein.ax, protein.ay, protein.az], [protein.bx, protein.by, protein.bz], [protein.ax + protein.bx, protein.ay + protein.by, protein.az + protein.bz],
          [protein.cx, protein.cy, protein.cz], [protein.cx + protein.ax, protein.cy + protein.ay,  protein.cz + protein.az], [protein.cx + protein.bx, protein.cy + protein.by, protein.cz + protein.bz], [protein.cx + protein.ax + protein.bx, protein.cy + protein.ay + protein.by, protein.cz + protein.az + protein.bz]];
    var edges = [0, 1, 0, 2, 1, 3, 2, 3, 4, 5, 4, 6, 5, 7, 6, 7, 0, 4, 1, 5, 2, 6, 3, 7];    

    var geo = new THREE.Geometry();
    for (var i = 0; i < edges.length; i++) {
       geo.vertices.push(new THREE.Vertex(new THREE.Vector3(vertices[edges[i]][0], vertices[edges[i]][1], vertices[edges[i]][2])));
    }
   var lineMaterial = new THREE.LineBasicMaterial({linewidth: 1, color: 0xcccccc});
   var line = new THREE.Line(geo, lineMaterial);
   line.type = THREE.Lines;
   group.add(line);
};


// FIXME: Very slow!
GLmol.prototype.drawBondsAsLine = function(group, atomlist, lineWidth) {
   var geo = new THREE.Geometry();   
   var nAtoms = atomlist.length;

   // MEMO: loop expanded to improve performance.
   for (var _i = 0; _i < nAtoms; _i++) {
      for (var _j = _i + 1; _j < _i + 40 && _j < nAtoms; _j++) {
         var i = atomlist[_i], j = atomlist[_j];
         var atom1 = this.atoms[i], atom2 = this.atoms[j];
         if (atom1 == undefined || atom2 == undefined) continue;
         if (!this.isConnected(atom1, atom2)) continue;
   
         var midpoint = new THREE.Vector3((atom1.x + atom2.x) / 2,
                 (atom1.y + atom2.y) / 2, (atom1.z + atom2.z) / 2);
         var color = new THREE.Color(atom1.color);
         geo.vertices.push(new THREE.Vertex(new THREE.Vector3(atom1.x, atom1.y, atom1.z)));
         geo.colors.push(color);
         geo.vertices.push(new THREE.Vertex(midpoint));
         geo.colors.push(color);

         color = new THREE.Color(atom2.color);
         geo.vertices.push(new THREE.Vertex(new THREE.Vector3(atom2.x, atom2.y, atom2.z)));
         geo.colors.push(color);
         geo.vertices.push(new THREE.Vertex(midpoint));
         geo.colors.push(color);
       }
    }
   var lineMaterial = new THREE.LineBasicMaterial({linewidth: lineWidth});
   lineMaterial.vertexColors = true;

   var line = new THREE.Line(geo, lineMaterial);
   line.type = THREE.Lines;
   group.add(line);
};

// MEMO: Shold I interpolate colors ?
GLmol.prototype.drawSmoothCurve = function(group, _points, width, colors) {
   if (_points.length == 0) return;

   var geo = new THREE.Geometry();
   var points = this.subdivide(_points, 5);
   console.log([_points.length, points.length, colors.length]);

   for (var i = 0; i < points.length; i++) {
      geo.vertices.push(new THREE.Vertex(points[i]));
      geo.colors.push(new THREE.Color(colors[Math.round((i - 1) / 5)]));
  }
  var lineMaterial = new THREE.LineBasicMaterial({linewidth: width});
  lineMaterial.vertexColors = true;
  var line = new THREE.Line(geo, lineMaterial);
  line.type = THREE.LineStrip;
  group.add(line);
};

// MEMO: This routine assumes chains and residues are ordered and
//  atoms N, CA, C and O come in this order. Is this OK?
GLmol.prototype.drawMainchainCurve = function(group, atomlist, curveWidth, atomName) {
   var points = [], colors = [];
   var currentChain, currentResi;
   for (var i in atomlist) {
      var atom = this.atoms[atomlist[i]];
      if (atom == undefined) continue;

      if ((atom.atom == atomName) && !atom.hetflag) {
         if (currentChain != atom.chain || currentResi + 1 != atom.resi) {
            this.drawSmoothCurve(group, points, curveWidth, colors);
            points = [];
            colors = [];
         }
         points.push(new THREE.Vector3(atom.x, atom.y, atom.z));
         colors.push(atom.color);
         currentChain = atom.chain;
         currentResi = atom.resi;
      }
   }
   this.drawSmoothCurve(group, points, curveWidth, colors);

   // TODO: Arrow heads
};

// MEMO: Should I interpolate colors, too?
// FIXME: Normals, doubleSided(bug in Three.js), vertex colors
GLmol.prototype.drawStrip = function(group, points1, points2, colors) {
   if ((points1.length) < 2) return;

   points1 = this.subdivide(points1, 5);
   points2 = this.subdivide(points2, 5);

   var geo = new THREE.Geometry();
   var i;
   for (i = 0; i < points1.length; i++) {
      geo.vertices.push(new THREE.Vertex(points1[i])); // 2i
      geo.vertices.push(new THREE.Vertex(points2[i])); // 2i + 1
   }
   for (i = 1; i < points1.length; i++) {
      var f;
      var diff = new THREE.Vector3().sub(points1[i], points2[i]);
         f = new THREE.Face4(2 * i, 2 * i + 1, 2 * i - 1, 2 * i - 2);
         f.color = new THREE.Color(colors[Math.round((i - 1)/ 5)]);
         geo.faces.push(f);
   }
   geo.computeFaceNormals();
   geo.computeVertexNormals(false);
   var material =  new THREE.MeshLambertMaterial();
   material.vertexColors = THREE.FaceColors; // FIXME: or VertexColors? which should I use?
   var mesh = new THREE.Mesh(geo, material);
   mesh.doubleSided = true;
   group.add(mesh);
};

GLmol.prototype.IcosahedronGeometry = function() {
   var geo = new THREE.Geometry();
   var m = 0.5;
   var n = (1 + Math.sqrt(5)) / 4;
   var norm = Math.sqrt(m * m + n * n);
   m /= norm; n /= norm;
   var vs = [[m, n, 0], [-m, n, 0], [-m, -n, 0], [m, -n, 0], 
             [0, m, n], [0, -m, n], [0, -m, -n], [0, m, -n],
             [n, 0, m], [n, 0, -m], [-n, 0, -m], [-n, 0, m]];
   var fs = [[5, 11, 2], [2, 11, 10], [10, 11, 1], [10, 1, 7], [7, 1, 0], [7, 0, 9], [9, 0, 8],
[8, 3, 9], [9, 3, 6], [3, 2, 6], [6, 2, 10], [10, 6, 7], [7, 6, 9], [2, 3, 5], [5, 3, 8],
[4, 0, 1], [0, 4, 8], [8, 4, 5], [5, 4, 11], [11, 4, 1]];
   for (var i = 0; i < 12; i++)
      geo.vertices.push(new THREE.Vertex(new THREE.Vector3(vs[i][0], vs[i][1], vs[i][2])));
   for (var i = 0; i < fs.length; i++) {
      var f = new THREE.Face3(fs[i][0], fs[i][1], fs[i][2]);
      f.vertexNormals = [geo.vertices[fs[i][0]].position, geo.vertices[fs[i][1]].position, geo.vertices[fs[i][2]].position];
      geo.faces.push(f);
  }
   return geo;
};

GLmol.prototype.drawCylinder = function(group, from, to, radius, color) {
   if (!from || !to) return;

   var midpoint = new THREE.Vector3().add(from, to).multiplyScalar(0.5);
   var color = new THREE.Color(color);

// MEMO: Three.js (r45) changed CylinderGeometry.
   var cylinderGeometry = new THREE.CylinderGeometry(radius, radius, 1, this.cylinderQuality, 1, true);
   var cylinderMaterial = new THREE.MeshLambertMaterial({color: color.getHex()});
// MEMO: now cylinder points to z direction so that we cannot use 'lookAt'.
//  any better way to do this?
   var cylinder = new THREE.Mesh(cylinderGeometry, cylinderMaterial);
   cylinder.doubleSided = true;
   cylinder.rotation.x = Math.PI / 2;
   cylinder.scale.y = from.distanceTo(to);
   var cylinder2 = new THREE.Object3D();
   cylinder2.add(cylinder);
   cylinder2.position = midpoint;
   cylinder2.lookAt(from);
   group.add(cylinder2);
};

GLmol.prototype.drawHelixAsCylinder = function(group, atomlist, radius) {
   var start = null;
   var currentChain, currentResi;

   for (var i in atomlist) {
      var atom = this.atoms[atomlist[i]];
      if (atom == undefined) continue;
      if (atom.hetflag || atom.atom != 'CA') continue;

      console.log(atom.chain, atom.resi, atom.ss, atom.begin, atom.end);
      if (atom.ss == 'h' && atom.ssend /* || currentChain != atom.chain || currentResi + 1 != atom.resi*/) {
         this.drawCylinder(group, start, new THREE.Vector3(atom.x, atom.y, atom.z), radius, atom.color);
         start = null;
      }
      currentChain = atom.chain;
      currentResi = atom.resi;
      if (start == null && atom.ss == 'h' && atom.ssbegin) start = new THREE.Vector3(atom.x, atom.y, atom.z);
   }
   this.drawCylinder(group, start, new THREE.Vector3(atom.x, atom.y, atom.z), radius, atom.color);
};

// MEMO: This routine assumes chains and residues are ordered and
//  atoms N, CA, C and O come in this order. Is this OK?
// MEMO: This routine draws FAKE cartoon representation, which is 
//  closer to Rasmol's "ribbon" representation.
// TODO: Implement true "Cartoon" with thickness and arrowhead.
// FIXME: 󼡹¤ζܤǿ߽ФˤĤƤƱ塣󼡹¤Ȥ褹٤Ǥ

GLmol.prototype.drawCartoon = function(group, atomlist) {
   var points1 = [], points2 = [];
   var colors = [];
   var currentChain, currentResi, currentCA;
   var prevCO = null; //new THREE.Vector3(1, 1, 1);

   for (var i in atomlist) {
      var atom = this.atoms[atomlist[i]];
      if (atom == undefined) continue;

      if ((atom.atom == 'O' || atom.atom == 'CA') && !atom.hetflag) {
         if (atom.atom == 'CA') {
            if (currentChain != atom.chain || currentResi + 1 != atom.resi) {
               this.drawStrip(group, points1, points2, colors);
               // MEMO: emphasize 'edges' 
               this.drawSmoothCurve(group, points1, 1 ,colors);
               this.drawSmoothCurve(group, points2, 1 ,colors);
               points1 = []; points2 = [];
               colors = [];
               prevCO = null;
            }
            currentCA = new THREE.Vector3(atom.x, atom.y, atom.z);
            currentChain = atom.chain;
            currentResi = atom.resi;
            colors.push(atom.color);
         } else { // O
            var O = new THREE.Vector3(atom.x, atom.y, atom.z);
            O.subSelf(currentCA);
/*            if (atom.ss == 's' && atom.ssend) {
                    var tmp = O.clone().multiplyScalar(0.75); // ΤǤϥᡣĥƤޤ
            if (prevCO != undefined && O.dot(prevCO) < 0) {
               O.negate();
            }

               var p1 = new THREE.Vector3();
               var p2 = new THREE.Vector3();
               points1.push(p1.add(currentCA, tmp));
               points2.push(p2.sub(currentCA, tmp));
               colors.push(colors[colors.size - 1]); // FIXME: 
            }*/
            O.multiplyScalar((atom.ss == 'c') ? 0.1 : 0.5); 
            if (prevCO != undefined && O.dot(prevCO) < 0) {
               O.negate();
            }
            prevCO = O;
            var p1 = new THREE.Vector3();
            var p2 = new THREE.Vector3();
            points1.push(p1.add(currentCA, prevCO));
            points2.push(p2.sub(currentCA, prevCO));
         }
      }
   }
   this.drawStrip(group, points1, points2, colors);
   this.drawSmoothCurve(group, points1, 1 ,colors);
   this.drawSmoothCurve(group, points2, 1 ,colors);
};

GLmol.prototype.drawCartoonNucleicAcid = function(group, atomlist) {
   var points1 = [], points2 = [];
   var colors = [];
   var currentChain, currentResi, currentCA;
   var prevCO = null; //new THREE.Vector3(1, 1, 1);

   for (var i in atomlist) {
      var atom = this.atoms[atomlist[i]];
      if (atom == undefined) continue;

      if ((atom.atom == 'O3\'' || atom.atom == 'OP2') && !atom.hetflag) { // or P?
         if (atom.atom == 'O3\'') {
            if (currentChain != atom.chain || currentResi + 1 != atom.resi) {
               this.drawStrip(group, points1, points2, colors);
               // MEMO: emphasize 'edges' 
               this.drawSmoothCurve(group, points1, 1 ,colors);
               this.drawSmoothCurve(group, points2, 1 ,colors);
               points1 = []; points2 = [];
               colors = [];
               prevCO = null; //new THREE.Vector3(1, 1, 1);
            }
            currentCA = new THREE.Vector3(atom.x, atom.y, atom.z);
            currentChain = atom.chain;
            currentResi = atom.resi;
            colors.push(atom.color); // FIXME: make atom.cartoonColor??
         } else { // OP2
            var O = new THREE.Vector3(atom.x, atom.y, atom.z);
            O.subSelf(currentCA);
            O.multiplyScalar(0.3);  // 0.6 if P
            if (prevCO != undefined && O.dot(prevCO) < 0) {
               O.negate();
            }
            prevCO = O;
            var p1 = new THREE.Vector3();
            var p2 = new THREE.Vector3();
            points1.push(p1.add(currentCA, prevCO));
            points2.push(p2.sub(currentCA, prevCO));
         }
      }
   }
   this.drawStrip(group, points1, points2, colors);
   this.drawSmoothCurve(group, points1, 1 ,colors);
   this.drawSmoothCurve(group, points2, 1 ,colors);
};

GLmol.prototype.getAllAtoms = function() {
   var ret = [];
   for (var i in this.atoms) {
      ret.push(this.atoms[i].serial);
   }
   return ret;
};

// Probably I can refactor using higher-order functions.
GLmol.prototype.getHetatms = function(atomlist) {
   var ret = [];
   for (var i in atomlist) {
      var atom = this.atoms[atomlist[i]]; if (atom == undefined) continue;

      if (atom.hetflag) ret.push(atom.serial);
   }
   return ret;
};

GLmol.prototype.removeSolvents = function(atomlist) {
   var ret = [];
   for (var i in atomlist) {
      var atom = this.atoms[atomlist[i]]; if (atom == undefined) continue;

      if (atom.resn != 'HOH') ret.push(atom.serial);
   }
   return ret;
};

GLmol.prototype.getProteins = function(atomlist) {
   var ret = [];
   for (var i in atomlist) {
      var atom = this.atoms[atomlist[i]]; if (atom == undefined) continue;

      if (!atom.hetflag) ret.push(atom.serial);
   }
   return ret;
};

GLmol.prototype.getSidechains = function(atomlist) {
   var ret = [];
   for (var i in atomlist) {
      var atom = this.atoms[atomlist[i]]; if (atom == undefined) continue;

      if (atom.hetflag) continue;
      if (atom.atom == 'C' || atom.atom == 'O' || atom.atom == 'N') continue;
      ret.push(atom.serial);
   }
   return ret;
};

GLmol.prototype.getExtent = function(atomlist) {
   var xmin = ymin = zmin = 9999;
   var xmax = ymax = zmax = -9999;

   for (var i in atomlist) {
      var atom = this.atoms[atomlist[i]]; if (atom == undefined) continue;

      xmin = (xmin < atom.x) ? xmin : atom.x;
      ymin = (ymin < atom.y) ? ymin : atom.y;
      zmin = (zmin < atom.z) ? zmin : atom.z;
      xmax = (xmax > atom.x) ? xmax : atom.x;
      ymax = (ymax > atom.y) ? ymax : atom.y;
      zmax = (zmax > atom.z) ? zmax : atom.z;
   }
   return [[xmin, ymin, zmin], [xmax, ymax, zmax]];
};

GLmol.prototype.getCenter = function(atomlist) {
   var tmp = this.getExtent(atomlist);

   return new THREE.Vector3((tmp[0][0] + tmp[1][0]) / 2, (tmp[0][1] + tmp[1][1]) / 2, (tmp[0][2] + tmp[1][2]) / 2);
};

GLmol.prototype.getResiduesById = function(atomlist, resi) {
   var ret = [];
   for (var i in atomlist) {
      var atom = this.atoms[atomlist[i]]; if (atom == undefined) continue;

      if (resi.indexOf(atom.resi) != -1) ret.push(atom.serial);
   }
   return ret;
};

GLmol.prototype.getChain = function(atomlist, chain) {
   var ret = [];
   for (var i in atomlist) {
      var atom = this.atoms[atomlist[i]]; if (atom == undefined) continue;

      if (chain.indexOf(atom.chain) != -1) ret.push(atom.serial);
   }
   return ret;
};

GLmol.prototype.colorByAtom = function(atomlist, colors) {
   for (var i in atomlist) {
      var atom = this.atoms[atomlist[i]]; if (atom == undefined) continue;

      var c = colors[atom.elem];
      if (c == undefined) c = this.ElementColors[atom.elem];
      if (c == undefined) c = this.defaultColor;
      atom.color = c;
   }
};

// MEMO: Color only CAs. maybe I should add atom.cartoonColor.
GLmol.prototype.colorByStructure = function(atomlist, helixColor, sheetColor) {
   for (var i in atomlist) {
      var atom = this.atoms[atomlist[i]]; if (atom == undefined) continue;

      if (atom.atom != 'CA' || atom.hetflag) continue;
      if (atom.ss[0] == 's') atom.color = sheetColor;
      else if (atom.ss[0] == 'h') atom.color = helixColor;
   }
};

// MEMO: Color only CAs. maybe I should add atom.cartoonColor.
GLmol.prototype.colorByChain = function(atomlist) {
    var Nucleotides = ['  G', '  A', '  T', '  C', '  U', ' DG', ' DA', ' DT', ' DC', ' DU'];
   for (var i in atomlist) {
      var atom = this.atoms[atomlist[i]]; if (atom == undefined) continue;

       if (atom.hetflag) continue;
//       if (atom.atom == 'P') console.log(atom);
       if (atom.atom == 'CA' || (atom.atom == 'P' && (Nucleotides.indexOf(atom.resn) != -1))) {
         var color = new THREE.Color(0);
         color.setHSV((atom.chain.charCodeAt(0)) % 15 / 15.0, 1, 0.9);
         atom.color = color.getHex();
       }
   }
};

GLmol.prototype.colorByResidue = function(atomlist, residueColors) {
   for (var i in atomlist) {
      var atom = this.atoms[atomlist[i]]; if (atom == undefined) continue;

      c = residueColors[atom.resn]
      if (c != undefined) atom.color = c;
   }
};

GLmol.prototype.colorByPolarity = function(atomlist, polar, nonpolar) {
   var polarResidues = ['ARG', 'HIS', 'LYS', 'ASP', 'GLU', 'SER', 'THR', 'ASN', 'GLN', 'CYS'];
   var nonPolarResidues = ['GLY', 'PRO', 'ALA', 'VAL', 'LEU', 'ILE', 'MET', 'PHE', 'TYR', 'TRP'];
   var colorMap = {};
   for (var i in polarResidues) colorMap[polarResidues[i]] = polar;
   for (i in nonPolarResidues) colorMap[nonPolarResidues[i]] = nonpolar;
   this.colorByResidue(atomlist, colorMap);   
};

// TODO: Add near(atomlist, neighbor, distanceCutoff)
// TODO: Add expandToResidue(atomlist)
// TODO: Add exclude(atomlist, atomlist)

GLmol.prototype.colorChainbow = function(atomlist) {
   var cnt = 0;
   var atom, i;
   for (i in atomlist) {
      atom = this.atoms[atomlist[i]]; if (atom == undefined) continue;

      if (atom.atom != 'CA' || atom.hetflag) continue;
      cnt++;
   }

   var total = cnt;
   cnt = 0;
   for (i in atomlist) {
      atom = this.atoms[atomlist[i]]; if (atom == undefined) continue;

      if (atom.atom != 'CA' || atom.hetflag) continue;
      var color = new THREE.Color(0);
      color.setHSV(240.0 / 360 * cnt / total, 1, 0.9);
      atom.color = color.getHex();
      cnt++;
   }
};

// FIXME: this is just for testing...
GLmol.prototype.drawSymmetryMates = function(group, atomlist, matrices) {
   if (matrices == undefined) return;

   for (var i = 0; i < matrices.length; i++) {
      var mat = matrices[i];
      if (mat == undefined || mat.isIdentity()) continue;
      var symmetryMate = new THREE.Object3D();
      this.drawMainchainCurve(symmetryMate, atomlist, this.curveWidth, 'CA');
      this.drawMainchainCurve(symmetryMate, atomlist, this.curveWidth, 'P');
      symmetryMate.matrixAutoUpdate = false;
      symmetryMate.matrix = mat;
      group.add(symmetryMate);
   }
};


GLmol.prototype.drawSymmetryMatesWithTranslation = function(group, atomlist, matrices) {
   if (matrices == undefined) return;

   for (var i = 0; i < matrices.length; i++) {
      var mat = matrices[i];
      if (mat == undefined) continue;

      for (var a = -1; a <= 0; a++) {
         for (var b = -1; b <= 0; b++) {
             for (var c = -1; c <= 0; c++) {
                var symmetryMate = new THREE.Object3D();
                this.drawMainchainCurve(symmetryMate, atomlist, this.curveWidth, 'CA');
                this.drawMainchainCurve(symmetryMate, atomlist, this.curveWidth, 'P');
                symmetryMate.matrixAutoUpdate = false;
                var translationMat = new THREE.Matrix4().setTranslation(
                   this.protein.ax * a + this.protein.bx * b + this.protein.cx * c,
                   this.protein.ay * a + this.protein.by * b + this.protein.cy * c,
                   this.protein.az * a + this.protein.bz * b + this.protein.cz * c);
                symmetryMate.matrix = mat.clone().multiplySelf(translationMat);
                if (symmetryMate.matrix.isIdentity()) continue;
                group.add(symmetryMate);
             }
         }
      }
   }
};

GLmol.prototype.defineRepresentation = function() {
   var all = this.getAllAtoms();
   var hetatm = this.removeSolvents(this.getHetatms(all));
   this.colorByAtom(all, {});
   this.colorByChain(all);

   this.drawAtomsAsSphere(this.modelGroup, hetatm, this.sphereRadius); 
   this.drawMainchainCurve(this.modelGroup, all, this.curveWidth, 'P');
   this.drawCartoon(this.modelGroup, all, this.curveWidth);

   // FIXME: support symmetry mates
   this.modelGroup.position = this.getCenter(all).multiplyScalar(-1);
};

GLmol.prototype.rebuildScene = function() {
   this.scene = new THREE.Scene();

   //TODO: Implement better fog. Must be correlated with SLAB. 
//   this.scene.fog = new THREE.Fog(0x000000, 50, 500);
   this.modelGroup = new THREE.Object3D();
   this.protein = {sheet: [], helix: [], scaleMatrix: new THREE.Matrix4().identity(), biomtMatrices: [], symmetryMatrices: []};
   this.atoms = [];

   var source = $('#' + this.id + '_src').val();
   
   this.parsePDB2(source);
   this.parseSDF(source);
  
   this.defineRepresentation();

   this.rotationGroup = new THREE.Object3D();
   this.rotationGroup.useQuaternion = true;
   this.rotationGroup.quaternion = new THREE.Quaternion(1, 0, 0, 0);
   this.rotationGroup.add(this.modelGroup);

   // Zoom adjustment
   // FIXME: better way to do?
   if (this.atoms.length > 10000) this.rotationGroup.position.z = 80;
   if (this.atoms.length < 150) this.rotationGroup.position.z = -120;

   this.scene.add(this.rotationGroup);
   this.setupLights(this.scene);

    this.show();
 };

GLmol.prototype.enableMouse = function() {
// FIXME: Why cannot I use 'this' inside closure?
   var parentObj = this; 

   this.container.mousedown(function(ev) {
      parentObj.mouseButton = ev.which;
      parentObj.isDragging = true;
      parentObj.mouseStartX = ev.pageX;
      parentObj.mouseStartY = ev.pageY;
      parentObj.cq = parentObj.rotationGroup.quaternion;
      parentObj.cz = parentObj.rotationGroup.position.z;
      parentObj.currentModelPos = parentObj.modelGroup.position.clone();
      ev.preventDefault();
    });

   // FIXME: Not working in Mac OS X?
   this.container.bind('DOMMouseScroll', function(ev) {ev.preventDefault();});
   this.container.bind("contextmenu", function(ev) {ev.preventDefault();});
   $('body').mouseup(function(ev) { // FIXME: Not working outside canvas?
      parentObj.isDragging = false;
   });

   $('#' + this.id + '_reload').click(function(ev) { parentObj.rebuildScene(); parentObj.show();});
   this.container.mousemove(function(ev) {
      if (!parentObj.isDragging) return;
      var dx = (ev.pageX - parentObj.mouseStartX) / parentObj.WIDTH;
      var dy = (ev.pageY - parentObj.mouseStartY) / parentObj.HEIGHT;
      var r = Math.sqrt(dx * dx + dy * dy);
      if (parentObj.mouseButton == 3 || ev.shiftKey) {
           parentObj.rotationGroup.position.z = parentObj.cz - dy * 100;
//           console.log(parentObj.rotationGroup.position.z);
      } else if (parentObj.mouseButton == 2 || ev.ctrlKey) {
           var translationByScreen = new THREE.Vector3(- dx * 100, - dy * 100, 0);
           var q = parentObj.rotationGroup.quaternion;
           var qinv = new THREE.Quaternion(q.x, q.y, q.z, q.w).inverse().normalize(); 
           var translation = qinv.multiplyVector3(translationByScreen);
           parentObj.modelGroup.position.x = parentObj.currentModelPos.x + translation.x;
           parentObj.modelGroup.position.y = parentObj.currentModelPos.y + translation.y;
           parentObj.modelGroup.position.z = parentObj.currentModelPos.z + translation.z;
      } else if (parentObj.mouseButton == 1 && r != 0) {
           var rs = Math.sin(r * Math.PI) / r;
           parentObj.dq.x = Math.cos(r * Math.PI); 
           parentObj.dq.y = 0;
           parentObj.dq.z =  rs * dx; 
           parentObj.dq.w =  rs * dy;
           parentObj.rotationGroup.quaternion = new THREE.Quaternion(1, 0, 0, 0); 
           parentObj.rotationGroup.quaternion.multiplySelf(parentObj.dq);
           parentObj.rotationGroup.quaternion.multiplySelf(parentObj.cq);
      }
     parentObj.show();
   });
};


GLmol.prototype.show = function() {
        this.renderer.render(this.scene, this.camera);
};

// For scripting
GLmol.prototype.doFunc = function(func) {
    func(this);
};

return GLmol;
}());
