/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hcatalog.hbase.snapshot;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import org.apache.hadoop.conf.Configuration;
import org.apache.hcatalog.hbase.snapshot.FamilyRevision;
import org.apache.hcatalog.hbase.snapshot.PathUtil;
import org.apache.hcatalog.hbase.snapshot.RevisionManager;
import org.apache.hcatalog.hbase.snapshot.TableSnapshot;
import org.apache.hcatalog.hbase.snapshot.Transaction;
import org.apache.hcatalog.hbase.snapshot.ZKUtil;
import org.apache.hcatalog.hbase.snapshot.lock.LockListener;
import org.apache.hcatalog.hbase.snapshot.lock.WriteLock;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooDefs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZKBasedRevisionManager
implements RevisionManager {
    private static final Logger LOG = LoggerFactory.getLogger(ZKBasedRevisionManager.class);
    private String zkHostList;
    private String baseDir;
    private ZKUtil zkUtil;
    private long writeTxnTimeout;

    @Override
    public void initialize(Configuration conf) {
        if ((conf = new Configuration(conf)).get("revision.manager.zk.hostList") == null) {
            String zkHostList = conf.get("hbase.zookeeper.quorum");
            int port = conf.getInt("hbase.zookeeper.property.clientPort", 2181);
            String[] splits = zkHostList.split(",");
            StringBuffer sb = new StringBuffer();
            for (String split : splits) {
                sb.append(split);
                sb.append(':');
                sb.append(port);
                sb.append(',');
            }
            sb.deleteCharAt(sb.length() - 1);
            conf.set("revision.manager.zk.hostList", sb.toString());
        }
        this.zkHostList = conf.get("revision.manager.zk.hostList");
        this.baseDir = conf.get("revision.manager.zk.dataDir");
        this.writeTxnTimeout = Long.parseLong(conf.get("revision.manager.writeTxn.timeout"));
    }

    @Override
    public void open() throws IOException {
        this.zkUtil = new ZKUtil(this.zkHostList, this.baseDir);
        this.zkUtil.createRootZNodes();
        LOG.info("Created root znodes for revision manager.");
    }

    @Override
    public void close() {
        this.zkUtil.closeZKConnection();
    }

    private void checkInputParams(String table, List<String> families) {
        if (table == null) {
            throw new IllegalArgumentException("The table name must be specified for reading.");
        }
        if (families == null || families.isEmpty()) {
            throw new IllegalArgumentException("At least one column family should be specified for reading.");
        }
    }

    @Override
    public void createTable(String table, List<String> columnFamilies) throws IOException {
        this.zkUtil.createRootZNodes();
        this.zkUtil.setUpZnodesForTable(table, columnFamilies);
    }

    @Override
    public void dropTable(String table) throws IOException {
        this.zkUtil.deleteZNodes(table);
    }

    @Override
    public Transaction beginWriteTransaction(String table, List<String> families, long keepAlive) throws IOException {
        this.checkInputParams(table, families);
        this.zkUtil.setUpZnodesForTable(table, families);
        long nextId = this.zkUtil.nextId(table);
        long expireTimestamp = this.zkUtil.getTimeStamp();
        Transaction transaction = new Transaction(table, families, nextId, expireTimestamp);
        if (keepAlive != -1L) {
            transaction.setKeepAlive(keepAlive);
        } else {
            transaction.setKeepAlive(this.writeTxnTimeout);
        }
        this.refreshTransactionList(transaction.getTableName());
        String lockPath = this.prepareLockNode(table);
        WriteLock wLock = new WriteLock(this.zkUtil.getSession(), lockPath, ZooDefs.Ids.OPEN_ACL_UNSAFE);
        RMLockListener myLockListener = new RMLockListener();
        wLock.setLockListener(myLockListener);
        try {
            boolean lockGrabbed = wLock.lock();
            if (!lockGrabbed) {
                throw new IOException("Unable to obtain lock while beginning transaction. " + transaction.toString());
            }
            List<String> colFamilies = transaction.getColumnFamilies();
            FamilyRevision revisionData = transaction.getFamilyRevisionInfo();
            for (String cfamily : colFamilies) {
                String path = PathUtil.getRunningTxnInfoPath(this.baseDir, table, cfamily);
                this.zkUtil.updateData(path, revisionData, ZKUtil.UpdateMode.APPEND);
            }
        }
        catch (KeeperException e) {
            throw new IOException("Exception while obtaining lock.", e);
        }
        catch (InterruptedException e) {
            throw new IOException("Exception while obtaining lock.", e);
        }
        finally {
            wLock.unlock();
        }
        return transaction;
    }

    @Override
    public Transaction beginWriteTransaction(String table, List<String> families) throws IOException {
        return this.beginWriteTransaction(table, families, -1L);
    }

    @Override
    public void commitWriteTransaction(Transaction transaction) throws IOException {
        this.refreshTransactionList(transaction.getTableName());
        String lockPath = this.prepareLockNode(transaction.getTableName());
        WriteLock wLock = new WriteLock(this.zkUtil.getSession(), lockPath, ZooDefs.Ids.OPEN_ACL_UNSAFE);
        RMLockListener myLockListener = new RMLockListener();
        wLock.setLockListener(myLockListener);
        try {
            boolean lockGrabbed = wLock.lock();
            if (!lockGrabbed) {
                throw new IOException("Unable to obtain lock while commiting transaction. " + transaction.toString());
            }
            String tableName = transaction.getTableName();
            List<String> colFamilies = transaction.getColumnFamilies();
            FamilyRevision revisionData = transaction.getFamilyRevisionInfo();
            for (String cfamily : colFamilies) {
                String path = PathUtil.getRunningTxnInfoPath(this.baseDir, tableName, cfamily);
                this.zkUtil.updateData(path, revisionData, ZKUtil.UpdateMode.REMOVE);
            }
        }
        catch (KeeperException e) {
            throw new IOException("Exception while obtaining lock.", e);
        }
        catch (InterruptedException e) {
            throw new IOException("Exception while obtaining lock.", e);
        }
        finally {
            wLock.unlock();
        }
        LOG.info("Write Transaction committed: " + transaction.toString());
    }

    @Override
    public void abortWriteTransaction(Transaction transaction) throws IOException {
        this.refreshTransactionList(transaction.getTableName());
        String lockPath = this.prepareLockNode(transaction.getTableName());
        WriteLock wLock = new WriteLock(this.zkUtil.getSession(), lockPath, ZooDefs.Ids.OPEN_ACL_UNSAFE);
        RMLockListener myLockListener = new RMLockListener();
        wLock.setLockListener(myLockListener);
        try {
            boolean lockGrabbed = wLock.lock();
            if (!lockGrabbed) {
                throw new IOException("Unable to obtain lock while aborting transaction. " + transaction.toString());
            }
            String tableName = transaction.getTableName();
            List<String> colFamilies = transaction.getColumnFamilies();
            FamilyRevision revisionData = transaction.getFamilyRevisionInfo();
            for (String cfamily : colFamilies) {
                String path = PathUtil.getRunningTxnInfoPath(this.baseDir, tableName, cfamily);
                this.zkUtil.updateData(path, revisionData, ZKUtil.UpdateMode.REMOVE);
                path = PathUtil.getAbortInformationPath(this.baseDir, tableName, cfamily);
                this.zkUtil.updateData(path, revisionData, ZKUtil.UpdateMode.APPEND);
            }
        }
        catch (KeeperException e) {
            throw new IOException("Exception while obtaining lock.", e);
        }
        catch (InterruptedException e) {
            throw new IOException("Exception while obtaining lock.", e);
        }
        finally {
            wLock.unlock();
        }
        LOG.info("Write Transaction aborted: " + transaction.toString());
    }

    @Override
    public void keepAlive(Transaction transaction) throws IOException {
        this.refreshTransactionList(transaction.getTableName());
        transaction.keepAliveTransaction();
        String lockPath = this.prepareLockNode(transaction.getTableName());
        WriteLock wLock = new WriteLock(this.zkUtil.getSession(), lockPath, ZooDefs.Ids.OPEN_ACL_UNSAFE);
        RMLockListener myLockListener = new RMLockListener();
        wLock.setLockListener(myLockListener);
        try {
            boolean lockGrabbed = wLock.lock();
            if (!lockGrabbed) {
                throw new IOException("Unable to obtain lock for keep alive of transaction. " + transaction.toString());
            }
            String tableName = transaction.getTableName();
            List<String> colFamilies = transaction.getColumnFamilies();
            FamilyRevision revisionData = transaction.getFamilyRevisionInfo();
            for (String cfamily : colFamilies) {
                String path = PathUtil.getRunningTxnInfoPath(this.baseDir, tableName, cfamily);
                this.zkUtil.updateData(path, revisionData, ZKUtil.UpdateMode.KEEP_ALIVE);
            }
        }
        catch (KeeperException e) {
            throw new IOException("Exception while obtaining lock.", e);
        }
        catch (InterruptedException e) {
            throw new IOException("Exception while obtaining lock.", e);
        }
        finally {
            wLock.unlock();
        }
    }

    @Override
    public TableSnapshot createSnapshot(String tableName) throws IOException {
        this.refreshTransactionList(tableName);
        long latestID = this.zkUtil.currentID(tableName);
        HashMap<String, Long> cfMap = new HashMap<String, Long>();
        List<String> columnFamilyNames = this.zkUtil.getColumnFamiliesOfTable(tableName);
        for (String cfName : columnFamilyNames) {
            long version;
            String cfPath = PathUtil.getRunningTxnInfoPath(this.baseDir, tableName, cfName);
            List<FamilyRevision> tranxList = this.zkUtil.getTransactionList(cfPath);
            if (!tranxList.isEmpty()) {
                Collections.sort(tranxList);
                long runningVersion = tranxList.get(0).getRevision();
                version = runningVersion - 1L;
            } else {
                version = latestID;
            }
            cfMap.put(cfName, version);
        }
        TableSnapshot snapshot = new TableSnapshot(tableName, cfMap, latestID);
        LOG.debug("Created snapshot For table: " + tableName + " snapshot: " + snapshot);
        return snapshot;
    }

    @Override
    public TableSnapshot createSnapshot(String tableName, long revision) throws IOException {
        long currentID = this.zkUtil.currentID(tableName);
        if (revision > currentID) {
            throw new IOException("The revision specified in the snapshot is higher than the current revision of the table.");
        }
        this.refreshTransactionList(tableName);
        HashMap<String, Long> cfMap = new HashMap<String, Long>();
        List<String> columnFamilies = this.zkUtil.getColumnFamiliesOfTable(tableName);
        for (String cf : columnFamilies) {
            cfMap.put(cf, revision);
        }
        return new TableSnapshot(tableName, cfMap, revision);
    }

    List<FamilyRevision> getRunningTransactions(String table, String columnFamily) throws IOException {
        String path = PathUtil.getRunningTxnInfoPath(this.baseDir, table, columnFamily);
        return this.zkUtil.getTransactionList(path);
    }

    @Override
    public List<FamilyRevision> getAbortedWriteTransactions(String table, String columnFamily) throws IOException {
        String path = PathUtil.getAbortInformationPath(this.baseDir, table, columnFamily);
        return this.zkUtil.getTransactionList(path);
    }

    private void refreshTransactionList(String tableName) throws IOException {
        String lockPath = this.prepareLockNode(tableName);
        WriteLock wLock = new WriteLock(this.zkUtil.getSession(), lockPath, ZooDefs.Ids.OPEN_ACL_UNSAFE);
        RMLockListener myLockListener = new RMLockListener();
        wLock.setLockListener(myLockListener);
        try {
            boolean lockGrabbed = wLock.lock();
            if (!lockGrabbed) {
                throw new IOException("Unable to obtain lock while refreshing transactions of table " + tableName + ".");
            }
            List<String> cfPaths = this.zkUtil.getColumnFamiliesOfTable(tableName);
            for (String cf : cfPaths) {
                String runningDataPath = PathUtil.getRunningTxnInfoPath(this.baseDir, tableName, cf);
                this.zkUtil.refreshTransactions(runningDataPath);
            }
        }
        catch (KeeperException e) {
            throw new IOException("Exception while obtaining lock.", e);
        }
        catch (InterruptedException e) {
            throw new IOException("Exception while obtaining lock.", e);
        }
        finally {
            wLock.unlock();
        }
    }

    private String prepareLockNode(String tableName) throws IOException {
        String txnDataPath = PathUtil.getTxnDataPath(this.baseDir, tableName);
        String lockPath = PathUtil.getLockManagementNode(txnDataPath);
        this.zkUtil.ensurePathExists(lockPath, null, ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
        return lockPath;
    }

    class RMLockListener
    implements LockListener {
        RMLockListener() {
        }

        @Override
        public void lockAcquired() {
        }

        @Override
        public void lockReleased() {
        }
    }
}

