/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import com.google.common.base.Preconditions;
import java.io.ByteArrayOutputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.MediumTests;
import org.apache.hadoop.hbase.ResourceCheckerJUnitRule;
import org.apache.hadoop.hbase.fs.HFileSystem;
import org.apache.hadoop.hbase.io.encoding.DataBlockEncoding;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.Compression;
import org.apache.hadoop.hbase.io.hfile.HFileBlock;
import org.apache.hadoop.hbase.io.hfile.HFileDataBlockEncoder;
import org.apache.hadoop.hbase.io.hfile.HFileDataBlockEncoderImpl;
import org.apache.hadoop.hbase.io.hfile.NoOpDataBlockEncoder;
import org.apache.hadoop.hbase.io.hfile.TestHFileBlock;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.ChecksumType;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.io.compress.CompressionOutputStream;
import org.apache.hadoop.io.compress.Compressor;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@Category(value={MediumTests.class})
@RunWith(value=Parameterized.class)
public class TestHFileBlockCompatibility {
    private static final boolean[] BOOLEAN_VALUES = new boolean[]{false, true};
    private static final Log LOG = LogFactory.getLog(TestHFileBlockCompatibility.class);
    private static final Compression.Algorithm[] COMPRESSION_ALGORITHMS = new Compression.Algorithm[]{Compression.Algorithm.NONE, Compression.Algorithm.GZ};
    private static int MINOR_VERSION = 0;
    private static final HBaseTestingUtility TEST_UTIL = new HBaseTestingUtility();
    private HFileSystem fs;
    private int uncompressedSizeV1;
    private final boolean includesMemstoreTS;
    @Rule
    public ResourceCheckerJUnitRule cu = new ResourceCheckerJUnitRule();

    public TestHFileBlockCompatibility(boolean includesMemstoreTS) {
        this.includesMemstoreTS = includesMemstoreTS;
    }

    @Parameterized.Parameters
    public static Collection<Object[]> parameters() {
        return HBaseTestingUtility.BOOLEAN_PARAMETERIZED;
    }

    @Before
    public void setUp() throws IOException {
        this.fs = (HFileSystem)HFileSystem.get((Configuration)TEST_UTIL.getConfiguration());
    }

    public byte[] createTestV1Block(Compression.Algorithm algo) throws IOException {
        Compressor compressor = algo.getCompressor();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        OutputStream os = algo.createCompressionStream((OutputStream)baos, compressor, 0);
        DataOutputStream dos = new DataOutputStream(os);
        BlockType.META.write((DataOutput)dos);
        TestHFileBlock.writeTestBlockContents(dos);
        this.uncompressedSizeV1 = dos.size();
        dos.flush();
        algo.returnCompressor(compressor);
        return baos.toByteArray();
    }

    private Writer createTestV2Block(Compression.Algorithm algo) throws IOException {
        BlockType blockType = BlockType.DATA;
        Writer hbw = new Writer(algo, null, this.includesMemstoreTS);
        DataOutputStream dos = hbw.startWriting(blockType);
        TestHFileBlock.writeTestBlockContents(dos);
        byte[] headerAndData = hbw.getHeaderAndData();
        Assert.assertEquals((long)4000L, (long)hbw.getUncompressedSizeWithoutHeader());
        hbw.releaseCompressor();
        return hbw;
    }

    private String createTestBlockStr(Compression.Algorithm algo, int correctLength) throws IOException {
        Writer hbw = this.createTestV2Block(algo);
        byte[] testV2Block = hbw.getHeaderAndData();
        int osOffset = 33;
        if (testV2Block.length == correctLength) {
            testV2Block[osOffset] = 3;
        }
        return Bytes.toStringBinary((byte[])testV2Block);
    }

    @Test
    public void testNoCompression() throws IOException {
        Assert.assertEquals((long)4000L, (long)this.createTestV2Block(Compression.Algorithm.NONE).getBlockForCaching().getUncompressedSizeWithoutHeader());
    }

    @Test
    public void testGzipCompression() throws IOException {
        String correctTestBlockStr = "DATABLK*\\x00\\x00\\x00:\\x00\\x00\\x0F\\xA0\\xFF\\xFF\\xFF\\xFF\\xFF\\xFF\\xFF\\xFF\\x1F\\x8B\\x08\\x00\\x00\\x00\\x00\\x00\\x00\\x03\\xED\\xC3\\xC1\\x11\\x00 \\x08\\xC00DD\\xDD\\x7Fa\\xD6\\xE8\\xA3\\xB9K\\x84`\\x96Q\\xD3\\xA8\\xDB\\xA8e\\xD4c\\xD46\\xEA5\\xEA3\\xEA7\\xE7\\x00LI\\x5Cs\\xA0\\x0F\\x00\\x00";
        int correctGzipBlockLength = 82;
        Assert.assertEquals((Object)"DATABLK*\\x00\\x00\\x00:\\x00\\x00\\x0F\\xA0\\xFF\\xFF\\xFF\\xFF\\xFF\\xFF\\xFF\\xFF\\x1F\\x8B\\x08\\x00\\x00\\x00\\x00\\x00\\x00\\x03\\xED\\xC3\\xC1\\x11\\x00 \\x08\\xC00DD\\xDD\\x7Fa\\xD6\\xE8\\xA3\\xB9K\\x84`\\x96Q\\xD3\\xA8\\xDB\\xA8e\\xD4c\\xD46\\xEA5\\xEA3\\xEA7\\xE7\\x00LI\\x5Cs\\xA0\\x0F\\x00\\x00", (Object)this.createTestBlockStr(Compression.Algorithm.GZ, 82));
    }

    @Test
    public void testReaderV1() throws IOException {
        for (Compression.Algorithm algo : COMPRESSION_ALGORITHMS) {
            for (boolean pread : new boolean[]{false, true}) {
                byte[] block = this.createTestV1Block(algo);
                Path path = new Path(TEST_UTIL.getDataTestDir(), "blocks_v1_" + algo);
                LOG.info((Object)("Creating temporary file at " + path));
                FSDataOutputStream os = this.fs.create(path);
                int totalSize = 0;
                int numBlocks = 50;
                for (int i = 0; i < numBlocks; ++i) {
                    os.write(block);
                    totalSize += block.length;
                }
                os.close();
                FSDataInputStream is = this.fs.open(path);
                HFileBlock.FSReaderV1 hbr = new HFileBlock.FSReaderV1(is, algo, (long)totalSize);
                int numBlocksRead = 0;
                long pos = 0L;
                while (pos < (long)totalSize) {
                    HFileBlock b = hbr.readBlockData(pos, (long)block.length, this.uncompressedSizeV1, pread);
                    b.sanityCheck();
                    pos += (long)block.length;
                    ++numBlocksRead;
                }
                Assert.assertEquals((long)numBlocks, (long)numBlocksRead);
                is.close();
            }
        }
    }

    @Test
    public void testReaderV2() throws IOException {
        for (Compression.Algorithm algo : COMPRESSION_ALGORITHMS) {
            for (boolean pread : new boolean[]{false, true}) {
                LOG.info((Object)("testReaderV2: Compression algorithm: " + algo + ", pread=" + pread));
                Path path = new Path(TEST_UTIL.getDataTestDir(), "blocks_v2_" + algo);
                FSDataOutputStream os = this.fs.create(path);
                Writer hbw = new Writer(algo, null, this.includesMemstoreTS);
                long totalSize = 0L;
                for (int blockId = 0; blockId < 2; ++blockId) {
                    DataOutputStream dos = hbw.startWriting(BlockType.DATA);
                    for (int i = 0; i < 1234; ++i) {
                        dos.writeInt(i);
                    }
                    hbw.writeHeaderAndData(os);
                    totalSize += (long)hbw.getOnDiskSizeWithHeader();
                }
                os.close();
                FSDataInputStream is = this.fs.open(path);
                HFileBlock.FSReaderV2 hbr = new HFileBlock.FSReaderV2(is, is, algo, totalSize, MINOR_VERSION, this.fs, path);
                HFileBlock b = hbr.readBlockData(0L, -1L, -1, pread);
                is.close();
                b.sanityCheck();
                Assert.assertEquals((long)4936L, (long)b.getUncompressedSizeWithoutHeader());
                Assert.assertEquals((long)(algo == Compression.Algorithm.GZ ? 2173L : 4936L), (long)(b.getOnDiskSizeWithoutHeader() - b.totalChecksumBytes()));
                String blockStr = b.toString();
                if (algo != Compression.Algorithm.GZ) continue;
                is = this.fs.open(path);
                hbr = new HFileBlock.FSReaderV2(is, is, algo, totalSize, MINOR_VERSION, this.fs, path);
                b = hbr.readBlockData(0L, (long)(2197 + b.totalChecksumBytes()), -1, pread);
                Assert.assertEquals((Object)blockStr, (Object)b.toString());
                int wrongCompressedSize = 2172;
                try {
                    b = hbr.readBlockData(0L, (long)(wrongCompressedSize + 24), -1, pread);
                    Assert.fail((String)"Exception expected");
                }
                catch (IOException ex) {
                    String expectedPrefix = "On-disk size without header provided is " + wrongCompressedSize + ", but block header contains " + b.getOnDiskSizeWithoutHeader() + ".";
                    Assert.assertTrue((String)("Invalid exception message: '" + ex.getMessage() + "'.\nMessage is expected to start with: '" + expectedPrefix + "'"), (boolean)ex.getMessage().startsWith(expectedPrefix));
                }
                is.close();
            }
        }
    }

    @Test
    public void testDataBlockEncoding() throws IOException {
        int numBlocks = 5;
        for (Compression.Algorithm algo : COMPRESSION_ALGORITHMS) {
            for (boolean pread : new boolean[]{false, true}) {
                for (DataBlockEncoding encoding : DataBlockEncoding.values()) {
                    LOG.info((Object)("testDataBlockEncoding algo " + algo + " pread = " + pread + " encoding " + encoding));
                    Path path = new Path(TEST_UTIL.getDataTestDir(), "blocks_v2_" + algo + "_" + encoding.toString());
                    FSDataOutputStream os = this.fs.create(path);
                    HFileDataBlockEncoderImpl dataBlockEncoder = new HFileDataBlockEncoderImpl(encoding);
                    Writer hbw = new Writer(algo, (HFileDataBlockEncoder)dataBlockEncoder, this.includesMemstoreTS);
                    long totalSize = 0L;
                    ArrayList<Integer> encodedSizes = new ArrayList<Integer>();
                    ArrayList<ByteBuffer> encodedBlocks = new ArrayList<ByteBuffer>();
                    for (int blockId = 0; blockId < 5; ++blockId) {
                        DataOutputStream dos = hbw.startWriting(BlockType.DATA);
                        TestHFileBlock.writeEncodedBlock(encoding, dos, encodedSizes, encodedBlocks, blockId, this.includesMemstoreTS);
                        hbw.writeHeaderAndData(os);
                        totalSize += (long)hbw.getOnDiskSizeWithHeader();
                    }
                    os.close();
                    FSDataInputStream is = this.fs.open(path);
                    HFileBlock.FSReaderV2 hbr = new HFileBlock.FSReaderV2(is, is, algo, totalSize, MINOR_VERSION, this.fs, path);
                    hbr.setDataBlockEncoder((HFileDataBlockEncoder)dataBlockEncoder);
                    hbr.setIncludesMemstoreTS(this.includesMemstoreTS);
                    int pos = 0;
                    for (int blockId = 0; blockId < 5; ++blockId) {
                        HFileBlock b = hbr.readBlockData((long)pos, -1L, -1, pread);
                        b.sanityCheck();
                        pos += b.getOnDiskSizeWithHeader();
                        Assert.assertEquals((long)((Integer)encodedSizes.get(blockId)).intValue(), (long)b.getUncompressedSizeWithoutHeader());
                        ByteBuffer actualBuffer = b.getBufferWithoutHeader();
                        if (encoding != DataBlockEncoding.NONE) {
                            Assert.assertEquals((long)0L, (long)actualBuffer.get(0));
                            Assert.assertEquals((long)encoding.getId(), (long)actualBuffer.get(1));
                            actualBuffer.position(2);
                            actualBuffer = actualBuffer.slice();
                        }
                        ByteBuffer expectedBuffer = (ByteBuffer)encodedBlocks.get(blockId);
                        expectedBuffer.rewind();
                        TestHFileBlock.assertBuffersEqual(expectedBuffer, actualBuffer, algo, encoding, pread);
                    }
                    is.close();
                }
            }
        }
    }

    public static final class Writer {
        private static final int HEADER_SIZE = 24;
        private static final boolean DONT_FILL_HEADER = false;
        private static final byte[] DUMMY_HEADER = HFileBlock.DUMMY_HEADER_NO_CHECKSUM;
        private State state = State.INIT;
        private final Compression.Algorithm compressAlgo;
        private final HFileDataBlockEncoder dataBlockEncoder;
        private ByteArrayOutputStream baosInMemory;
        private Compressor compressor;
        private CompressionOutputStream compressionStream;
        private ByteArrayOutputStream compressedByteStream;
        private BlockType blockType;
        private DataOutputStream userDataStream;
        private byte[] onDiskBytesWithHeader;
        private byte[] uncompressedBytesWithHeader;
        private long startOffset;
        private long[] prevOffsetByType;
        private long prevOffset;
        private boolean includesMemstoreTS;

        public Writer(Compression.Algorithm compressionAlgorithm, HFileDataBlockEncoder dataBlockEncoder, boolean includesMemstoreTS) {
            this.compressAlgo = compressionAlgorithm == null ? Compression.Algorithm.NONE : compressionAlgorithm;
            this.dataBlockEncoder = dataBlockEncoder != null ? dataBlockEncoder : NoOpDataBlockEncoder.INSTANCE;
            this.baosInMemory = new ByteArrayOutputStream();
            if (this.compressAlgo != Compression.Algorithm.NONE) {
                this.compressor = compressionAlgorithm.getCompressor();
                this.compressedByteStream = new ByteArrayOutputStream();
                try {
                    this.compressionStream = compressionAlgorithm.createPlainCompressionStream((OutputStream)this.compressedByteStream, this.compressor);
                }
                catch (IOException e) {
                    throw new RuntimeException("Could not create compression stream for algorithm " + compressionAlgorithm, e);
                }
            }
            this.prevOffsetByType = new long[BlockType.values().length];
            for (int i = 0; i < this.prevOffsetByType.length; ++i) {
                this.prevOffsetByType[i] = -1L;
            }
            this.includesMemstoreTS = includesMemstoreTS;
        }

        public DataOutputStream startWriting(BlockType newBlockType) throws IOException {
            if (this.state == State.BLOCK_READY && this.startOffset != -1L) {
                this.prevOffsetByType[this.blockType.getId()] = this.startOffset;
            }
            this.startOffset = -1L;
            this.blockType = newBlockType;
            this.baosInMemory.reset();
            this.baosInMemory.write(DUMMY_HEADER);
            this.state = State.WRITING;
            this.userDataStream = new DataOutputStream(this.baosInMemory);
            return this.userDataStream;
        }

        DataOutputStream getUserDataStream() {
            this.expectState(State.WRITING);
            return this.userDataStream;
        }

        private void ensureBlockReady() throws IOException {
            Preconditions.checkState((this.state != State.INIT ? 1 : 0) != 0, (Object)("Unexpected state: " + (Object)((Object)this.state)));
            if (this.state == State.BLOCK_READY) {
                return;
            }
            this.finishBlock();
        }

        private void finishBlock() throws IOException {
            this.userDataStream.flush();
            this.uncompressedBytesWithHeader = this.baosInMemory.toByteArray();
            LOG.warn((Object)("Writer.finishBlock user data size with header before compression " + this.uncompressedBytesWithHeader.length));
            this.prevOffset = this.prevOffsetByType[this.blockType.getId()];
            this.state = State.BLOCK_READY;
            this.encodeDataBlockForDisk();
            this.doCompression();
            this.putHeader(this.uncompressedBytesWithHeader, 0, this.onDiskBytesWithHeader.length, this.uncompressedBytesWithHeader.length);
        }

        private void doCompression() throws IOException {
            if (this.compressAlgo != Compression.Algorithm.NONE) {
                this.compressedByteStream.reset();
                this.compressedByteStream.write(DUMMY_HEADER);
                this.compressionStream.resetState();
                this.compressionStream.write(this.uncompressedBytesWithHeader, 24, this.uncompressedBytesWithHeader.length - 24);
                this.compressionStream.flush();
                this.compressionStream.finish();
                this.onDiskBytesWithHeader = this.compressedByteStream.toByteArray();
                this.putHeader(this.onDiskBytesWithHeader, 0, this.onDiskBytesWithHeader.length, this.uncompressedBytesWithHeader.length);
            } else {
                this.onDiskBytesWithHeader = this.uncompressedBytesWithHeader;
            }
        }

        private void encodeDataBlockForDisk() throws IOException {
            if (this.blockType != BlockType.DATA) {
                return;
            }
            ByteBuffer rawKeyValues = ByteBuffer.wrap(this.uncompressedBytesWithHeader, 24, this.uncompressedBytesWithHeader.length - 24).slice();
            Pair encodingResult = this.dataBlockEncoder.beforeWriteToDisk(rawKeyValues, this.includesMemstoreTS, DUMMY_HEADER);
            BlockType encodedBlockType = (BlockType)encodingResult.getSecond();
            if (encodedBlockType == BlockType.ENCODED_DATA) {
                this.uncompressedBytesWithHeader = ((ByteBuffer)encodingResult.getFirst()).array();
                this.blockType = BlockType.ENCODED_DATA;
            } else {
                if (encodedBlockType != BlockType.DATA) {
                    throw new IOException("Unexpected block type coming out of data block encoder: " + encodedBlockType);
                }
                if (this.userDataStream.size() != this.uncompressedBytesWithHeader.length - 24) {
                    throw new IOException("Uncompressed size mismatch: " + this.userDataStream.size() + " vs. " + (this.uncompressedBytesWithHeader.length - 24));
                }
            }
        }

        private void putHeader(byte[] dest, int offset, int onDiskSize, int uncompressedSize) {
            offset = this.blockType.put(dest, offset);
            offset = Bytes.putInt((byte[])dest, (int)offset, (int)(onDiskSize - 24));
            offset = Bytes.putInt((byte[])dest, (int)offset, (int)(uncompressedSize - 24));
            Bytes.putLong((byte[])dest, (int)offset, (long)this.prevOffset);
        }

        public void writeHeaderAndData(FSDataOutputStream out) throws IOException {
            long offset = out.getPos();
            if (this.startOffset != -1L && offset != this.startOffset) {
                throw new IOException("A " + this.blockType + " block written to a " + "stream twice, first at offset " + this.startOffset + ", then at " + offset);
            }
            this.startOffset = offset;
            this.writeHeaderAndData((DataOutputStream)out);
        }

        private void writeHeaderAndData(DataOutputStream out) throws IOException {
            this.ensureBlockReady();
            out.write(this.onDiskBytesWithHeader);
        }

        public byte[] getHeaderAndData() throws IOException {
            this.ensureBlockReady();
            return this.onDiskBytesWithHeader;
        }

        public void releaseCompressor() {
            if (this.compressor != null) {
                this.compressAlgo.returnCompressor(this.compressor);
                this.compressor = null;
            }
        }

        public int getOnDiskSizeWithoutHeader() {
            this.expectState(State.BLOCK_READY);
            return this.onDiskBytesWithHeader.length - 24;
        }

        public int getOnDiskSizeWithHeader() {
            this.expectState(State.BLOCK_READY);
            return this.onDiskBytesWithHeader.length;
        }

        public int getUncompressedSizeWithoutHeader() {
            this.expectState(State.BLOCK_READY);
            return this.uncompressedBytesWithHeader.length - 24;
        }

        public int getUncompressedSizeWithHeader() {
            this.expectState(State.BLOCK_READY);
            return this.uncompressedBytesWithHeader.length;
        }

        public boolean isWriting() {
            return this.state == State.WRITING;
        }

        public int blockSizeWritten() {
            if (this.state != State.WRITING) {
                return 0;
            }
            return this.userDataStream.size();
        }

        private byte[] getUncompressedDataWithHeader() {
            this.expectState(State.BLOCK_READY);
            return this.uncompressedBytesWithHeader;
        }

        private void expectState(State expectedState) {
            if (this.state != expectedState) {
                throw new IllegalStateException("Expected state: " + (Object)((Object)expectedState) + ", actual state: " + (Object)((Object)this.state));
            }
        }

        public ByteBuffer getUncompressedBufferWithHeader() {
            byte[] b = this.getUncompressedDataWithHeader();
            return ByteBuffer.wrap(b, 0, b.length);
        }

        public void writeBlock(HFileBlock.BlockWritable bw, FSDataOutputStream out) throws IOException {
            bw.writeToBlock((DataOutput)this.startWriting(bw.getBlockType()));
            this.writeHeaderAndData(out);
        }

        public HFileBlock getBlockForCaching() {
            return new HFileBlock(this.blockType, this.getOnDiskSizeWithoutHeader(), this.getUncompressedSizeWithoutHeader(), this.prevOffset, this.getUncompressedBufferWithHeader(), false, this.startOffset, this.includesMemstoreTS, MINOR_VERSION, 0, ChecksumType.NULL.getCode(), this.getOnDiskSizeWithoutHeader());
        }

        private static enum State {
            INIT,
            WRITING,
            BLOCK_READY;

        }
    }
}

