'-------------------------------------------------------------------------------
'--                                                                           --
'--  FILE          :  MenuManagerSingleton.vb                                 --
'--                                                                           --
'--  Author(s)     :  Yin Xuebin (Three Swordsmen Team)                       --
'--                                                                           --
'--  NameSpace     :  WomenWC.Managers                                        --
'--                                                                           --
'--  Project       :  WomenWC                                                 --
'--                                                                           --
'--  Solution      :  Women Working with COBOL                                --
'--                                                                           --
'--  Creation Date :  2007/04/10                                              --
'-------------------------------------------------------------------------------
'--  Modifications :                                                          --
'--                                                                           --
'--                                                                           --
'--                                                                           --
'-------------------------------------------------------------------------------
'-- Copyright(C) 2007, Three Swordsmen, Beijing China                         --
'--                                                                           --
'-- This software is released under the GNU General Public License            --
'-------------------------------------------------------------------------------

Imports WomenWC.Menu
Imports WomenWC.Forms
Imports WomenWCCore.Interfaces.Menu
Imports WomenWCCore.Infos.Menu
Imports Common

Namespace Managers
    Public Class MenuManagerSingleton
        Private _menuItemProcessList As Dictionary(Of ToolStripMenuItem, IMenuItemProcess)
        Private _editorContextMenu As ContextMenuStrip
        Private _projectNodeContextMenu As ContextMenuStrip
        Private _programNodeContextMenu As ContextMenuStrip
        Private Shared _menuManager As MenuManagerSingleton

        Private Sub New()            
            _menuItemProcessList = New Dictionary(Of ToolStripMenuItem, IMenuItemProcess)

            _editorContextMenu = New ContextMenuStrip
            AddHandler _editorContextMenu.Opening, AddressOf OnContextMenuOpening

            _projectNodeContextMenu = New ContextMenuStrip
            AddHandler _projectNodeContextMenu.Opening, AddressOf OnContextMenuOpening

            _programNodeContextMenu = New ContextMenuStrip
            AddHandler _programNodeContextMenu.Opening, AddressOf OnContextMenuOpening
        End Sub

        Public Shared ReadOnly Property MenuManager() As MenuManagerSingleton
            Get
                If _menuManager Is Nothing Then
                    _menuManager = New MenuManagerSingleton
                End If
                Return _menuManager
            End Get
        End Property

        Public ReadOnly Property EditorContextMenu() As ContextMenuStrip
            Get
                Return _editorContextMenu
            End Get
        End Property

        Public ReadOnly Property ProjectNodeContextMenu() As ContextMenuStrip
            Get
                Return _projectNodeContextMenu
            End Get
        End Property

        Public ReadOnly Property ProgramNodeContextMenu() As ContextMenuStrip
            Get
                Return _programNodeContextMenu
            End Get
        End Property

        Public Sub Initialize(ByVal womenWCMainForm As WomenWCMainForm)
            CreateProcessList(womenWCMainForm)
            CreateEditorContextMenu(womenWCMainForm)
            CreateProjectNodeContextMenu(womenWCMainForm)
            CreateProgramNodeContextMenu(womenWCMainForm)
        End Sub

        Public Sub AddMenuItemProcess(ByVal menuItem As ToolStripMenuItem, ByVal menuItemProcess As IMenuItemProcess)
            If Not _menuItemProcessList.ContainsKey(menuItem) Then
                _menuItemProcessList.Add(menuItem, menuItemProcess)
                AddHandler menuItem.Click, AddressOf OnMenuItemClick
                AddHandler menuItem.DropDownOpening, AddressOf OnMenuItemDropDownOpening
                AddHandler menuItem.DropDownClosed, AddressOf OnMenuItemDropDownClosed
                AddHandler menuItem.MouseEnter, AddressOf OnMenuItemMouseEnter
            Else
            End If
        End Sub

        Public Sub DelMenuItemProcess(ByVal menuItem As ToolStripMenuItem)
            If _menuItemProcessList.ContainsKey(menuItem) Then
                _menuItemProcessList.Remove(menuItem)
                RemoveHandler menuItem.Click, AddressOf OnMenuItemClick
                RemoveHandler menuItem.DropDownOpening, AddressOf OnMenuItemDropDownOpening
                RemoveHandler menuItem.DropDownClosed, AddressOf OnMenuItemDropDownClosed
                RemoveHandler menuItem.MouseEnter, AddressOf OnMenuItemMouseEnter
            Else
            End If
        End Sub

        Public Sub AddContextMenuItemProcess(ByVal mainMenuItem As ToolStripMenuItem, ByVal contextMenu As ContextMenuStrip)
            Dim contextMenuItem As ToolStripMenuItem

            If _menuItemProcessList.ContainsKey(mainMenuItem) Then
                contextMenuItem = New ToolStripMenuItem(mainMenuItem.Text, mainMenuItem.Image)
                contextMenu.Items.Add(contextMenuItem)
                AddMenuItemProcess(contextMenuItem, _menuItemProcessList(mainMenuItem))
            Else
            End If
        End Sub

        Private Sub CreateProcessList(ByVal womenWCMainForm As WomenWCMainForm)

            AddMenuItemProcess(womenWCMainForm._menuFile, Nothing)
            AddMenuItemProcess(womenWCMainForm._menuNewProject, New File.OnNewProject(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuOpenProject, New File.OnOpenProject(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuOpenFile, New File.OnOpenFile(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuCloseProject, New File.OnCloseProject(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuCloseFile, New File.OnCloseFile(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuSave, New File.OnSave(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuSaveAs, New File.OnSaveAs(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuSaveAll, New File.OnSaveAll(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuPageSetup, New File.OnPageSetup(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuPrintPreview, New File.OnPrintPreview(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuPrint, New File.OnPrint(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuRecentProjects, New File.OnRecentProjects(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuRecentFiles, New File.OnRecentFiles(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuExit, New File.OnExit(womenWCMainForm))

            AddMenuItemProcess(womenWCMainForm._menuEdit, Nothing)
            AddMenuItemProcess(womenWCMainForm._menuUndo, New Edit.OnUndo(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuRedo, New Edit.OnRedo(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuCut, New Edit.OnCut(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuCopy, New Edit.OnCopy(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuPaste, New Edit.OnPaste(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuCommentOut, New Edit.OnCommentOut(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuCommentCancel, New Edit.OnCommentCancel(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuSelectAll, New Edit.OnSelectAll(womenWCMainForm))

            AddMenuItemProcess(womenWCMainForm._menuSearch, Nothing)
            AddMenuItemProcess(womenWCMainForm._menuFind, New Search.OnFind(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuClearBookMark, New Search.OnClearBookMark(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuNextBookMark, New Search.OnNextBookMark(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuFindNext, New Search.OnNextFind(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuPrevBookMark, New Search.OnPrevBookMark(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuFindPrev, New Search.OnPrevFind(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuReplace, New Search.OnReplace(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuSetBookMark, New Search.OnSetBookMark(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuJump, New Search.OnJump(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuGoToDefine, New Search.OnGoToDefine(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuUnGoTo, New Search.OnUnGoTo(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuReGoTo, New Search.OnReGoTo(womenWCMainForm))

            AddMenuItemProcess(womenWCMainForm._menuProject, Nothing)
            AddMenuItemProcess(womenWCMainForm._menuAddNewFile, New Project.OnAddNewFile(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuAddExistingFile, New Project.OnAddExistingFile(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuDeleteFile, New Project.OnDeleteFile(womenWCMainForm))

            AddMenuItemProcess(womenWCMainForm._menuSetting, Nothing)
            AddMenuItemProcess(womenWCMainForm._menuOption, New Setting.OnOption(womenWCMainForm))

            AddMenuItemProcess(womenWCMainForm._menuWindow, Nothing)
            AddMenuItemProcess(womenWCMainForm._menuNextWindow, New Window.OnNextWindow(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuPrevWindow, New Window.OnPrevWindow(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuWindowList, New Window.OnWindowList(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuVerticalWindow, New Window.OnVerticalWindow(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuHorizontalWindow, New Window.OnHorizontalWindow(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuMinimizeWindow, New Window.OnMinimizeWindow(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuRevertWindow, New Window.OnRevertWindow(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuCloseWindow, New Window.OnCloseWindow(womenWCMainForm))

            AddMenuItemProcess(womenWCMainForm._menuHelp, Nothing)
            AddMenuItemProcess(womenWCMainForm._menuTechnology, New Help.OnTechnology(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuHistory, New Help.OnHistory(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuGPL, New Help.OnGPL(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuGPLNative, New Help.OnGPLNative(womenWCMainForm))
            AddMenuItemProcess(womenWCMainForm._menuAbout, New Help.OnAbout(womenWCMainForm))

        End Sub

        Private Sub CreateEditorContextMenu(ByVal womenWCMainForm As WomenWCMainForm)

            EditorContextMenu.Items.Clear()

            AddContextMenuItemProcess(womenWCMainForm._menuCut, EditorContextMenu)
            AddContextMenuItemProcess(womenWCMainForm._menuCopy, EditorContextMenu)
            AddContextMenuItemProcess(womenWCMainForm._menuPaste, EditorContextMenu)

            EditorContextMenu.Items.Add(New ToolStripSeparator())
            AddContextMenuItemProcess(womenWCMainForm._menuGoToDefine, EditorContextMenu)

            EditorContextMenu.Items.Add(New ToolStripSeparator())
            AddContextMenuItemProcess(womenWCMainForm._menuCommentOut, EditorContextMenu)
            AddContextMenuItemProcess(womenWCMainForm._menuCommentCancel, EditorContextMenu)

        End Sub

        Private Sub CreateProjectNodeContextMenu(ByVal womenWCMainForm As WomenWCMainForm)
            ProjectNodeContextMenu.Items.Clear()

            AddContextMenuItemProcess(womenWCMainForm._menuAddNewFile, ProjectNodeContextMenu)
            AddContextMenuItemProcess(womenWCMainForm._menuAddExistingFile, ProjectNodeContextMenu)

        End Sub

        Private Sub CreateProgramNodeContextMenu(ByVal womenWCMainForm As WomenWCMainForm)
            ProgramNodeContextMenu.Items.Clear()

            AddContextMenuItemProcess(womenWCMainForm._menuDeleteFile, ProgramNodeContextMenu)

        End Sub

        Private Sub OnMenuItemClick(ByVal sender As Object, ByVal e As System.EventArgs)
            If _menuItemProcessList.ContainsKey(sender) AndAlso _menuItemProcessList(sender) IsNot Nothing Then
                If _menuItemProcessList(sender).IsEnable Then
                    Try
                        _menuItemProcessList(sender).Execute()
                    Catch myex As MyException
                        Common.Message.ShowMessage(myex)
                    Catch ex As Exception
                        Common.Message.ShowMessage(ex)
                    End Try
                Else
                End If
            End If
        End Sub

        Private Sub OnMenuItemDropDownOpening(ByVal sender As Object, ByVal e As System.EventArgs)
            Dim menuItem As ToolStripMenuItem
            menuItem = DirectCast(sender, ToolStripMenuItem)
            For Each submenuItem As ToolStripItem In menuItem.DropDownItems
                If TypeOf submenuItem Is ToolStripMenuItem AndAlso _
                   _menuItemProcessList.ContainsKey(submenuItem) AndAlso _
                   _menuItemProcessList(submenuItem) IsNot Nothing Then

                    submenuItem.Enabled = _menuItemProcessList(submenuItem).IsEnable
                    If _menuItemProcessList(submenuItem).HasDynamicMenuItem Then
                        DelSubMenuItem(submenuItem)
                        AddSubMenuItem(submenuItem)
                    Else
                    End If

                Else
                End If
            Next
        End Sub

        Private Sub OnMenuItemDropDownClosed(ByVal sender As Object, ByVal e As System.EventArgs)
            Dim menuItem As ToolStripMenuItem
            menuItem = DirectCast(sender, ToolStripMenuItem)
            For Each submenuItem As ToolStripItem In menuItem.DropDownItems
                If TypeOf submenuItem Is ToolStripMenuItem AndAlso _
                   _menuItemProcessList.ContainsKey(submenuItem) AndAlso _
                   _menuItemProcessList(submenuItem) IsNot Nothing Then

                    submenuItem.Enabled = True
                End If
            Next
        End Sub

        Private Sub OnMenuItemMouseEnter(ByVal sender As Object, ByVal e As System.EventArgs)
            Dim menuItem As ToolStripMenuItem
            menuItem = DirectCast(sender, ToolStripMenuItem)
            If _menuItemProcessList.ContainsKey(menuItem) AndAlso _menuItemProcessList(menuItem) IsNot Nothing Then
                WomenWCMainForm._statusBarComment.Text = _menuItemProcessList(menuItem).Comment
            Else
                WomenWCMainForm._statusBarComment.Text = String.Empty
            End If
        End Sub

        Private Sub AddSubMenuItem(ByVal menuItem As ToolStripMenuItem)
            For Each dynamicMenuItemInfo As DynamicMenuItemInfo In _menuItemProcessList(menuItem).DynamicMenuItemInfos
                menuItem.DropDownItems.Add(dynamicMenuItemInfo.ToolStripMenuItem)
                AddMenuItemProcess(dynamicMenuItemInfo.ToolStripMenuItem, dynamicMenuItemInfo.MenuItemProcess)
            Next
        End Sub

        Private Sub DelSubMenuItem(ByVal menuItem As ToolStripMenuItem)
            For Each subMenuItem As ToolStripMenuItem In menuItem.DropDownItems
                DelMenuItemProcess(subMenuItem)
            Next
            menuItem.DropDownItems.Clear()
        End Sub

        Private Sub OnContextMenuOpening(ByVal sender As Object, ByVal e As System.ComponentModel.CancelEventArgs)
            Dim contextMenu As ContextMenuStrip
            contextMenu = DirectCast(sender, ContextMenuStrip)

            For Each menuItem As ToolStripItem In contextMenu.Items
                If TypeOf menuItem Is ToolStripMenuItem AndAlso _
                   _menuItemProcessList.ContainsKey(menuItem) AndAlso _
                   _menuItemProcessList(menuItem) IsNot Nothing Then

                    menuItem.Enabled = _menuItemProcessList(menuItem).IsEnable
                Else
                End If
            Next
        End Sub

    End Class
End Namespace
