/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.config;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

/**
 * {@link Map}をキャッシュする{@link ConfigExpression}。
 * 委譲先で生成される{@link Map}が{@link #getCacheKey()}で返されるキーに対して一意であることを想定しているため並行性制御を行っていない。
 * @author nakamura
 *
 */
public class CacheMap implements ConfigExpression {
	private static final Object DEFAULT_KEY = new Object();
	
	private final ConfigExpression mapMaker;
	private final Map<Object,Map<String,Object>> cacheMap;

	/**
	 * コンストラクタ。
	 * @param maxEntries キャッシュに保持する最大{@link Map}数。
	 * @param mapMaker {@link Map}を生成し加工する委譲先の{@link ConfigExpression}。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public CacheMap(final int maxEntries, final ConfigExpression mapMaker){
		mapMaker.getClass();
		this.mapMaker = mapMaker;
		
		if(maxEntries > 0){
			cacheMap = new LinkedHashMap<Object,Map<String,Object>>(){
				/**
				 * 
				 */
				private static final long serialVersionUID = 1L;

				@Override protected boolean removeEldestEntry(Entry eldest) {
					return size() > maxEntries;
				}
			};
		}else{
			cacheMap = null;
		}
	}

	/**
	 * @throws NullPointerExcpeption 引数が null の場合。
	 */
	public void interpret(final ConfigContext context) {
		if(cacheMap == null){
			mapMaker.interpret(context);
		}else{
			final Object key = getCacheKey();
			Map<String,Object> map = cacheMap.get(key);
			if(map == null){
				mapMaker.interpret(context);
				map = context.getMap();
				cacheMap.put(key, map);
			}
			context.setMap(map);
		}
	}
	
	/**
	 * 委譲先で生成する{@link Map}を一意に識別するためのキーを取得するために呼び出される(called)。
	 * デフォルトは常に同一のインスタンスを返す。
	 * {@link java.util.Locale}や端末ごとに{@link Map}をキャッシュする際にオーバライドする。
	 * オーバライドする際は{@link PutResourceBundle#getResourceBundle(String)}と整合性を保つ必要がある。
	 * @return cache 委譲先で生成する{@link Map}を一意に識別するためのキー。
	 */
	public Object getCacheKey(){
		return DEFAULT_KEY;
	}
}
