/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;


import java.util.Collection;


import org.w3c.dom.Element;

import woolpack.el.GettingEL;
import woolpack.utils.UtilsConstants;

/**
 * DOM ノードの属性値が値の一覧に存在することを検証して委譲先を分岐する{@link DomExpression}。
 * @author nakamura
 *
 */
public class EqualsAttrValue implements DomExpression {
	private final String attrName;
	private final GettingEL valueEL;
	private final DomExpression trueExpression;
	private final DomExpression falseExpression;

	/**
	 * コンストラクタ。
	 * @param attrName 属性名。
	 * @param valueEL 値の一覧への参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @param trueExpression 属性値と等しい値が値の一覧に存在した場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException attrName が空の場合。
	 */
	public EqualsAttrValue(
			final String attrName, 
			final GettingEL valueEL, 
			final DomExpression trueExpression){
		this(attrName, valueEL, trueExpression, DomConstants.NULL);
	}
	
	/**
	 * コンストラクタ。
	 * @param attrName 属性名。
	 * @param valueEL 値の一覧への参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @param trueExpression 属性値と等しい値が値の一覧に存在した場合の委譲先。
	 * @param falseExpression 属性値と等しい値が値の一覧に存在しない場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException attrName が空の場合。
	 */
	public EqualsAttrValue(
			final String attrName, 
			final GettingEL valueEL, 
			final DomExpression trueExpression, 
			final DomExpression falseExpression){
		attrName.charAt(0);
		valueEL.getClass();
		trueExpression.getClass();
		falseExpression.getClass();
		
		this.attrName = attrName;
		this.valueEL = valueEL;
		this.trueExpression = trueExpression;
		this.falseExpression = falseExpression;
	}

	/**
	 * @throws NullPointerException 引数または{@link DomContext#getNode()}が null の場合。
	 * @throws ClassCastException {@link DomContext#getNode()}が{@link Element}でない場合。
	 */
	public void interpret(final DomContext context) {
		final Object value = valueEL.getValue(context);
		final Collection collection = UtilsConstants.toCollection(value);
		final String elementValue = ((Element)context.getNode()).getAttribute(attrName);
		
		boolean containsFlag = false;
		for(final Object o:collection){
			// 属性値は文字列なので文字列に変換して比較する
			if(o != null && o.toString().equals(elementValue)){
				containsFlag = true;
				break;
			}
		}
		
		if(containsFlag){
			trueExpression.interpret(context);
		}else{
			falseExpression.interpret(context);
		}
	}
}
