/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.HashMap;
import java.util.Map;

import woolpack.el.GettingEL;


/**
 * {@link DomContext}と固定のオブジェクトの組に対して{@link GettingEL#getValue(Object)}を実行し、
 * 返却値により委譲先を分岐する{@link DomExpression}。
 * 
 * @author nakamura
 *
 */
public class Exec implements DomExpression {
	/**
	 * {@link DomContext}にアクセスするためのキー。
	 * {@link woolpack.el.PathEL}の "context" などにより{@link DomContext}にアクセスする。
	 */
	public static final String CONTEXT = "context";
	
	/**
	 * 固定のオブジェクトにアクセスするためのキー。
	 * {@link woolpack.el.PathEL}の "resource" などにより固定のオブジェクトにアクセスする。
	 */
	public static final String RESOURCE = "resource";
	
	private final Object resource;
	private final GettingEL el;
	private final DomExpression trueExpression;
	private final DomExpression falseExpression;

	/**
	 * コンストラクタ。
	 * @param resource 固定のオブジェクト。
	 * @param el 作用への参照。
	 * @param trueExpression 作用結果が{@link Boolean#TRUE}の場合の委譲先。
	 * @param falseExpression 作用結果が{@link Boolean#FALSE}の場合の委譲先。
	 * @throws NullPointerException el,trueExpression,falseExpression のいずれかが null の場合。
	 */
	public Exec(
			final Object resource,
			final GettingEL el, 
			final DomExpression trueExpression, 
			final DomExpression falseExpression){
		el.getClass();
		trueExpression.getClass();
		falseExpression.getClass();
		
		this.resource = resource;
		this.el = el;
		this.trueExpression = trueExpression;
		this.falseExpression = falseExpression;
	}
	/**
	 * コンストラクタ。
	 * @param el 作用への参照。
	 * @param trueExpression 作用結果が{@link Boolean#TRUE}の場合の委譲先。
	 * @param falseExpression 作用結果が{@link Boolean#FALSE}の場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public Exec(final GettingEL el, final DomExpression trueExpression, final DomExpression falseExpression){
		this(null, el, trueExpression, falseExpression);
	}

	/**
	 * コンストラクタ。
	 * 作用結果が{@link Boolean#FALSE}の場合は委譲しない。
	 * @param el 作用への参照。
	 * @param trueExpression 作用結果が{@link Boolean#TRUE}の場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public Exec(final GettingEL el, final DomExpression trueExpression){
		this(el, trueExpression, DomConstants.NULL);
	}
	
	/**
	 * コンストラクタ。
	 * 作用結果にかかわらず委譲しない。
	 * @param el 作用への参照。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public Exec(final GettingEL el){
		this(el, DomConstants.NULL);
	}

	public void interpret(final DomContext context) {
		final Map<String,Object> map = new HashMap<String,Object>(2);
		map.put(RESOURCE, resource);
		map.put(CONTEXT, context);
		
		final Object value = el.getValue(map);
		if(Boolean.TRUE.equals(value)){
			trueExpression.interpret(context);
		}else if(Boolean.FALSE.equals(value)){
			falseExpression.interpret(context);
		}
	}
}
