/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.regex.Pattern;

/**
 * {@link DomContext#getId()}を正規表現で照合して委譲先を分岐する{@link DomExpression}。
 * @author nakamura
 *
 */
public class RegExpId implements DomExpression {
	private final Pattern pattern;
	private final DomExpression trueExpression;
	private final DomExpression falseExpression;
	
	/**
	 * コンストラクタ。
	 * @param pattern 正規表現。
	 * @param trueExpression {@link DomContext#getId()}が正規表現にマッチする場合の委譲先。
	 * @param falseExpression {@link DomContext#getId()}が正規表現にマッチしない場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public RegExpId(
			final Pattern pattern, 
			final DomExpression trueExpression, 
			final DomExpression falseExpression){
		pattern.getClass();
		trueExpression.getClass();
		falseExpression.getClass();
		
		this.pattern = pattern;
		this.trueExpression = trueExpression;
		this.falseExpression = falseExpression;
	}

	/**
	 * コンストラクタ。
	 * @param pattern 正規表現。
	 * @param trueExpression {@link DomContext#getId()}が正規表現にマッチする場合の委譲先。
	 * @param falseExpression {@link DomContext#getId()}が正規表現にマッチしない場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws PatternSyntaxException 表現の構文が無効である場合。
	 */
	public RegExpId(
			final String pattern, 
			final DomExpression trueExpression, 
			final DomExpression falseExpression){
		this(Pattern.compile(pattern), trueExpression, falseExpression);
	}
	
	
	/**
	 * コンストラクタ。
	 * @param pattern 正規表現。
	 * @param trueExpression {@link DomContext#getId()}が正規表現にマッチする場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public RegExpId(
			final Pattern pattern, 
			final DomExpression trueExpression){
		this(pattern, trueExpression, DomConstants.NULL);
	}
	
	/**
	 * コンストラクタ。
	 * @param pattern 正規表現。
	 * @param trueExpression {@link DomContext#getId()}が正規表現にマッチする場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws PatternSyntaxException 表現の構文が無効である場合。
	 */
	public RegExpId(
			final String pattern, 
			final DomExpression trueExpression){
		this(Pattern.compile(pattern), trueExpression);
	}

	/**
	 * @throws NullPointerException 引数または{@link DomContext#getId()}が null の場合。
	 */
	public void interpret(final DomContext context) {
		if(pattern.matcher(context.getId()).matches()){
			trueExpression.interpret(context);
		}else{
			falseExpression.interpret(context);
		}
	}
}
