/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.concurrent.Semaphore;

import woolpack.utils.Switchable;

/**
 * id 毎にセマフォ(同時実行スレッド数)を制御する{@link DomExpression}。
 * セマフォが設定されていない id に対するリクエストはパーミットの取得に成功した場合の委譲先に委譲する。
 * 同時実行数を越えている場合のリクエストはスレッドをブロックせずにエラーとする。
 * @author nakamura
 *
 */
public class TrySemaphore implements DomExpression {
	private final Switchable<String,Semaphore> map;
	private final DomExpression trueExpression;
	private final DomExpression falseExpression;
	
	/**
	 * コンストラクタ。
	 * @param map id とセマフォの{@link Switchable}。
	 * @param trueExpression パーミットの取得に成功した場合の委譲先。
	 * @param falseExpression パーミットの取得に失敗した場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public TrySemaphore(final Switchable<String,Semaphore> map, final DomExpression trueExpression, final DomExpression falseExpression){
		map.getClass();
		trueExpression.getClass();
		falseExpression.getClass();
		
		this.map = map;
		this.trueExpression = trueExpression;
		this.falseExpression = falseExpression;
	}

	public void interpret(final DomContext context) {
		final Semaphore semaphore = map.get(context.getId());
		if(semaphore != null){
			if(semaphore.tryAcquire()){
				try{
					trueExpression.interpret(context);
				}finally{
					semaphore.release();
				}
			}else{
				falseExpression.interpret(context);
			}
		}else{
			trueExpression.interpret(context);
		}
	}
}
