/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el;

import java.util.HashMap;
import java.util.Map;

/**
 * {@link Map}または get/set メソッドのみをサポートする{@link EL}。
 * @author nakamura
 *
 */
public class PropertyEL extends ELAdapter {
	private static final Map<Class,Map<String,GettingEL>> getterMap = new HashMap<Class,Map<String,GettingEL>>();
	private static final Map<Class,Map<String,EL>> setterMap = new HashMap<Class,Map<String,EL>>();
	
	private final String propertyName;
	private final TypeConverter converter;
	private final PropertyELFactory factory;
	
	private final EL mapEL;
	
	/**
	 * コンストラクタ。
	 * @param propertyName プロパティ名。
	 * @param converter 型変換器。
	 * @param factory プロパティにアクセスする{@link GettingEL}と{@link EL}のファクトリ。
	 * @throws NullPointerException 引数が null の場合。
	 * @throws StringIndexOutOfBoundsException 引数のいずれかが空の場合。
	 */
	public PropertyEL(final String propertyName, final TypeConverter converter, final PropertyELFactory factory){
		propertyName.charAt(0);
		converter.getClass();
		factory.getClass();
		
		this.propertyName = propertyName;
		this.converter = converter;
		this.factory = factory;
		mapEL = new MapEL(propertyName);
	}

	/**
	 * コンストラクタ。
	 * @param propertyName プロパティ名。
	 * @param converter 型変換器。
	 * @throws NullPointerException 引数が null の場合。
	 * @throws StringIndexOutOfBoundsException propertyName が空の場合。
	 */
	public PropertyEL(final String propertyName, final TypeConverter converter){
		this(propertyName, converter, new ReflectionPropertyELFactory(converter));
	}
	
	/**
	 * コンストラクタ。
	 * @param propertyName プロパティ名。
	 * @throws NullPointerException 引数が null の場合。
	 * @throws StringIndexOutOfBoundsException 引数が空の場合。
	 */
	public PropertyEL(final String propertyName){
		this(propertyName, ELConstants.COLLECTION_CONVERTER);
	}

	@Override public Object getValue(final Object root, final Class clazz) throws ELTargetRuntimeException {
		if(root instanceof Map){
			final Object result = mapEL.getValue(root, clazz);
			return converter.convert(propertyName, result, clazz);
		}
		
		GettingEL getter = null;
		{
			final Class rootClass = root.getClass();
			Map<String,GettingEL> map1 = getterMap.get(rootClass);
			if(map1 != null){
				getter = map1.get(propertyName);
			}
			if(getter == null){
				getter = factory.newGetter(rootClass, propertyName);
				if(map1 == null){
					map1 = new HashMap<String,GettingEL>();
					getterMap.put(rootClass, map1);
				}
				map1.put(propertyName, getter);
			}
		}
		return getter.getValue(root, clazz);
	}

	@Override public void setValue(final Object root, final Object value) throws ELTargetRuntimeException {
		if(root instanceof Map){
			// TODO 汎用型を取得する方法を検討する
			mapEL.setValue(root, value);
			return;
		}
		
		EL setter = null;
		{
			final Class rootClass = root.getClass();
			Map<String,EL> map1 = setterMap.get(rootClass);
			if(map1 != null){
				setter = map1.get(propertyName);
			}
			if(setter == null){
				setter = factory.newSetter(rootClass, propertyName);
				if(map1 == null){
					map1 = new HashMap<String,EL>();
					setterMap.put(rootClass, map1);
				}
				map1.put(propertyName, setter);
			}
		}
		setter.setValue(root, value);
	}
	
	@Override public int hashCode(){
		return propertyName.hashCode() + 3*converter.hashCode() + 5*factory.hashCode();
	}
	
	@Override public boolean equals(final Object other){
		if(other instanceof PropertyEL){
			final PropertyEL o = (PropertyEL)other;
			return propertyName.equals(o.propertyName) && converter.equals(o.converter) && factory.equals(o.factory);
		}
		return false;
	}
}
