/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import java.util.Collection;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.Loop;
import woolpack.dom.NodeLoop;
import woolpack.el.ArrayPathEL;
import woolpack.el.EL;
import woolpack.el.GettingEL;
import woolpack.el.MapEL;
import woolpack.el.PropertyEL;


/**
 * DOM エレメントの属性値をプロパティ名として、コンポーネントから値を取得し、 DOM ノードに自動設定する{@link DomExpression}。
 * @author nakamura
 *
 */
public class AutoUpdate extends NodeLoop {
	private static final EL COMPONENT_EL = new ArrayPathEL(DomConstants.LOCAL_EL, new MapEL("autoUpdateComponent"));
	
	private final Iterable<String> attrNames;
	private final GettingEL componentEL;
	private final boolean selectFlag;
	
	AutoUpdate(
			final Iterable<String> attrNames, 
			final GettingEL componentEL, 
			final boolean selectMode){
		attrNames.getClass();
		componentEL.getClass();
		
		this.attrNames = attrNames;
		this.componentEL = componentEL;
		this.selectFlag = selectMode;
	}
	
	/**
	 * コンストラクタ。
	 * @param attrNames 属性名の一覧。
	 * @param componentEL コンポーネントへの参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public AutoUpdate(
			final Iterable<String> attrNames, 
			final GettingEL componentEL){
		this(attrNames, componentEL, true);
	}

	/**
	 * @throws NullPointerException 引数または{@link DomContext#getNode()}が null の場合。
	 * @throws ClassCastException {@link DomContext#getNode()}が{@link Element}でない場合。
	 */
	@Override public void interpret(final DomContext context) {
		if(context.getNode().getNodeType() == Node.ELEMENT_NODE){
			final Element e = (Element)context.getNode();
			final String attrName = getAttrName(e);
			if(attrName != null){
				final String attrValue = e.getAttribute(attrName);
				final GettingEL valueEL = new ArrayPathEL(componentEL, new PropertyEL(attrValue));
				final Object value = getValue(context, valueEL);
				if(value != null){
					if(isBeanCollection(value)){
						new Loop(valueEL, COMPONENT_EL, 
								new AutoUpdate(attrNames, COMPONENT_EL, false)).interpret(context);
					}else{
						// TODO メモリ効率を挙げるためにUpdateValueを再設計する。
						new UpdateValue(
								valueEL, 
								new ArrayPathEL(DomConstants.CONFIG_EL, new MapEL(attrValue)), 
								selectFlag).interpret(context);
					}
					return;
				}
			}
		}
		super.interpret(context);
	}
	
	private static Object getValue(final DomContext context, final GettingEL valueEL){
		try{
			return valueEL.getValue(context);
		}catch(RuntimeException exception){
			// TODO 無視以外の方法がないか考察する。
			return null;
		}
	}
	
	private String getAttrName(final Element e){
		for(final String attrName:attrNames){
			if(e.hasAttribute(attrName)){
				return attrName;
			}
		}
		return null;
	}
	
	private static boolean isBeanCollection(final Object value){
		if(value instanceof Collection){
			final Collection c = (Collection)value;
			if(c.size() == 0){
				return true;
			}
			return !isAtom(c.iterator().next().getClass());
		}
		if(value.getClass().isArray()){
			return !isAtom(value.getClass().getComponentType());
		}
		return false;
	}
	
	private static boolean isAtom(final Class clazz){
		// TODO 原子クラス抽出ロジックを再検討する。(日付型も)
		return clazz.isPrimitive() || clazz.getPackage().getName().equals("java.lang");
	}
}
