/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack;

public class TestConcurrent {
	private final String label;
	private final Object lock;
	private final int minConcurrent;
	private final int maxConcurrent;
	private final int threshold;
	private final double failOdds;
	private final int[] concurrentArray;
	private final int[] threadArray;
	private final ThreadLocal<Integer> threadLocal;
	
	private int currentConcurrent;
	private int count;
	private boolean minValidFlag;
	private boolean maxValidFlag;

	public TestConcurrent(
			final String label,
			final Object lock,
			final int minConcurrent, 
			final int maxConcurrent,
			final int threadCount, 
			final int threshold){
		this(label, lock, minConcurrent, maxConcurrent, threadCount, threshold, 0.5);
	}
			
	public TestConcurrent(
			final String label,
			final Object lock,
			final int minConcurrent, 
			final int maxConcurrent, 
			final int threadCount, 
			final int threshold,
			final double failOdds){
		this.label = label;
		this.lock = lock;
		this.minConcurrent = minConcurrent;
		this.maxConcurrent = maxConcurrent;
		this.threshold = threshold;
		this.failOdds = failOdds;
		concurrentArray = new int[maxConcurrent];
		threadArray = new int[threadCount];
		threadLocal = new ThreadLocal<Integer>();
		
		currentConcurrent = 0;
		count = 0;
		minValidFlag = true;
		maxValidFlag = true;
	}
	
	public int getCount(){
		return count;
	}
	
	public void setThreadId(int i){
		threadLocal.set(i);
	}
	
	private void increaseCount(){
		synchronized(lock){
			count++;
		}
	}
	
	private void recodeThread(){
		synchronized(lock){
			threadArray[threadLocal.get()]++;
		}
	}
	
	private void recodeConcurrent(){
		synchronized(lock){
			concurrentArray[currentConcurrent-1]++;
		}
	}
	
	private void increaseConcurrent(){
		synchronized(lock){
			currentConcurrent++;
		}
	}
	
	private void decreaseConcurrent(){
		synchronized(lock){
			currentConcurrent--;
		}
	}
	
	private void check(){
		synchronized(lock){
			minValidFlag &= (minConcurrent <= currentConcurrent);
			maxValidFlag &= (currentConcurrent <= maxConcurrent);
		}
	}
	
	private void failRandom(){
		if(Math.random() < failOdds){
			throw new RuntimeException("random fail.");
		}
	}
	
	public void execute(){
		Thread.yield();
		increaseCount();
		Thread.yield();
		recodeThread();
		Thread.yield();
		increaseConcurrent();
		Thread.yield();
		recodeConcurrent();
		Thread.yield();
		check();
		Thread.yield();
		decreaseConcurrent();
		Thread.yield();
		failRandom();
		Thread.yield();
	}
	
	public void print(){
		System.out.print(label);
		System.out.print(": ");
		
		if(count == 0){
			System.out.print("count:0");
		}else{
			System.out.print("thread");
			print(threadArray);
			System.out.print(", ");
			
			System.out.print("mean:");
			System.out.print(culcMean(threadArray));
			System.out.print(", ");
			
			System.out.print("concurrent");
			print(concurrentArray);
			System.out.print(", ");

			System.out.print("min:");
			System.out.print(minValidFlag);
			System.out.print(", ");
			System.out.print("max:");
			System.out.print(maxValidFlag);
		}

		System.out.println();
	}
	
	public boolean assertValid(){
		return minValidFlag && maxValidFlag && assertThreshold(threadArray);
	}
	
	private void print(final int[] array){
		for(int i=0; i<array.length; i++){
			System.out.print("-");
			System.out.print(array[i]);
		}
	}
	
	private int culcMean(final int[] array){
		if(array.length == 0){
			return 0;
		}
		int mean = 0;
		for(int i=0; i<array.length; i++){
			mean += array[i];
		}
		mean /= array.length;
		return mean;
	}
	
	private boolean assertThreshold(final int[] array){
		final int mean = culcMean(array);
		for(int i=0; i<array.length; i++){
			if(Math.abs(mean - array[i]) > threshold){
				return false;
			}
		}
		return true;
	}
}
