/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import junit.framework.TestCase;
import woolpack.TestUtils;
import woolpack.text.RegExpFormat;
import woolpack.utils.FixSwitch;
import woolpack.utils.MapBuilder;
import woolpack.utils.SwitchBuilder;

public class BranchByNameValidatorTest extends TestCase {

	public void testConstructor(){
		try{
			new BranchByNameValidator(null, new FixSwitch<String,ValidatorExpression>(null));
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new BranchByNameValidator(ValidatorConstants.AND, null);
			fail();
		}catch(final NullPointerException expected){
		}
	}

	public void testNotMatch(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter1 = new CountValidator(ValidatorConstants.TRUE);
		final ValidatorExpression expression = new BranchByNameValidator(ValidatorConstants.AND, 
				new SwitchBuilder<String,ValidatorExpression>()
				.put("a", counter0)
				.put("b", counter1)
				.get());
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(
				MapBuilder.get(new HashMap<String,List<Object>>())
				.put("x", Arrays.asList((Object)"vx"))
				.put("y", Arrays.asList((Object)"vy")).get());
		assertFalse(expression.interpret(context));
		
		assertEquals(1, counter0.getIntepretCount());
		assertEquals("a", counter0.getContextList().getLast().getTmpKey());
		assertEquals(0, counter0.getContextList().getLast().getTmpIndex());
		
		assertEquals(1, counter1.getIntepretCount());
		assertEquals("b", counter1.getContextList().getLast().getTmpKey());
		assertEquals(0, counter1.getContextList().getLast().getTmpIndex());
	}

	public void testMatchFalse(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter1 = new CountValidator(ValidatorConstants.TRUE);
		final ValidatorExpression expression = new BranchByNameValidator(ValidatorConstants.AND, 
				new SwitchBuilder<String,ValidatorExpression>()
				.put("a", counter0)
				.put("b", counter1)
				.get());
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(
				MapBuilder.get(new HashMap<String,List<Object>>())
				.put("a", Arrays.asList((Object)"va"))
				.put("b", Arrays.asList((Object)"vb")).get());
		assertFalse(expression.interpret(context));
		
		assertEquals(1, counter0.getIntepretCount());
		assertEquals("a", counter0.getContextList().getLast().getTmpKey());
		assertEquals(0, counter0.getContextList().getLast().getTmpIndex());
		
		assertEquals(1, counter1.getIntepretCount());
		assertEquals("b", counter1.getContextList().getLast().getTmpKey());
		assertEquals(0, counter1.getContextList().getLast().getTmpIndex());
	}

	public void testMatchTrue(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter1 = new CountValidator(ValidatorConstants.TRUE);
		final ValidatorExpression expression = new BranchByNameValidator(ValidatorConstants.AND, 
				new SwitchBuilder<String,ValidatorExpression>()
				.put("a", counter0)
				.put("b", counter1)
				.get());
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(
				MapBuilder.get(new HashMap<String,List<Object>>())
				.put("x", Arrays.asList((Object)"vx"))
				.put("b", Arrays.asList((Object)"vb")).get());
		assertFalse(expression.interpret(context));
		
		assertEquals(1, counter0.getIntepretCount());
		assertEquals("a", counter0.getContextList().getLast().getTmpKey());
		assertEquals(0, counter0.getContextList().getLast().getTmpIndex());
		
		assertEquals(1, counter1.getIntepretCount());
		assertEquals("b", counter1.getContextList().getLast().getTmpKey());
		assertEquals(0, counter1.getContextList().getLast().getTmpIndex());
	}

	public void testMatchFalse2(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter1 = new CountValidator(ValidatorConstants.TRUE);
		final ValidatorExpression expression = new BranchByNameValidator(ValidatorConstants.AND, 
				new SwitchBuilder<String,ValidatorExpression>()
				.put("a", counter0)
				.put("b", counter1)
				.get());
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(
				MapBuilder.get(new HashMap<String,List<Object>>())
				.put("a", Arrays.asList((Object)"va"))
				.put("b", Arrays.asList((Object)"vb")).get());
		assertFalse(expression.interpret(context));
		
		assertEquals(1, counter0.getIntepretCount());
		assertEquals("a", counter0.getContextList().getLast().getTmpKey());
		assertEquals(0, counter0.getContextList().getLast().getTmpIndex());
		
		assertEquals(1, counter1.getIntepretCount());
		assertEquals("b", counter1.getContextList().getLast().getTmpKey());
		assertEquals(0, counter1.getContextList().getLast().getTmpIndex());
	}

	public void testParse(){
		final ValidatorExpression expression = new BranchByNameValidator(
				new SwitchBuilder<String,ValidatorExpression>()
				.put("a", new ParseValidator(new RegExpFormat("a", "A")))
				.put("b", new ParseValidator(new RegExpFormat("b", "B")))
				.get());
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(
				MapBuilder.get(new HashMap<String,List<Object>>())
				.put("a", Arrays.asList((Object)"va"))
				.put("b", Arrays.asList((Object)"vb")).get());
		assertTrue(expression.interpret(context));
		assertTrue(TestUtils.equals(
				MapBuilder.get(new HashMap<String,Object>())
				.put("a", Arrays.asList("vA"))
				.put("b", Arrays.asList("vB"))
				.get()
				, context.getInputMap()));
	}
	
	public void testAppendTo() throws IOException{
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter1 = new CountValidator(ValidatorConstants.TRUE);
		final ValidatorExpression expression = new BranchByNameValidator(ValidatorConstants.AND, 
				new SwitchBuilder<String,ValidatorExpression>()
				.put("a", counter0)
				.put("b", counter1)
				.get()
		);
		final StringBuilder sb = new StringBuilder();
		expression.appendTo(sb);
		assertEquals("new BranchByNameValidator({\"a\":new FalseValidator(),\"b\":new TrueValidator()})", sb.toString());
		assertEquals(1, counter0.getAppendToCount());
		assertEquals(1, counter1.getAppendToCount());
		assertTrue(sb == counter0.getSbList().getLast());
		assertTrue(sb == counter1.getSbList().getLast());
	}
	
	public void testAddMessageTo(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter1 = new CountValidator(ValidatorConstants.FALSE);
		final ValidatorExpression expression = new BranchByNameValidator(ValidatorConstants.AND, 
				new SwitchBuilder<String,ValidatorExpression>()
				.put("a", counter0)
				.put("b", counter1)
				.get()
		);
		final Collection<String> expectedCollection = new ArrayList<String>();
		expression.addMessageTo(expectedCollection);
		assertEquals(1, counter0.getAddMessageToCount());
		assertEquals(1, counter1.getAddMessageToCount());
		assertTrue(expectedCollection == counter0.getCollectionList().getLast());
		assertTrue(expectedCollection == counter1.getCollectionList().getLast());
	}
}
