/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import woolpack.utils.NodeFindable;

/**
 * XPath でノードを検索し委譲先を分岐する{@link DomExpression}。
 * 委譲先でノードを操作した場合、次の検索結果に影響するかは{@link NodeFindable}の実装に依存する。
 * は次の XPath 検索に反映される(スナップショットではない)。
 * 適用しているパターン：Proxy、フィルタ。
 * @author nakamura
 *
 */
public class XPath implements DomExpression {
	private final NodeFindable findable;
	private final DomExpression firstExpression;
	private final DomExpression pluralExpression;
	
	/**
	 * コンストラクタ。
	 * @param findable XPath。
	 * @param firstExpression 最初の検索結果に対する委譲先。
	 * @param pluralExpression 2番目以降の検索結果に対する委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public XPath(final NodeFindable findable, final DomExpression firstExpression, final DomExpression pluralExpression){
		findable.getClass();
		firstExpression.getClass();
		pluralExpression.getClass();
		
		this.findable = findable;
		this.firstExpression = firstExpression;
		this.pluralExpression = pluralExpression;
	}

	
	/**
	 * コンストラクタ。
	 * @param findable XPath。
	 * @param expression 検索結果に対する委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public XPath(final NodeFindable findable, final DomExpression expression){
		this(findable, expression, expression);
	}

	/**
	 * @throws NullPointerException 引数または{@link DomContext#getNode()}が null の場合。
	 */
	public void interpret(final DomContext context) {
		final NodeList nodeList = findable.evaluateList(context.getNode());
		final Node[] beforeNodeList = new Node[nodeList.getLength()];
		for(int i=0; i<nodeList.getLength(); i++){
			beforeNodeList[i] = nodeList.item(i);
		}
		
		for (int i=0; i < beforeNodeList.length ; i++) {
			final DomContext newContext = context.copy();
			newContext.setNode(beforeNodeList[i]);
			if(i == 0){
				firstExpression.interpret(newContext);
			}else{
				pluralExpression.interpret(newContext);
			}
		}
	}
}
