/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el;

import java.util.Arrays;
import java.util.Collection;

/**
 * {@link GettingEL}の一覧の順に基点をたどっていく{@link EL}。
 * @author nakamura
 *
 */
public class ArrayPathEL extends ELAdapter {
	private final GettingEL[] getting;
	private final EL setting;
	
	/**
	 * コンストラクタ。
	 * @param getting 基点をたどっていくための{@link GettingEL}の一覧。本クラスはこの引数の状態を変化させない。
	 * @param setting 値を設定または取得するための{@link EL}。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public ArrayPathEL(final GettingEL[] getting, final EL setting){
		getting.getClass();
		setting.getClass();
		this.getting = (GettingEL[])getting.clone();
		this.setting = setting;
	}
	
	/**
	 * コンストラクタ。
	 * @param getting 基点をたどっていくための{@link GettingEL}。
	 * @param setting 値を設定または取得するための{@link EL}。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public ArrayPathEL(final GettingEL getting, final EL setting){
		this(new GettingEL[]{getting}, setting);
		getting.getClass();
	}
	
	/**
	 * コンストラクタ。
	 * @param getting 基点をたどっていくための{@link GettingEL}の一覧。本クラスはこの引数の状態を変化させない。
	 * @param setting 値を設定または取得するための{@link EL}。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public ArrayPathEL(final Collection<? extends GettingEL> getting, final EL setting){
		this(getting.toArray(new GettingEL[0]), setting);
	}
	
	/**
	 * コンストラクタ。
	 * @param setting 値を設定または取得するための{@link EL}の一覧。一覧の末尾の{@link EL}は値を設定または取得するために使用する。本クラスはこの引数の状態を変化させない。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public ArrayPathEL(final EL... setting){
		this(Arrays.asList(setting).subList(0, setting.length-1), setting[setting.length-1]);
	}
	
	/**
	 * コンストラクタ。
	 * @param setting 値を設定または取得するための{@link EL}の一覧。一覧の末尾の{@link EL}は値を設定または取得するために使用する。本クラスはこの引数の状態を変化させない。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public ArrayPathEL(final Collection<? extends EL> setting){
		this(setting.toArray(new EL[0]));
	}
	
	private Object getLocal(final Object root){
		Object r = root;
		for(int i=0; i<getting.length; i++){
			r = getting[i].getValue(r);
		}
		return r;
	}

	@Override public void setValue(final Object root, final Object value)
			throws ELTargetRuntimeException {
		setting.setValue(getLocal(root), value);
	}

	@Override public Object getValue(final Object root, final Class toType) throws ELTargetRuntimeException {
		return setting.getValue(getLocal(root), toType);
	}

	public GettingEL[] getGetting() {
		return (GettingEL[])getting.clone();
	}

	public EL getSetting() {
		return setting;
	}
}
