/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.text;

import java.text.FieldPosition;
import java.text.Format;
import java.text.ParsePosition;

import woolpack.utils.Switchable;

/**
 * 有限個の文字列を文字列にフォーマットする変換器。
 * {@link #format(Object, StringBuffer, FieldPosition)}・{@link #parseObject(String, ParsePosition)}
 * とも同一の変換を行うため、可逆ではない。
 * このクラスは入力チェックで使用することを想定していない。
 * このクラスは対応する Javascript クラスを持たない。
 * @author nakamura
 *
 */
public class SwitchableFormat extends Format {
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	
	private final Switchable<String,String> switchable;
	
	/**
	 * コピーコンストラクタ。
	 * {@link #clone()}から呼び出される(called)。
	 * @param format コピー元。
	 */
	protected SwitchableFormat(final SwitchableFormat format){
		this.switchable = format.switchable;
	}
	
	/**
	 * コンストラクタ。
	 * @param switchable 変換元と変換先の{@link Switchable}。
	 * @throws NullPointerException parseMap が null の場合。
	 */
	public SwitchableFormat(final Switchable<String,String> switchable){
		switchable.getClass();
		this.switchable = switchable;
	}
	
	@Override
	public StringBuffer format(final Object obj, final StringBuffer toAppendTo,
			final FieldPosition pos) {
		final int start = toAppendTo.length();
		String o = switchable.get((String)obj);
		toAppendTo.append(o);
		pos.setBeginIndex(start);
		pos.setEndIndex(toAppendTo.length());
		return toAppendTo;
	}

	@Override
	public Object parseObject(final String source, final ParsePosition pos) {
		for(final String key:switchable.keys()){
			if(source.startsWith(key, pos.getIndex())){
				pos.setIndex(pos.getIndex()+key.length());
				return switchable.get(key);
			}
		}
		String o = switchable.getDefault();
		pos.setIndex(source.length());
		return o;
	}

	@Override public Object clone(){
		return new SwitchableFormat(this);
	}

	public Switchable<String, String> getSwitchable() {
		return switchable;
	}
}
