/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.text;

import java.text.DateFormat;
import java.text.FieldPosition;
import java.text.ParsePosition;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;

/**
 * {@link DateFormat}の一覧を順にパースし、最初に成功した場合の値を返す{@link DateFormat}。
 * 適用しているパターン：Proxy。
 * @author nakamura
 *
 */
public class TrysDateFormat extends DateFormat {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	
	private final Iterable<DateFormat> iterable;
	
	/**
	 * コピーコンストラクタ。
	 * {@link #clone()}から呼び出される(called)。
	 * @param format コピー元。
	 */
	protected TrysDateFormat(final TrysDateFormat format, final boolean dummy){
		iterable = format.iterable;
	}
	
	/**
	 * コンストラクタ。
	 * @param iterable {@link DateFormat}の一覧。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public TrysDateFormat(final Iterable<DateFormat> iterable){
		iterable.getClass();
		this.iterable = iterable;
	}

	/**
	 * コンストラクタ。
	 * @param array {@link DateFormat}の一覧。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public TrysDateFormat(final DateFormat... array){
		this(Arrays.asList(array));
	}

	@Override
	public StringBuffer format(final Date date, final StringBuffer toAppendTo,
			final FieldPosition fieldPosition) {
		return copy(iterable.iterator().next()).format(date, toAppendTo, fieldPosition);
	}

	@Override
	public Date parse(final String source, final ParsePosition pos) {
		int errorIndex = pos.getErrorIndex();
		for(final DateFormat f:iterable){
			final ParsePosition pp = new ParsePosition(0);
			pp.setIndex(pos.getIndex());
			pp.setErrorIndex(pos.getErrorIndex());
			final Date d = copy(f).parse(source, pp);
			if(d != null){
				pos.setIndex(pp.getIndex());
				return d;
			}
			errorIndex = pp.getErrorIndex();
		}
		pos.setErrorIndex(errorIndex);
		return null;
	}

	@Override public Object clone(){
		return new TrysDateFormat(this, false);
	}
	
	// 定義しないとDateFormatでNullPointerExceptionが発生する
	@Override public int hashCode(){
		// TODO 本来はiterable の中身でハッシュコードを計算する必要がある
		return iterable.hashCode();
	}
	
	private static DateFormat copy(final DateFormat f){
		return ((DateFormat)f.clone());
	}

	public List<DateFormat> getIterable() {
		final List<DateFormat> list = new ArrayList<DateFormat>();
		for(final DateFormat f:iterable){
			list.add(copy(f));
		}
		return Collections.unmodifiableList(list);
	}	
}
