/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils;

import java.util.Map;

/**
 * メッセージの埋め込みを解決するリゾルバを表す。
 * @author nakamura
 *
 */
public interface EmbeddingResolveable {

	/**
	 * 埋め込みを解決する。
	 * このメソッドは{@link Map}の状態を変化させる実装を前提にしているため、ひとつの{@link Map}に対して一度だけ実行すべき。
	 * @param map キーとメッセージが埋め込まれた値の{@link Map}。本クラスはこの引数の状態を変化させる。
	 */
	public void resolve(final Map<String,Object> map);
	
	/**
	 * valueの埋め込みを解決する。
	 * 引数の{@link Map}の埋め込みが既に解決済みの場合は複数のスレッドで同時に使用可能だが、
	 * 解決していない場合は{@link Map}に対して更新時異状が発生する可能性がある。
	 * @param map キーとメッセージが埋め込まれた値の{@link Map}。本クラスはこの引数の状態を変化させる可能性がある。
	 * @param value 解決対象のメッセージの値。
	 * @return 解決されたメッセージの値。
	 */
	public String resolve(final Map<String,Object> map, final String value);
}
