/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.Arrays;
import java.util.HashMap;

import junit.framework.TestCase;
import woolpack.LoadAllTests;
import woolpack.el.PathEL;
import woolpack.test.DomExpressionRunnable;
import woolpack.test.RunnableDomExpression;
import woolpack.test.RunnableGate;
import woolpack.test.TestUtils;
import woolpack.test.RunnableGate.Context;
import woolpack.utils.SwitchBuilder;
import woolpack.utils.UtilsConstants;
import woolpack.utils.concurrent.TrySemaphoreFactory;

public class DoAcquireSessionTest extends TestCase {

	public void testConstructor(){
		try{
			new DoAcquireSession(null, new TrySemaphoreFactory(1, true), DomConstants.NULL, DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new DoAcquireSession("", new TrySemaphoreFactory(1, true), DomConstants.NULL, DomConstants.NULL);
			fail();
		}catch(final StringIndexOutOfBoundsException expected){
		}
		try{
			new DoAcquireSession("key0", new TrySemaphoreFactory(1, true), null, DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new DoAcquireSession("key0", new TrySemaphoreFactory(1, true), DomConstants.NULL, null);
			fail();
		}catch(final NullPointerException expected){
		}
	}
	
	public void testNormal() throws InterruptedException{
		final DomExpression thrower = new Branch<Object>(
				new EvalEL(null, new PathEL("context.request.empty")), 
				new SwitchBuilder<Object,DomExpression>().get());
		
		final RunnableGate gate = new RunnableGate(LoadAllTests.THREAD_POOL);
		final DomExpression expression = new DoAcquireSession(new TrySemaphoreFactory(1, true),
				new Serial(new RunnableDomExpression(gate.getPause("true_start", "true_end")), thrower), 
				new RunnableDomExpression(gate.getGate("false")));

		// セマフォを取得するのに成功する。
		final DomContext c0 = new DomContext();
		c0.setSession(UtilsConstants.concurrentMap(new HashMap<String,Object>(), new Object()));
		c0.setRequest(new HashMap<String,Object>());
		final Context t0 = gate.execute(new DomExpressionRunnable(expression, c0));
		assertTrue(t0.waitFor("true_start"));
		assertFalse(t0.waitFor("true_end"));

		{
			// セマフォを取得するのに失敗する。処理をブロックしない。
			final Context t = gate.execute(new DomExpressionRunnable(expression, c0));
			assertTrue(TestUtils.equals(Arrays.asList("false"), t.getEndList()));
		}

		{
			// 別セッションでセマフォを取得するのに成功する。
			final Context t;
			{
				final DomContext c = new DomContext();
				c.setSession(UtilsConstants.concurrentMap(new HashMap<String,Object>(), new Object()));
				c.setRequest(new HashMap<String,Object>());
				t = gate.execute(new DomExpressionRunnable(expression, c));
			}
			assertTrue(t.waitFor("true_start"));
			assertFalse(t.waitFor("true_end"));
			t.resume();
			assertTrue(TestUtils.equals(Arrays.asList("true_start", "true_end"), t.getEndList()));
		}

		// 取得したセマフォを開放する。
		t0.resume();
		assertTrue(TestUtils.equals(Arrays.asList("true_start", "true_end"), t0.getEndList()));

		{
			// セマフォを再取得するのに成功する。
			final Context t = gate.execute(new DomExpressionRunnable(expression, c0));
			assertTrue(t.waitFor("true_start"));
			assertFalse(t.waitFor("true_end"));
			t.resume();
			assertTrue(TestUtils.equals(Arrays.asList("true_start", "true_end"), t.getEndList()));
		}
		{
			// 委譲先で異常が発生した場合。
			final Context t;
			{
				final DomContext c = new DomContext();
				c.setSession(UtilsConstants.concurrentMap(new HashMap<String,Object>(), new Object()));
				c.setRequest(null);
				t = gate.execute(new DomExpressionRunnable(expression, c));
			}
			assertTrue(t.waitFor("true_start"));
			assertFalse(t.waitFor("true_end"));
			t.resume();
			assertTrue(TestUtils.equals(Arrays.asList("true_start", "true_end"), t.getEndList()));
		}
	}
}
