/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.visitor.impl;

import java.text.SimpleDateFormat;
import java.util.LinkedHashMap;

import junit.framework.TestCase;
import woolpack.dom.DomContext;
import woolpack.test.TestUtils;
import woolpack.text.FixFormat;
import woolpack.text.LimitedValueFormat;
import woolpack.text.RegExpFormat;
import woolpack.text.SwitchableFormat;
import woolpack.text.ToBigDecimalFormat;
import woolpack.text.ToBigIntegerFormat;
import woolpack.text.ToByteFormat;
import woolpack.text.ToCharacterFormat;
import woolpack.text.ToDoubleFormat;
import woolpack.text.ToFloatFormat;
import woolpack.text.ToIntegerFormat;
import woolpack.text.ToLongFormat;
import woolpack.text.ToSQLDateFormat;
import woolpack.text.ToShortFormat;
import woolpack.text.ToStringFormat;
import woolpack.text.TrysDateFormat;
import woolpack.utils.MapBuilder;
import woolpack.utils.SwitchBuilder;
import woolpack.visitor.Visitor;

public class DocTextConstantsTest extends TestCase {

	private void scenario(final Object expression, final String expected){
		final DomContext context = new DomContext();
		TestUtils.getToNode("<HTML><BODY></BODY></HTML>").interpret(context);
		final Visitor visitor = new Visitor();
		visitor.setMap(DocValidatorConstants.MAP);
		visitor.setContext(context.getNode().getFirstChild().getFirstChild());
		visitor.visit(expression);
		assertTrue(TestUtils.equalsBody(context, expected));
	}
	
	public void testToCharacterFormat(){
		scenario(new ToCharacterFormat(), "文字に変換するフォーマット");
	}
	
	public void testToStringFormat(){
		scenario(new ToStringFormat(), "文字列に変換するフォーマット");
	}
	
	public void testFixFormat(){
		scenario(new FixFormat("message0"), "message0に変換するフォーマット");
	}
	
	public void testLimitedValueFormat(){
		scenario(new LimitedValueFormat(MapBuilder.get(new LinkedHashMap<String,String>())
				.put("f0", "p0")
				.put("f1", "p1")
				.get(), "defaultParse", "defaultFormat"),
				"次の定義で変換するフォーマット、フォーマットとパースの対応：" +
				"<dl><dt>f0</dt><dd>p0</dd><dt>f1</dt><dd>p1</dd></dl>、" +
				"デフォルトのフォーマット値：defaultFormat、デフォルトのパース値：defaultParse");
	}
	
	public void testRegExpFormat(){
		scenario(new RegExpFormat("me(ssag)e0", "$1"), 
				"正規表現\"me(ssag)e0\"にマッチした文字列を\"$1\"のパターンに変換するフォーマット");
	}
	
	public void testSwitchableFormat(){
		scenario(new SwitchableFormat(new SwitchBuilder<String,String>()
				.put("f0", "p0")
				.put("f1", "p1")
				.get()), 
				"次の対応で変換するフォーマット<dl><dt>f0</dt><dd>p0</dd><dt>f1</dt><dd>p1</dd></dl>");
	}
	
	public void testToBigDecimalFormat(){
		scenario(new ToBigDecimalFormat(), 
				"パターン\"#,##0.###\"の数値フォーマットによる変換結果を BigDecimal に変換するフォーマット");
	}
	
	public void testToBigIntegerFormat(){
		scenario(new ToBigIntegerFormat(), 
				"パターン\"#,##0.###\"の数値フォーマットによる変換結果を BigInteger に変換するフォーマット");
	}
	
	public void testToByteFormat(){
		scenario(new ToByteFormat(), 
				"パターン\"#,##0.###\"の数値フォーマットによる変換結果を Byte に変換するフォーマット");
	}
	
	public void testToDoubleFormat(){
		scenario(new ToDoubleFormat(), 
				"パターン\"#,##0.###\"の数値フォーマットによる変換結果を Double に変換するフォーマット");
	}
	
	public void testToFloatFormat(){
		scenario(new ToFloatFormat(), 
				"パターン\"#,##0.###\"の数値フォーマットによる変換結果を Float に変換するフォーマット");
	}
	
	public void testToIntegerFormat(){
		scenario(new ToIntegerFormat(), 
				"パターン\"#,##0.###\"の数値フォーマットによる変換結果を Integer に変換するフォーマット");
	}
	
	public void testToLongFormat(){
		scenario(new ToLongFormat(), 
				"パターン\"#,##0.###\"の数値フォーマットによる変換結果を Long に変換するフォーマット");
	}
	
	public void testToShortFormat(){
		scenario(new ToShortFormat(), 
				"パターン\"#,##0.###\"の数値フォーマットによる変換結果を Short に変換するフォーマット");
	}
	
	public void testToSQLDateFormat(){
		scenario(new ToSQLDateFormat(new SimpleDateFormat("yyyyMMdd")), 
				"パターン\"yyyyMMdd\"の日付フォーマットによる変換結果を SQL 用の日付型(java.sql.Date)に変換するフォーマット");
	}
	
	public void testTrysDateFormat(){
		scenario(new TrysDateFormat(new SimpleDateFormat("yyyyMMddHHmmss"), new SimpleDateFormat("yyyyMMdd")), 
				"次の一覧を順にパースし、最初に成功した変換値を適用するフォーマット, パターン\"yyyyMMddHHmmss\"の日付フォーマット, パターン\"yyyyMMdd\"の日付フォーマット");
	}
}
