/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.acquirable;

import java.util.concurrent.Semaphore;

/**
 * {@link #acquire()}で{@link Semaphore#acquire()}を実行し、
 * {@link #release()}で{@link Semaphore#release()}を実行する{@link Acquirable}です。
 * <br/>適用しているデザインパターン：Adapter, Template Method。
 * 
 * @author nakamura
 * 
 */
public class DoSemaphore implements Acquirable {
	private Semaphore semaphore;

	public DoSemaphore(final Semaphore semaphore) {
		this.semaphore = semaphore;
	}

	public boolean acquire() {
		try {
			semaphore.acquire();
		} catch (final InterruptedException e) {
			processInterruptedException(e);
		}
		return true;
	}

	public void release() {
		semaphore.release();
	}

	/**
	 * {@link #acquire()}で{@link InterruptedException}
	 * が発生した場合に{@link #acquire()}から呼び出されます(called)。
	 * 必要に応じてオーバライドすることができます。
	 * デフォルトは現在のスレッドに対して{@link Thread#interrupt()}を実行した後、
	 * 原因が引数である{@link IllegalStateException}を throw します。
	 * 
	 * @param e 原因。
	 * @throws IllegalStateException ({@link InterruptedException})デフォルト実装では常に throw します。
	 */
	public void processInterruptedException(final InterruptedException e) {
		Thread.currentThread().interrupt();
		throw new IllegalStateException(e);
	}

	public Semaphore getSemaphore() {
		return semaphore;
	}
	public void setSemaphore(final Semaphore semaphore) {
		this.semaphore = semaphore;
	}
}
