/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.factory;

import java.util.concurrent.ConcurrentMap;

import woolpack.fn.Fn;

/**
 * 委譲先で生成したオブジェクトを{@link ConcurrentMap}にキャッシュして返す{@link Fn}です。
 * {@link ConcurrentMap#putIfAbsent(Object, Object)}を使用して返却値の一意性を保証しますが、
 * ひとつの返却値に対応するインスタンス生成の委譲が複数回発生する可能性があります。
 * @author nakamura
 *
 * @param <C>
 * @param <K>
 */
public class ConcurrentMapCache<C, K> implements Fn<C, Object> {
	private Fn<? super C, ? extends ConcurrentMap<? super K, Object>> mapGetter;
	private Fn<? super C, ? extends K> keyGetter;
	private Fn<? super C, ?> maker;
	
	public ConcurrentMapCache(
			final Fn<? super C, ? extends ConcurrentMap<? super K, Object>> mapGetter,
			final Fn<? super C, K> keyGetter,
			final Fn<? super C, ?> maker) {
		this.mapGetter = mapGetter;
		this.keyGetter = keyGetter;
		this.maker = maker;
	}

	public Object exec(final C c) {
		final ConcurrentMap<? super K, Object> map = mapGetter.exec(c);
		final K key = keyGetter.exec(c);
		while (true) {
			Object value = map.get(key);
			if (value == null) {
				value = maker.exec(c);
				map.putIfAbsent(key, value);
			} else {
				return value;
			}
		}
	}
	public Fn<? super C, ? extends K> getKeyGetter() {
		return keyGetter;
	}
	public void setKeyGetter(final Fn<? super C, ? extends K> keyGetter) {
		this.keyGetter = keyGetter;
	}
	public Fn<? super C, ?> getMaker() {
		return maker;
	}
	public void setMaker(final Fn<? super C, ?> maker) {
		this.maker = maker;
	}
	public Fn<? super C, ? extends ConcurrentMap<? super K, Object>> getMapGetter() {
		return mapGetter;
	}
	public void setMapGetter(
			final Fn<? super C, ? extends ConcurrentMap<? super K, Object>> mapGetter) {
		this.mapGetter = mapGetter;
	}
}
