/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils;

import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;

class SynchronizedConcurrentMap<K, V> implements ConcurrentMap<K, V> {
	private final Map<K, V> map;
	private final Object lock;

	SynchronizedConcurrentMap(final Map<K, V> map, final Object lock) {
		this.map = map;
		this.lock = lock;
	}

	public V putIfAbsent(final K key, final V value) {
		synchronized (lock) {
			if (!map.containsKey(key)) {
				return map.put(key, value);
			} else {
				return map.get(key);
			}
		}
	}

	public boolean remove(final Object key, final Object value) {
		synchronized (lock) {
			if (map.containsKey(key) && map.get(key).equals(value)) {
				map.remove(key);
				return true;
			} else {
				return false;
			}
		}
	}

	public boolean replace(final K key, final V oldValue, final V newValue) {
		synchronized (lock) {
			if (map.containsKey(key) && map.get(key).equals(oldValue)) {
				map.put(key, newValue);
				return true;
			} else {
				return false;
			}
		}
	}

	public V replace(final K key, final V value) {
		synchronized (lock) {
			if (map.containsKey(key)) {
				return map.put(key, value);
			} else {
				return null;
			}
		}
	}

	public int size() {
		return map.size();
	}

	public boolean isEmpty() {
		return map.isEmpty();
	}

	public boolean containsKey(final Object key) {
		return map.containsKey(key);
	}

	public boolean containsValue(final Object value) {
		return map.containsValue(value);
	}

	public V get(final Object key) {
		return map.get(key);
	}

	public V put(final K key, final V value) {
		return map.put(key, value);
	}

	public V remove(final Object key) {
		return map.remove(key);
	}

	public void putAll(final Map<? extends K, ? extends V> t) {
		map.putAll(t);
	}

	public void clear() {
		map.clear();
	}

	public Set<K> keySet() {
		return map.keySet();
	}

	public Collection<V> values() {
		return map.values();
	}

	public Set<Entry<K, V>> entrySet() {
		return map.entrySet();
	}

	@Override
	public boolean equals(final Object o) {
		return map.equals(o);
	}

	@Override
	public int hashCode() {
		return map.hashCode();
	}

	@Override
	public String toString() {
		return map.toString();
	}
}
