/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.visitor;



import woolpack.fn.Fn;

/**
 * ユーティリティです。
 * @author nakamura
 *
 */
public final class VisitorUtils {

	public static final Fn<Visitor<?>, Void> ECHO_ACCEPTOR = new EchoAcceptor();
	public static final Fn<Visitor<?>, Void> MAP_ACCEPTOR = new MapAcceptor();
	public static final Fn<Visitor<?>, Void> ITERABLE_ACCEPTOR = new IterableAcceptor();
	public static final Fn<Visitor<?>, Void> ARRAY_ACCEPTOR = new ArrayAcceptor();
	public static final Fn<Visitor<?>, Void> BEAN_ACCEPTOR = new BeanAcceptor();
	public static final Fn<Visitor<?>, Void> COMPLEX_ACCEPTOR = new ComplexAcceptor();
	
	private VisitorUtils() {
	}
	
	/**
	 * {@link Visitor#setSubContext(Object)}を一時的に設定して委譲し、
	 * 委譲先から復帰したときに呼び出し時の状態に初期化します。
	 * @param <V>
	 * @param c 一時的に置き換えるコンテキスト。
	 * @param visitor ビジター。
	 * @param acceptor 委譲先。
	 */
	public static <V> void localContext(
			final V c,
			final Visitor<V> visitor,
			final Fn<? super Visitor<V>, Void> acceptor) {
		final V baseContext = visitor.getSubContext();
		visitor.setSubContext(c);
		try {
			acceptor.exec(visitor);
		} finally {
			visitor.setSubContext(baseContext);
		}
	}
	
	/**
	 * {@link Visitor#setElement(Object)}を一時的に設定して委譲し、
	 * 委譲先から復帰したときに呼び出し時の状態に初期化します。
	 * @param <V>
	 * @param element 一時的に置き換えるエレメント。
	 * @param visitor ビジター。
	 * @param acceptor 委譲先。
	 */
	public static <V> void localElement(
			final Object element,
			final Visitor<V> visitor,
			final Fn<? super Visitor<V>, Void> acceptor) {
		final Object baseElement = visitor.getElement();
		visitor.setElement(element);
		try {
			acceptor.exec(visitor);
		} finally {
			visitor.setElement(baseElement);
		}
	}
}
