/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.typeconvert;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.DecimalFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;

import junit.framework.TestCase;
import woolpack.fn.Fn;

public class TypeConvertUtilsTest extends TestCase {
	private final Fn<ConvertContext, Void> c = TypeConvertUtils.getSimpleConverter(
			new DecimalFormat(), new SimpleDateFormat("yyyyMMdd"));
	
	private void scenario(final Object expected, final Object before, final Class toType, final String name) {
		final ConvertContext context = new ConvertContext();
		context.setToType(toType);
		context.setPropertyName(name);
		context.setValue(before);
		c.exec(context);
		assertEquals(expected, context.getValue());
	}

	public void testNormal() throws ParseException {

		scenario("true", Boolean.TRUE, String.class, null);
		scenario("20060708", new SimpleDateFormat("yyyyMMdd").parse("20060708"), String.class, null);
		scenario("1", (byte) 1, String.class, null);
		scenario("1,000", (short) 1000, String.class, null);
		scenario("1,000", (int) 1000, String.class, null);
		scenario("1,000", (long) 1000, String.class, null);
		scenario("1,000", (float) 1000, String.class, null);
		scenario("1,000", (double) 1000, String.class, null);
		scenario("1,000", BigInteger.valueOf(1000), String.class, null);
		scenario("1,000", new BigDecimal(1000), String.class, null);
		scenario("c", 'c', String.class, null);

		scenario(Boolean.TRUE, "true", Boolean.class, null);
		scenario(Boolean.FALSE, "false", Boolean.class, null);
		scenario(Boolean.FALSE, Boolean.FALSE, Boolean.class, null);
		scenario(Character.valueOf('a'), "abc", Character.class, null);
		scenario(new java.sql.Date(new SimpleDateFormat("yyyyMMdd").parse("20060708").getTime()), "20060708", java.sql.Date.class, null);
		scenario(new SimpleDateFormat("yyyyMMdd").parse("20060708"), "20060708", java.util.Date.class, null);

		scenario(Byte.valueOf((byte) 1), "1", Byte.class, null);
		scenario(Short.valueOf((short) 1), "1", Short.class, null);
		scenario(Integer.valueOf(1), "1", Integer.class, null);
		scenario(Long.valueOf(1), "1", Long.class, null);
		scenario(Float.valueOf(1), "1", Float.class, null);
		scenario(Double.valueOf(1), "1", Double.class, null);
		scenario(BigInteger.valueOf(1), "1", BigInteger.class, null);
		scenario(new BigDecimal(1), "1", BigDecimal.class, null);

		scenario(Byte.valueOf((byte) 1), Integer.valueOf(1), Byte.class, null);
		scenario(Short.valueOf((short) 1), Integer.valueOf(1), Short.class, null);
		scenario(Integer.valueOf(1), Long.valueOf(1), Integer.class, null);
		scenario(Long.valueOf(1), Integer.valueOf(1), Long.class, null);
		scenario(Float.valueOf(1), Integer.valueOf(1), Float.class, null);
		scenario(Double.valueOf(1), Integer.valueOf(1), Double.class, null);
		scenario(BigInteger.valueOf(1), Integer.valueOf(1), BigInteger.class, null);
		scenario(new BigDecimal(1), Integer.valueOf(1), BigDecimal.class, null);

		{
			final Object o = new Object();
			scenario(o, o, Object.class, null);
		}

		try {
			scenario(null, "abs", Integer.class, null);
			fail();
		} catch (final IllegalArgumentException e) {
			assertTrue(e.getCause() instanceof ParseException);
		}

		try {
			scenario(null, "", Character.class, null);
			fail();
		} catch (final IllegalArgumentException e) {
		}
	}
	
	public void testGetPropertyName() {
		final ConvertContext context = new ConvertContext();
		context.setPropertyName("hoge");
		assertEquals("hoge", TypeConvertUtils.GET_PROPERTY_NAME.exec(context));
	}
}
