class Book

 private
	def initialize(vpath)
		@vpath = vpath
	end

	def _pagelist
		return []
	end

 public
	attr_reader :vpath
#	def type
#		return :nil
#	end

	def self.factory(vpath)
		case type(vpath)
		 when :images
			return BookOfFiles.new(vpath)
		 when :pdf
			return BookOfPdf.new(vpath)
		 when :zip
			return BookOfZip.new(vpath)
		 else
			return Book.new(vpath)
		end
	end

	# 本の種類を返す ※本の実在確認はしない
	def self.type(vpath)
		b = vpath.basename
		if b == "*img"
			return :images
		elsif extname_match?(YOMEYA_IMAGE_EXT, b)
			return :image
		elsif extname_match?(YOMEYA_PDF_EXT, b)
			return :pdf
		elsif extname_match?(YOMEYA_ZIP_EXT, b)
			return :zip
		elsif extname_match?(YOMEYA_TXT_EXT, b)
			return :text
		elsif extname_match?(YOMEYA_HTML_EXT, b)
			return :html
		elsif extname_match?(YOMEYA_AUDIO_EXT, b)
			return :audio
		elsif extname_match?(YOMEYA_VIDEO_EXT, b)
			return :video
		else
			return nil
		end
	end

#	def bookinfo
#		type = class.type(@rpath)
#		pl = pagelist
#		pages = pl.count
#		if pages > 0
#
#
#		else
#			location = ""
#			page = 0
#			read_start = nil
#			read_end = nil
#		end
#
#		return {
#			type: type,
#			pages: pages,
#			location: location,
#			page: page,
#			read_start: read_start,
#			read_end: read_end,
#		}
#	end

	def pagelist
		Cache.transaction do
			return Cache.fetch_pagelist(@vpath) do
				_pagelist.sort_by! { |v| key4sort(v) }
			end
		end
	end

	# pagelist中の何枚目の画像かを得る
	# 一致するものがなかった場合、最も近いものを得る
	def index(location)
		pl = pagelist
		return nil  if pl.empty?

		r = pl.index(location)
		return r  if r

		location_key = key4sort(location)
		pl.each.with_index do |loc, i|
			if (key4sort(loc) <=> location_key) != -1
				return i
			end
		end

		return pl.size - 1
	end

	def extract(location);					nil; end
	def extract_thumbnail(w, h, l=nil);		nil; end
	def writable?;							false; end
	def delete(location);					false; end
	def update(location, data);				false; end
	def rotate(location, degree);			false; end
end

############################################################

class BookOfFiles < Book

 private
	def initialize(vpath)
		super
		@rpath = File.dirname(vpath.r)	# *img を除く
	end

	def _pagelist
		r = []
		Dir.foreach(@rpath) do |entry|
			if entry.start_with?(".")
				# NOP
			elsif extname_match?(YOMEYA_IMAGE_EXT, entry)
				path = File.join(@rpath, entry)
				if File.file?(path)
					r << entry
				end
			end
		end
		return r
	end

 public
#	def type
#		return :images
#	end

	def extract(location)
		r = File.binread(File.join(@rpath, location))
		return r  if r.present?

		return asset_image_data("icon_file_missing.png")
	end

	def extract_thumbnail(width, height, location=nil)
		if location.present?
			p = location
		else
			p = pagelist.first
		end
		return extract(p)  if p.present?

		return asset_image_data("icon_file_missing.png")
	end

	def delete(location)
		src_filename = File.join(@rpath, location)
		return false  if !File.file?(src_filename)

		n = 1
		while true
			dst_filename = src_filename + ".skip" + ((n == 1) ? "" : n.to_s)
			break  if !File.exist?(dst_filename)
			n += 1
		end

		FileUtils.mv(src_filename, dst_filename)
		return true
	end

	def update(location, data)
		filename = File.join(@rpath, location)
		org_filename = filename + ".org"

		if !File.exist?(org_filename)
			FileUtils.cp(filename, org_filename, preserve:true)
		end

		File.open(filename, "wb:BINARY") do |file|
			file.write(data)
		end
	end

	def rotate(location, degree)
		src_filename = File.join(@rpath, location)
		return false  if !File.file?(src_filename)
		org_filename = src_filename + ".org"

		temp = Tempfile.new(YOMEYA_NAME)
		temp.binmode

		cmd = [JPEGTRAN_CMD]
		cmd += ["-perfect", "-optimize", "-progressive"]
		cmd += ["-rotate", degree.to_s]
		cmd += ["-outfile", temp.path]
		cmd += [src_filename]
		r = system(*cmd)
		if !r
			cmd = [CONVERT_CMD]
			cmd += ["-define", "jpeg:preserve-settings"]
			cmd += ["-interlace", "line"]
			cmd += ["-rotate", degree.to_s]
			cmd += [src_filename]
			cmd += [temp.path]
			r = system(*cmd)
		end
		if r
			if !File.exist?(org_filename)
				FileUtils.cp(src_filename, org_filename, preserve:true)
			end
			FileUtils.mv(temp.path, src_filename)
		end
		temp.close!
		return r
	end

end

############################################################

class BookOfPdf < Book

 private
	def _pagelist
		n = PDF::Reader.new(@vpath.r).page_count
		return (1 .. n).map { |n| "#{n}p" }
	end

 public
#	def type
#		return :pdf
#	end

	def extract(location)
		filename = "#{@vpath.r}[#{location.to_i-1}]"
		filename.encode!(Encoding.find("locale"))
		imagelist = Magick::Image.read(filename) do
			self.density = "288"
		end
		image = imagelist[0]

		# 背景を白に
		bg = Magick::Image.new(image.columns, image.rows) do
			self.background_color = "white"
		end
		image.composite!(bg, 0, 0, Magick::DstOverCompositeOp)

		r = image.to_blob do
			self.format = "PNG"
		end
		return r
	end

	def extract_thumbnail(width, height, location=nil)
		if location.present?
			p = location.to_i - 1
		else
			p = 0
		end

		filename = "#{@vpath.r}[#{p}]"
		filename.encode!(Encoding.find("locale"))
		imagelist = Magick::Image.read(filename) do
			self.density = "36"
		end
		image = imagelist[0]

		# 背景を白に
		bg = Magick::Image.new(image.columns, image.rows) do
			self.background_color = "white"
		end
		image.composite!(bg, 0, 0, Magick::DstOverCompositeOp)

		r = image.to_blob do
			self.format = "PNG"
		end
		return r  if r.present?

		return asset_image_data("icon_file_pdf.png")
	end
end

############################################################

class BookOfZip < Book

 private
	FN_ENC = Encoding::WINDOWS_31J

	def zip_encode(s)
		if !s.valid_encoding?
			tmp = s.dup
			tmp.force_encoding(Encoding::ASCII_8BIT)
			tmp.gsub!("/", "\\")			# なみだぐましい
			tmp.force_encoding(FN_ENC)
			if tmp.valid_encoding?
				tmp.gsub!("\\", "/")			# なみだぐましい
				s = tmp
			end
		end
		return s.encode
	end

	def _pagelist
		r = []
		Zip::File.open(@vpath.r) do |zipfile|
			zipfile.each do |entry|
				if entry.ftype == :file
					entry_name = zip_encode(entry.name)
					basename = File.basename(entry_name)
					if basename.start_with?(".")
						# NOP
					elsif extname_match?(YOMEYA_IMAGE_EXT, basename)
						r << entry_name
					end
				end
			end
		end

		return r
	end

 public
#	def type
#		return :zip
#	end

	def extract(location)
		r = nil

		Zip::File.open(@vpath.r) do |zipfile|
			zipfile.each do |entry|
				if entry.ftype == :file
					entry_name = zip_encode(entry.name)
					if entry_name == location
						r = zipfile.read(entry)
						break
					end
				end
			end
		end

		return r
	end

	def extract_thumbnail(width, height, location=nil)
		if location.present?
			p = location
		else
			p = pagelist.first
		end
		return extract(p)  if p.present?

		return asset_image_data("icon_file_zip.png")
	end
end
